// $Id: error_estimator.h 3428 2009-07-17 01:51:26Z roystgnr $

// The libMesh Finite Element Library.
// Copyright (C) 2002-2008 Benjamin S. Kirk, John W. Peterson, Roy H. Stogner
  
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
  
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
  
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA



#ifndef __error_estimator_h__
#define __error_estimator_h__

// C++ includes
#include <map>
#include <string>
#include <vector>

// Local Includes
#include "libmesh_common.h"
#include "system_norm.h"

// Forward Declarations
class ErrorVector;
class EquationSystems;
class System;
template <typename T> class NumericVector;

/**
 * This class holds functions that will estimate the error
 * in a finite element solution on a given mesh.  These error
 * estimates can be useful in their own right, or may be used
 * to guide adaptive mesh refinement.  Note that in general
 * the computed errors are stored as floats rather than doubles
 * since the required precision is low.
 *
 * @author Benjamin S. Kirk, 2003.
 */
class ErrorEstimator
{
public:

  /**
   * Constructor.  Empty.  Derived classes should reset error_norm as
   * appropriate.
   */
  ErrorEstimator() : error_norm() {}
  
  /**
   * Destructor.  
   */
  virtual ~ErrorEstimator() {}


  /**
   * This pure virtual function must be redefined
   * in derived classes to compute the error for each
   * active element and place it in the "error_per_cell" vector.
   *
   * If solution_vector is not NULL, the estimator will
   * (if able) attempt to estimate an error in that field
   * instead of in system.solution.
   *
   * If estimate_parent_error is not false, the estimator will (if
   * able) attempt to give a consistent estimate of errors in parent
   * elements that would be generated by coarsening.
   */
  virtual void estimate_error (const System& system,
			       ErrorVector& error_per_cell,
			       const NumericVector<Number>* solution_vector = NULL,
			       bool estimate_parent_error = false) = 0;

  /**
   * This virtual function can be redefined
   * in derived classes, but by default computes the sum of
   * the error_per_cell for each system in the equation_systems.
   *
   * Currently this function ignores the error_norm member variable,
   * and uses the function argument error_norms instead.
   *
   * This function is named estimate_errors instead of estimate_error
   * because otherwise C++ can get confused.
   */
  virtual void estimate_errors (const EquationSystems& equation_systems,
			        ErrorVector& error_per_cell,
			        const std::map<const System*, SystemNorm>& error_norms,
			        const std::map<const System*, const NumericVector<Number>* >* solution_vectors = NULL,
			        bool estimate_parent_error = false);

  /**
   * When calculating many error vectors at once, we need a data structure to
   * hold them all
   */
  typedef std::map<std::pair<const System*, unsigned int>, ErrorVector*> ErrorMap;

  /**
   * This virtual function can be redefined
   * in derived classes, but by default it calls estimate_error
   * repeatedly to calculate the requested error vectors.
   *
   * Currently this function ignores the error_norm.weight() values
   * because it calculates each variable's error individually, unscaled.
   * 
   * The user selects which errors get computed by filling a map with error
   * vectors: If errors_per_cell[&system][v] exists, it will be filled with the
   * error values in variable \p v of \p system
   */
  virtual void estimate_errors (const EquationSystems& equation_systems,
			        ErrorMap& errors_per_cell,
			        const std::map<const System*, const NumericVector<Number>* >* solution_vectors = NULL,
			        bool estimate_parent_error = false);

  /**
   * When estimating the error in a single system, the \p error_norm
   * is used to control the scaling and norm choice for each variable.
   * Not all estimators will support all norm choices.  The default
   * scaling is for all variables to be weighted equally.  The default
   * norm choice depends on the error estimator.
   *
   * Part of this functionality was supported via component_scale and
   * sobolev_order in older libMesh versions, and a small part was
   * supported via component_mask in even older versions.  Hopefully
   * the encapsulation here will allow us to avoid changing this API
   * again.
   */
  SystemNorm error_norm;

protected:

  /**
   * This method takes the local error contributions in
   * \p error_per_cell from each processor and combines
   * them to get the global error vector.
   */
  void reduce_error (std::vector<float>& error_per_cell) const;
};


#endif

