#include "../config.h"
#include <sys/types.h>
#include "byteorder.h"

/*
 * We want libnjb to be "endianness agnostic" i.e. the byte-ordering
 * of the libnjb host platform shall not affect its functionality.
 * These routines are written using shifting and byte operations that
 * will produce the same result regardless of whether the host
 * platform is little-endian, big-endian or even mixed-endian.
 *
 * NJB1 and the "NJB3 family" (NJB2, NJB3, NJB Zen, NJB Zen 2.0) have
 * different byte ordering. NJB1 is essentially big-endian, and the NJB3
 * family little-endian. The terminology could be confusing, so we refer
 * to the different endiannesses as "njb1-endian" and "njb3-endian".
 */

/*
 * This function will take 8 bytes from the njb1-endian byte array 
 * pointed to by *dp and transform it to a u_int64_t on the host
 * platform.
 */
u_int64_t njb1_bytes_to_64bit(unsigned char *dp)
{
  u_int64_t ret;

  ret = ((u_int64_t) dp[3] << 56);
  ret = ret | ((u_int64_t) dp[2] << 48);
  ret = ret | ((u_int64_t) dp[1] << 40);
  ret = ret | ((u_int64_t) dp[0] << 32);
  ret = ret | ((u_int64_t) dp[7] << 24);
  ret = ret | ((u_int64_t) dp[6] << 16);
  ret = ret | ((u_int64_t) dp[5] << 8);
  ret = ret | (u_int64_t) dp[4];
  return ret;
}

/*
 * This function will write the u_int64_t of the host platform, "val"
 * as njb1-endian bytes beginning at the first byte in the byte 
 * array pointed to by *dp.
 */
void from_64bit_to_njb1_bytes(u_int64_t val, unsigned char *dp)
{
  dp[0] = (val >> 32) & 255;
  dp[1] = (val >> 40) & 255;
  dp[2] = (val >> 48) & 255;
  dp[3] = (val >> 56) & 255;
  dp[4] = val & 255;
  dp[5] = (val >> 8) & 255;
  dp[6] = (val >> 16) & 255;
  dp[7] = (val >> 24) & 255;
}

/*
 * This function will take 4 bytes from the njb1-endian byte array 
 * pointed to by *dp and transform it to a u_int32_t on the host
 * platform.
 */
u_int32_t njb1_bytes_to_32bit(unsigned char *dp)
{
  u_int32_t ret;

  ret = ((u_int32_t) dp[3] << 24);
  ret = ret | ((u_int32_t) dp[2] << 16);
  ret = ret | ((u_int32_t) dp[1] << 8);
  ret = ret | (u_int32_t) dp[0];  
  return ret;
}

u_int32_t njb3_bytes_to_32bit(unsigned char *dp)
{
  u_int32_t ret;

  ret = ((u_int32_t) dp[0] << 24);
  ret = ret | ((u_int32_t) dp[1] << 16);
  ret = ret | ((u_int32_t) dp[2] << 8);
  ret = ret | (u_int32_t) dp[3];  
  return ret;
}


/*
 * This function will write the u_int32_t of the host platform, "val"
 * as 4 njb1-endian bytes beginning at the first byte in the byte 
 * array pointed to by *dp.
 */
void from_32bit_to_njb1_bytes(u_int32_t val, unsigned char *dp)
{
  dp[0] = val & 255;
  dp[1] = (val >> 8) & 255;
  dp[2] = (val >> 16) & 255;
  dp[3] = (val >> 24) & 255;
}

void from_32bit_to_njb3_bytes(u_int32_t val, unsigned char *dp)
{
  dp[0] = (val >> 24) & 255;
  dp[1] = (val >> 16) & 255;
  dp[2] = (val >> 8) & 255;
  dp[3] = val & 255;
}

/*
 * This function will take 2 bytes from the njb1-endian byte array 
 * pointed to by *dp and transform it to a u_int16_t on the host
 * platform.
 */
u_int16_t njb1_bytes_to_16bit(unsigned char *dp)
{
  u_int16_t ret;

  ret = ((u_int16_t) dp[1] << 8);
  ret = ret | (u_int16_t) dp[0];
  return ret;
}

u_int16_t njb3_bytes_to_16bit(unsigned char *dp)
{
  u_int16_t ret;

  ret = ((u_int16_t) dp[0] << 8);
  ret = ret | (u_int16_t) dp[1];
  return ret;
}


/*
 * This function will write the u_int16_t of the host platform, "val"
 * as 2 njb1-endian bytes beginning at the first byte in the byte 
 * array pointed to by *dp.
 */
void from_16bit_to_njb1_bytes(u_int16_t val, unsigned char *dp)
{
  dp[0] = val & 255;
  dp[1] = (val >> 8) & 255;
}

void from_16bit_to_njb3_bytes(u_int16_t val, unsigned char *dp)
{
  dp[0] = (val >> 8) & 255;
  dp[1] = val & 255;
}

/*
 * These simply extract the most/least significant 16 bit parts of
 * a 32bit word. Not endianess-related at all actually.
 */
u_int16_t get_msw (u_int32_t word)
{
  return (u_int16_t) (word >> 16) & 0xffff;
}

u_int16_t get_lsw (u_int32_t word)
{
  return (u_int16_t) word & 0xffff;
}

u_int64_t make64 (u_int32_t msdw, u_int32_t lsdw)
{
	u_int64_t val;

	val= msdw * 0x100000000ULL + lsdw;
	return val;
}

void split64 (u_int64_t num, u_int32_t *msdw, u_int32_t *lsdw)
{
	u_int64_t val= num/0x100000000ULL;

	*msdw= (u_int32_t) val;
	*lsdw= (u_int32_t) ( num - val*0x100000000ULL );
}
