#ifndef __NJB__PROTO3__H
#define __NJB__PROTO3__H

/* Buffer for short reads */
#define NJB3_SHORTREAD_BUFSIZE 1024

/* Transfer block size (adding and fetching tracks, files */
#define NJB3_CHUNK_SIZE 0x100000
#define NJB3_GET_FILE_BLOCK_SIZE    0x2000
#define NJB3_SEND_FILE_BLOCK_SIZE   0x2000

/* These are the metadata frame IDs used by NJB3 */
#define NJB3_CODECS_FRAME_ID    0x0001 /* List of supported audio file types/codecs */
#define NJB3_DISKUTIL_FRAME_ID  0x0002 /* 14 bytes disk utilization information */
#define NJB3_PRODID_FRAME_ID    0x0003 /* String - product ID */
#define NJB3_LOCKED_FRAME_ID    0x0006 /* 16 bit word */
#define NJB3_FNAME_FRAME_ID     0x0007 /* String: Original filename on host */
#define NJB3_KEY_FRAME_ID       0x000a /* 4 bytes string "AR00", "PL00", "SG00", "LG00" known */
#define NJB3_CODEC_FRAME_ID     0x000b /* 16 bit word */
#define NJB3_POSTID_FRAME_ID    0x000c /* 16 bit word TrackID on tracks, PlaylistID on playlists */
#define NJB3_DIR_FRAME_ID       0x000d /* String: Original directory on host */
#define NJB3_FILESIZE_FRAME_ID  0x000e /* 32 bit word */
#define NJB3_FILECOUNT_FRAME_ID 0x0013 /* 6 bytes: at least top 2 bytes are file & directory count */
#define NJB3_VALUE_FRAME_ID     0x0014 /* 8 bytes, 2*32 bit words */
#define NJB3_JUKEBOXID_FRAME_ID 0x0015 /* 16 bytes - unique device ID */
#define NJB3_FILETIME_FRAME_ID  0x0016 /* 32 bit word timestamp (UNIX format) */
#define NJB3_UNKNOWN6_FRAME_ID  0x0017 /* 32 bit word */
#define NJB3_UNKNOWN8_FRAME_ID  0x0018 /* 32 bit word set on files, often 0x2000 0x0000 - permissions? */
#define NJB3_ALBUM_FRAME_ID     0x0101 /* String */
#define NJB3_ARTIST_FRAME_ID    0x0102 /* String */
#define NJB3_GENRE_FRAME_ID     0x0103 /* String */
#define NJB3_TITLE_FRAME_ID     0x0104 /* String */
#define NJB3_LENGTH_FRAME_ID    0x0105 /* String */
#define NJB3_TRACKNO_FRAME_ID   0x0106 /* 16 bit word */
#define NJB3_YEAR_FRAME_ID      0x0107 /* 16 bit word */
#define NJB3_TRACKINFO_FRAME_ID 0x010b /* 16 bit word: Info about current track on the playlist */
#define NJB3_SEEKTRACK_FRAME_ID 0x010c /* 32 bit word: seek to position (in ms) in current track */
#define NJB3_EAX_TYPENAME       0x010e /* String with the name of an EAX type */
#define NJB3_PLNAME_FRAME_ID    0x010f /* String: playlist name */
#define NJB3_TIME_FRAME_ID      0x0110 /* 10 bytes */
#define NJB3_OWNER_FRAME_ID     0x0113 /* String: owner name */
#define NJB3_UNKNOWN3_FRAME_ID  0x0115 /* 32bit word - unknown use,
					 requested after reading tracks value on NJB3 0x0000 0x0003 */
#define NJB3_PLAYTRACK_FRAME_ID 0x0119 /* 32 bit word track ID to play */
#define NJB3_PLTRACKS_FRAME_ID  0x011c /* Array of 16bit words */
/* 
   0200 EAX read command?
   0201 EAX write command?
 */ 
#define NJB3_MINMAX_ID          0x0201 /* 2x16 bit values, max and min */
#define NJB3_EAX_ACTIVE_ID      0x0202 /* 16 bit word - this EAX type is active/to 
					* be activated (0x0000 = off, 0x0001 = on) */
#define NJB3_VOLUME_FRAME_ID    0x0203 /* 16 bit word */
#define NJB3_ENV_FRAME_ID       0x0204 /* 16 bit word - environment setting */
#define NJB3_EQ_FRAME_ID        0x0205 /* 16 bit word - equalizer setting */
#define NJB3_SPAT_FRAME_ID      0x0206 /* 16 bit word - spatialization 2 = full */
#define NJB3_TSCALE_FRAME_ID    0x0207 /* 16 bit word - time scaling factor */
#define NJB3_SMARTVOL_FRAME_ID  0x0208 /* 16 bit word - smart volume setting */
#define NJB3_EAXID_FRAME_ID     0x020b /* 16 bit word with the numerical ID of a certain EAX type */
#define NJB3_EAX_INDEX_ID       0x020c /* 16 bit word - currently selected effect in a set of effects */
#define NJB3_KEYVALUE_FRAME_ID  0x1400 /* Array of value-key-pairs, requested in a 
					  subrequest parameter to this request */

/* File type IDs, used for eg create_file */
#define NJB3_FILETYPE_FILE     0x0000
#define NJB3_FILETYPE_TRACK    0x0002

/* Codec IDs */
#define NJB3_CODEC_MP3_ID_OLD        0x0000 /* Used on NJB3/Zen FW? */
#define NJB3_CODEC_WAV_ID            0x0001
#define NJB3_CODEC_MP3_ID            0x0002
#define NJB3_CODEC_WMA_ID            0x0003

/* Stop, pause and resume are very much alike. */
#define NJB3_START_PLAY     0x00
#define NJB3_STOP_PLAY      0x01
#define NJB3_PAUSE_PLAY     0x02
#define NJB3_RESUME_PLAY    0x03

/* Status codes */
#define NJB3_STATUS_OK                0x0000
#define NJB3_STATUS_TRANSFER_ERROR    0x0002 /* Error during read or write */
#define NJB3_STATUS_NOTEXIST          0x0005 /* Tried to access nonexistant track */
#define NJB3_STATUS_PROTECTED         0x000c /* Tried to access protected object */
#define NJB3_STATUS_EMPTY_CHUNK       0x000e /* Appear when requesting empty metadata lists
					      * or beyond the end of files. */

/*
 * Status codes that must exist, find by trial-and-error:
 * - postid invalid (does not exist)
 * - disk full
 * - file path / track name etc too long 
 * - playing (cannot transfer when playing)
 * - tried to skip to position outside file in playback mode
 * - too many files/tracks etc - limit reached.
 * - filename or track+artist+(all metadata) is the same as
 *   one already present on the player
 * - transfer in progress (a track/file is transferring in
 *   either direction
 */

#define njb3_start_play(njb) njb3_ctrl_playing(njb, NJB3_START_PLAY)
#define njb3_stop_play(njb) njb3_ctrl_playing(njb, NJB3_STOP_PLAY)
#define njb3_pause_play(njb) njb3_ctrl_playing(njb, NJB3_PAUSE_PLAY)
#define njb3_resume_play(njb) njb3_ctrl_playing(njb, NJB3_RESUME_PLAY)

typedef struct njb3_state_t njb3_state_t;
typedef struct njb3_keyval_t njb3_keyval_t;

/* Structure to hold protocol3 states */
struct njb3_state_t {
  songid_t *first_songid;
  songid_t *next_songid;
  playlist_t *first_plid;
  playlist_t *next_plid;
  datafile_t *first_dfid;
  datafile_t *next_dfid;
  int current_playing_track;
  njb3_keyval_t *first_key;
  njb3_keyval_t *next_key;
};

/* Local struct to hold key/value pairs */
struct njb3_keyval_t {
  char key[5];
  u_int32_t value1;
  u_int32_t value2;
  unsigned char deviceid[16];
  njb3_keyval_t *next;
};


/* NJB3 functions */
int njb3_init_state (njb_t *njb);
int njb2_set_bitmap (njb_t *njb, unsigned char * bitmap);
int njb3_current_track (njb_t *njb, u_int16_t * track);
int njb3_elapsed_time (njb_t *njb, u_int16_t * elapsed, int * change);
int njb3_play_track (njb_t *njb, u_int32_t trackid);
int njb3_queue_track (njb_t *njb, u_int32_t trackid);
int njb3_ctrl_playing (njb_t *njb, int cmd);
int njb3_seek_track (njb_t *njb, u_int32_t position);
int njb3_get_codecs(njb_t *njb);
int njb3_ping (njb_t *njb, njbid_t *njbid, int type);
int njb3_readid (njb_t *njb, njbid_t *njbid);
int njb3_capture (njb_t *njb);
int njb3_release (njb_t *njb);
int njb3_get_disk_usage (njb_t *njb, u_int64_t *totalbytes, u_int64_t *freebytes);
int njb3_get_owner_string (njb_t *njb, char *name);
int njb3_set_owner_string (njb_t *njb, const char *name);
njb_time_t *njb3_get_time(njb_t *njb);
int njb3_set_time(njb_t *njb, njb_time_t *time);
songid_t *njb3_get_first_track_tag (njb_t *njb, int extended);
songid_t *njb3_get_next_track_tag (njb_t *njb);
playlist_t *njb3_get_first_playlist_tag (njb_t *njb);
playlist_t *njb3_get_next_playlist_tag (njb_t *njb);
datafile_t *njb3_get_first_datafile_tag (njb_t *njb);
datafile_t *njb3_get_next_datafile_tag (njb_t *njb);
int njb3_read_keys(njb_t *njb);
int njb3_request_file_chunk(njb_t *njb, u_int32_t fileid, u_int32_t offset);
int njb3_get_file_block(njb_t *njb, unsigned char *data, u_int32_t maxsize);
u_int32_t njb3_create_file(njb_t *njb, unsigned char *ptag, u_int32_t tagsize, u_int16_t filetype);
int njb3_send_file_chunk(njb_t *njb, unsigned char *chunk, u_int32_t chunksize, u_int32_t fileid);
int njb3_send_file_complete(njb_t *njb, u_int32_t fileid);
int njb3_delete_item (njb_t *njb, u_int32_t itemid);
int njb3_update_16bit_frame(njb_t *njb, u_int32_t itemid, u_int16_t frameid, u_int16_t value);
int njb3_update_string_frame(njb_t *njb, u_int32_t itemid, u_int16_t frameid, unsigned char *str);
unsigned char *njb3_create_update_command(u_int32_t *packedlen, u_int32_t itemid);
unsigned char *njb3_add_16bit_frame(unsigned char *packedframe, u_int32_t *packedlen, u_int16_t frameid, u_int16_t value);
unsigned char *njb3_add_string_frame(unsigned char *packedframe, u_int32_t *packedlen, u_int16_t frameid, unsigned char *str);
int njb3_send_packed_frames(njb_t *njb, unsigned char *packedframe, u_int32_t packedlen);
int njb3_create_playlist (njb_t *njb, char *name, u_int32_t *plid);
int njb3_add_multiple_tracks_to_playlist (njb_t *njb, u_int32_t plid, u_int32_t *trids, u_int16_t ntracks);
int njb3_adjust_volume(njb_t *njb, u_int16_t value);
int njb3_select_eax(njb_t *njb, u_int16_t effect, u_int16_t index);
eax_t *njb3_get_eax (njb_t *njb);
void njb3_destroy_state(njb_t *njb);

#endif
