/* --- BEGIN COPYRIGHT BLOCK ---
 * Copyright (C) 2015  Red Hat
 * see files 'COPYING' and 'COPYING.openssl' for use and warranty
 * information
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Additional permission under GPLv3 section 7:
 * 
 * If you modify this Program, or any covered work, by linking or
 * combining it with OpenSSL, or a modified version of OpenSSL licensed
 * under the OpenSSL license
 * (https://www.openssl.org/source/license.html), the licensors of this
 * Program grant you additional permission to convey the resulting
 * work. Corresponding Source for a non-source form of such a
 * combination shall include the source code for the parts that are
 * licensed under the OpenSSL license as well as that of the covered
 * work.
 * --- END COPYRIGHT BLOCK ---
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include "nspr.h"
#include "pratom.h"
#include "private/pprio.h"
#include "liblfds.h"
#include "ns_event_fw.h"

/*
 * Threadpool
 * PR_Atomic*() operates on PRInt32
 */
struct ns_thrpool_t {
    PRInt32 init_threads;
    PRInt32 max_threads;
    PRInt32 idle_threads;
    PRUint32 stacksize;
    struct stack_state *thread_stack; /* stack of ns_thread_t */
    PRInt32 current_threads;
    PRThread *event_thread; /* the event loop thread */
    struct queue_state *work_q; /* queue of jobs to be executed by the thread pool */
    PRInt32 work_q_size; /* number of elements in work_q */
    PRLock *work_q_lock;
    PRCondVar *work_q_cv;
    struct queue_state *event_q; /* queue of jobs to return to the event loop */
    PRInt32 event_q_size; /* number of elements in event_q */
    PRFileDesc *event_q_wakeup_pipe_read;
    PRFileDesc *event_q_wakeup_pipe_write;
    ns_job_t *event_q_wakeup_job;
    PRInt32 shutdown;
    PRInt32 shutdown_event_loop;
    ns_event_fw_t *ns_event_fw;
    ns_event_fw_ctx_t *ns_event_fw_ctx;
    /* stats */
    PRInt32 min_idle_threads; /* minimum number of idle threads */
    PRInt32 max_work_q_size; /* maximum size of work q */
    PRInt32 max_event_q_size; /* maximum size of event q */
};

struct ns_thread_t {
    PRThread *thr; /* the thread */
    struct ns_thrpool_t *tp; /* pointer back to thread pool */
};

#define ATOMIC_VAL_IS_ZERO(val) 0 == PR_AtomicAdd((val), 0)
#define ERRNO_WOULD_BLOCK(iii) (iii == EWOULDBLOCK) || (iii == EAGAIN)

/* logging function pointers */
static void (*logger)(int, const char *, va_list) = NULL;
static void (*log_start)( void ) = NULL;
static void (*log_close)( void ) = NULL;

/* memory function pointers */
static void *(*malloc_fct)(size_t) = NULL;
static void *(*calloc_fct)(size_t, size_t) = NULL;
static void *(*realloc_fct)(void *, size_t) = NULL;
static void (*free_fct)(void *) = NULL;

/* syslog functions */
static void
ns_syslog(int priority, const char *fmt, va_list varg)
{
    vsyslog(priority, fmt, varg);
}

static void
ns_syslog_start( void )
{
    openlog("nunc-stans", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);
}

static void
ns_syslog_close( void )
{
    closelog();
}

/* default memory functions */
static void*
os_malloc(size_t size)
{
    return malloc(size);
}

static void*
os_calloc(size_t count, size_t size)
{
    return calloc(count, size);
}

static void*
os_realloc(void *ptr, size_t size)
{
    return realloc(ptr, size);
}

static void
os_free(void *ptr)
{
    free(ptr);
}

static void
work_q_wait(ns_thrpool_t *tp)
{
    PR_Lock(tp->work_q_lock);
    PR_WaitCondVar(tp->work_q_cv, PR_INTERVAL_NO_TIMEOUT);
    PR_Unlock(tp->work_q_lock);
}

static void
work_q_notify(ns_job_t *job)
{
    if (NS_JOB_IS_THREAD(job->job_type)) {
        queue_enqueue(job->tp->work_q, job);
        (void)PR_AtomicIncrement(&job->tp->work_q_size);
        if (job->tp->work_q_size > job->tp->max_work_q_size) {
            job->tp->max_work_q_size = job->tp->work_q_size;
        }
        PR_Lock(job->tp->work_q_lock);
        PR_NotifyCondVar(job->tp->work_q_cv);
        PR_Unlock(job->tp->work_q_lock);
        PR_Sleep(PR_INTERVAL_NO_WAIT); /* yield to allow worker thread to pick up job */
    } else {
        /* execute in same thread */
        ns_job_func_t jobfunc = job->func;
        jobfunc(job);
    }
}

/*
 * worker thread function
 */
static void
worker_thread_func(void *arg)
{
    ns_thread_t *thr = (ns_thread_t *)arg;
    ns_thrpool_t *tp = thr->tp;

    /*
     * Execute jobs until shutdown is set and the queues are empty.
     */
    while (ATOMIC_VAL_IS_ZERO(&tp->shutdown) || !(ATOMIC_VAL_IS_ZERO(&tp->work_q_size)) ||
           !(ATOMIC_VAL_IS_ZERO(&tp->event_q_size))) {
        ns_job_t *job = NULL;
        /* NGK - could this cause a tight loop if shutdown is set and nothing is in the work queue,
         * but we have a queued event?  I don't think this scenario would happen very long, so
         * maybe this is OK? */
        while(!queue_dequeue(tp->work_q, (void **)&job) &&
               ATOMIC_VAL_IS_ZERO(&tp->shutdown)) {
            work_q_wait(tp);
        }
        if (job) {
            ns_job_func_t jobfunc = job->func;
            (void)PR_AtomicDecrement(&tp->work_q_size);
            (void)PR_AtomicDecrement(&tp->idle_threads);
            if (tp->idle_threads < tp->min_idle_threads) {
                tp->min_idle_threads = tp->idle_threads;
            }
            jobfunc(job);
            (void)PR_AtomicIncrement(&tp->idle_threads);
        }
    }
    (void)PR_AtomicDecrement(&tp->current_threads);
    (void)PR_AtomicDecrement(&tp->idle_threads);
}

static void
internal_ns_job_done(ns_job_t *job)
{
    (void)PR_AtomicIncrement(&job->delete_state);
    if (job->ns_event_fw_fd) {
        job->tp->ns_event_fw->ns_event_fw_io_event_done(job->tp->ns_event_fw_ctx, job);
    }
    if (job->ns_event_fw_time) {
        job->tp->ns_event_fw->ns_event_fw_timer_event_done(job->tp->ns_event_fw_ctx, job);
    }
    if (job->ns_event_fw_sig) {
        job->tp->ns_event_fw->ns_event_fw_signal_event_done(job->tp->ns_event_fw_ctx, job);
    }

    if (job->fd && !NS_JOB_IS_PRESERVE_FD(job->job_type)) {
        PR_Close(job->fd);
    } /* else application is responsible for fd */
    ns_free(job);
}

/*
 * Add a new event, or modify or delete an existing event
 */
static void
update_event(ns_job_t *job)
{
    if (job->delete_state == NS_JOB_NEEDS_DELETE) {
        internal_ns_job_done(job);
    } else if (NS_JOB_IS_IO(job->job_type) || job->ns_event_fw_fd) {
        if (!job->ns_event_fw_fd) {
            job->tp->ns_event_fw->ns_event_fw_add_io(job->tp->ns_event_fw_ctx, job);
        } else {
            job->tp->ns_event_fw->ns_event_fw_mod_io(job->tp->ns_event_fw_ctx, job);
        }
    } else if (NS_JOB_IS_TIMER(job->job_type) || job->ns_event_fw_time) {
        if (!job->ns_event_fw_time) {
            job->tp->ns_event_fw->ns_event_fw_add_timer(job->tp->ns_event_fw_ctx, job);
        } else {
            job->tp->ns_event_fw->ns_event_fw_mod_timer(job->tp->ns_event_fw_ctx, job);
        }
    } else if (NS_JOB_IS_SIGNAL(job->job_type) || job->ns_event_fw_sig) {
        if (!job->ns_event_fw_sig) {
            job->tp->ns_event_fw->ns_event_fw_add_signal(job->tp->ns_event_fw_ctx, job);
        } else {
            job->tp->ns_event_fw->ns_event_fw_mod_signal(job->tp->ns_event_fw_ctx, job);
        }
    }

    return;
}

static void
event_q_wait(ns_thrpool_t *tp)
{
    /* unused for now */
    /* the main event loop will do our "waiting" - waiting for events
       to happen, or we can trigger an event to "wakeup" the event
       loop (see event_q_notify) */
}

static void
event_q_wake(ns_thrpool_t *tp)
{
    PRInt32 len;

    /* NSPR I/O doesn't allow non-blocking signal pipes, so use write instead of PR_Write */
    len = write(PR_FileDesc2NativeHandle(tp->event_q_wakeup_pipe_write),
                "a", 1);
    if (1 != len) {
        if ((errno == 0) || ERRNO_WOULD_BLOCK(errno)) {
            ns_log(LOG_DEBUG, "Write blocked for wakeup pipe - ignore %d\n",
                   errno);
        } else {
            ns_log(LOG_ERR, "Error: could not write wakeup pipe: %d:%s\n",
                   errno, PR_ErrorToString(errno, PR_LANGUAGE_I_DEFAULT));
        }
    }
    PR_Sleep(PR_INTERVAL_NO_WAIT); /* yield to allow event thread to pick up event */
}

static void
event_q_notify(ns_job_t *job)
{
    ns_thrpool_t *tp = job->tp;
    /* if we are being called from a thread other than the
       event loop thread, we have to notify that thread to
       perform the event work */
    if (PR_GetCurrentThread() == tp->event_thread) {
        /* If we are being run from the same thread as the event
           loop thread, we can just update the event here */
        update_event(job);
    } else {
        /* The event loop may be waiting for events, and may wait a long
           time by default if there are no events to process - since we
           want to add an event, we have to "wake up" the event loop by
           posting an event - this will cause the wakeup_cb to be called
           which will empty the event_q and add all of the events
        */
        /* NOTE: once job is queued, it may be deleted immediately in
         * another thread, if the event loop picks up the deletion
         * job before we can notify it below - so make sure not to
         * refer to job after the enqueue.
         */
        queue_enqueue(tp->event_q, job);
        (void)PR_AtomicIncrement(&tp->event_q_size);
        if (tp->event_q_size > tp->max_event_q_size) {
            tp->max_event_q_size = tp->event_q_size;
        }
        event_q_wake(tp);
    }
}

/* This is run inside the event loop thread, and only in the
   event loop thread
   This function pulls the io/timer/signal event requests off
   the request queue, formats the events in the format required
   by the event framework, and adds them
*/
static void
get_new_event_requests(ns_thrpool_t *tp)
{
    ns_job_t *job = NULL;
    while (queue_dequeue(tp->event_q, (void **)&job)) {
        (void)PR_AtomicDecrement(&tp->event_q_size);
        update_event(job);
    }
}

static void
event_loop_thread_func(void *arg)
{
    ns_thrpool_t *tp = (ns_thrpool_t *)arg;
    while (ATOMIC_VAL_IS_ZERO(&tp->shutdown_event_loop)) {
        int rc;
        /* get new event requests */
        get_new_event_requests(tp);
        /* process events */
        /* return 1 - no events ; 0 - normal exit ; -1 - error */
        rc = tp->ns_event_fw->ns_event_fw_loop(tp->ns_event_fw_ctx);
        if (rc == -1) { /* error */
        } else if (rc == 0) { /* exiting */
        } else { /* no events to process */
            event_q_wait(tp);
        }
    }
}

/*
 * The event framework calls this function when it receives an
 * event - the event framework does all of the work to map
 * the framework flags, etc. into the job
 * This function is called only in the event loop thread.
 * If the THREAD flag is set, this means the job is to be
 * handed off to the work q - otherwise, execute it now
 * in this thread (the event loop thread)
 * the function must be careful not to block the event
 * loop thread or starvation will occur
 */
static void
event_cb(ns_job_t *job)
{
    if (NS_JOB_IS_THREAD(job->job_type)) {
        work_q_notify(job);
    } else {
        job->func(job);
    }
}

static void
wakeup_cb(ns_job_t *job)
{
    PRInt32 len;
    char buf[1];

    /* NSPR I/O doesn't allow non-blocking signal pipes, so use read instead of PR_Read */
    len = read(PR_FileDesc2NativeHandle(job->tp->event_q_wakeup_pipe_read),
               buf, 1);
    if (1 != len) {
        if ((errno == 0) || ERRNO_WOULD_BLOCK(errno)) {
            ns_log(LOG_DEBUG, "Read blocked for wakeup pipe - ignore %d\n",
                   errno);
        } else {
            ns_log(LOG_ERR, "Error: could not read wakeup pipe: %d:%s\n",
                   errno, PR_ErrorToString(errno, PR_LANGUAGE_I_DEFAULT));
        }
    }
    /* wakeup_cb is usually called because a worker thread has posted a new
       event we need to add - get all new event requests */
    get_new_event_requests(job->tp);
}

/* convenience function for the event fw to use to allocate
   space for its event fw event object
   it is assumed that eventually job will have a memory region/arena
   to use
   these functions will be called in the event loop thread from
   the event framework function that adds a new event
*/
static void *
alloc_event_context(size_t size, ns_job_t *job)
{
    return malloc(size);
}

static void
free_event_context(void *ev_ctx, ns_job_t *job)
{
    ns_free(ev_ctx);
}

static ns_job_t *
new_ns_job(ns_thrpool_t *tp, PRFileDesc *fd, ns_job_type_t job_type, ns_job_func_t func, struct ns_job_data_t *data)
{
    ns_job_t *job = ns_calloc(1, sizeof(ns_job_t));
    job->tp = tp;
    job->ns_event_fw_ctx = tp->ns_event_fw_ctx;
    job->fd = fd;
    job->func = func;
    job->data = data;
    job->alloc_event_context = alloc_event_context;
    job->free_event_context = free_event_context;
    job->event_cb = event_cb;
    job->job_type = job_type;
    return job;
}

static ns_job_t *
alloc_io_context(ns_thrpool_t *tp, PRFileDesc *fd, ns_job_type_t job_type,
        ns_job_func_t func, struct ns_job_data_t *data)
{
    ns_job_t *job = new_ns_job(tp, fd, job_type, func, data);

    return job;
}

static ns_job_t *
alloc_timeout_context(ns_thrpool_t *tp, struct timeval *tv, ns_job_type_t job_type,
        ns_job_func_t func, struct ns_job_data_t *data)
{
    ns_job_t *job = new_ns_job(tp, NULL, NS_JOB_TIMER | job_type, func, data);
    job->tv = *tv;

    return job;
}

static ns_job_t *
alloc_signal_context(ns_thrpool_t *tp, PRInt32 signum, ns_job_type_t job_type,
        ns_job_func_t func, struct ns_job_data_t *data)
{
    ns_job_t *job = new_ns_job(tp, NULL, NS_JOB_SIGNAL | job_type, func, data);
    job->signal = signum;

    return job;
}

void
ns_job_done(ns_job_t *job)
{
    if (!job || job->delete_state){
        /* Just return if the job has been marked for deletion */
        return;
    }
    (void)PR_AtomicIncrement(&job->delete_state);
    event_q_notify(job);
}

/* queue a file descriptor to listen for and accept new connections */
PRStatus
ns_add_io_job(ns_thrpool_t *tp, PRFileDesc *fd, ns_job_type_t job_type,
        ns_job_func_t func, void *data, ns_job_t **job)
{
    ns_job_t *_job = NULL;

    if (job) {
        *job = NULL;
    }

    /* Don't allow a job to be added if the threadpool is being shut down. */
    if (ns_thrpool_is_shutdown(tp)) {
        return PR_FAILURE;
    }

    /* Don't allow an accept job to be run outside of the event thread. 
     * We do this so a listener job won't shut down while still processing
     * current connections in other threads.
     * TODO: Need to be able to have multiple threads accept() at the same time
     * This is fine - just have to remove the listen job in the polling thread
     * immediately after receiving notification - then call the job to do the
     * accept(), which will add back the persistent listener job immediately after
     * doing the accept()
     * This will be a combination of a non-threaded job and a threaded job
     *
     */
    if (NS_JOB_IS_ACCEPT(job_type) && NS_JOB_IS_THREAD(job_type)) {
        return PR_FAILURE;
    }

    /* get an event context for an accept */
    _job = alloc_io_context(tp, fd, job_type, func, data);
    if (!_job) {
        return PR_FAILURE;
    }

    event_q_notify(_job);

    /* fill in a pointer to the job for the caller if requested */
    if (job) {
        *job = _job;
    }

    return PR_SUCCESS;
}

PRStatus
ns_add_timeout_job(ns_thrpool_t *tp, struct timeval *tv, ns_job_type_t job_type,
        ns_job_func_t func, void *data, ns_job_t **job)
{
    ns_job_t *_job = NULL;

    if (job) {
        *job = NULL;
    }

    /* Don't allow a job to be added if the threadpool is being shut down. */
    if (ns_thrpool_is_shutdown(tp)) {
        return PR_FAILURE;
    }

    /* get an event context for a timer job */
    _job = alloc_timeout_context(tp, tv, job_type, func, data);
    if (!_job) {
        return PR_FAILURE;
    }

    event_q_notify(_job);

    /* fill in a pointer to the job for the caller if requested */
    if (job) {
        *job = _job;
    }

    return PR_SUCCESS;
}

/* queue a file descriptor to listen for and accept new connections */
PRStatus
ns_add_io_timeout_job(ns_thrpool_t *tp, PRFileDesc *fd, struct timeval *tv,
        ns_job_type_t job_type, ns_job_func_t func, void *data, ns_job_t **job)
{
    ns_job_t *_job = NULL;

    if (job) {
        *job = NULL;
    }

    /* Don't allow a job to be added if the threadpool is being shut down. */
    if (ns_thrpool_is_shutdown(tp)) {
        return PR_FAILURE;
    }

    /* Don't allow an accept job to be run outside of the event thread.
     * We do this so a listener job won't shut down while still processing
     * current connections in other threads.
     * TODO: Need to be able to have multiple threads accept() at the same time
     * This is fine - just have to remove the listen job in the polling thread
     * immediately after receiving notification - then call the job to do the
     * accept(), which will add back the persistent listener job immediately after
     * doing the accept()
     * This will be a combination of a non-threaded job and a threaded job
     *
     */
    if (NS_JOB_IS_ACCEPT(job_type) && NS_JOB_IS_THREAD(job_type)) {
        return PR_FAILURE;
    }

    /* get an event context for an accept */
    _job = alloc_io_context(tp, fd, job_type|NS_JOB_TIMER, func, data);
    if (!_job) {
        return PR_FAILURE;
    }
    _job->tv = *tv;

    event_q_notify(_job);

    /* fill in a pointer to the job for the caller if requested */
    if (job) {
        *job = _job;
    }

    return PR_SUCCESS;
}

PRStatus
ns_add_signal_job(ns_thrpool_t *tp, PRInt32 signum, ns_job_type_t job_type,
        ns_job_func_t func, void *data, ns_job_t **job)
{
    ns_job_t *_job = NULL;

    if (job) {
        *job = NULL;
    }

    /* Don't allow a job to be added if the threadpool is being shut down. */
    if (ns_thrpool_is_shutdown(tp)) {
        return PR_FAILURE;
    }

    /* get an event context for a signal job */
    _job = alloc_signal_context(tp, signum, job_type, func, data);
    if (!_job) {
        return PR_FAILURE;
    }

    event_q_notify(_job);

    /* fill in a pointer to the job for the caller if requested */
    if (job) {
        *job = _job;
    }

    return PR_SUCCESS;
}

PRStatus
ns_add_job(ns_thrpool_t *tp, ns_job_type_t job_type, ns_job_func_t func, void *data, ns_job_t **job)
{
    ns_job_t *_job = NULL;

    if (job) {
        *job = NULL;
    }

    /* Don't allow a job to be added if the threadpool is being shut down. */
    if (ns_thrpool_is_shutdown(tp)) {
        return PR_FAILURE;
    }

    _job = new_ns_job(tp, NULL, job_type, func, data);
    if (!_job) {
        return PR_FAILURE;
    }
    /* fill in a pointer to the job for the caller if requested */
    if (job) {
        *job = _job;
    }

    work_q_notify(_job);
    return PR_SUCCESS;
}

void *
ns_job_get_data(ns_job_t *job)
{
    return job->data;
}

ns_thrpool_t *
ns_job_get_tp(ns_job_t *job)
{
    return job->tp;
}

ns_job_type_t
ns_job_get_output_type(ns_job_t *job)
{
    return job->output_job_type;
}

ns_job_type_t
ns_job_get_type(ns_job_t *job)
{
    return job->job_type;
}

PRFileDesc *
ns_job_get_fd(ns_job_t *job)
{
    return job->fd;
}

/*
 * This is a convenience function - use if you need to re-arm the same event
 * usually not needed for persistent jobs
 */
void
ns_job_rearm(ns_job_t *job)
{
    event_q_notify(job);
}

void
ns_job_modify(ns_job_t *job, ns_job_type_t job_type)
{
    ns_job_type_t cur_job_type = job->job_type;
    int disable_only = NS_JOB_IS_DISABLE_ONLY(job_type);

    NS_JOB_UNSET_DISABLE_ONLY(job_type); /* clear the flag before setting in the job */
    job->job_type = job_type;
    /* if we are converting a job from an event job to a non-event job, or
       converting from a non-event to an event job, or
       changing the type of event (e.g. from read to write)
       we have to notify the event fw
       we need to let the event q know if the current job is threaded and
       if the new job is threaded
    */
    if (NS_JOB_IS_IO(job_type) || NS_JOB_IS_TIMER(job_type) || NS_JOB_IS_SIGNAL(job_type) ||
        NS_JOB_IS_IO(cur_job_type) || NS_JOB_IS_TIMER(cur_job_type) || NS_JOB_IS_SIGNAL(cur_job_type)) {
        event_q_notify(job);
    }

    if (!disable_only && !(NS_JOB_IS_IO(job_type) || NS_JOB_IS_TIMER(job_type) || NS_JOB_IS_SIGNAL(job_type))) {
        /* if this is a non event task, just queue it on the work q */
        work_q_notify(job);
    }
}

static void
ns_thrpool_delete(ns_thrpool_t *tp)
{
    ns_free(tp);
}

static ns_thrpool_t *
ns_thrpool_alloc(void)
{
    ns_thrpool_t *tp;

    tp = ns_calloc(1, sizeof(struct ns_thrpool_t));
    if (NULL == tp) {
        goto failed;
    }

    return tp;
failed:
    ns_thrpool_delete(tp);
    PR_SetError(PR_OUT_OF_MEMORY_ERROR, 0);
    return NULL;
}

/* libevent does not make public the file descriptors used to
   wakeup epoll_wait - you have to send a signal - instead of
   that just create our own wakeup pipe */
static void
setup_event_q_wakeup(ns_thrpool_t *tp)
{
    ns_job_t *job;
    PR_CreatePipe(&tp->event_q_wakeup_pipe_read,
                  &tp->event_q_wakeup_pipe_write);
    /* setting options is not supported on NSPR pipes - use fcntl
    PRSocketOptionData prsod = {PR_SockOpt_Nonblocking, {PR_TRUE}};
    PR_SetSocketOption(tp->event_q_wakeup_pipe_read, &prsod);
    PR_SetSocketOption(tp->event_q_wakeup_pipe_write, &prsod);
    */
    if (fcntl(PR_FileDesc2NativeHandle(tp->event_q_wakeup_pipe_read),
              F_SETFD, O_NONBLOCK) == -1) {
        ns_log(LOG_ERR, "setup_event_q_wakeup(): could not make read pipe non-blocking: %d\n",
               PR_GetOSError());
    }
    if (fcntl(PR_FileDesc2NativeHandle(tp->event_q_wakeup_pipe_write),
              F_SETFD, O_NONBLOCK) == -1) {
        ns_log(LOG_ERR, "setup_event_q_wakeup(): could not make write pipe non-blocking: %d\n",
               PR_GetOSError());
    }
    /* wakeup events are processed inside the event loop thread */
    job = alloc_io_context(tp, tp->event_q_wakeup_pipe_read,
                            NS_JOB_READ|NS_JOB_PERSIST|NS_JOB_PRESERVE_FD,
                            wakeup_cb, NULL);
    tp->ns_event_fw->ns_event_fw_add_io(tp->ns_event_fw_ctx, job);
    /* Stash the wakeup job in tp so we can release it later. */
    tp->event_q_wakeup_job = job;
}

/* Initialize the thrpool config */
#define NS_INIT_MAGIC 0xdefa014

void
ns_thrpool_config_init(struct ns_thrpool_config *tp_config)
{
    tp_config->init_flag = NS_INIT_MAGIC;
    tp_config->initial_threads = 1;
    tp_config->max_threads = 1;
    tp_config->stacksize = 0;
    tp_config->event_queue_size = 1024;
    tp_config->work_queue_size = 1024;
    tp_config->log_fct = NULL;
    tp_config->log_start_fct = NULL;
    tp_config->log_close_fct = NULL;
    tp_config->malloc_fct = NULL;
    tp_config->calloc_fct = NULL;
    tp_config->realloc_fct = NULL;
    tp_config->free_fct = NULL;
}

/*
 * Process the config and set the pluggable function pointers
 */
static int
ns_thrpool_process_config(struct ns_thrpool_config *tp_config)
{
    /* Check that the config has been properly initialized */
    if (!tp_config || tp_config->init_flag != NS_INIT_MAGIC){
        return -1;
    }
    /*
     * Assign our logging function pointers
     */
    if (tp_config->log_fct){
        /* Set a logger function */
        logger = tp_config->log_fct;
        if (tp_config->log_start_fct){
            log_start = tp_config->log_start_fct;
        }
        if (tp_config->log_close_fct){
            log_close = tp_config->log_close_fct;
        }
    } else {
        /* Default to syslog */
        logger = ns_syslog;
        log_start = ns_syslog_start;
        log_close = ns_syslog_close;
    }
    if(log_start){
        /* Start logging */
        (log_start)();
    }

    /*
     * Set the memory function pointers
     */
    /* malloc */
    if (tp_config->malloc_fct){
        malloc_fct = tp_config->malloc_fct;
    } else {
        malloc_fct = os_malloc;
    }
    /* calloc */
    if (tp_config->calloc_fct){
        calloc_fct = tp_config->calloc_fct;
    } else {
        calloc_fct = os_calloc;
    }
    /* realloc */
    if (tp_config->realloc_fct){
        realloc_fct = tp_config->realloc_fct;
    } else {
        realloc_fct = os_realloc;
    }
    /* free */
    if (tp_config->free_fct){
        free_fct = tp_config->free_fct;
    } else {
        free_fct = os_free;
    }

    return 0;
}

ns_thrpool_t *
ns_thrpool_new(struct ns_thrpool_config *tp_config)
{
    ns_thrpool_t *tp = NULL;
    ns_thread_t *thr;
    PRThread *event_thr;
    int ii;

    if(ns_thrpool_process_config(tp_config) == -1){
        ns_log(LOG_ERR, "ns_thrpool_new(): config has not been properly initialized\n");
        goto failed;
    }

    tp = ns_thrpool_alloc();
    if (NULL == tp) {
        ns_log(LOG_ERR, "ns_thrpool_new(): failed to allocate thread pool\n");
        goto failed;
    }

    ns_log(LOG_DEBUG, "ns_thrpool_new():  initial threads (%d), max threads, (%d)\n"
           "stacksize (%d), event q size (%d), work q size (%d)\n",
           tp_config->initial_threads,tp_config->max_threads,
           tp_config->stacksize,tp_config->event_queue_size,
           tp_config->work_queue_size);

    tp->init_threads = tp_config->initial_threads;
    /* NGK - how does the thread pool increase the threads if max_threads is set higher than initial_threads? */
    tp->max_threads = tp_config->max_threads;
    tp->stacksize = tp_config->stacksize;
    if (!stack_new(&tp->thread_stack, tp_config->max_threads) || !tp->thread_stack) {
        goto failed;
    }

    if (!queue_new(&tp->work_q, tp_config->work_queue_size) || !tp->work_q) {
        goto failed;
    }
    if (!(tp->work_q_lock = PR_NewLock())) {
        goto failed;
    }
    if (!(tp->work_q_cv = PR_NewCondVar(tp->work_q_lock))) {
        goto failed;
    }

    if (!queue_new(&tp->event_q, tp_config->event_queue_size) || !tp->event_q) {
        goto failed;
    }

    /* NGK TODO - add tevent vs. libevent switch */
    /* tp->ns_event_fw = get_event_framework_tevent(); */
    tp->ns_event_fw = get_event_framework_event();
    tp->ns_event_fw_ctx = tp->ns_event_fw->ns_event_fw_init();

    setup_event_q_wakeup(tp);

    for (ii = 0; ii < tp_config->initial_threads; ++ii) {
        thr = ns_calloc(1, sizeof(ns_thread_t));
        PR_ASSERT(thr);
        thr->tp = tp;
        thr->thr = PR_CreateThread(PR_USER_THREAD, worker_thread_func,
                                   thr, PR_PRIORITY_NORMAL, PR_GLOBAL_THREAD,
                                   PR_JOINABLE_THREAD, tp_config->stacksize);
        PR_ASSERT(thr->thr);
        stack_push(tp->thread_stack, thr);
    }
    tp->current_threads = tp->idle_threads = tp->min_idle_threads = tp_config->initial_threads;

    event_thr = PR_CreateThread(PR_USER_THREAD, event_loop_thread_func,
                                tp, PR_PRIORITY_NORMAL, PR_GLOBAL_THREAD,
                                PR_JOINABLE_THREAD, tp_config->stacksize);
    PR_ASSERT(event_thr);

    /* We keep the event thread separate from the stack of worker threads. */
    tp->event_thread = event_thr;

    return tp;
failed:
    ns_thrpool_destroy(tp);
    return NULL;
}

static void
print_stats(struct ns_thrpool_t *tp)
{
    printf("min idle threads: %d\n", tp->min_idle_threads);
    printf("max work q size: %d\n", tp->max_work_q_size);
    printf("max event q size: %d\n", tp->max_event_q_size);
}

void
ns_thrpool_destroy(struct ns_thrpool_t *tp)
{
    if (tp) {
        print_stats(tp);
        /* Set the flag to shutdown the event loop. */
        PR_AtomicSet(&tp->shutdown_event_loop, 1);

        /* Finish the event queue wakeup job.  This has the
         * side effect of waking up the event loop thread, which
         * will cause it to exit since we set the event loop
         * shutdown flag.  Fake the job to be a threaded job
         * so that we can run it from outside the event loop,
         * and use it to wake up the event loop.
         */
        tp->event_q_wakeup_job->job_type |= NS_JOB_THREAD;
        ns_job_done(tp->event_q_wakeup_job);

        /* Wait for the event thread to finish before we free the
         * internals of tp. */
        PR_JoinThread(tp->event_thread);

        /* Free the work queue. */
        if (tp->work_q) {
            struct ns_job_t *job = NULL;
            int njobs = 0;
            while (queue_dequeue(tp->work_q, (void **)&job) && job) {
                njobs++;
                ns_job_done(job);
            }
            ns_log(LOG_DEBUG, "ns_thrpool_destroy: destroyed [%d] work_q jobs\n", njobs);
            queue_delete(tp->work_q, NULL, NULL);
            tp->work_q = NULL;
        }

        /* Free the thread stack. */
        if (tp->thread_stack) {
            /* The stack should be empty if ns_thrpool_wait() returned successfully
             * already.  For this reason, we don't need to supply a data delete callback. */
            stack_delete(tp->thread_stack, NULL, NULL);
            tp->thread_stack = NULL;
        }

        /* Free the work queue lock. */
        if (tp->work_q_lock) {
            PR_DestroyLock(tp->work_q_lock);
            tp->work_q_lock = NULL;
        }

        /* Free the work queue condition variable. */
        if (tp->work_q_cv) {
            PR_DestroyCondVar(tp->work_q_cv);
            tp->work_q_cv = NULL;
        }

        /* Free the event queue. */
        if (tp->event_q) {
            struct ns_job_t *job = NULL;
            int njobs = 0;
            while (queue_dequeue(tp->event_q, (void **)&job) && job) {
                njobs++;
                ns_job_done(job);
            }
            ns_log(LOG_DEBUG, "ns_thrpool_destroy: destroyed [%d] event_q jobs\n", njobs);
            queue_delete(tp->event_q, NULL, NULL);
            tp->event_q = NULL;
        }

        /* Free the event queue wakeup pipe/job. */
        if (tp->event_q_wakeup_pipe_read) {
            PR_Close(tp->event_q_wakeup_pipe_read);
            tp->event_q_wakeup_pipe_read = NULL;
        }

        if (tp->event_q_wakeup_pipe_write) {
            PR_Close(tp->event_q_wakeup_pipe_write);
            tp->event_q_wakeup_pipe_write = NULL;
        }
        internal_ns_job_done(tp->event_q_wakeup_job);
        tp->event_q_wakeup_job = NULL;

        /* Free the event framework context. */
        if (tp->ns_event_fw_ctx) {
            tp->ns_event_fw->ns_event_fw_destroy(tp->ns_event_fw_ctx);
            tp->ns_event_fw_ctx = NULL;
        }

        /* Free the thread pool struct itself. */
        ns_thrpool_delete(tp);
    }
    /* Stop logging */
    if(log_close){
        (log_close)();
    }
}

/* Triggers the pool of worker threads to shutdown after finishing the remaining work. 
 * This must be run in a worker thread, not the event thread.  Running it in the event
 * thread could cause a deadlock. */
void
ns_thrpool_shutdown(struct ns_thrpool_t *tp)
{
    /* Set the shutdown flag.  This will cause the worker
     * threads to exit after they finish all remaining work. */
    PR_AtomicSet(&tp->shutdown, 1);

    /* Wake up the idle worker threads so they can exit. */
    PR_Lock(tp->work_q_lock);
    PR_NotifyAllCondVar(tp->work_q_cv);
    PR_Unlock(tp->work_q_lock);
}

PRInt32
ns_thrpool_is_shutdown(struct ns_thrpool_t *tp)
{
    return ATOMIC_VAL_IS_ZERO(&tp->shutdown) ? 0 : 1;
}

PRStatus
ns_thrpool_wait(ns_thrpool_t *tp)
{
    PRStatus retval = PR_SUCCESS;
    ns_thread_t *thr;
    struct ns_job_t *job = NULL;
    int njobs = 0;

    while (stack_pop(tp->thread_stack, (void **)&thr) && thr) {
        PRStatus rc = PR_JoinThread(thr->thr);
        if (rc != PR_SUCCESS) {
            /* NGK TODO - this is unused right now. */
            PRErrorCode prerr = PR_GetError();
            (void)prerr;
        }
        ns_free(thr);
    }
    while (queue_dequeue(tp->work_q, (void **)&job) && job) {
        njobs++;
        ns_job_done(job);
    }
    ns_log(LOG_DEBUG, "ns_thrpool_wait: thread [0x%p] deleted [%d] jobs\n",
           thr, njobs);

    return retval;
}

/*
 * nunc stans logger
 */
void
ns_log(int priority, const char *fmt, ...)
{
    va_list varg;

    va_start(varg, fmt);
    ns_log_valist(priority, fmt, varg);
    va_end(varg);

}

void
ns_log_valist(int priority, const char *fmt, va_list varg)
{
    (logger)(priority, fmt, varg);
}

/*
 * Pluggable memory functions
 */
void *
ns_malloc(size_t size)
{
    return (malloc_fct)(size);
}

void *
ns_calloc(size_t count, size_t size)
{
    return (calloc_fct)(count, size);
}

void *
ns_realloc(void *ptr, size_t size)
{
    return (realloc_fct)(ptr, size);
}

void
ns_free(void *ptr)
{
    (free_fct)(ptr);
}
