/* Hash tables for Objective C internal structures
   Copyright (C) 1993, 1996, 1997 Free Software Foundation, Inc.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/* As a special exception, if you link this library with files
   compiled with GCC to produce an executable, this does not cause
   the resulting executable to be covered by the GNU General Public License.
   This exception does not however invalidate any other reasons why
   the executable file might be covered by the GNU General Public License.  */

#include "assert.h"
#include "hash.h"
#include "runtime.h"		/* for DEBUG_PRINTF */
#include "objc-api.h"

/* These two macros determine when a hash table is full and
   by how much it should be expanded respectively.

   These equations are percentages.  */
#define FULLNESS(cache)    ((((cache)->size * 75) / 100) <= (cache)->used)
#define EXPANSION(cache)   ((cache)->size * 2)

cache_ptr hash_new (unsigned int size, char hashType)
{
  cache_ptr cache;

  /* Pass me a value greater than 0 and a power of 2.  */
  assert (size);
  assert (!(size & (size - 1)));

  /* Allocate the cache structure.  calloc insures
     its initialization for default values.  */
  cache = (cache_ptr)OBJC_CALLOC_UNCOLLECTABLE(sizeof (struct cache));
  assert (cache);

  /* Allocate the array of buckets for the cache.
     calloc initializes all of the pointers to NULL.  */
  cache->node_table =
    (node_ptr *)OBJC_CALLOC_UNCOLLECTABLE(size * sizeof (node_ptr));

  assert (cache->node_table);

  cache->size  = size;

  /* This should work for all processor architectures? */
  cache->mask = (size - 1);

  cache->hashType = hashType;
  
  return cache;
}


void
hash_add (cache_ptr *cachep, const void *key, void *value)
{
  size_t indx = objc_call_hash(*cachep, key);
  //hh: size_t indx = (*(*cachep)->hash_func)(*cachep, key);
  node_ptr node;

  node = (node_ptr)OBJC_CALLOC_UNCOLLECTABLE(sizeof (struct cache_node));
  assert (node);

  /* Initialize the new node.  */
  node->key    = key;
  node->value  = value;
  node->next  = (*cachep)->node_table[indx];

  /* Debugging.
     Check the list for another key.  */
#if DEBUG_OBJC
  { node_ptr node1 = (*cachep)->node_table[indx];

    while (node1) {

      assert (node1->key != key);
      node1 = node1->next;
    }
  }
#endif

  /* Install the node as the first element on the list.  */
  (*cachep)->node_table[indx] = node;

  /* Bump the number of entries in the cache.  */
  ++(*cachep)->used;

  /* Check the hash table's fullness.   We're going
     to expand if it is above the fullness level.  */
  if (FULLNESS (*cachep)) {

    /* The hash table has reached its fullness level.  Time to
       expand it.

       I'm using a slow method here but is built on other
       primitive functions thereby increasing its
       correctness.  */
    node_ptr node1 = NULL;
    cache_ptr new = hash_new (EXPANSION (*cachep), (*cachep)->hashType);
    
    DEBUG_PRINTF ("Expanding cache 0x%08X from %d to %d\n",
		  (size_t)*cachep, (*cachep)->size, new->size);

    /* Copy the nodes from the first hash table to the new one.  */
    while ((node1 = hash_next (*cachep, node1)))
      hash_add (&new, node1->key, node1->value);

    /* Trash the old cache.  */
    hash_delete (*cachep);

    /* Return a pointer to the new hash table.  */
    *cachep = new;
  }
}
