(**
   A nice analog clock, to be placed everywhere you like.
**)

MODULE VO:Clock;

(*
    Demo for VisualOberon. A clock.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VO:Base:Display,
       F  := VO:Base:Frame,
       O  := VO:Base:Object,
       U  := VO:Base:Util,

       G  := VO:Object,

       SC := SysClock;

CONST
  callPeriod = 1;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
               END;


  Clock*     = POINTER TO ClockDesc;
  ClockDesc* = RECORD (G.GadgetDesc)
                 timeOut   : D.TimeOut;
                 framed    : BOOLEAN;
               END;

VAR
  prefs* : Prefs;
  sinTab : ARRAY 16 OF LONGINT;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DOut;
  END Init;

  PROCEDURE (c : Clock) Init*;

  BEGIN
    c.Init^;

    c.SetPrefs(prefs);

    c.timeOut:=NIL;

    c.framed:=TRUE;
  END Init;

  (**
    Tell if the clock should be framed. Defaults to TRUE.
  **)

  PROCEDURE (c : Clock) ShowFrame*(framed : BOOLEAN);

  BEGIN
    (* We cannot switch back to using no fraame if we already generated one *)
    ASSERT(c.objectFrame=NIL);

    c.framed:=framed;
  END ShowFrame;

  PROCEDURE (c : Clock) CalcSize*;

  BEGIN

    IF c.framed THEN
      c.SetObjectFrame(c.prefs.frame);
    ELSE
      c.SetObjectFrame(F.none);
    END;

    c.width:=15;
    c.height:=15;

    c.minWidth:=c.width;
    c.minHeight:=c.height;

    c.CalcSize^;
  END CalcSize;

  PROCEDURE sin(grad : LONGINT):LONGINT;

  BEGIN
    IF (grad>=0) & (grad<=90) THEN
      RETURN sinTab[grad DIV 6];
    ELSIF (grad>=91) & (grad<=179) THEN
      RETURN sin(-grad+180);
    ELSIF (grad>=180) & (grad<=269) THEN
      RETURN -sin(grad-180);
    ELSE (*grad>270*)
      RETURN -sin(360-grad);
    END;
  END sin;

  PROCEDURE cos(grad : LONGINT):LONGINT;

  BEGIN
    RETURN sin((grad+90) MOD 360);
  END cos;

  PROCEDURE (c : Clock) Draw*(x,y,w,h : LONGINT);

  VAR
    zx,zy,r,
    height   : LONGINT;
    sTime    : SC.DateTime;
    draw     : D.DrawInfo;

  BEGIN
    c.Draw^(x,y,w,h);

    IF ~c.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=c.GetDrawInfo();

    c.DrawBackground(c.x,c.y,c.width,c.height);

    SC.GetClock(sTime);

    height:=c.height;

    zx:=c.x+c.width DIV 2;
    zy:=c.y+height DIV 2;

    r:=U.MinLong(c.width-D.display.spaceWidth,height-D.display.spaceHeight) DIV 2;

    draw.PushForeground(D.textColor);
    draw.DrawLine(zx,zy,
                  zx+(r * sin((LONG(sTime.hour) MOD 12)*30+sTime.minute DIV 2) DIV 1500),
                  zy-(r * cos((LONG(sTime.hour) MOD 12)*30+sTime.minute DIV 2) DIV 1500));
    draw.DrawLine(zx,zy,
                  zx+(r * sin(LONG(sTime.minute)*6) DIV 1000),
                  zy-(r * cos(LONG(sTime.minute)*6) DIV 1000));
    draw.PopForeground;

    draw.PushForeground(D.warnColor);
    draw.DrawLine(zx,zy,
                  zx+(r * sin(LONG(sTime.second)*6) DIV 1200),
                  zy-(r * cos(LONG(sTime.second)*6) DIV 1200));
    draw.PopForeground;

    IF c.timeOut=NIL THEN
      c.timeOut:=D.display.AddTimeOut(callPeriod,0,c);
    END;
  END Draw;

  PROCEDURE (c : Clock) Hide*;

  BEGIN
    IF c.timeOut#NIL THEN
      D.display.RemoveTimeOut(c.timeOut);
      c.timeOut:=NIL;
    END;
    IF c.visible THEN
      c.DrawHide;
      c.Hide^;
    END;
  END Hide;

  PROCEDURE (c : Clock) Receive*(msg : O.Message);

  BEGIN
    WITH msg:
      D.TimeOutMsg DO
        IF c.visible THEN
          c.timeOut:=NIL;
          c.Redraw;
        END;
    ELSE
    END;
  END Receive;

  PROCEDURE CreateClock*():Clock;

  VAR
    clock : Clock;

  BEGIN
    NEW(clock);
    clock.Init;

    RETURN clock;
  END CreateClock;

BEGIN
  sinTab[ 0]:= 000; (*  0 *)
  sinTab[ 1]:= 105; (*  6 *)
  sinTab[ 2]:= 208; (* 12 *)
  sinTab[ 3]:= 309; (* 18 *)
  sinTab[ 4]:= 407; (* 24 *)
  sinTab[ 5]:= 500; (* 30 *)
  sinTab[ 6]:= 588; (* 36 *)
  sinTab[ 7]:= 669; (* 42 *)
  sinTab[ 8]:= 743; (* 48 *)
  sinTab[ 9]:= 809; (* 54 *)
  sinTab[10]:= 866; (* 60 *)
  sinTab[11]:= 914; (* 66 *)
  sinTab[12]:= 951; (* 72 *)
  sinTab[13]:= 978; (* 78 *)
  sinTab[14]:= 995; (* 84 *)
  sinTab[15]:=1000; (* 90 *)

  NEW(prefs);
  prefs.Init;
END VO:Clock.