(**
  Module for handling of the various preferences GUIs.
  Offers a flexible settings dialog.
**)

MODULE VO:Prefs:GUI;

(*
    Implements a unvisible tab gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       O   := VO:Base:Object,
       Z   := VO:Base:Size,

       TM  := VO:Model:Table,

       PP  := VO:Prefs:Parser,

       B   := VO:Button,
       BR  := VO:ButtonRow,
       G   := VO:Object,
       M   := VO:Multi,
       P   := VO:Panel,
       S   := VO:Space,
       T   := VO:Text,
       TA  := VO:Table,
       W   := VO:Window,
       WG  := VO:WindowGroup,

              XML:Builder,
(*              XML:Builder:Validation,*)
              XML:DTD,
              XML:Parser,
              XML:UnicodeCodec:UTF8,
              XML:UnicodeCodec:ImportAll,
       LS  := LongStrings,
              Msg,
              StdChannels,
              Err,
       f   := Files,
              Channel,
              Rts,
       str := Strings,
       t   := TextRider;

CONST
  prefsNameSize* = 20;

  useId    = 1;
  saveId   = 2;
  closeId  = 3;

TYPE
  PrefsItem*     = POINTER TO PrefsItemDesc;
  PrefsItemDesc* = RECORD (O.MsgObjectDesc)
                     window-  : W.Window;
                     next,
                     last     : PrefsItem;
                     name*    : ARRAY prefsNameSize OF CHAR;
                   END;

  Settings*      = POINTER TO SettingsDesc;
  SettingsDesc*  = RECORD ( W.WindowDesc)
                     tableModel : TM.LTableModel;
                     table      : TA.Table;
                     multi      : M.Multi;

                     itemList,
                     itemLast   : PrefsItem;

                     exitIt     : BOOLEAN;

                     top        : PP.Item;
                   END;

  AboutItem     = POINTER TO AboutItemDesc;
  AboutItemDesc = RECORD (PrefsItemDesc);
                  END;


  Sel2Set*     = POINTER TO Sel2SetDesc;
  Sel2SetDesc* = RECORD (O.HandlerDesc)
                 END;

  Prefs        = POINTER TO PrefsDesc;
  PrefsDesc    = RECORD (Builder.BuilderDesc)
                   top : PP.Item;
                 END;

  PrefsCallback     = POINTER TO PrefsCallbackDesc;
  PrefsCallbackDesc = RECORD (D.PrefsCallbackDesc)
                      END;

VAR
  settings*     : Settings;
  prefsCallback : PrefsCallback;

  home-    : ARRAY 256 OF CHAR;  (** The name of the directory for all preferences files *)
  fileName : ARRAY 256 OF CHAR;


  PROCEDURE (p : Prefs) StartDocument* (documentEntity: DTD.ExternalEntity);

  VAR
    block : PP.BlockItem;

  BEGIN
    NEW(block);
    block.Init;
    NEW(block.name,str.Length("TOP")+1);
    COPY("TOP",block.name^);
    p.top:=block;
  END StartDocument;

  PROCEDURE (p : Prefs) StartElement* (namespaceDecl: DTD.NamespaceDeclaration;
                                        localName: DTD.String);

  VAR
    block : PP.BlockItem;

  BEGIN
    NEW(block);
    block.Init;
    NEW(block.name,LS.Length(localName^)+1);
    LS.Short(localName^,"",block.name^);
    p.top.AddItem(block);
    p.top:=block;
  END StartElement;

  PROCEDURE (p : Prefs) EndElement* (namespaceDecl: DTD.NamespaceDeclaration;
                                      localName: DTD.String);

  BEGIN
    p.top:=p.top.parent;
  END EndElement;

  PROCEDURE (p : Prefs) Attribute* (namespaceDecl: DTD.NamespaceDeclaration;
                                     localName: DTD.String;
                                     attrDecl: DTD.AttrDecl;
                                     value: DTD.AttValue;
                                     specified: BOOLEAN);

  VAR
    item : PP.ValueItem;
    res  : DTD.String;
    bool : BOOLEAN;

  BEGIN
    NEW(item);
    item.Init;
    NEW(item.name,LS.Length(localName^)+1);
    LS.Short(localName^,"",item.name^);
    res:=value.FlattenValue(TRUE,bool);
    NEW(item.value,LS.Length(res^)+1);
    LS.Short(res^,"",item.value^);

    p.top.AddItem(item);
  END Attribute;

  PROCEDURE (p : Prefs) AttributesDone*;

  BEGIN
    (* nothing to do *)
  END AttributesDone;

  PROCEDURE (s : Settings) AddItem(item : PrefsItem);

  BEGIN
    IF s.itemList=NIL THEN
      s.itemList:=item;
    ELSE
      s.itemLast.next:=item;
      item.last:=s.itemLast;
    END;
    s.itemLast:=item;
  END AddItem;

  PROCEDURE (p : PrefsItem) Init*;

  BEGIN
    p.Init^;
        
    p.next:=NIL;
    p.last:=NIL;
    p.name[0]:=0X;
    settings.AddItem(p);
  END Init;

  PROCEDURE  (p : PrefsItem) ErrorWrongData*(name,value : ARRAY OF CHAR);

  BEGIN
    Err.String("Unknown value for '");
    Err.String(p.name);
    Err.String("/");
    Err.String(name);
    Err.String("': ");
    Err.String(value);
    Err.Ln;
  END ErrorWrongData;

  PROCEDURE (i : PrefsItem) GetObject*():G.Object;

  VAR
    panel : P.Panel;

  BEGIN

    panel:=P.CreateVPanel();

    RETURN panel;
  END GetObject;

  PROCEDURE (o : PrefsItem) Apply*;

  BEGIN
  END Apply;

  PROCEDURE (o : PrefsItem) Refresh*;

  BEGIN
  END Refresh;

  PROCEDURE (p : PrefsItem) LoadPrefs*(top : PP.Item);
  (**
    All preferences should have a load-method. The baseclass
    does nothing.
  *)

  BEGIN
  END LoadPrefs;

  PROCEDURE (o : PrefsItem) SavePrefs*(block : PP.BlockItem);

  BEGIN
  END SavePrefs;

  PROCEDURE (s : Settings) Init*;

  VAR
    about : AboutItem;

  BEGIN
    s.Init^;

    s.SetTitle("VisualOberon - Settings");

    s.exitIt:=FALSE;

    s.itemList:=NIL;

    NEW(about);
    about.Init;
    s.AddItem(about);
  END Init;

  PROCEDURE (s : Settings) ExitOnClose*(exit : BOOLEAN);
  (**
    If you want the dialog to quit the application, set this to true.

    This is usefull, if the configuration dialog is a standalone
    window.
  *)

  BEGIN
    s.exitIt:=exit;
  END ExitOnClose;

  PROCEDURE (s : Settings) PreInit*;

  VAR
    top,vert    : P.Panel;
    row         : BR.ButtonRow;
    button      : B.Button;
    msg2Exit    : W.Msg2Exit;
    space       : S.Space;
    item        : PrefsItem;
    listHandler : Sel2Set;
    wGroup      : WG.WindowGroup;

  BEGIN
    s.SetBackground(D.backgroundColor); (* Hack! *)

    top:=P.CreateHPanel();
    top.SetFlags({G.horizontalFlex,G.verticalFlex});

      NEW(s.tableModel);
      s.tableModel.Init;
      s.tableModel.SetSelectionType(TM.singleLineSelect);

      s.table:=TA.CreateTable();
      s.table.SetModel(s.tableModel);
      s.table.SetFlags({G.verticalFlex});
      s.table.SetMinWidth(Z.unit,prefsNameSize+5);
      s.table.SetMinHeight(Z.unit,20);
      s.table.SetShowHeader(FALSE);
      s.table.SetShowScroller(FALSE,TRUE);
      s.AddFocusObject(s.table.table);
    top.Add(s.table);


      space:=S.CreateHSpace();
    top.Add(space);

      vert:=P.CreateVPanel();
      vert.SetFlags({G.horizontalFlex,G.verticalFlex});

        s.multi:=M.CreateMulti();
        s.multi.SetFlags({G.horizontalFlex,G.verticalFlex});
        NEW(listHandler);
        listHandler.destination:=s.multi;
        s.tableModel.AddHandler(listHandler,TM.selectionMsg);

        item:=s.itemList;
        WHILE item#NIL DO
          s.tableModel.AppendEntry;
          s.tableModel.SetString(1,s.tableModel.GetRows(),item.name);
          item.window:=s;
          s.multi.Add(item.GetObject());
          item:=item.next;
        END;

        s.tableModel.SelectRow(1);
      vert.Add(s.multi);

        space:=S.CreateVSpace();
      vert.Add(space);

        row:=BR.CreateButtonRow();
        row.SetFlags({G.horizontalFlex});

          button:=B.CreateButton();
          button.SetId(saveId);
          button.SetFlags({G.horizontalFlex});
          button.SetLabelText("_Save");
          button.Forward(B.pressedMsg,s);
          s.AddFocusObject(button);
        row.Add(button);

          button:=B.CreateButton();
          button.SetId(useId);
          button.SetFlags({G.horizontalFlex});
          button.SetLabelText("_Apply");
          button.Forward(B.pressedMsg,s);
          s.AddFocusObject(button);
        row.Add(button);

          button:=B.CreateButton();
          button.SetId(closeId);
          button.SetFlags({G.horizontalFlex});
          button.SetLabelText("_Close^");
          NEW(msg2Exit);
          msg2Exit.destination:=s;
          button.AddHandler(msg2Exit,B.pressedMsg);
          s.AddFocusObject(button);
        row.Add(button);

      vert.Add(row);
    top.Add(vert);

    wGroup:=WG.CreateWindowGroup();
    wGroup.Set(NIL,top,TRUE);

    s.SetTop(wGroup);

    NEW(msg2Exit);
    msg2Exit.destination:=s;
    s.AddHandler(msg2Exit,W.closeMsg);

    s.PreInit^;
  END PreInit;

  PROCEDURE (s : Settings) Open*;

  VAR
    item : PrefsItem;

  BEGIN
    s.Open^;

    item:=s.itemList;
    WHILE item#NIL DO
      item.Refresh;
      item:=item.next;
    END;
  END Open;

  PROCEDURE (s : Settings) ApplyPrefs*;

  VAR
    item : PrefsItem;

  BEGIN
    item:=s.itemList;
    WHILE item#NIL DO
      item.Apply;
      item:=item.next;
    END;
  END ApplyPrefs;

  PROCEDURE (s : Settings) LoadXMLPrefsFile(name : ARRAY OF CHAR):PP.Item;

  VAR
    canon   : Parser.Parser;
    builder : Builder.Builder;
    prefs   : Prefs;
    error   : ARRAY 256 OF CHAR;

    PROCEDURE CreateParser(): Parser.Parser;

    VAR
      p   : Parser.Parser;
      res : Msg.Msg;

    BEGIN
      NEW (canon);

      NEW(prefs);
      (*  medium.Init;*)
      builder:=prefs;

      p := Parser.NewFile(fileName, NIL, NIL, NIL, builder, res);
      IF p#NIL THEN
        p. followExternalRef := TRUE;
        p. enforceQNames := FALSE;   (* enable QName and NCName checks *)
        p. validating := TRUE; (* enable parser-level validation *)
      END;
      IF (res#NIL) & (res.code#f.noSuchFile) THEN
        Err.String("Error loading '"); Err.String(fileName); Err.String("': ");
        res.GetText(error); Err.String(error); Err.Ln;
      END;
      RETURN p
    END CreateParser;

  BEGIN
    COPY(home,fileName);
    str.Append(name,fileName);
    str.Append(".res",fileName);
    canon:=CreateParser();
    IF canon=NIL THEN
      RETURN NIL;
    END;

    canon.ParseDocument;

    IF (canon.errList.msgCount # 0) THEN
      canon.errList.Write (StdChannels.stderr);
      RETURN NIL;
    ELSE
      RETURN prefs.top;
    END
  END LoadXMLPrefsFile;

  PROCEDURE (s : Settings) SaveXMLPrefsToFile(name : ARRAY OF CHAR; top : PP.Item);

  VAR
    file    : f.File;
    writer  : t.Writer;
    res     : f.Result;
    text    : ARRAY 256 OF CHAR;
    current : PP.Item;

  BEGIN
    COPY(home,fileName);
    str.Append(name,fileName);
    str.Append(".res",fileName);
    file:=f.New(fileName,{f.write},res);
    IF file#NIL THEN
      writer:=t.ConnectWriter(file);
      IF writer#NIL THEN
        current:=top;
        WHILE current#NIL DO
          current.PrintXML(writer,0);
          current:=current.next;
          IF current#NIL THEN
            writer.WriteLn;
          END;
        END;
        IF writer.res#Channel.done THEN
          writer.res.GetText(text);
          Err.String("Error saving '"); Err.String(fileName);
          Err.String("': "); Err.String(text); Err.Ln;
        END;
      ELSE
        file.res.GetText(text);
        Err.String("Error saving '"); Err.String(fileName);
        Err.String("': "); Err.String(text); Err.Ln;
      END;
      file.Close;
    ELSE
      res.GetText(text);
      Err.String("Error saving '"); Err.String(fileName);
      Err.String("': "); Err.String(text); Err.Ln;
    END;
  END SaveXMLPrefsToFile;

  PROCEDURE (s : Settings) SavePrefs*;

  VAR
    item  : PrefsItem;
    top,
    block : PP.BlockItem;

  BEGIN
    NEW(top);
    top.Init;
    top.SetName("Config");

    item:=s.itemList;
    WHILE item#NIL DO
      NEW(block);
      block.Init;
      block.SetName(item.name);

      item.SavePrefs(block);

      top.AddItem(block);

      item:=item.next;
    END;

    s.SaveXMLPrefsToFile("VisualOberon",top);
  END SavePrefs;

  PROCEDURE (s : Settings) Receive*(message : O.Message);

  BEGIN
    WITH
      message : B.PressedMsg DO
        CASE message.source.id OF
          closeId:
            IF s.exitIt THEN
              D.display.Exit;
            ELSE
              s.Exit;
            END;
        | useId:
            s.ApplyPrefs;
            D.display.ReinitWindows;
        | saveId:
            s.ApplyPrefs;
            D.display.ReinitWindows;
            s.SavePrefs;
        END;
    | message : W.ExitMsg DO
        IF s.exitIt THEN
          D.display.Exit;
        ELSE
          s.Exit;
        END;
    ELSE
      s.Receive^(message);
    END;
  END Receive;

  PROCEDURE (a : AboutItem) Init*;

  BEGIN
    a.Init^;
    a.name:="About";
  END Init;


  PROCEDURE (a : AboutItem) GetObject*():G.Object;

  BEGIN
    RETURN T.MakeLeftText("\e\c\e\s\e\9VisualOberon\e\p\n\en\n\e\cCopyright 1997-2000\nTim Teulings\n(rael@edge.ping.de)");
  END GetObject;


  PROCEDURE (h : Sel2Set) Convert*(message : O.Message):O.Message;

  VAR
    new : M.SetMsg;

  BEGIN
    NEW(new);
    new.pos:=message(TM.SelectionMsg).y;
    RETURN new;
  END Convert;

  PROCEDURE (s : Settings) LoadPrefs(appName : ARRAY OF CHAR);

  BEGIN
    s.top:=s.LoadXMLPrefsFile("VisualOberon");

    IF s.top=NIL THEN
      Err.String("Cannot load preferences"); Err.Ln;
      RETURN;
    END;

    s.top:=s.top.GetEntry("Config");
  END LoadPrefs;

  PROCEDURE (s : Settings) ReadDisplayPrefs;

  VAR
    item  : PrefsItem;
    entry : PP.Item;

  BEGIN
    IF s.top=NIL THEN
      RETURN;
    END;

    item:=s.itemList;
    WHILE item#NIL DO
      IF item.name="Display" THEN
        entry:=s.top.GetEntry(item.name);
        IF entry#NIL THEN
          item.LoadPrefs(entry);
        END;

        RETURN;
      END;
      item:=item.next;
    END;
  END ReadDisplayPrefs;

  PROCEDURE (s : Settings) ReadOtherPrefs;

  VAR
    item  : PrefsItem;
    entry : PP.Item;

  BEGIN
    IF s.top=NIL THEN
      RETURN;
    END;

    item:=s.itemList;
    WHILE item#NIL DO
      IF item.name#"Display" THEN
        entry:=s.top.GetEntry(item.name);
        IF entry#NIL THEN
          item.LoadPrefs(entry);
        END;
      END;
      item:=item.next;
    END;
  END ReadOtherPrefs;

  PROCEDURE (s : Settings) FreePrefs;

  BEGIN
    s.top:=NIL;
  END FreePrefs;

  PROCEDURE (p : PrefsCallback) LoadPrefs*(appName : ARRAY OF CHAR);

  BEGIN
    settings.LoadPrefs(appName);
  END LoadPrefs;

  PROCEDURE (p : PrefsCallback) ReadDisplayPrefs*;

  BEGIN
    settings.ReadDisplayPrefs;
  END ReadDisplayPrefs;

  PROCEDURE (p : PrefsCallback) ReadOtherPrefs*;

  BEGIN
    settings.ReadOtherPrefs;
  END ReadOtherPrefs;

  PROCEDURE (p : PrefsCallback) FreePrefs*;

  BEGIN
    settings.FreePrefs;
  END FreePrefs;

BEGIN
  Rts.GetUserHome(home,"");
  str.Append("/.VisualOberon/",home);

  NEW(settings);
  settings.Init;

  NEW(prefsCallback);
  D.prefsCallback:=prefsCallback;
END VO:Prefs:GUI.