/***************************************************************************
 $RCSfile: api.cpp,v $
                             -------------------
    cvs         : $Id: api.cpp,v 1.45 2003/07/03 00:05:00 aquamaniac Exp $
    begin       : Sat Jun 08 2002
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "assert.h"

#include "api.h"
#include <hbcistring.h>
#include <accountimpl.h>
#include <mediumrdhbase.h>
#include <openhbci/connection.h>
#include <openhbci/directory.h>
#include "parser.h"
#include "init_plugins.h"


namespace HBCI {


unsigned API::_nextTransactionId=1;


API::API(bool readonly,
	 bool retrievalonly)
  :Hbci(readonly, retrievalonly)
{
  Error err;

  _loader=new Loader(this);
  _loader.setDescription("API::_loader");
  _loader.setObjectDescription("Loader");
  _monitor=new ProgressMonitor();
  _monitor.setDescription("API::_monitor");
  _monitor.setObjectDescription("ProgressMonitor");
  _queue=new Outbox(this);
  _queue.setDescription("API::_queue");
  _queue.setObjectDescription("Outbox");

  err=registerLinkedPlugins(this);
  if (!err.isOk()) {
    fprintf(stderr,"Error registering plugins: %s\n",
	    err.errorString().c_str());
    throw Error("API::API",err);
  }
}


API::~API() {
  //fprintf(stderr,"Destroying API\n");

  // Manually delete the Pointer<Bank> list since we own these
  // objects but due to cross-referencing of Bank<->Account *and*
  // Bank<->User the Bank's reference counters won't reach zero
  // automatically.
  list<Pointer<Bank> >::iterator it;

  for (it=_banks.begin();
       it!=_banks.end();
       it++) {
      // delete object without destroying the admin structures. This keeps
      // the pointer itself valid, but if some pointer tries to use this
      // pointer we will get an exception
      (*it).release();
  }

  _mediumPlugins.clear();

  _pluginFiles.clear();
}


Pointer<ProgressMonitor> API::monitor() const {
  return _monitor;
}


void API::setMonitor(Pointer<ProgressMonitor> monitor){
    _monitor=monitor;
}


void API::addBank(Pointer<Bank> b){
    Pointer<Bank> nb;

    for(list<Pointer<Bank> >::const_iterator iter=_banks.begin();
        iter!=_banks.end(); iter++) {
        /* first check if the very same object already exists.
         * It's not a problem if it does, since it is the SAME.
         */
        if (*iter==b) {
#if DEBUGMODE>0
            fprintf(stderr,
                    "Will not add bank, since it already IS listed.\n");
#endif
            return;
        }
        /* then check if there already is an object with the same id data.
         * This WOULD be a problem, since that would corrupt the list.
         * The second occurence of an equal account could rarely be accessed.
         */
        if ((*iter).ref().countryCode()==b.ref().countryCode())
            if ((*iter).ref().bankCode()==b.ref().bankCode())
	      throw Error("BankImpl::addBank()",
			  ERROR_LEVEL_NORMAL,
			  HBCI_ERROR_CODE_EXISTS,
			  ERROR_ADVISE_DONTKNOW,
			  "bank already exists.");
    } // for

    // otherwise simply add the bank
    nb=b;
    nb.setDescription("Entry of API::_banks");
    _banks.push_back(nb);
}


Pointer<Bank> API::findBank(int country,
                                        const string &instcode) const {
    for(list<Pointer<Bank> >::const_iterator iter=_banks.begin();
        iter!=_banks.end(); iter++) {
        if ((*iter).ref().countryCode()==country)
            if ((*iter).ref().bankCode()==instcode)
                return (*iter);
    }
    return 0;
}


Pointer<Account> API::findAccount(int country,
                                              const string &instcode,
                                              const string &accnr,
                                              const string &suffix) const {
  // ok, this is crappy:
  // there might be accounts that have another institute id than the 
  // bank they belong to (Hypovereinsbank) !?!
  // So we need to loop over all banks, check if any of their accounts
  // match "accnr"
  for(list<Pointer<Bank> >::const_iterator iter=_banks.begin();
	  iter!=_banks.end(); iter++) {

	Pointer<Account> account = 
	  (*iter).ref().findAccount(accnr, suffix);
	
	if (account.isValid() &&
	    account.cast<AccountImpl>().ref().instituteCode() == instcode &&
	    account.cast<AccountImpl>().ref().countryCode() == country) {
	  return account;
	}
  }

  // not found
  return NULL;
}


Pointer<User> API::findUser(int country,
                                        const string &instcode,
                                        const string &user) const {
    Pointer<Bank> bank;

    bank=findBank(country, instcode);
    if (!bank.isValid())
        return 0;
    return bank.ref().findUser(user);
}


Pointer<Customer> API::findCustomer(int country,
                                                const string &instcode,
                                                const string &cust) const {
    Pointer<Bank> bank;

    bank=findBank(country, instcode);
    if (!bank.isValid())
        return 0;
    return bank.ref().findCustomer(cust);
}

int API::totalAccounts () const 
{
    list<Pointer<Bank> >::const_iterator iter, end;
    int ret = 0;

    end = banks().end();
    for ( iter = banks().begin(); iter != end; iter++ ) {
	ret += (*iter).ref().accounts().size();
    }
    return ret;
}
int API::totalUsers () const 
{
    list<Pointer<Bank> >::const_iterator iter, end;
    int ret = 0;

    end = banks().end();
    for ( iter = banks().begin(); iter != end; iter++ ) {
	ret += (*iter).ref().users().size();
    }
    return ret;
}


Error API::loadEnvironment(const string &f,
			   unsigned int fl){
    SimpleConfig cfg;
    Error err;

    cfg.setMode(HBCIAPP_CONFIG_MODE);
    err=cfg.readFile(f);
    if (!err.isOk())
        return err;
    err=_loader.ref().loadAll(cfg,cfg.root(),fl);

    return err;
}


Error API::saveEnvironment(const string &f,
			   unsigned int fl) const {
    SimpleConfig cfg;
    Error err;

    cfg.setMode(HBCIAPP_CONFIG_MODE);
    err=_loader.ref().saveAll(cfg,cfg.root(),fl);
    if (!err.isOk())
        return err;
    err=cfg.writeFile(f);
    if (!err.isOk())
        return err;

    return err;
}


Error API::postProcessInitJob(JOBDialogInit &job) {
    Pointer<userParams> upd;
    Pointer<bankParams> bpd;
    Pointer<Bank> bank;
    Pointer<Customer> cust;
    Pointer<RSAKey> signKey;
    Pointer<RSAKey> cryptKey;

    cust=job.owner();
    User &user = cust.ref().user().ref();

    bpd = job.bpd();
    upd = job.upd();
    signKey = job.serverSignKey();
    cryptKey = job.serverCryptKey();

    // check for BPD
    if (bpd.isValid()) {
        // check if the bank already exists
        bank=findBank(bpd.ref().countryCode(),
                      bpd.ref().instituteCode());
        if (bank.isValid()) {
            // it does, so set the new values
            // do not overwrite server (WORKAROUND)
            bpd.ref().setAddr(bank.ref().addr());
            dynamic_cast<BankImpl&>(bank.ref())=bpd.ref();
        }
        else {
            // does not exists, add it
            addBank(new BankImpl(this,bpd.ref()));
        }
    }
    // check for keys
    if (cryptKey.isValid())
        cust.ref().user().ref().medium().cast<MediumRDHBase>().ref()
            .setInstituteCryptKey(cryptKey);
    if (signKey.isValid())
        cust.ref().user().ref().medium().cast<MediumRDHBase>().ref()
            .setInstituteSignKey(signKey);
    job.clearKeys();

    // check for UPD
    if (upd.isValid()) {
        list<accountParams> accs;
        list<accountParams>::const_iterator ait;
        Pointer<Account> acc;
        Pointer<AccountImpl> aimpl;

        bank=user.bank();
        user.setVersion(upd.ref().version());
		user.setKnowsSupportedJobs(upd.ref().knowsSupportedJobs());

        accs=upd.ref().accounts();
        for (ait=accs.begin(); ait!=accs.end(); ait++) {
            acc=bank.ref().findAccount((*ait).accountNumber(),
                                       (*ait).accountSuffix());
            if (acc.isValid()) {
                // account exists, so set new basic params
		// Downcast of existing account
		aimpl = acc.cast<AccountImpl>();
		aimpl.ref() = *ait;
                //dynamic_cast<AccountImpl&>(acc.ref())=*ait;
                aimpl.ref().addAuthorizedCustomer(cust);
            }
            else {
                // otherwise simply add it
                Pointer<Account> acc;
                aimpl=new AccountImpl(bank,*ait);
                aimpl.ref().addAuthorizedCustomer(cust);
		// Upcast to store only the reference to the base
		// class.
		acc = aimpl.cast<Account>();
                acc.setObjectDescription("Account");
                bank.ref().addAccount(acc);
            }
        } // for
    }

    return Error();
}


Error
API::processInstituteMessages(Pointer<MessageQueue> mbox){
    list<instituteMessage>::const_iterator it;
    Pointer<Bank> bank;
    bool iserr=false;

    // check all messages and copy them to the appropriate bank
    for (it=mbox.ref().bankMessages().begin();
         it!=mbox.ref().bankMessages().end();
         it++) {
        bank=findBank((*it).country(),(*it).bankCode());
        if (!bank.isValid())
            iserr=true;
        else {
            // store message in the bank
            BankImpl &ba = dynamic_cast<BankImpl&> (bank.ref());
            ba.addInstituteMessage(*it);
        }
    } // for
    if (iserr)
        return Error("API::processInstituteMessages()",
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_INEXISTENT,
		     ERROR_ADVISE_DONTKNOW,
		     "bank does not exist.");
    else
        return Error();
}


Error API::_handleMessageQueue(Pointer<Connection> conn,
                                       Pointer<MessageQueue> mbox){
    Error err;

    try {
        if (mbox.ref().empty()) {
            // empty queue, finished ;-)
            _monitor.ref().actionStarted(ACT_SENDINGMESSAGE,
					 "Sending message (nothing to do)");
            _monitor.ref().actionFinished();
            _monitor.ref().actionStarted(ACT_WAITRESPONSE,
					 "Waiting for response (nothing to do)");
            _monitor.ref().actionFinished();
            return Error();
        }

        _monitor.ref().actionStarted(ACT_SENDINGMESSAGE,
				     "Sending message");
        // send this message
        if (!conn.ref().sendMessage(mbox))
	    return Error("API::_handleMessageQueue()",
			 ERROR_LEVEL_INTERNAL,
			 HBCI_ERROR_CODE_SOCKET_ERROR_UNKNOWN,
			 ERROR_ADVISE_ABORT,
			 "could not send");
	_monitor.ref().actionFinished();
        _monitor.ref().actionStarted(ACT_WAITRESPONSE,
				     "Waiting for response");
        // await response
        if (!conn.ref().getResponse())
            return Error("API::_handleMessageQueue()",
                         ERROR_LEVEL_INTERNAL,
			 HBCI_ERROR_CODE_SOCKET_ERROR_UNKNOWN,
                         ERROR_ADVISE_ABORT,
                         "nothing received");
        _monitor.ref().actionFinished();
    }
    catch(Error xerr) {
        err=xerr;
    }
    if (!err.isOk())
        _logMessage(1,"RESULT (handleMessage): "+err.errorString());
    //fprintf(stderr,"RESULT (handleMessage): %s\n",
    //        err.errorString().c_str());
    return err;
}


void API::_logMessage(int loglevel, const string &msg) {
    if (_monitor.isValid()) {
        if (loglevel<debugLevel())
            _monitor.ref().logMessage(msg);
    }
}


Error API::_handleJobQueue(const list<Pointer<OutboxJob> > &jq,
                                   Pointer<Connection> conn,
                                   Pointer<MessageQueue> mbox,
                                   bool dlg,
                                   bool chg){
    list<Pointer<OutboxJob> >::const_iterator it;
    Error err;
    int jobErrors=0;

    // handle the jobs
    for (it=jq.begin(); it!=jq.end(); it++) {
#if DEBUGMODE>0
        fprintf(stderr,"HANDLE_QUEUE: job is %s\n",
                (*it).ref().description().c_str());
#endif
        (*it).ref().setMessageReference(mbox.ref().messageReference());
        if ((*it).ref().isDialogJob()==dlg &&
            (*it).ref().status()==HBCI_JOB_STATUS_TODO) {
            _monitor.ref().jobStarted((*it).ref().type(),
				      (*it).ref().description(),
				      (*it).ref().messages()*3+(chg?2:1));

            // let the job create its messages and send them
            int i=-1;
            while ((*it).ref().stillMessagesToSend(++i)) {
                // create real jobs and add them to the message queue
                _monitor.ref().actionStarted(ACT_CREATEHBCIJOB,
					     "Creating hbci jobs "+
                                             String::num2string(i));
                if (!(*it).ref().createHBCIJobs(mbox,i)) {
                    // error on creation, break
                    jobErrors++;
                    break;
                }
                _monitor.ref().actionFinished();

                // now send the message queue to the server
                err=_handleMessageQueue(conn,mbox);
                if (!err.isOk())
                    return err;

                // during a dialog-job, new data may be returned (upd, bpd,
                // signaturekeys, system-id...)
                // some of them are necessary in the dialog-end-message
                // to finish the dialog correctly
                // if we are allowed to we let the job change this data
                // within the current context
                if (chg)
                    (*it).ref().commit(i);

                // reset mbox for next use
                mbox.ref().reset();
            } // while

            // all jobs handled ?
            if ((*it).ref().stillMessagesToSend(i))
                // no, stop handling this queue
                break;

            // mark job as done
            (*it).ref().setStatus(HBCI_JOB_STATUS_DONE);

            // evaluate this job
            _monitor.ref().actionStarted(ACT_CHKRESULT,
					 "Checking job result.");
            if (!(*it).ref().evaluate())
                jobErrors++;
            _monitor.ref().actionFinished();

            // commit changes if wanted
            if (chg) {
                _monitor.ref().actionStarted(ACT_UPDATESYSTEM,
					     "Updating system.");
                if (!(*it).ref().commit(-1))
                    jobErrors++;
                _monitor.ref().actionFinished();
            }
            _monitor.ref().jobFinished();
        }
    } // for

    if (jobErrors>0)
        return ( (jobErrors == 1) ?
		 err :
		 Error("API::_handleJobQueue()",
		       ERROR_LEVEL_NORMAL,
		       1,
		       ERROR_ADVISE_DONTKNOW,
		       "more than one job failed"));
    return Error();
}


Error API::_handleUserQueue(Pointer<customerQueue> q,
                                    Pointer<Connection> conn,
                                    bool chg){
    list<Pointer<OutboxJob> > jq;
    Error err;
    int jobErrors=0;
    int j;
    Pointer<Customer> cust;
    Pointer<MessageQueue> mbox;
    JOBDialogInit *initJobImpl;
    Pointer<Job> initJob;
    Pointer<Job> exitJob;

    // get queue size
    j=q.ref().sizeByStatus(HBCI_JOB_STATUS_TODO);
    if (!j)
        // nothing to be done
        return Error();

    // get corresponding customer
    cust=q.ref().customer();
    if (!cust.isValid())
        return Error("API::_handleUserQueue()",
		     ERROR_LEVEL_INTERNAL,
		     HBCI_ERROR_CODE_INEXISTENT,
		     ERROR_ADVISE_ABORT,
		     "no customer in customerQueue");

    if (Hbci::debugLevel()>2)
	fprintf(stderr,"HANDLEUSERQUEUE: customer is %s\n",
		cust.ref().custId().c_str());

    jq=q.ref().jobs();
    mbox=new MessageQueue(cust);

    // first handle all dialog jobs
    err=_handleJobQueue(jq,conn,mbox,true,chg);
    if (!err.isOk()) {
	jobErrors++;
	if (err.advise()==ERROR_ADVISE_ABORT)
            return err;
    }

    // get queue size
    j=q.ref().sizeByStatus(HBCI_JOB_STATUS_TODO);
    if (!j)
        // nothing to be done
        return Error();

    // otherwise handle non-dialog jobs
    _monitor.ref().jobStarted(JOB_OPENINGDIALOG,
			      "Opening dialog",chg ? 3:2);

    // create non anonymous dialog init job
    initJobImpl=new JOBDialogInit(cust,
				  false,  // anon
				  true,   // crypt
				  true,   // sign
				  false,  // getkeys
				  false); // sync
    initJob = initJobImpl;
    mbox.ref().addJob(initJob);

    err=_handleMessageQueue(conn,mbox);
    if (!err.isOk())
        return err;

    /* commit possibly pending changes inflicted by the servers answer
     * only if wanted */
    if (chg) {
        _monitor.ref().actionStarted(ACT_UPDATESYSTEM,
				     "Updating system.");
        err=postProcessInitJob(*initJobImpl);
        if (!err.isOk())
            return err;
        // handle institute messages
        err=processInstituteMessages(mbox);
        if (!err.isOk())
            return err;
        _monitor.ref().actionFinished();
    }
    _monitor.ref().jobFinished();

    // reset mbox for next use
    mbox.ref().reset();

    // now handle normal jobs
    err=_handleJobQueue(jq,conn,mbox,false,chg);
    if (!err.isOk()) {
	jobErrors++;
	if (err.advise()==ERROR_ADVISE_ABORT)
	    return err;
    }

    // create exit job
    exitJob=new JOBDialogEnd(cust,mbox.ref().dialogId());
    mbox.ref().addJob(exitJob);

    _monitor.ref().jobStarted(JOB_CLOSINGDIALOG,
			      "Closing dialog",2);
    // send this message
    err=_handleMessageQueue(conn,mbox);
    _monitor.ref().jobFinished();
    if (jobErrors>0)
        return ( (jobErrors == 1) ?
		 err :
		 Error("API::_handleUserQueue()",
		       ERROR_LEVEL_NORMAL,
		       1,
		       ERROR_ADVISE_DONTKNOW,
		       "more than one job failed"));
    return err;
}


Error API::_handleBankQueue(Pointer<bankQueue> q, bool chg){
    list<Pointer<customerQueue> > cq;
    list<Pointer<customerQueue> >::const_iterator it;
    Pointer<Connection> conn;
    int j;
    Pointer<Bank> bank;
    Error err;
    int jobErrors=0;
    int connectionErrors=0;

    // get queue size
    j=q.ref().sizeByStatus(HBCI_JOB_STATUS_TODO);
    if (!j)
        // nothing to be done
        return Error();

    // get corresponding bank
    bank=q.ref().bank();
    if (!bank.isValid())
        return Error("API::_handleBankQueue()",
		     ERROR_LEVEL_INTERNAL,
		     HBCI_ERROR_CODE_INEXISTENT,
		     ERROR_ADVISE_ABORT,
		     "no bank in bankQueue");
#if DEBUGMODE>0
    fprintf(stderr,"HANDLEBANKQUEUE: bank is %s\n",
	    bank.ref().bankCode().c_str());
#endif

    cq=q.ref().customerQueues();
    for (it=cq.begin(); it!=cq.end(); it++) {
        _monitor.ref().jobStarted(JOB_OPENINGNETWORK,
				  "Opening network connection",1);
        _monitor.ref().actionStarted(ACT_CONTACTINGSERVER,
				     "Contacting server "+
                                     bank.ref().addr());
        conn=new Connection(this,bank.ref().addr());
	err = conn.ref().open();
	if ( !err.isOk() ){
            _logMessage(1,"Could not connect to "+bank.ref().name()+
                        " ("+bank.ref().bankCode()+")");
            // skip all jobs in this queue
            connectionErrors++;
            break;
        } // if !open
        _monitor.ref().actionFinished();
        _monitor.ref().jobFinished();

        // handle user jobs
        err=_handleUserQueue(*it, conn, chg);
        if (!err.isOk()) {
            _logMessage(1,"HandleUserQueue: "+err.errorString());
            jobErrors++;
	    if (err.advise()==ERROR_ADVISE_ABORT)
		break;
	}
        // close physical connection
        _monitor.ref().jobStarted(JOB_CLOSINGNETWORK,
				  "Closing network connection",1);
        _monitor.ref().actionStarted(ACT_CLOSECONNECTION,
				     "Closing connection");
        if (!conn.ref().close())
            connectionErrors++;
        _monitor.ref().actionFinished();
        _monitor.ref().jobFinished();
    } // for

    // check for errors
    if (jobErrors>0)
        return ( (jobErrors == 1) ?
		 err :
		 Error("API::_handleBankQueue()",
		       ERROR_LEVEL_NORMAL,
		       1,
		       ERROR_ADVISE_DONTKNOW,
		       "more than one job failed"));
    if (connectionErrors > 0)
        return ( (connectionErrors == 1) ?
		 err :
		 Error("API::_handleBankQueue()",
		       ERROR_LEVEL_NORMAL,
		       2,
		       ERROR_ADVISE_DONTKNOW,
		       "some connections failed"));
    return Error();
}


Error API::_handleQueue(Pointer<Outbox> q, bool chg){
    list<Pointer<bankQueue> > bq;
    list<Pointer<bankQueue> >::const_iterator it;
    int j;
    int jobErrors=0;
    Error err;

    // get queue size
    j=q.ref().sizeByStatus(HBCI_JOB_STATUS_TODO);
    if (!j)
        // nothing to be done
        return Error();
    bq=q.ref().bankQueues();

    // add jobs for connection opening and closing
    j+=q.ref().bankCount()*2;

    // add jobs for dialog opening and closing
    if (!q.ref().allDialogJobs())
	j+=q.ref().customerCount()*2;

    _monitor.ref().transactionStarted(TRANS_JOBQUEUE,
				      "Executing job queue",j);
    for (it=bq.begin(); it!=bq.end(); it++) {
        err=_handleBankQueue(*it,chg);
        if (!err.isOk()){
            _logMessage(1,"Bank: "+err.errorString());
            jobErrors++;
        }
    }

    _monitor.ref().transactionFinished();
    
    if (jobErrors>0)
        return ( (jobErrors == 1) ?
		 err :
		 Error("API::_handleQueue()",
		       ERROR_LEVEL_NORMAL,
		       1,
		       ERROR_ADVISE_DONTKNOW,
		       "more than one job failed"));
    return Error();
}


Error API::addJob(Pointer<OutboxJob> job){
    _queue.ref().addJob(job);

    return Error();
}


Error API::executeQueue(bool chg){
    Error err;

    err= _handleQueue(_queue,chg);
    return err;
}


list<Pointer<Account> > 
API::getAccounts(int country,
		 const string &instid,
		 const string &accnr) const {
  list<Pointer<Bank> >::const_iterator it;
  list<Pointer<Account> > accs;
  list<Pointer<Account> >::const_iterator ait;;

  for (it=_banks.begin(); it!=_banks.end();it++) {
    if ((country==0 || (*it).ref().countryCode()) &&
	(-1!=parser::cmpPattern((*it).ref().bankCode(),
				instid,false))) {
      // add matching accounts to mine
      for (ait=(*it).ref().accounts().begin();
	   ait!=(*it).ref().accounts().end(); ait++) {
	string n1, n2;
	unsigned int i, j;

	n1=(*ait).ref().accountId();
	j=n1.length();
	for (i=0; i<j; i++)
	  if (n1[i]!='0')
	    break;
	if (i)
	  n1=n1.substr(i);
        n2=string(10-n1.length(),'0')+n1;

	if (-1!=parser::cmpPattern((*ait).ref().accountId(),
				   accnr,false) ||
	    -1!=parser::cmpPattern(n1,
				   accnr,false) ||
	    -1!=parser::cmpPattern(n2,
				   accnr,false))
	  accs.push_back(*ait);
      } // for
    } // if bank matches
  } // for
  return accs;
}


list<Pointer<User> > 
API::getUsers(int country,
                  const string &instid,
                  const string &userid) const {
    list<Pointer<Bank> >::const_iterator it;
    list<Pointer<User> > custs;
    list<Pointer<User> >::const_iterator cit;;

    for (it=_banks.begin(); it!=_banks.end();it++) {
        if ((country==0 || (*it).ref().countryCode()) &&
            (-1!=parser::cmpPattern((*it).ref().bankCode(),
                                      instid,false))) {
            // add matching customers to mine
            for (cit=(*it).ref().users().begin();
                 cit!=(*it).ref().users().end(); cit++)
                if (-1!=parser::cmpPattern((*cit).ref().userId(),
                                             userid,false))
                    custs.push_back(*cit);
        } // if bank matches
    } // for
    return custs;
}


list<Pointer<Customer> >
API::getCustomers(int country,
                      const string &instid,
                      const string &custid) const {
    list<Pointer<Bank> >::const_iterator it;
    list<Pointer<User> >::const_iterator uit;;
    list<Pointer<Customer> >::const_iterator cit;;
    list<Pointer<Customer> > custs;

    // iterate through banks
    for (it=_banks.begin(); it!=_banks.end();it++) {
        if ((country==0 || (*it).ref().countryCode()) &&
            (-1!=parser::cmpPattern((*it).ref().bankCode(),
                                      instid,false))) {
            // iterate through users
            for (uit=(*it).ref().users().begin();
                 uit!=(*it).ref().users().end(); uit++) {
                // add matching customers to mine
                for (cit=(*uit).ref().customers().begin();
                     cit!=(*uit).ref().customers().end(); cit++)
                    if (-1!=parser::cmpPattern((*cit).ref().custId(),
                                                 custid,false))
                        custs.push_back(*cit);
            } // for user
        } // if bank matches
    } // for bank
    return custs;
}


Pointer<Medium> API::findMedium(const string &name) const {
  list<Pointer<Bank> >::const_iterator bit;
  list<Pointer<User> >::const_iterator cit;
  Pointer<Medium> medium;

  for (bit=_banks.begin(); bit!=_banks.end();bit++) {
    // add matching accounts to mine
    for (cit=(*bit).ref().users().begin();
	 cit!=(*bit).ref().users().end(); cit++) {
      medium=(*cit).ref().medium();
      if (medium.isValid())
	if (medium.ref().mediumName()==name)
	  return medium;
    } // for

  } // for

  // not found
  return 0;
}


Pointer<MediumPlugin> API::_findMediumPlugin(const string &name) {
  list<Pointer<MediumPlugin> >::iterator it;

  for (it=_mediumPlugins.begin();
       it!=_mediumPlugins.end();
       it++)
    if (strcasecmp((*it).ref().mediumTypeName().c_str(), name.c_str())==0)
      return *it;

  return 0;
}


Error API::registerMediumPlugin(Pointer<MediumPlugin> mp){
  Pointer<MediumPlugin> tmp;

  // check if a plugin of that type already exists
  tmp=_findMediumPlugin(mp.ref().mediumTypeName());
  if (tmp.isValid())
    return Error("BankImpl::addBank()",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_EXISTS,
		 ERROR_ADVISE_DONTKNOW,
		 "medium plugin already exists.");
  _mediumPlugins.push_back(mp);
  return Error();
}


string API::_getLibraryPrefix(const string &path) {
  string prefix;
  unsigned int pp;

  // get basename
  pp=path.rfind("/");
  if (pp==string::npos)
    prefix=path;
  else
    prefix=path.substr(pp+1);
  pp=prefix.rfind(PLUGIN_EXT);
  if (pp!=string::npos)
    prefix=prefix.substr(0,pp);
  return prefix;
}


Error API::loadMediumPlugin(const string &path) {
  Pointer<MediumPluginFile> f;
  Error err;

  try {
    f=new MediumPluginFile(path, _getLibraryPrefix(path));
  }
  catch(HBCI::Error xerr) {
    return Error("API::loadMediumPlugin",xerr);
  }

  err=f.ref().registerYourSelf(this);
  if (!err.isOk())
    return Error("API::loadMediumPlugin",err);

  _pluginFiles.push_back(f);
  if (debugLevel()>0)
    fprintf(stderr," Loaded plugin \"%s\"\n", path.c_str());
  return Error();
}


Error API::_tryToLoadPlugin(const list<string> &dnames, const string &mtype) {
  int soversion;
  list<string>::const_iterator sit;
  Error err;

  for (soversion=OPENHBCI_SO_VERSION;
       soversion>=OPENHBCI_SO_VERSION-OPENHBCI_SO_AGE;
       soversion--) {
    for (sit=dnames.begin();
	 sit!=dnames.end();
	 sit++) {
      if (debugLevel()>1)
	fprintf(stderr, "Checking directory \"%s\"\n", (*sit).c_str());

      // try to load the plugin
      string mpath;
      string t;
      unsigned int i;

      mpath=*sit;
      mpath+="/";
      mpath+=String::num2string(soversion);
      mpath+="/media/";
      t=mtype;
      for (i=0; i<t.length(); i++)
	t[i]=tolower(t[i]);
      mpath+=t;
      mpath+=PLUGIN_EXT;
      err=loadMediumPlugin(mpath);
      if (err.isOk()) {
	return Error();
      }
      else {
	if (debugLevel()>0) {
	  fprintf(stderr,
		  "API::_tryToLoadPlugin: Error loading plugin (%s)\n",
		  err.errorString().c_str());
	}
      }
    } // for
  } // for

  return Error("API::tryToLoadPlugin",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_INVALID,
	       ERROR_ADVISE_DONTKNOW,
	       "medium type not found");
}


Pointer<MediumPlugin> API::_ensureMediumPlugin(const string &mtype) {
  Pointer<MediumPlugin> tmp;

  tmp=_findMediumPlugin(mtype);
  if (!tmp.isValid()) {
#ifdef PLUGINLOADING_SUPPORTED
    Error err;
    list<string> dnames;

    dnames.push_back(PLUGIN_PATH);
    dnames.push_back("/usr/share/openhbci/plugins");
    dnames.push_back("/usr/local/share/openhbci/plugins");

    err=_tryToLoadPlugin(dnames, mtype);
    if (!err.isOk()) {
      throw Error("API::mediumFromConfig",
		  ERROR_LEVEL_NORMAL,
		  HBCI_ERROR_CODE_INVALID,
		  ERROR_ADVISE_DONTKNOW,
		  "medium type not found");
    }

    tmp=_findMediumPlugin(mtype);
    if (!tmp.isValid()) {
      throw Error("API::mediumFromConfig",
		  ERROR_LEVEL_NORMAL,
		  HBCI_ERROR_CODE_INVALID,
		  ERROR_ADVISE_DONTKNOW,
		  "medium type not found");
    }
#else  // ifdef PLUGINLOADING_SUPPORTED
    throw Error("API::mediumFromConfig",
		ERROR_LEVEL_NORMAL,
		HBCI_ERROR_CODE_INVALID,
		ERROR_ADVISE_DONTKNOW,
		"medium type not found");
#endif // ifdef PLUGINLOADING_SUPPORTED
  }

  return tmp;
}


Pointer<Medium> API::mediumFromConfig(SimpleConfig &cfg,
				      cfgPtr group){
  Pointer<MediumPlugin> tmp;
  string mtype;
  Error err;

  mtype=cfg.getVariable("mediumtype","", group);
  if (mtype.empty())
    throw Error("API::mediumFromConfig",
		ERROR_LEVEL_NORMAL,
		HBCI_ERROR_CODE_INVALID,
		ERROR_ADVISE_DONTKNOW,
		"no medium type given");

  tmp=_ensureMediumPlugin(mtype);
  return tmp.ref().mediumFromConfig(cfg, group);
}


Error API::mediumToConfig(Pointer<Medium> m,
			  SimpleConfig &cfg,
			  cfgPtr group){
  Pointer<MediumPlugin> tmp;

  //fprintf(stderr,"Saving medium %s\n", m.ref().mediumTypeName().c_str());
  tmp=_findMediumPlugin(m.ref().mediumTypeName());
  if (!tmp.isValid()) {
    return Error("API::mediumToConfig",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "medium type not found");
  }

  return tmp.ref().mediumToConfig(m, cfg, group);
}


MediumType API::mediumType(const string &mtype) {
  Pointer<MediumPlugin> tmp;

  tmp=_ensureMediumPlugin(mtype);
  return tmp.ref().mediumType();

}


Pointer<Medium> API::mediumFactory(const string &name,
				   const string &mtype) {
  Pointer<MediumPlugin> tmp;

  tmp=_ensureMediumPlugin(mtype);
  return tmp.ref().mediumFactory(name);
}


Pointer<Medium> API::createNewMedium(const string &mtype,
                                     bool readonly,
				     int country,
				     const string &bankId,
				     const string &userId,
				     const string &name){
  Pointer<MediumPlugin> tmp;

  // first check arguments
  if (bankId.empty() || userId.empty() || mtype.empty())
    throw Error("API::createNewMedium()",
		ERROR_LEVEL_NORMAL,
		HBCI_ERROR_CODE_INVALID,
		ERROR_ADVISE_DONTKNOW,
		"empty argument");

  tmp=_ensureMediumPlugin(mtype);
  return tmp.ref().createNewMedium(readonly, country, bankId, userId, name);
}


Pointer<User> API::userFactory(Pointer<Bank> b,
			       Pointer<Medium> m,
			       const string &userid,
			       int version,
			       const string &userName,
			       bool knowsSupportedJobs) {
  return new User(b,m,userid,version, userName, knowsSupportedJobs);
}


Pointer<Customer> API::customerFactory(Pointer<User> u,
				       const string &id,
				       const string &role) {
    return new Customer(u,id,role);
}


Pointer<Bank> API::bankFactory(int country,
                               const string &bankCode,
                               const string &server,
                               int hbciVersion) const {
  Pointer<Bank> newBank = 
	new BankImpl(this,country,bankCode,server,hbciVersion);

  // set the default-language to german
  newBank.ref().setLanguage(HBCI_LANGUAGE_GERMAN);

  return newBank;
}


void API::removeBank(Pointer<Bank> b) {
    list<Pointer<Bank> >::iterator it1;

    // find bank in list and remove it
    for(it1=_banks.begin();
        it1!=_banks.end(); it1++) {
        if ((*it1)==b) {
            _banks.erase(it1);
            break;
        }
    } // for
}



Pointer<Account>
API::accountFactory(Pointer<Bank> b,
                    const string &accountId,
                    const string &accountSubId) {
    return new AccountImpl(b,accountId,accountSubId);
}


list<Pointer<OutboxJob> > API::queuedJobs() const {
    return _queue.ref().jobs();
}


void API::clearQueueByResult(OutboxJob_Result result) {
    _queue.ref().removeByResult(result);
}


void API::clearQueueByStatus(OutboxJob_Status status){
    _queue.ref().removeByState(status);
}


void API::removeQueuedJob(Pointer<OutboxJob> job){
    _queue.ref().removeJob(job);
}


string API::pluginPath() const {
    return PLUGIN_PATH;
}


void API::_addPluginsFromDirs(Pointer<MediumPluginList> pl,
			      const list<string> &dnames) {
  Pointer<Directory> dir;
  string path;
  string tmp;
  string entry;
  Error err;
  bool goon;
  int soversion;
  list<string>::const_iterator sit;


  for (soversion=OPENHBCI_SO_VERSION;
       soversion>=OPENHBCI_SO_VERSION-OPENHBCI_SO_AGE;
       soversion--) {
    for (sit=dnames.begin();
	 sit!=dnames.end();
	 sit++) {
      if (debugLevel()>1)
	fprintf(stderr, "Checking directory \"%s\"\n", (*sit).c_str());
      path=*sit;
      path+="/";
      path+=String::num2string(soversion);
      path+="/media";

      dir=new Directory(path);
      err=dir.ref().openDirectory();
      if (!err.isOk()) {
	if (debugLevel()>0)
	  fprintf(stderr, "Error opening directory \"%s\" (%s)\n",
		  path.c_str(),
		  err.errorString().c_str());
	continue;
      }

      goon=true;
      while(goon) {
	err=dir.ref().readEntry(entry);
	if (!err.isOk())
	  goon=false;
	else {
	  if (-1!=parser::cmpPattern(entry,
				     "*"PLUGIN_EXT,
				     false)) {
	    Pointer<MediumPluginFile> mpf;

	    try {
	      tmp=path+"/"+entry;
	      mpf=new MediumPluginFile(tmp, _getLibraryPrefix(entry));
	      if (debugLevel()>0)
		fprintf(stderr, "Loaded pluginfile \"%s\"\n",
			tmp.c_str());
	      err=pl.ref().addPluginFile(this, mpf);
	      if (!err.isOk()) {
		fprintf(stderr, "Error adding pluginfile \"%s\" (%s)\n",
			entry.c_str(),
			err.errorString().c_str());
	      }
	    }
	    catch (Error xerr) {
	      err=xerr;
	      fprintf(stderr, "Error opening pluginfile \"%s\" (%s)\n",
		      entry.c_str(),
		      xerr.errorString().c_str());
	    }
	    if (err.isOk()) {
	      if (debugLevel()>1)
		fprintf(stderr,"Handled pluginfile \"%s\"\n",
			entry.c_str());
	    }
	  } // if filename ends with system EXT pattern
	}
      } // while

      err=dir.ref().closeDirectory();
      if (!err.isOk()) {
	fprintf(stderr, "Error closing directory \"%s\" (%s)\n",
		path.c_str(),
		err.errorString().c_str());
      }
    } // for dnames
  } // for versions

  return;
}



Pointer<MediumPluginList> API::enumerateMediumPlugins() {
  Pointer<MediumPluginList> pl;
  list<Pointer<MediumPlugin> >::iterator it;
  list<string> dnames;

  pl=new MediumPluginList();

  /* add all plugins we already know */
  for (it=_mediumPlugins.begin(); it!=_mediumPlugins.end(); it++)
    pl.ref().addPlugin(this, *it);

  /* check for loadable media, if supported */
#ifdef PLUGINLOADING_SUPPORTED
  /* */

  dnames.push_back(PLUGIN_PATH);
  dnames.push_back("/usr/share/openhbci/plugins");
  dnames.push_back("/usr/local/share/openhbci/plugins");
  _addPluginsFromDirs(pl, dnames);
#endif // #ifdef PLUGINLOADING_SUPPORTED

  return pl;
}


unsigned API::nextTransactionId(){
  return _nextTransactionId++;
}


unsigned API::lastTransactionId(){
  return _nextTransactionId;
}

void API::setLastTransactionId(unsigned int i){
  _nextTransactionId=i;
}







} // namespace HBCI


