/***************************************************************************
                          hbcistring.cpp  -  description
                             -------------------
    begin                : Sun Jul 1 2001
    copyright            : (C) 2001 by fabian kaiser
    email                : fabian.kaiser@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "hbcistring.h"
#include "error.h"
#include "value.h"

#include <stdlib.h>
#include <stdio.h>
#include <ctime>
#include <cctype>

namespace HBCI {

String::String(){
}


String::~String(){
}


string String::nextDE(const string &data, int startIndex) {
    unsigned int i;
    bool found = false;
    bool esc=false;
    unsigned char c;

    for (i=startIndex; i<data.length(); i++) {
        c = data.at(i);
        if (esc)
            esc=false;
        else {
            switch (c) {
            case '?':  // escape !!
                esc=true;
                break;
            case '\\': // escape
                esc=true;
                break;
            case '@': // start of binary-data
                int startOfBinaryData;
                startOfBinaryData = data.find('@', i+1) + 1;
                i = startOfBinaryData + lengthOfBinaryData(data, i);
                i--;
                break;
            case '\'': // segment-end
                found = true;
                break;
            case '+': // dataelement-end
                found = true;
                break;
            default:
                break;
            }

            if (found) break;
        }
    } // for

    if (found || (i == data.length()))
        return data.substr(startIndex, i-startIndex);
    else
        return "";
}


string String::nextDEG(const string &data, int startIndex) {
    unsigned int i;
    bool found = false;
    unsigned char c;
    bool esc=false;

    for (i=startIndex; i<data.length(); i++) {
        c = data.at(i);

        if (esc)
            esc=false;
        else {
            switch (c) {
            case '?':  // escape !!
                esc=true;
                break;
            case '\\': // escape
                esc=true;
                break;
            case '@': // start of binary-data
                int startOfBinaryData;
                startOfBinaryData = data.find('@', i+1) + 1;
                i = startOfBinaryData + lengthOfBinaryData(data, i);
                i--;
                break;
            case '\'': // segment-end
                found = true;
                break;
            case '+': // dataelement-end
                found = true;
                break;
            case ':': // dataelementgroup-end
                found = true;
                break;
            default:
                break;
            }

            if (found) break;
        }
    } // for

    if (found || (i == data.length()))
        return data.substr(startIndex, i-startIndex);
    else
        return "";
}


string String::nextSEG(const string &data, int startIndex) {
    unsigned int i;
    bool found = false;
    unsigned char c;
    string result;
    bool esc=false;

    for (i=startIndex; i<data.length(); i++) {
        c = data.at(i);

        if (esc)
            esc=false;
        else {
            switch (c) {
            case '?':  // escape !!
                esc=true;
                break;
            case '\\': // escape
                esc=true;
                break;
            case '@': // start of binary-data
                int startOfBinaryData;
                startOfBinaryData = data.find('@', i+1) + 1;
                i = startOfBinaryData + lengthOfBinaryData(data, i);
                i--;
                break;
            case '\'': // segment-end
                found = true;
                break;
            default:
                break;
            } // switch
            if (found) break;
        }
    } // for

    if (found || (i == data.length()))
        result = data.substr(startIndex, i-startIndex);
    else
        result = "";

    return result;
}


string String::escape(const string &unescaped) {
    string result;
    unsigned char c;
    for (unsigned int i=0; i<unescaped.length(); i++) {
        c = unescaped[i];
        if ((c == '+') || (c == ':') || (c == '?') || (c == '\''))
            result += "?";

        result += c;
    }

    return result;
}


string String::unEscape(const string &escaped) {
    string result;
    for (unsigned int i=0; i<escaped.length(); i++) {
        if (escaped[i] == '?')
            i++;
        if (i < escaped.length())
            result += escaped[i];
    }

    return result;
}


int String::lengthOfBinaryData(const string &data, int startIndex) {
    unsigned int i = data.find('@', startIndex + 1);

    string tmp = data.substr(startIndex + 1,  i - (startIndex + 1));
    return atoi(tmp.c_str()) ;
}


string String::num2string(int number, bool fillWithZero, int length) {
    string snumber;
    char numberText[20];
    sprintf(numberText, "%i", number);
    snumber = numberText;

    //mit 0 auffllen (von links)
    if (fillWithZero && (int) snumber.length() < length) {
        string tmp(length - (int) snumber.length(),'0');
        snumber = tmp + snumber;
    }

    return snumber;
}


string String::date2string() {
    //JJJJMMTT
    time_t now;
    time(&now);
    tm *localNow = localtime(&now);
    string result;

    result = String::num2string(localNow->tm_year + 1900);
    result += String::num2string(localNow->tm_mon + 1, true, 2);
    result += String::num2string(localNow->tm_mday, true, 2);

    return result;
}


double String::string2double(const string &d) 
{
    Value v(d);
    return v.getValue();
}


/*string String::double2string(double number,int pre, int post){
    string fmt;
    char buffer[256];

    if (pre+post+1>(int)sizeof(buffer)-1)
        return "BUFFER OVERFLOW";

    fmt="%";
    fmt+=num2string(pre+post);
    fmt+=".";
    fmt+=num2string(post);
    fmt+="f";
#ifdef HAVE_SNPRINTF
    snprintf(buffer,sizeof(buffer),fmt.c_str(),number);
#else
    sprintf(buffer,fmt.c_str(),number);
#endif
    return buffer;
    }*/


string String::time2string() {
    //HHMMSS
    time_t now;
    time(&now);
    tm *localNow = localtime(&now);
    string result;

    result = String::num2string(localNow->tm_hour, true, 2);
    result += String::num2string(localNow->tm_min, true, 2);
    result += String::num2string(localNow->tm_sec, true, 2);

    return result;
}


/*bool String::toStream(ostream *f, const string &data) {
    if (!  (*f << "@") )
        return false;
    if (!  (*f << data.length()) )
        return false;
    if (!  (*f << "@") )
        return false;
    if (!  (*f << data) )
        return false;

    return true;
    }*/


/*string String::fromStream(istream *f) {
    string tmp;
    char c;
    unsigned int len;
    char buffer[4096];

    // get length of stored string
    if (! ((*f >> c)  && (c == '@')) )
        throw Error("String::fromStream()",
                        "Could not read from stream",0);
    tmp = "";
    while (*f) {
        *f >> c;
        if (c != '@')
            tmp += c;
        else
            break;
    }
    if (c != '@')
        return "";
    len = atoi(tmp.c_str());

    if (len == 0)
        return "";
    if (len>sizeof(buffer)) {
        fprintf(stderr,"INTERNAL ERROR: String::fromString(): buffer too small\n");
        return "";
    }
    f->read(buffer, len);
    tmp = string(buffer, len);

    // was reading successfull?
    if (len == tmp.length()) {
        return tmp;
    } else
        throw Error("String::fromStream()",
                        "Could not read from stream",0);
}
*/

/*
string String::readWholeFile(const string &filename) {
    string result;
    char c;
    ifstream f(filename.c_str());
    while (f) {
        c=f.get();
        result += c;
    }

    f.close();

    return result;
}
*/

/* blank . , & - + * % / $     */
const unsigned char dtaus0Special[] = {'\x20', '\x2e', '\x2c' , '\x26', '\x2d', '\x2b', '\x2a', '\x25', '\x2f', '\x24' , '\x5b', '\x5c', '\x5d', '\x7e' };

string String::transformToDTAUS0(const string &m) {
    string result = "";
    bool found;
    unsigned char c;
    string msg;

    msg=m;
    // to uppercase
    for (unsigned int i=0; i<msg.length(); i++)
      msg.at(i)=toupper(msg.at(i));
    //    transform(msg.begin(), msg.end(), msg.begin(), toupper);

    // check each character if it is valid (else substitute with blank)
    for (unsigned int i=0; i<msg.length(); i++) {
	c = (unsigned char)(msg[i]);
        found = false;
        // c in 0..9 or A..Z?
	if ( (c >= (unsigned char)'\x30' ) && (c <= (unsigned char)'\x39') ||
	    ( (c >= (unsigned char)'\x41' ) && (c <= (unsigned char)'\x5a') ) )
	    found = true;
	else
            // one of the special characters that are allowed?
            for (unsigned int j = 0; j < sizeof(dtaus0Special); j++) {
                found = (found || (c == dtaus0Special[j]));
		if (c == (unsigned char)'') {
                    c = ''; found = true;
                }
                if (c == (unsigned char)'') {
                    c = ''; found = true;
                }
                if (c == (unsigned char)'') {
                    c = ''; found = true;
                }
            }

        if (! found)
            c = ' ';
        result += c;
    }

    // delete leading blanks
    unsigned int i;
    for (i=0; i < result.length(); i++)
        if (result[i] != ' ')
            break;
    result = result.substr(i);

    // delete ending blanks
    for (i=0; i < result.length();  i++)
        if (result[ result.length() - (i + 1) ] != ' ')
            break;
    result = result.substr(0, result.length() - i);


    return escape(result);
}


string String::transformToHBCIBinData(const string &data) {
    string result = "@" + num2string(data.length()) + "@" + data;
    return result;
}


string String::extractBinData(const string &data) {
    unsigned int startOfBinaryData;

    startOfBinaryData = data.find('@', 1) + 1;
    if (startOfBinaryData == string::npos)
        throw Error("String::extractBinData()",
                        "Could not read binary data", 0);

    return data.substr(startOfBinaryData);
}


string String::nextTLV(const string &data, int startIndex){
    unsigned short size;

    if (startIndex+2>=(int)data.length())
        return "";
    size=(unsigned char)(data.at(startIndex+2))<<8;
    size+=(unsigned char)(data.at(startIndex+1))& 0xff;
    return data.substr(startIndex,size+3);
}


int String::sizeTLV(const string &tlv) {
    unsigned short size;

    if (tlv.length()<3)
        return -1;
    size=(unsigned char)(tlv.at(2))<<8;
    size+=(unsigned char)(tlv.at(1))& 0xff;
    return (int)size;
}


unsigned int String::typeTLV(const string &tlv) {
    if (tlv.length()<3)
        return 0xff;
    return (unsigned char) tlv.at(0);
}


string String::dataTLV(const string &tlv) {
    int size;

    if (tlv.length()<3)
        return "";
    size=sizeTLV(tlv);
    return tlv.substr(3,size);
}


string String::newTLV(int tag, const string &data) {
    unsigned short size;
    string result;

    size=data.length();
    if (size==0)
        return "";
    result=(unsigned char) tag;
    result+=(char)((unsigned char)(size & 0xff));
    result+=(char)((unsigned char)(size>>8));
    result+=data;
    return result;
}


string String::dumpToString(const string &s)
{
    bool seghead = true;
    int  segstart = 0;
    string out;
    for (int i=0; i<(int)s.length(); i++)
    {
	if (s[i] == ':' && seghead)
	{
	    string segname = s.substr(segstart, i-segstart);
	    string segdesc = string(hbciSegmentName(segname.c_str())).substr(0, 34);
	    out += "(" + segdesc + ")" 
		+ string("                              ").substr(0, 36 - segdesc.length());
	    seghead = false;
	}

	if (s[i] == '@')
	{
	    long int number = 0;
	    i++;
	    while (isdigit(s[i]))
	    {
		number = number * 10 + (s[i] - '0');
		i++;
	    }
	    i += number; // skip second '@' and binary data
	    out += "@" + String::num2string(number) + "@<...>";
	}
	else if (s[i] == '\'') {
	    out += "'\n";
	    seghead = true;
	    segstart = i+1;
	}
	else 
	    out += s.substr(i, 1);
    }
    return out;
}

struct hbcisegment {
    const char *segment;
    const char *name;
};

hbcisegment hbcisegments[] = 
{
    { "HIAOMS", "Auslandsberweisung ohne Meldeteil Parameter" },
    { "HIAUBS", "Auslandsberweisung Parameter" },
    { "HIAZK", "Kartenanzeige" },
    { "HIAZKS", "Kartenanzeige Parameter" },
    { "HIBPA", "Bankparameter allgemein" },
    { "HIDAA", "Dauerauftragsnderungsvormerkungen" },
    { "HIDAAS", "Dauerauftragsnderungsvormerkungen" },
    { "HIDAB", "Dauerauftragsbestand rckmelden" },
    { "HIDABS", "Dauerauftragsbestand Parameter" },
    { "HIDAE", "Dauerauftragseinrichtung besttigen" },
    { "HIDAES", "Dauerauftrag einrichten Parameter" },
    { "HIDALS", "Dauerauftrag lschen Parameter" },
    { "HIDAN", "Dauerauftragsnderung besttigen" },
    { "HIDANS", "Dauerauftrag ndern Parameter" },
    { "HIDAS", "Dauerauftragsaussetzung besttigen" },
    { "HIDASS", "Dauerauftrag aussetzen Parameter" },
    { "HIDVK", "Devisenkurse rckmelden" },
    { "HIDVKS", "Devisenkurse Parameter" },
    { "HIFDA", "Finanzdatenformat rckmelden" },
    { "HIFDAS", "Finanzdatenformat anfordern Parameter" },
    { "HIFDB", "Bearbeitungsstatus Finanzdatenformat" },
    { "HIFDBS", "Bearbeitungsstatus Finanzdatenformat" },
    { "HIFDL", "Finanzdatenformatliste rckmelden" },
    { "HIFDLS", "Finanzdatenformatliste anfordern Parameter" },
    { "HIFDSS", "Finanzdatenformat senden Parameter" },
    { "HIFGA", "Festgeldnderung besttigen" },
    { "HIFGAS", "Festgeld ndern Parameter" },
    { "HIFGB", "Festgeldbestand rckmelden" },
    { "HIFGBS", "Festgeldbestand Parameter" },
    { "HIFGK", "Festgeldkonditionen rckmelden" },
    { "HIFGKS", "Festgeldkonditionen Parameter" },
    { "HIFGN", "Festgeldneuanlage besttigen" },
    { "HIFGNS", "Festgeldneuanlage Parameter" },
    { "HIFGP", "Festgeldprolongation besttigen" },
    { "HIFGPS", "Festgeldprolongation Parameter" },
    { "HIFGWS", "Festgeldneuanlage widerrufen Parameter" },
    { "HIFPO", "Festpreisordereinreichung besttigen" },
    { "HIFPOS", "Festpreisorder Parameter" },
    { "HIFPWS", "Festgeldprolongation widerrufen Parameter" },
    { "HIFRD", "Freistellungsdaten rckmelden" },
    { "HIFRDS", "Freistellungsdaten Parameter" },
    { "HIGAMS", "Gastmeldung Parameter" },
    { "HIINF", "Informationen rckmelden" },
    { "HIINFS", "Informationsanforderung Parameter" },
    { "HIISA", "bermittlung eines ffentlichen Schlssels" },
    { "HIKAN", "Kontoumstze rckmelden/neue Umstze" },
    { "HIKANS", "Kontoumstze/neu Parameter" },
    { "HIKASS", "Kartensperre beantragen Parameter" },
    { "HIKAZ", "Kontoumstze rckmelden/Zeitraum" },
    { "HIKAZS", "Kontoumstze/Zeitraum Parameter" },
    { "HIKDMS", "Kundenmeldung Parameter" },
    { "HIKIA", "Kreditinstitutsangebote rckmelden" },
    { "HIKIAS", "Kreditinstitutsangebote Parameter" },
    { "HIKIF", "Kontoinformationen rckmelden" },
    { "HIKIFS", "Kontoinformationen Parameter" },
    { "HIKIM", "Kreditinstitutsmeldung" },
    { "HIKOM", "Kommunikationszugang rckmelden" },
    { "HIKOMS", "Kommunikationszugang Parameter" },
    { "HIKPV", "Komprimierungsverfahren" },
    { "HILASS", "Lastschrift Parameter" },
    { "HINEA", "Liste Neuemissionen" },
    { "HINEAS", "Liste Neuemissionen Parameter" },
    { "HINEZ", "Einreichung Zeichnung besttigen" },
    { "HINEZS", "Neuemission zeichnen Parameter" },
    { "HIOAN", "Orderanzeige" },
    { "HIOANS", "Orderanzeige Parameter" },
    { "HIPRO", "Statusprotokoll rckmelden" },
    { "HIPROS", "Statusprotokoll Parameter" },
    { "HIRMG", "Rckmeldungen zur Gesamtnachricht" },
    { "HIRMS", "Rckmeldung zu Segmenten" },
    { "HISAL", "Saldenrckmeldung" },
    { "HISALS", "Saldenabfrage Parameter" },
    { "HISHV", "Sicherheitsverfahren" },
    { "HISLAS", "Sammellastschrift Parameter" },
    { "HISLB", "Bestand terminierter Sammellastschriften" },
    { "HISLBS", "Bestand terminierter Sammellastschriften" },
    { "HISLE", "Einreichung terminierter Sammellastschrift" },
    { "HISLES", "Terminierte Sammellastschrift einreichen" },
    { "HISLLS", "Terminierte Sammellastschrift lschen Parameter" },
    { "HISRBS", "Sorten- und Reisescheckbestellung Parameter" },
    { "HISRK", "Sorten- und Reisescheckkonditionen" },
    { "HISRKS", "Sorten- und Reisescheckkonditionen Parameter" },
    { "HISSP", "Besttigung der Schlsselsperrung" },
    { "HISUBS", "Sammelberweisung Parameter" },
    { "HISYN", "Synchronisierungsantwort" },
    { "HITMVS", "Terminvereinbarung Parameter" },
    { "HITSB", "Bestand terminierter Sammelberweisungen rckmelden" },
    { "HITSBS", "Bestand terminierter Sammelberweisungen Parameter" },
    { "HITSE", "Einreichung terminierter Sammelberweisung besttigen" },
    { "HITSES", "Terminierte Sammelberweisung einreichen Parameter" },
    { "HITSLS", "Terminierte Sammelberweisung lschen" },
    { "HITUA", "nderung terminierter berweisung besttigen" },
    { "HITUAS", "Terminierte berweisung ndern Parameter" },
    { "HITUB", "Bestand terminierter berweisungen rckmelden" },
    { "HITUBS", "Bestand terminierter berweisungen Parameter" },
    { "HITUE", "Einreichung terminierter berweisung" },
    { "HITUES", "Terminierte berweisung einreichen Parameter" },
    { "HITULS", "Terminierte berweisung lschen Parameter" },
    { "HIUEBS", "Einzelberweisung Parameter" },
    { "HIUMBS", "Umbuchung Parameter" },
    { "HIUPA", "Userparameter allgemein" },
    { "HIUPD", "Kontoinformation" },
    { "HIVDBS", "Vordruckbestellung Parameter" },
    { "HIWDU", "Depotumstze rckmelden" },
    { "HIWDUS", "Depotumstze Parameter" },
    { "HIWFO", "Fondsordereinreichung besttigen" },
    { "HIWFOS", "Fondsorder Parameter" },
    { "HIWFP", "Festpreisangebote rckmelden" },
    { "HIWFPS", "Festpreisangebote Parameter" },
    { "HIWOA", "Wertpapierordernderung besttigen" },
    { "HIWOAS", "Wertpapierordernderung Parameter" },
    { "HIWOH", "Wertpapierorderhistorie" },
    { "HIWOHS", "Wertpapierorderhistorie Parameter" },
    { "HIWPD", "Depotaufstellung rckmelden" },
    { "HIWPDS", "Depotaufstellung Parameter" },
    { "HIWPH", "Wichtige Informationen rckmelden" },
    { "HIWPHS", "Wichtige Informationen Parameter" },
    { "HIWPI", "Wertpapierinformationen rckmelden" },
    { "HIWPIS", "Wertpapierinformationen Parameter" },
    { "HIWPK", "Wertpapierkurse rckmelden" },
    { "HIWPKS", "Wertpapierkurse Parameter" },
    { "HIWPO", "Wertpapierordereinreichung besttigen" },
    { "HIWPOS", "Wertpapierorder Parameter" },
    { "HIWPR", "Wertpapierreferenznummern rckmelden" },
    { "HIWPRS", "Wertpapierreferenznummern Parameter" },
    { "HIWPS", "Wertpapierorderstreichung besttigen" },
    { "HIWPSS", "Wertpapierorderstreichung Parameter" },
    { "HIWSD", "Wertpapierstammdaten rckmelden" },
    { "HIWSDS", "Wertpapierstammdaten Parameter" },
    { "HIWSO", "Orderstatus" },
    { "HIWSOS", "Orderstatus Parameter" },
    { "HKAOM", "Auslandsberweisung ohne Meldeteil" },
    { "HKAUB", "Auslandsberweisung" },
    { "HKAZK", "Kartenanzeige anfordern" },
    { "HKDAA", "Dauerauftragsnderungsvormerkungen" },
    { "HKDAB", "Dauerauftragsbestand anfordern" },
    { "HKDAE", "Dauerauftrag einrichten" },
    { "HKDAL", "Dauerauftrag lschen" },
    { "HKDAN", "Dauerauftrag ndern" },
    { "HKDAS", "Dauerauftrag aussetzen" },
    { "HKDVK", "Devisenkurse anfordern" },
    { "HKEND", "Dialogende" },
    { "HKFDA", "Finanzdatenformat anfordern" },
    { "HKFDB", "Bearbeitungsstatus Finanzdatenformat" },
    { "HKFDL", "Finanzdatenformatliste anfordern" },
    { "HKFDS", "Finanzdatenformat senden" },
    { "HKFGA", "Festgeldanlage ndern" },
    { "HKFGB", "Festgeldbestand anfordern" },
    { "HKFGK", "Festgeldkonditionen anfordern" },
    { "HKFGN", "Festgeldneuanlage" },
    { "HKFGP", "Festgeldanlage prolongieren" },
    { "HKFGW", "Festgeldneuanlage widerrufen" },
    { "HKFPO", "Festpreisorder" },
    { "HKFPW", "Festgeldprolongation widerrufen" },
    { "HKFRD", "Freistellungsdaten abfragen" },
    { "HKGAM", "Gastmeldung" },
    { "HKIDN", "Identifikation" },
    { "HKINF", "Informationen anfordern" },
    { "HKISA", "Anforderung eines ffentlichen Schlssels" },
    { "HKKAN", "Kontoumstze anfordern/neue Umstze" },
    { "HKKAS", "Kartensperre beantragen" },
    { "HKKAZ", "Kontoumstze anfordern/Zeitraum" },
    { "HKKDM", "Kundenmeldung" },
    { "HKKIA", "Kreditinstitutsangebote anfordern" },
    { "HKKIF", "Kontoinformationen anfordern" },
    { "HKKOM", "Kommunikationszugang anfordern" },
    { "HKLAS", "Einzellastschrift" },
    { "HKNEA", "Liste Neuemissionen anfordern" },
    { "HKNEZ", "Neuemission zeichnen" },
    { "HKOAN", "Orderanzeige anfordern" },
    { "HKPRO", "Statusprotokoll anfordern" },
    { "HKSAK", "Schlsselnderung" },
    { "HKSAL", "Saldenabfrage" },
    { "HKSLA", "Sammellastschrift" },
    { "HKSLB", "Bestand terminierter Sammellastschriften" },
    { "HKSLE", "Terminierte Sammellastschrift einreichen" },
    { "HKSLL", "Terminierte Sammellastschrift lschen" },
    { "HKSRB", "Sorten- und Reisescheckbestellung" },
    { "HKSRK", "Sorten- und Reisescheckkonditionen anfordern" },
    { "HKSSP", "Schlsselsperrung" },
    { "HKSUB", "Sammelberweisung" },
    { "HKSYN", "Synchronisierung" },
    { "HKTMV", "Terminvereinbarung" },
    { "HKTSB", "Bestand terminierter Sammelberweisungen anfordern" },
    { "HKTSE", "Terminierte Sammelberweisung einreichen" },
    { "HKTSL", "Terminierte Sammelberweisung lschen" },
    { "HKTUA", "Terminierte berweisung ndern" },
    { "HKTUB", "Bestand terminierter berweisungen anfordern" },
    { "HKTUE", "Terminierte berweisung einreichen" },
    { "HKTUL", "Terminierte berweisung lschen" },
    { "HKUEB", "Einzelberweisung" },
    { "HKUMB", "Umbuchung" },
    { "HKVDB", "Vordruckbestellung" },
    { "HKVVB", "Verarbeitungsvorbereitung" },
    { "HKWDU", "Depotumstze anfordern" },
    { "HKWFO", "Fondsorder einreichen" },
    { "HKWFP", "Festpreisangebote anfordern" },
    { "HKWOA", "Wertpapierordernderung" },
    { "HKWOH", "Wertpapierorderhistorie anfordern" },
    { "HKWPD", "Depotaufstellung anfordern" },
    { "HKWPH", "Wichtige Informationen anfordern" },
    { "HKWPI", "Wertpapierinformationen anfordern" },
    { "HKWPK", "Wertpapierkurse anfordern" },
    { "HKWPO", "Wertpapierorder einreichen" },
    { "HKWPR", "Wertpapierreferenznummern anfordern" },
    { "HKWPS", "Wertpapierorderstreichung" },
    { "HKWSD", "Wertpapierstammdaten anfordern" },
    { "HKWSO", "Orderstatus anfordern" },
    { "HNHBK", "Nachrichtenkopf" },
    { "HNHBS", "Nachrichtenabschluss" },
    { "HNSHA", "Signaturabschluss" },
    { "HNSHK", "Signaturkopf" },
    { "HNVSD", "Verschlsselte Daten" },
    { "HNVSK", "Verschlsselungskopf" }
};

const char *String::hbciSegmentName(const char *segment)
{
    for (unsigned int i=0; i< sizeof(hbcisegments) / sizeof(hbcisegment); i++)
	if (!strcmp(hbcisegments[i].segment, segment)) return hbcisegments[i].name;
    return "";
}


void String::simpleDump(const string &s, FILE *f){
    unsigned int i;
    unsigned int j;
    unsigned int pos;
    string result;

    pos=0;
    fprintf(f,"String size is %d:\n",s.length());
    while(pos<s.length()) {
        fprintf(f,"%04x: ",pos);
        j=pos+16;
        if (j>=s.length())
            j=s.length();

        // show hex dump
        for (i=pos; i<j; i++) {
            fprintf(f,"%02x ",(unsigned char)s[i]);
        }
        if (j-pos<16)
            for (i=0; i<16-(j-pos); i++)
                fprintf(f,"   ");
        // show text
        for (i=pos; i<j; i++) {
	    if ((unsigned char)(s[i])<32)
		fprintf(f,".");
            else
                fprintf(f,"%c",s[i]);
        }
        fprintf(f,"\n");
        pos+=16;
    }
}


} // namespace HBCI

char* hbci_strdup(const char *c)
{
    char * p = (char*)malloc(strlen(c) + 1);
    strcpy(p, c);
    return p;
}
char* hbci_strdup(const string &s)
{
    char * p = (char*)malloc(s.length() + 1);
    s.copy(p, string::npos);
    p[s.length()] = 0;
    return p;
}
