#include "openingBookConverter.h"

#include "osl/state/simpleState.h"
#include "osl/apply_move/applyMove.h"
#include "osl/stl/vector.h"
#include "osl/record/compactBoard.h"
#include "osl/record/record.h"
#include <iostream>

OpeningBookConverter::OpeningBookConverter(const char *filename)
{
  std::ifstream ifs(filename);
  int nStates = osl::record::readInt(ifs);
  states.reserve(nStates);
  for (int i = 0 ; i < nStates; i++)
  {
    int blackWin = osl::record::readInt(ifs);
    int whiteWin = osl::record::readInt(ifs);
    int nMove = osl::record::readInt(ifs);
    int index = osl::record::readInt(ifs);
    states.push_back(OBState(index, nMove, blackWin, whiteWin));
  }
  while (true)
  {
    osl::Move move=osl::Move::makeDirect(osl::record::readInt(ifs));
    int stateIndex=osl::record::readInt(ifs);
    if (!ifs)
      break;
    moves.push_back(osl::record::opening::OBMove(move,stateIndex));
  }
}

void
OpeningBookConverter::write(const char* filename)
{
  std::ofstream ofs(filename);
  osl::record::writeInt(ofs, states.size());
  for (osl::vector<OBState>::const_iterator it = states.begin();
       it != states.end(); ++it)
  {
    osl::record::writeInt(ofs, it->getOBMoveIndex());
    osl::record::writeInt(ofs, it->getNOBMove());
    osl::record::writeInt(ofs, it->getBlackWinCount());
    osl::record::writeInt(ofs, it->getWhiteWinCount());
  }
  for (osl::vector<osl::record::opening::OBMove>::const_iterator it = moves.begin();
       it != moves.end(); ++it)
  {
    osl::record::writeInt(ofs, it->getMove().intValue());
    osl::record::writeInt(ofs, it->getStateIndex());
  }
}

void
OpeningBookConverter::writeInNewFormat(std::ofstream& ofs)
{
  osl::vector<int> weights(moves.size());
  osl::record::writeInt(ofs, 1);		// version number
  osl::record::writeInt(ofs, states.size());
  osl::record::writeInt(ofs, moves.size());
  osl::record::writeInt(ofs, 0);		// Start state index
  for (osl::vector<OBState>::const_iterator it = states.begin();
       it != states.end(); ++it)
  {
    osl::record::writeInt(ofs, it->getOBMoveIndex());
    osl::record::writeInt(ofs, it->getNOBMove());
    int total_wins = 0;
    osl::vector<int> wins;
    wins.reserve(it->getNOBMove());
    for (int i = 0; i < it->getNOBMove(); i++)
    {
      const osl::record::opening::OBMove& move
	= moves.at(i + it->getOBMoveIndex());

      const OBState& state = states.at(move.getStateIndex());
      if (move.getMove().player() == osl::BLACK)
	wins.push_back(state.getBlackWinCount());
      else
	wins.push_back(state.getWhiteWinCount());

      total_wins += wins.at(i);
    }
    for (int i = 0; i < it->getNOBMove(); i++)
    {
      if (total_wins != 0)
	weights.at(i + it->getOBMoveIndex()) = (wins.at(i) * 10000 / total_wins);
      else
	weights.at(i + it->getOBMoveIndex()) = 0;
    }
    osl::record::writeInt(ofs, it->getBlackWinCount());
    osl::record::writeInt(ofs, it->getWhiteWinCount());
  }
  int i = 0;
  for (osl::vector<osl::record::opening::OBMove>::const_iterator it = moves.begin();
       it != moves.end(); ++it, ++i)
  {
    osl::record::opening::WMove wmove(it->getMove(),
				      it->getStateIndex(),
				      weights.at(i));
    ofs << wmove;
  }
}

void
OpeningBookConverter::writeInNewFormat(const char* filename)
{
  std::ofstream ofs(filename);
  writeInNewFormat(ofs);
}

void
OpeningBookConverter::writeInNewEditFormat(const char* filename)
{
  std::ofstream ofs(filename);
  writeInNewFormat(ofs);
  osl::vector<osl::SimpleState> simpleStates(states.size());
  osl::vector<bool> visited(states.size());
  osl::vector<int> toTraceIndex;

  for (unsigned int i = 0; i < states.size(); i++)
    visited[i] = false;

  assert(states.size() >= 1);

  // First entry is assmed to be the start state.
  toTraceIndex.push_back(0);
  simpleStates[0] = osl::SimpleState(osl::HIRATE);
  visited[0] = true;

  while (!toTraceIndex.empty())
  {
    const int index = toTraceIndex.back();
    toTraceIndex.pop_back();
    const OBState& s = states.at(index);
    const int moveIndex = s.getOBMoveIndex();
    for (int i = 0; i < s.getNOBMove(); i++)
    {
      const osl::record::opening::OBMove& m = moves.at(moveIndex + i);
      const int nextState = m.getStateIndex();
      if (!visited[nextState])
      {
	toTraceIndex.push_back(nextState);
	osl::SimpleState newState(simpleStates[index]);
	osl::ApplyMoveOfTurn::doMove(newState, m.getMove());
	simpleStates[nextState] = newState;
	visited[nextState] = true;
      }
    }
  }
  for (unsigned int i = 0; i < states.size(); i++)
  {
    osl::record::CompactBoard board(simpleStates[i]);
    ofs << board;
  }
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
