/* progressEval.t.cc
 */
#include "osl/eval/endgame/attackDefense.h"
#include "osl/eval/progressEval.h"
#include "osl/record/csaString.h"
#include "consistencyTest.h"

#include <cppunit/TestCase.h>

class ProgressEvalTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(ProgressEvalTest);
  CPPUNIT_TEST(testLoad);
  // 以降は testLoad の後にテストすること
  CPPUNIT_TEST(testBeginning);
  CPPUNIT_TEST(testConsistentUpdate);
  CPPUNIT_TEST(testConsistentExpect);
  // 
  CPPUNIT_TEST(testInitial);
  CPPUNIT_TEST(testPinBonus);
  CPPUNIT_TEST(testSeeScale);
  CPPUNIT_TEST(testSymmetry);
  CPPUNIT_TEST_SUITE_END();
public:
  void testLoad();
  void testBeginning();
  void testConsistentUpdate();
  void testConsistentExpect();
  void testInitial();
  void testPinBonus();
  void testSeeScale();
  void testSymmetry();
};

CPPUNIT_TEST_SUITE_REGISTRATION(ProgressEvalTest);

using namespace osl;
using namespace osl::eval;

void ProgressEvalTest::testBeginning()
{
  const NumEffectState state(CsaString(
			       "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			       "P2 * -HI *  *  *  *  * -KA * \n"
			       "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			       "P4 *  *  *  *  *  *  *  *  * \n"
			       "P5 *  *  *  *  *  *  *  *  * \n"
			       "P6 *  *  *  *  *  *  *  *  * \n"
			       "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			       "P8 * +KA *  *  *  *  * +HI * \n"
			       "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			       "+\n").getInitialState());
  ProgressEval eval(state);
  CPPUNIT_ASSERT_EQUAL(0, eval.progress16().value());
  ProgressEval::opening_eval_t opening(state);
#ifdef EXPERIMENTAL_EVAL
  CPPUNIT_ASSERT(opening.value() == 0);
#else
  CPPUNIT_ASSERT(opening.value() != 0);
#endif
  CPPUNIT_ASSERT(abs(opening.value()*16 - eval.value())
		 < ProgressEval::ROUND_UP);
}

void ProgressEvalTest::testInitial()
{
  const NumEffectState state(CsaString(
			       "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			       "P2 * -HI *  *  *  *  * -KA * \n"
			       "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			       "P4 *  *  *  *  *  *  *  *  * \n"
			       "P5 *  *  *  *  *  *  *  *  * \n"
			       "P6 *  *  *  *  *  *  *  *  * \n"
			       "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			       "P8 * +KA *  *  *  *  * +HI * \n"
			       "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			       "+\n").getInitialState());
  ProgressEval eval(state);
  CPPUNIT_ASSERT_EQUAL(0, eval.endgameValue());
}


void ProgressEvalTest::testLoad()
{
  CPPUNIT_ASSERT(ProgressEval::setUp());
}

void ProgressEvalTest::testConsistentUpdate()
{
  consistencyTestUpdate<ProgressEval>();
}

void ProgressEvalTest::testConsistentExpect()
{
#ifndef EXPERIMENTAL_EVAL
  consistencyTestExpect<ProgressEval>(16000); // 48000
#endif
}

void ProgressEvalTest::testPinBonus()
{
  {
    const NumEffectState state(
      CsaString("P1-KY-KE *  *  *  *  * -KE-KY\n"
		"P2 * -HI *  *  *  * -KI-OU * \n"
		"P3-FU-FU-FU-FU * -KI-GI-FU-FU\n"
		"P4 *  *  *  * -FU-KA-FU *  * \n"
		"P5 *  *  *  *  * -FU *  *  * \n"
		"P6 *  * +FU+FU+FU *  *  *  * \n"
		"P7+FU+FU * +KI * +FU+FU+FU+FU\n"
		"P8 * +OU+GI *  *  *  * -HI * \n"
		"P9+KY+KE+KA+KI *  *  * +KE+KY\n"
		"P+00GI\n"
		"P-00GI\n"
		"+\n").getInitialState());
    ProgressEval eval(state);
    endgame::AttackDefense endgame(state);
    const Piece black_king = state.getKingPiece<BLACK>();
    const Piece white_king = state.getKingPiece<WHITE>();
    const int bonus = eval.calculatePinBonus(state);
    CPPUNIT_ASSERT(bonus < 0);
    CPPUNIT_ASSERT_EQUAL(
      -(endgame.valueOf(black_king, white_king,
			state.getPieceAt(Position(6, 6))) +
	endgame.valueOf(black_king, white_king,
			state.getPieceAt(Position(7, 8)))) / 4 *
      eval.progress16().value() / 16,
      bonus);
  }

  {
    const NumEffectState state(
      CsaString("P1-KY-KE *  *  *  *  * -KE-KY\n"
		"P2 * +HI *  *  *  * -KI-OU * \n"
		"P3-FU-FU-FU-FU * -KI-GI-FU-FU\n"
		"P4 *  *  *  * -FU-KA-FU *  * \n"
		"P5 *  *  *  *  * -FU *  *  * \n"
		"P6 *  * +FU+FU+FU *  *  *  * \n"
		"P7+FU+FU+KE+KI * +FU+FU+FU+FU\n"
		"P8 * +OU+GI *  *  *  * +HI * \n"
		"P9+KY * +KA+KI *  *  * +KE+KY\n"
		"P+00GI\n"
		"P-00GI\n"
		"+\n").getInitialState());
    ProgressEval eval(state);
    endgame::AttackDefense endgame(state);
    const Piece black_king = state.getKingPiece<BLACK>();
    const Piece white_king = state.getKingPiece<WHITE>();
    const int bonus = eval.calculatePinBonus(state);
    CPPUNIT_ASSERT(bonus > 0);
    CPPUNIT_ASSERT_EQUAL(
      -(endgame.valueOf(black_king, white_king,
		       state.getPieceAt(Position(3, 2))) / 4) *
      eval.progress16().value() / 16,
      bonus);
  }
}

void ProgressEvalTest::testSeeScale()
{
  CPPUNIT_ASSERT_EQUAL(16, ProgressEval::seeScale());
}

void ProgressEvalTest::testSymmetry()
{
  using namespace osl;

  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0;i<(isShortTest ? 10 : 200) && (ifs >> file_name) ; i++)
  {
    if (file_name == "") 
      break;
    file_name = OslConfig::testCsaFile(file_name);

    const Record rec=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=rec.getMoves();

    NumEffectState state(rec.getInitialState());
    ProgressEval eval(state);
    for (unsigned int i=0; i<moves.size(); i++){
      const Move m = moves[i];
      ApplyMoveOfTurn::doMove(state, m);
      eval.update(state, m);

      NumEffectState state_r(state.rotate180());
      ProgressEval eval_r(state_r);
      ProgressDebugInfo debug = eval.debugInfo(state);
      ProgressDebugInfo debug_r = eval_r.debugInfo(state_r);
      // No opening value here because it's known to by assymetric.
      CPPUNIT_ASSERT_EQUAL(debug.endgame, -debug_r.endgame);
      CPPUNIT_ASSERT_EQUAL(debug.progress_bonus, -debug_r.progress_bonus);
      CPPUNIT_ASSERT_EQUAL(debug.progress_independent_bonus,
			   -debug_r.progress_independent_bonus);
      if (debug.minor_piece_bonus !=
	  -debug_r.minor_piece_bonus)
      {
	std::cout << state << moves[i]
		  << std::endl
		  << debug.minor_piece_bonus << " " << debug_r.minor_piece_bonus<< std::endl;
      }
      CPPUNIT_ASSERT_EQUAL(debug.progress_dependent_bonus,
			   -debug_r.progress_dependent_bonus);
      CPPUNIT_ASSERT_EQUAL(debug.minor_piece_bonus,
			   -debug_r.minor_piece_bonus);
    }
  }
}
/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
