#include "osl/move_generator/addEffect_.h"
#if (__GNUC__ >= 4 && __GNUC_MINOR__ >=3)
#  include "osl/move_generator/addEffect_.tcc"
#endif
#include "osl/move_generator/allMoves.h"
#include "osl/move_action/store.h"
#include "osl/record/record.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/container/moveVector.h"
#include "osl/state/simpleState.h"
#include "osl/effect_util/effectUtil.h"
#include "osl/apply_move/applyMove.h"
#include "osl/state/numEffectState.h"
#include "osl/search/shouldPromoteCut.h"
#include "osl/oslConfig.h"

typedef osl::NumEffectState test_state_t;

#include <algorithm>
#include <iostream>
#include <fstream>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class AddEffectTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(AddEffectTest);
  CPPUNIT_TEST(testOne);
  CPPUNIT_TEST(testShort);
  CPPUNIT_TEST(testNear);
  CPPUNIT_TEST(testLong);
  CPPUNIT_TEST(testKing);
  CPPUNIT_TEST(testMove);
  CPPUNIT_TEST(testDrop);
  CPPUNIT_TEST(testOpen);
  CPPUNIT_TEST(testAdditional);
  CPPUNIT_TEST(testShadow);
  CPPUNIT_TEST(testMoveMember);
  CPPUNIT_TEST_SUITE_END();
private:
  osl::SimpleState state;
public:
  AddEffectTest();
  void testOne();
  void testShort();
  void testNear();
  void testLong();
  void testKing();
  void testMove();
  void testDrop();
  void testOpen();
  void testAdditional();
  void testShadow();
  void testMoveMember();
};

CPPUNIT_TEST_SUITE_REGISTRATION(AddEffectTest);

using namespace osl;
using namespace osl::move_action;
using namespace osl::move_generator;

AddEffectTest::AddEffectTest()
{
  state=CsaString(
"P1+NY+TO *  *  *  * -OU-KE-KY\n"
"P2 *  *  *  *  * -GI-KI *  *\n"
"P3 * +RY *  * +UM * -KI-FU-FU\n"
"P4 *  * +FU-FU *  *  *  *  *\n"
"P5 *  * -KE * +FU *  * +FU *\n"
"P6+KE *  * +FU+GI-FU *  * +FU\n"
"P7 *  * -UM *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n"
).getInitialState();
}

void AddEffectTest::testOne()
{
  {
    NumEffectState state(CsaString(
"P1 *  *  *  * -KA * -OU-KE-KY\n"
"P2-KY * -KI * -HI *  *  *  * \n"
"P3 *  * -KE-GI-KI *  * -FU-FU\n"
"P4-FU *  *  *  * -FU-FU *  * \n"
"P5 * -FU+KA-FU *  * -GI+FU+KY\n"
"P6+FU * +FU * +FU+FU *  * +HI\n"
"P7 * +FU * +FU * +GI *  *  * \n"
"P8 * +OU+KI+GI+KI *  *  *  * \n"
"P9+KY+KE *  *  *  *  *  *  * \n"
"P+00FU00FU\n"
"P-00KE00FU00FU\n"
"-\n"
				   ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(WHITE,state,Position(5,6),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,3),Position(6,4),GOLD,PTYPE_EMPTY,false,WHITE)) ||
		   (std::cerr << moves << std::endl,0));
  }
}

void AddEffectTest::testNear()
{
  const Position from = Position(7,4);
  const Position target = Position(6,2);
  CPPUNIT_ASSERT_EQUAL(NearMask::makeDirect(4),
		       Add_Effect_Table.getNearMaskWithPromote<BLACK>(PAWN,from,target));
}

void AddEffectTest::testShort()
{
  MoveVector moves;
  NumEffectState state(this->state);
  const Position target = Position(6,2);
  NearMask nearMask=NearMask::make<BLACK>(state,target);
  {
    move_action::Store store(moves);
    AddEffectShort<BLACK,PAWN>::generate(state, target, store, nearMask);
  }
  CPPUNIT_ASSERT(moves.size()==1);
}

void AddEffectTest::testLong()
{
  {
    NumEffectState state(CsaString(
				   "P1 *  *  *  * +HI *  * -KE-KY\n"
				   "P2 *  *  *  *  *  *  *  *  * \n"
				   "P3 * +NK * +TO *  * -KI-FU * \n"
				   "P4+OU+GI *  *  *  * -FU * -FU\n"
				   "P5-FU *  *  *  * -KI-KY-RY * \n"
				   "P6 *  *  *  * +FU+GI * -KY-OU\n"
				   "P7+FU-UM *  * +KI+GI *  *  * \n"
				   "P8 *  *  *  *  *  *  *  * -KI\n"
				   "P9 *  *  *  *  *  *  *  *  * \n"
				   "P+00GI00FU00FU\n"
				   "P-00KA00KE00KE00KY00FU00FU00FU00FU00FU00FU00FU00FU00FU\n"
				   "-\n"
				   ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(WHITE,state,Position(3,7),store);
    }
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,5),Position(3,5),PROOK,LANCE,false,WHITE)) ||
		   (std::cerr << moves << std::endl,0));
  }
  MoveVector moves;
  NumEffectState state(this->state);
  const Position target = Position(6,2);
  NearMask nearMask=NearMask::make<BLACK>(state,target);
  {
    move_action::Store store(moves);
    AddEffectLong<BLACK,ROOK>::generate(state, target, store, nearMask);
  }
  CPPUNIT_ASSERT(moves.size() == 11);
  {
    NumEffectState state(CsaString(
				   "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
				   "P2 * -HI *  *  *  *  * -KA * \n"
				   "P3-FU-FU-FU-FU-FU-FU * -FU-FU\n"
				   "P4 *  *  *  *  *  * -FU *  * \n"
				   "P5 *  *  *  *  *  *  *  *  * \n"
				   "P6 *  * +FU *  *  *  *  *  * \n"
				   "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
				   "P8 * +KA *  *  *  *  * +HI * \n"
				   "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
				   "+\n"
				   ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,state,Position(1,1),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(8,8),Position(2,2),PBISHOP,BISHOP,true,BLACK))||
		   (std::cerr << moves << std::endl,0));
  }
}

void AddEffectTest::testKing()
{
  MoveVector moves;
  NumEffectState state(this->state);
  const Position target = Position(9,7);
  NearMask nearMask=NearMask::make<BLACK>(state,target);
  {
    move_action::Store store(moves);
    AddEffect<BLACK>::generateKing(state, target, store, nearMask);
  }
  CPPUNIT_ASSERT(moves.size() == 2);
}

template<typename State>
static void testDrop(SimpleState& state){
  State eState(state);
  MoveVector moves;
  {
    move_action::Store store(moves);
    GenerateAddEffect<false>::
      generate(BLACK,eState,Position(5,3),store);
  }
  // 2歩は生成しない
  CPPUNIT_ASSERT(!moves.isMember(Move(Position(5,4),PAWN,BLACK)));
}

void AddEffectTest::testDrop(){
  ::testDrop<test_state_t>(state); 
}

template<typename State>
static void testMove(SimpleState& state){
  {
    SimpleState state1=CsaString(
"P1-KY-KE *  *  * -OU * -KE-KY\n"
"P2 *  *  *  *  *  * -KI-KA * \n"
"P3-FU * -FU-FU-HI-GI * +RY * \n"
"P4 * -FU *  *  * -FU+FU * -FU\n"
"P5 *  *  *  * -GI *  *  *  * \n"
"P6 *  * +FU+FU *  *  *  *  * \n"
"P7+FU+FU+GI *  * +FU+GI * +FU\n"
"P8 *  * +KI * +KI *  *  *  * \n"
"P9+KY+KE * +OU *  *  * +KE+KY\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P+00KI\n"
"P-00KA\n"
"+\n").getInitialState();
    State eState(state1);
    MoveVector moves;
  // 龍
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,eState,Position(3,1),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,3),Position(2,2),PROOK,BISHOP,false,BLACK)) ||
		   (std::cerr << moves << std::endl,0));
  }
  {
    SimpleState state1=CsaString(
"P1-KY-KE * -KI *  * -GI+RY-KY\n"
"P2 *  *  * -OU *  * -KI *  * \n"
"P3-FU * -GI-FU-FU-FU *  * -FU\n"
"P4 *  * -FU *  *  *  *  *  * \n"
"P5 *  *  *  * +KA *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
"P8 * +GI+KI *  *  *  *  *  * \n"
"P9+KY+KE *  * +OU+KI+GI * +KY\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P+00KE\n"
"P-00KE\n"
"P-00KA\n"
"P-00HI\n"
"+\n").getInitialState();
    State eState(state1);
    MoveVector moves;
    {
      Store store(moves);
  // 龍
      GenerateAddEffect<false>::
	generate(BLACK,eState,Position(4,1),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,1),Position(3,2),PROOK,GOLD,false,BLACK)) ||
		   (std::cerr << moves << std::endl,0));
  }
  State eState(state);
  MoveVector moves;
  {
    Store store(moves);
  // ケイマ
    GenerateAddEffect<false>::
      generate(BLACK,eState,Position(9,2),store);
  }
  CPPUNIT_ASSERT(moves.isMember(Move(Position(9,6),Position(8,4),KNIGHT,PTYPE_EMPTY,false,BLACK)));

  // promoteして利きをつける
  moves.clear();
  {
    Store store(moves);
    GenerateAddEffect<false>::
      generate(BLACK,eState,Position(6,2),store);
  }
  CPPUNIT_ASSERT(moves.isMember(Move(Position(7,4),Position(7,3),PPAWN,PTYPE_EMPTY,true,BLACK)));

}

void AddEffectTest::testMove(){
  ::testMove<test_state_t>(state);
}

template<typename State>
static void testOpen(SimpleState& state){
  {
    State eState(state);
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,eState,Position(3,3),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,3),Position(5,2),PBISHOP,PTYPE_EMPTY,false,BLACK))||
		   (std::cerr << moves << std::endl,0));
  }
  {
    SimpleState state1=CsaString(
"P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
"P2 * -HI *  *  *  *  * -KA * \n"
"P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
"P4 *  *  *  *  *  *  *  *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  * +FU *  *  *  *  *  * \n"
"P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
"P8 * +KA *  *  *  *  * +HI * \n"
"P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
"-\n"
).getInitialState();
    State eState(state1);
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(WHITE,eState,Position(5,5),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,3),Position(3,4),PAWN,PTYPE_EMPTY,false,WHITE))||
		   (std::cerr << moves << std::endl,0));
  }
}

// 開き王手
void AddEffectTest::testOpen(){
  ::testOpen<test_state_t>(state);
}

template<typename State>
static void testAdditional()
{
  {
    State state(CsaString(
			  "P1-KY+RY * +UM *  *  *  * -KY\n"
			  "P2 * -FU *  *  *  * +NK *  * \n"
			  "P3-FU * -GI-FU-FU-FU *  * -FU\n"
			  "P4 * -OU-FU *  *  *  *  *  * \n"
			  "P5 *  *  *  * +KA *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			  "P8 *  * -NK * +OU *  *  *  * \n"
			  "P9+KY+KE * -HI * +KI+GI * +KY\n"
			  "P+00KI00KE00FU00FU\n"
			  "P-00KI00KI00GI00GI00FU00FU\n"
			  "+\n"
			  ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,state,Position(2,1),store);
    }
    // double count +6143UM
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(4,3),PBISHOP,PAWN,false,BLACK)));
  }
  {
    State state(CsaString(
			  "P1-KY *  *  *  * -KI * -KE-KY\n"
			  "P2 *  *  *  * -GI *  *  * -OU\n"
			  "P3-FU *  *  * -KI-FU-FU-FU-FU\n"
			  "P4 *  * -FU * -FU *  *  *  * \n"
			  "P5 * -KE * -FU+FU *  * +KA * \n"
			  "P6+FU * +FU *  * +FU+HI * +FU\n"
			  "P7 * +FU * +FU-GI+UM * +FU * \n"
			  "P8 *  * +OU *  *  *  *  *  * \n"
			  "P9+KY+KE * -KI *  *  *  * +KY\n"
			  "P+00HI00KI00KE\n"
			  "P-00GI00GI00FU00FU\n"
			  "+\n").getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,state,Position(6,9),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,6),Position(3,3),PROOK,PAWN,true,BLACK))||
		   (std::cerr << moves << std::endl,0));
  }
}

// 追加利き
void AddEffectTest::testAdditional(){
  ::testAdditional<test_state_t>();
}

template<typename State>
static void testShadow()
{
  {
    State state(CsaString(
"P1-KY-KE *  *  * -OU-GI-KE-KY\n"
"P2 *  * -HI * -KI * -KI *  * \n"
"P3-FU * -FU-FU *  *  * -FU * \n"
"P4 *  *  *  * +FU+FU-FU * -FU\n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  * +FU * -KE+GI+FU * +FU\n"
"P7+FU+FU * +FU *  *  *  *  * \n"
"P8 * +GI+KI *  * +KI-UM *  * \n"
"P9+KY+KE * +OU+HI *  *  * +KY\n"
"P+00GI00FU00FU00FU\n"
"P-00KA00FU\n"
"-\n"
			  ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(WHITE,state,Position(5,7),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,8),Position(3,9),PBISHOP,PTYPE_EMPTY,false,WHITE)) ||
		   (std::cerr << moves << std::endl,0));
  }
  {
    State state(CsaString(
			  "P1-KY-KE * -KI *  * -GI+RY-KY\n"
			  "P2 *  *  * -OU *  * -KI *  * \n"
			  "P3-FU * -GI-FU-FU-FU *  * -FU\n"
			  "P4 *  * -FU *  *  *  *  *  * \n"
			  "P5 *  *  *  * +KA *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			  "P8 * +GI+KI *  *  *  *  *  * \n"
			  "P9+KY+KE *  * +OU+KI+GI * +KY\n"
			  "P+00KE00FU00FU\n"
			  "P-00HI00KA00KE00FU00FU00FU\n"
			  "+\n"
			  ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,state,Position(4,2),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(2,1),Position(1,2),PROOK,PTYPE_EMPTY,false,BLACK)) ||
		   (std::cerr << moves << std::endl,0));
  }
  {
    State state(CsaString(
			  "P1-KY-KE * -KI-OU * -GI-KE-KY\n"
			  "P2 * -GI *  *  *  * -KI *  * \n"
			  "P3-FU *  * -FU-FU-FU *  * -FU\n"
			  "P4 *  * -FU *  *  *  *  *  * \n"
			  "P5 *  *  *  * +KA *  *  *  * \n"
			  "P6 *  *  *  *  *  *  *  *  * \n"
			  "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			  "P8 * +GI+KI *  *  *  *  *  * \n"
			  "P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
			  "P+00HI00FU\n"
			  "P-00HI00KA00FU00FU00FU00FU\n"
			  "-\n"
			  ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(WHITE,state,Position(1,1),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,6),BISHOP,WHITE))||
		   (std::cerr << moves << std::endl,0));
  }
  {
    State state(CsaString(
			  "P1-KY *  *  *  * -KI * -KE-KY\n"
			  "P2 *  *  *  * -GI *  *  * -OU\n"
			  "P3-FU *  *  * -KI-FU-FU-FU-FU\n"
			  "P4 *  * -FU * -FU *  *  *  * \n"
			  "P5 * -KE * -FU+FU *  * +KA * \n"
			  "P6+FU * +FU *  * +FU+HI * +FU\n"
			  "P7 * +FU * +FU-GI-UM * +FU * \n"
			  "P8 *  * +OU *  *  *  *  *  * \n"
			  "P9+KY+KE * +KI *  *  *  * +KY\n"
			  "P+00HI00KI00KE\n"
			  "P-00GI00GI00FU00FU\n"
			  "+\n").getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,state,Position(6,9),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(3,6),Position(3,3),PROOK,PAWN,true,BLACK))||
		   (std::cerr << moves << std::endl,0));
  }
  {
    State state(CsaString(
			  "P1-OU-KE-KI * +KA *  * +RY-KY\n"
			  "P2-KY-GI+GI *  *  *  *  *  * \n"
			  "P3 * -FU *  *  *  *  *  *  * \n"
			  "P4-FU * -FU+KE * +UM *  *  * \n"
			  "P5 * +FU * -FU * -FU * -FU-FU\n"
			  "P6+FU * +FU * +FU-KY+FU *  * \n"
			  "P7 *  * +KE+FU *  * +KE+FU+FU\n"
			  "P8 *  * -HI *  * -GI+GI+OU * \n"
			  "P9 *  *  *  *  *  * +KI * +KY\n"
			  "P+00KI00FU\n"
			  "P-00KI00FU00FU\n"
			  "+\n"
			  ).getInitialState());
    MoveVector moves;
    {
      Store store(moves);
      GenerateAddEffect<false>::
	generate(BLACK,state,Position(3,9),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Position(5,1),Position(8,4),PBISHOP,PTYPE_EMPTY,true,BLACK))||
		   (std::cerr << moves << std::endl,0));
  }
}

// 追加利き
void AddEffectTest::testShadow(){
  ::testShadow<test_state_t>();
}

template<typename State> 
static bool hasShadowEffect(State& state, PtypeO ptypeO,Position from, Position to)
{
  if(!from.isOnBoard()) return false;
  EffectContent ec=Ptype_Table.getEffect(ptypeO,from,to);
  if(ec.hasBlockableEffect()){
    Offset offset=ec.offset();
    Piece p;
    for(from+=offset;(p=state.getPieceAt(from)).isEmpty();from+=offset)
      ;
    if(!p.isEdge() && state.hasEffectByPiece(p,to)) return true;
  }
  return false;
}

template<typename State> 
static bool isAddEffectMove(const State& state_org,Position pos,Move move){
  // 自分で動いて元のpositionに利きをつける手は含まない
  if(move.from()==pos) return false;
  bool shadowFlag=hasShadowEffect(state_org,move.ptypeO(),move.from(),pos);
  State state = state_org;
  ApplyMoveOfTurn::doMove(state, move);
  bool flag=state.hasEffectBy(move.player(),pos);
  if(!flag && state.getPieceAt(pos).ptype()!=KING){ // additional or shadow
    // check direct
    if(!shadowFlag && hasShadowEffect(state,move.ptypeO(),move.to(),pos))
      flag=true;
    // check open
    Position pos1=move.from();
    if(pos1.isOnBoard()){
      Offset offset=Board_Table.getShortOffset(Offset32(pos1,pos));
      if(!offset.zero()){
	Piece p;
	Position pos2;
	for(pos2=pos+offset;(p=state.getPieceAt(pos2)).isEmpty();pos2+=offset)
	  ;
	if(!p.isEdge() && state.hasEffectByPiece(p,pos)){
	  for(pos2+=offset;pos2!=pos1 && (p=state.getPieceAt(pos2)).isEmpty();pos2+=offset)
	    ;
	  if(pos2==pos1){
	    for(pos2+=offset;(p=state.getPieceAt(pos2)).isEmpty();pos2+=offset)
	      ;
	    if(pos2!=move.to() && p.isOnBoardByOwner(move.player()) &&
	       Ptype_Table.getEffect(p.ptypeO(),pos2,pos).hasEffect())
	      flag=true;
	  }
	}
      }
    }
  }
  return flag
    && (! state.inCheck(alt(state.getTurn()))); // 自殺
}

template<typename State>
static void testMoveFile(const std::string& fileName){
  using osl::search::ShouldPromoteCut;
  Record rec=CsaFile(fileName).getRecord();
  State state(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    if (! state.inCheck())
    {				// 王手以外の状況でテスト
      MoveVector allMoves;
      {
	Store store(allMoves);
	AllMoves<Store>::
	  generate(state.getTurn(),state,store);
      }
      for(int y=1;y<=9;y++)
	for(int x=9;x>0;x--){
	  Position pos=Position(x,y);
	  // 次の1行はaddEffect.hのバグの範囲を突き止めるためのテスト
	  //	if(state.hasEffectBy(state.getTurn(),pos)) continue;
	  MoveVector effectMoves;
	  {
	  Store storeEffect(effectMoves);
	  if(pos==state.getKingPosition(alt(state.getTurn()))){
	    continue;
	  }
	  else
	    GenerateAddEffect<false>::
	      generate(state.getTurn(),state,pos,storeEffect);
	  }
	  effectMoves.unique();
	
	  size_t count1=0;
	  for(size_t j=0;j<effectMoves.size();j++){
	    CPPUNIT_ASSERT((state.isValidMove(effectMoves[j],true)  
			    && isAddEffectMove<State>(state,pos,effectMoves[j]))
			   || (std::cerr << effectMoves[j] << std::endl << state << std::endl << "pos=" << pos << std::endl,0)
			    );
	    if(!ShouldPromoteCut::canIgnoreAndNotDrop(effectMoves[j])) count1++;
	  }
	  if(state.hasEffectBy(state.getTurn(),pos)) continue;
	  // 
	  size_t count=0;
	  MoveVector tmpMoves;
	  for(size_t j=0;j<allMoves.size();j++){
	    if(isAddEffectMove<State>(state,pos,allMoves[j])){
	      if(ShouldPromoteCut::canIgnoreAndNotDrop(allMoves[j])) continue;
	      tmpMoves.push_back(allMoves[j]);
	      count++;
	      CPPUNIT_ASSERT(effectMoves.isMember(allMoves[j]) ||
			     (std::cerr << state << std::endl << pos << "," <<
			      allMoves[j] << std::endl,1)
			     );
	    }
	  }
	  CPPUNIT_ASSERT(count == count1 ||
			 (std::cerr << state << std::endl << "count=" << count << ",count1=" << count1 << std::endl,0)
	    );
	}
    }
    Move move=moves[i];
    ApplyMoveOfTurn::doMove(state, move);
  }
}

void AddEffectTest::testMoveMember(){
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=20;
  if (isShortTest) 
    count=2;
  std::string fileName;
  while((ifs >> fileName) && ++i<count){
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testMoveFile<test_state_t>(OslConfig::testCsaFile(fileName));
  }
}


// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
