/* usi.t.cc
 */
#include "osl/record/usi.h"
#include "osl/state/simpleState.h"
#include "osl/record/csaString.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class UsiTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(UsiTest);
  CPPUNIT_TEST(testMove);
  CPPUNIT_TEST(testParseBoard);
  CPPUNIT_TEST(testParseBoard2);
  CPPUNIT_TEST(testParse);
  CPPUNIT_TEST(testParseBoardError);
  CPPUNIT_TEST(testParseError);
  CPPUNIT_TEST(testKingAbsence);
  CPPUNIT_TEST(testShowPosition);
  CPPUNIT_TEST_SUITE_END();
public:
  void testMove();
  void testParseBoard();
  void testParseBoard2();
  void testParse();
  void testParseBoardError();
  void testParseError();
  void testKingAbsence();
  void testShowPosition();
};

CPPUNIT_TEST_SUITE_REGISTRATION(UsiTest);

using namespace osl;
using namespace osl::record;

void UsiTest::testMove()
{
  SimpleState state(HIRATE);
  const Move m76fu = Move(Position(7,7), Position(7,6), PAWN, 
			  PTYPE_EMPTY, false, BLACK);
  const std::string usi76fu = "7g7f";
  const std::string usi76fu2 = usi::show(m76fu);
  CPPUNIT_ASSERT_EQUAL(usi76fu, usi76fu2);

  const Move m76fu2 = usi::strToMove(usi76fu, state);
  CPPUNIT_ASSERT_EQUAL(m76fu, m76fu2);
}

void UsiTest::testParseBoard()
{
  SimpleState state(HIRATE);
  const std::string hirate = "lnsgkgsnl/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL";

  SimpleState state2;
  usi::parseBoard(hirate, state2);
  CPPUNIT_ASSERT_EQUAL(state, state2);
}

void UsiTest::testParseBoard2()
{
  SimpleState state(HIRATE);
  const std::string hirate = "8l/1l+R2P3/p2pBG1pp/kps1p4/Nn1P2G2/P1P1P2PP/1PS6/1KSG3+r1/LN2+p3L";

  SimpleState state2;
  CPPUNIT_ASSERT_NO_THROW(usi::parseBoard(hirate, state2));
}

void UsiTest::testParse()
{
  {
    SimpleState state(CsaString(
			"P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			"P2 * -HI *  *  *  *  * -KA * \n"
			"P3-FU-FU-FU-FU-FU-FU * -FU-FU\n"
			"P4 *  *  *  *  *  * -FU *  * \n"
			"P5 *  *  *  *  *  *  *  *  * \n"
			"P6 *  * +FU *  *  *  * +FU * \n"
			"P7+FU+FU * +FU+FU+FU+FU * +FU\n"
			"P8 * +KA *  *  *  *  * +HI * \n"
			"P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			"-\n").getInitialState());

    const std::string s763426 = "startpos moves 7g7f 3c3d 2g2f";

    SimpleState state2;
    usi::parse(s763426, state2);
    CPPUNIT_ASSERT_EQUAL(state, state2);
  }
  {
    SimpleState state(CsaString(
			"P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			"P2 * -HI *  *  *  *  *  *  * \n"
			"P3 *  *  * -FU-FU-FU-FU-FU-FU\n"
			"P4 *  *  *  *  *  *  *  *  * \n"
			"P5 *  *  *  *  *  *  *  *  * \n"
			"P6 *  *  *  *  *  *  *  *  * \n"
			"P7+FU+FU+FU+FU+FU+FU+FU *  * \n"
			"P8 * +KA *  *  *  *  * +HI * \n"
			"P9+KY+KE * +KI+OU+KI+GI+KE+KY\n"
			"P+00GI00FU00FU\n"
			"P-00KA00FU00FU00FU\n"
			"-\n").getInitialState());

    const std::string stest = "sfen lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL w S2Pb3p 1";

    SimpleState state2;
    usi::parse(stest, state2);
    CPPUNIT_ASSERT_EQUAL(state, state2);

    const std::string stest3 = "sfen lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL w S2Pb3p";
    SimpleState state3;
    usi::parse(stest3, state3);
    CPPUNIT_ASSERT_EQUAL(state, state3);
  }
}

void UsiTest::testParseBoardError()
{
  SimpleState state;

  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsgkgsna/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL", state), // invalid ascii character a
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("", state), // empty
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsgkgsnl/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL+", state), // ends with +
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("8l/1l+R2P3/p2pBG1pp/kps1p4/Nn1P2G2/P1P1P2PP/1PS6/1KSG3+r1/LN2+3L", state), // should be ...+p3L
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsg+kgsnl/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL", state), // k can not be promoted
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsgkgsnl/1r5b2/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL", state), // too many columns at the second row
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsgkgsnl/1r5b1/pppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL", state), // too many columns at the third row
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsgkgsnl/1r5b10/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL", state), // 0 is invalid
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parseBoard("lnsgkgsn?/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL", state), // ? is invalid
    osl::record::usi::ParseError);
}

void UsiTest::testParseError()
{
  SimpleState state;

  CPPUNIT_ASSERT_THROW(usi::parse("lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL w S2Pb3p 1", state), // should start with sfen
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parse("sfen lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL l S2Pb3p 1", state), // should be black or white
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parse("sfen lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL w S2Ab3p 1", state), // invalid ascii character in hands
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parse("sfen lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL w S2#b3p 1", state), // invalid character in hands
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_THROW(usi::parse("sfen lnsgkgsnl/1r7/3pppppp/9/9/9/PPPPPPP2/1B5R1/LN1GKGSNL w S0Pb3p 1", state), // 0 is invalid
    osl::record::usi::ParseError);
  CPPUNIT_ASSERT_NO_THROW(usi::parse("sfen lnsgkgsnl/1r5b1/ppppppppp/9/9/9/PPPPPPPPP/1B5R1/LNSGKGSNL w - a", state)); // invalid moves are not respected
}

void UsiTest::testKingAbsence()
{
#ifdef ALLOW_KING_ABSENCE
  SimpleState state;
  usi::parse("sfen 3l+R4/1l1g5/2k1n4/1p+p2s3/4+r+B3/1G1s+p4/9/2N2+p3/9 b BGS2NPgs2l13p 1", state);
  usi::parse("sfen 6n2/+PS1+Ps1k2/3s+B1l2/1Lp1G1p1S/3p1n1+R+p/3+p1+nG2/4Pp1G1/6G+r1/5L1N+B b 2Pl7p 1", state);
#endif
}

void UsiTest::testShowPosition()
{
  {
    SimpleState state(HIRATE);
    CPPUNIT_ASSERT_EQUAL(std::string("startpos"), usi::show(state));
  }
  {
    std::string str = "sfen kng5l/ls1S2r2/pppPp2Pp/7+B1/7p1/2Pp2P2/PP3P2P/L2+n5/K1G5L b B2S2N2Pr2gp 1";
    SimpleState state(CsaString(
			"P1-OU-KE-KI *  *  *  *  * -KY\n"
			"P2-KY-GI * +GI *  * -HI *  * \n"
			"P3-FU-FU-FU+FU-FU *  * +FU-FU\n"
			"P4 *  *  *  *  *  *  * +UM * \n"
			"P5 *  *  *  *  *  *  * -FU * \n"
			"P6 *  * +FU-FU *  * +FU *  * \n"
			"P7+FU+FU *  *  * +FU *  * +FU\n"
			"P8+KY *  * -NK *  *  *  *  * \n"
			"P9+OU * +KI *  *  *  *  * +KY\n"
			"P+00KA00GI00GI00KE00KE00FU00FU\n"
			"P-00HI00KI00KI00FU\n"
			"+").getInitialState());
    CPPUNIT_ASSERT_EQUAL(str, usi::show(state));
    SimpleState test;
    usi::parse(str, test);
    CPPUNIT_ASSERT_EQUAL(state, test);
  }
  {
    std::string str = "sfen lnsg4l/2k2R+P2/pppp1s2p/9/4p1P2/2P2p+n2/PP1P4P/2KSG1p2/LN1G4L w BG2Prbsnp 1";
    SimpleState state(CsaString(
			"P1-KY-KE-GI-KI *  *  *  * -KY\n"
			"P2 *  * -OU *  * +HI+TO *  * \n"
			"P3-FU-FU-FU-FU * -GI *  * -FU\n"
			"P4 *  *  *  *  *  *  *  *  * \n"
			"P5 *  *  *  * -FU * +FU *  * \n"
			"P6 *  * +FU *  * -FU-NK *  * \n"
			"P7+FU+FU * +FU *  *  *  * +FU\n"
			"P8 *  * +OU+GI+KI * -FU *  * \n"
			"P9+KY+KE * +KI *  *  *  * +KY\n"
			"P+00KA00KI00FU00FU\n"
			"P-00HI00KA00GI00KE00FU\n"
			"-\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(str, usi::show(state));
    SimpleState test;
    usi::parse(str, test);
    CPPUNIT_ASSERT_EQUAL(state, test);
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
