/* breakThreatmate.t.cc
 */
#include "osl/search/breakThreatmate.h"
#include "osl/state/numEffectState.h"
#include "osl/record/csaString.h"
#include "osl/container/moveLogProbVector.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

namespace osl
{
  namespace search
  {
    class BreakThreatmateTest : public CppUnit::TestFixture 
    {
      CPPUNIT_TEST_SUITE(BreakThreatmateTest);
      CPPUNIT_TEST(testGenerate);
      CPPUNIT_TEST(testBlock);
      CPPUNIT_TEST(testDropValidity);
      CPPUNIT_TEST(testDropCandidate);
      CPPUNIT_TEST_SUITE_END();
    public:
      void testGenerate();
      void testBlock();
      void testDropValidity();
      void testDropCandidate();
    };
  }
}

CPPUNIT_TEST_SUITE_REGISTRATION(osl::search::BreakThreatmateTest);

void osl::search::BreakThreatmateTest::testBlock()
{
  {
    NumEffectState state(CsaString(
			   "P1-OU-KE *  *  *  *  *  * +TO\n"
			   "P2-KY * -KI+TO+KI * -FU *  * \n"
			   "P3 * -FU *  *  * -FU-KE *  * \n"
			   "P4-FU * +FU-UM * -RY *  *  * \n"
			   "P5 *  *  * -FU * -GI *  *  * \n"
			   "P6+FU * +KI * -KA-KY *  *  * \n"
			   "P7 * +FU *  *  *  *  * -FU+FU\n"
			   "P8+KY+GI-GI *  *  *  *  *  * \n"
			   "P9+OU+KE * +KY * +FU *  *  * \n"
			   "P+00HI00GI00KE00FU00FU00FU00FU00FU\n"
			   "P-00KI\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    BreakThreatmate::generate(500, state, Move(Position(7,8), Position(8,9), PSILVER, KNIGHT, true, WHITE), moves);
    CPPUNIT_ASSERT(! moves.empty());    
    CPPUNIT_ASSERT(moves.find(Move(Position(6,7), PAWN, BLACK)));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-HI *  *  *  *  *  * +TO\n"
			   "P2 * -OU-FU *  * +RY *  *  * \n"
			   "P3 *  *  *  *  * -KE *  *  * \n"
			   "P4 *  * +KI * -GI * +KA-FU-FU\n"
			   "P5-FU * +GI * -FU *  *  *  * \n"
			   "P6 *  *  *  *  * -FU+FU-KY+FU\n"
			   "P7+FU *  *  *  *  *  *  *  * \n"
			   "P8+KY-FU *  *  * +KI+GI *  * \n"
			   "P9 * -NG-KA * -TO * +OU+KE+KY\n"
			   "P+00KE\n"
			   "P-00KI00KI00KE00FU00FU00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    BreakThreatmate::generate(500, state, Move(Position(2,8), GOLD, WHITE), moves);
    CPPUNIT_ASSERT(! moves.empty());    
    CPPUNIT_ASSERT(moves.find(Move(Position(2,7), KNIGHT, BLACK)));
  }
  {
    NumEffectState state(CsaString(
			   "P1-OU-KE *  *  * +HI *  * -KY\n"
			   "P2-KY-GI+KI-FU *  * +FU *  * \n"
			   "P3 *  *  *  *  * -FU *  * -FU\n"
			   "P4-FU-FU * -KI-FU * -FU-FU * \n"
			   "P5 *  *  * -GI *  *  *  * +FU\n"
			   "P6+FU * -FU-KE+FU *  *  *  * \n"
			   "P7 * +FU+KI *  * +FU *  *  * \n"
			   "P8+KY+GI *  *  *  *  *  *  * \n"
			   "P9+OU+KE+KI *  * -HI *  * +KY\n"
			   "P+00FU00FU\n"
			   "P-00KA00KA00GI00KE00FU\n"
			   "-\n").getInitialState());
    MoveLogProbVector moves;
    const Move m81ki(Position(7,2), Position(8,1), GOLD, KNIGHT, false, BLACK);
    BreakThreatmate::generate(500, state, m81ki, moves);
    const Move m71gi(Position(7,1), SILVER, WHITE); // Ϥϵͤ
    CPPUNIT_ASSERT(moves.isMember(MoveLogProb(m71gi, 300)));
  }
}

void osl::search::BreakThreatmateTest::testDropValidity()
{
  {
    NumEffectState state(CsaString(
			   "P1+OU *  *  *  *  * -KI-KE-OU\n"
			   "P2+FU-FU-GI *  *  * -KI-GI-KY\n"
			   "P3 *  *  *  *  *  * -FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  *  *  *  *  *  *  *  * \n"
			   "P+00KE00KY00FU\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    BreakThreatmate::generate(500, state, Move(Position(8,1), ROOK, WHITE), moves);
    CPPUNIT_ASSERT(! moves.empty());    
    CPPUNIT_ASSERT(moves.size() == 3); // 73KE, 93KE, 82OU
  }
}

void osl::search::BreakThreatmateTest::testGenerate()
{
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -KI-KE-OU\n"
			   "P2 *  *  *  *  *  * -KI-GI-KY\n"
			   "P3 *  *  *  *  *  * -FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7-FU *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9+OU *  *  *  *  *  *  *  * \n"
			   "P-00KI\n"
			   "P+00AL\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    BreakThreatmate::generate(500, state, Move(Position(9,8), GOLD, WHITE), 
				   moves);
    CPPUNIT_ASSERT(! moves.empty());    
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -KI-KE-OU\n"
			   "P2 *  *  *  *  *  * -KI-GI-KY\n"
			   "P3 *  *  *  *  *  * -FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU+FU+FU+FU *  *  *  *  * \n"
			   "P8+OU * +KI *  * -HI *  *  * \n"
			   "P9+KY+KE-GI *  *  *  *  *  * \n"
			   "P+00AL\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    const Move threatmate_move(Position(4,8), Position(7,8), PROOK, 
			       GOLD, false, WHITE);
    BreakThreatmate::generate(500, state, threatmate_move, moves);
    CPPUNIT_ASSERT(! moves.empty());    
    CPPUNIT_ASSERT(moves.isMember(MoveLogProb(Move(Position(5,8), GOLD, BLACK),
					      400))); 
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -KI-KE-OU\n"
			   "P2 *  *  *  *  *  * -KI-GI-KY\n"
			   "P3 *  *  *  *  *  * -FU-FU-FU\n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 * -FU-KE-FU *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU+FU+FU *  *  *  *  *  * \n"
			   "P8+KY+KA *  *  *  *  *  *  * \n"
			   "P9+OU+KE *  *  *  *  *  *  * \n"
			   "P+00AL\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    const Move threatmate_move(Position(7,5), Position(8,7), KNIGHT, 
			       PAWN, false, WHITE);
    BreakThreatmate::generate(500, state, threatmate_move, moves);
    CPPUNIT_ASSERT(! moves.empty());    
    CPPUNIT_ASSERT(moves.isMember(MoveLogProb(Move(Position(8,8),Position(7,9),
						   BISHOP, PTYPE_EMPTY, false,
						   BLACK), 100))); 
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE *  *  *  *  *  * -KY\n"
			   "P2 *  *  * -FU+FU *  *  *  * \n"
			   "P3-FU+RY *  *  *  * +TO+HI-FU\n"
			   "P4 *  *  * +FU * -GI+KA *  * \n"
			   "P5 *  * +FU * -GI-FU-FU *  * \n"
			   "P6 * +KI *  * -OU * -KE *  * \n"
			   "P7+FU+FU-KI *  * -TO *  * +FU\n"
			   "P8+OU * -KI *  *  *  *  *  * \n"
			   "P9+KY *  *  *  *  * +GI * +KY\n"
			   "P+00KA00KI00GI00KE00FU00FU00FU\n"
			   "P-00KE00FU00FU\n"
			   "+\n").getInitialState());
    MoveLogProbVector moves;
    const Move m88ki(Position(7,8), Position(8,8), GOLD, PTYPE_EMPTY, false, WHITE);
    BreakThreatmate::generate(500, state, m88ki, moves);
    const Move m96fu(Position(9,7), Position(9,6), PAWN, PTYPE_EMPTY, false, BLACK);
    CPPUNIT_ASSERT(moves.isMember(MoveLogProb(m96fu, 100)));
  }
}

void osl::search::BreakThreatmateTest::testDropCandidate()
{
  {
    NumEffectState state(CsaString(
			   "P1-OU-KE *  *  *  *  *  * -KY\n"
			   "P2-KY * +KI-FU *  * +FU *  * \n"
			   "P3 *  *  * -KI * -FU *  * -FU\n"
			   "P4-FU-FU * -GI-FU * -FU-FU * \n"
			   "P5 *  *  * -GI *  *  *  * +FU\n"
			   "P6+FU * -FU-KE+FU+KA *  *  * \n"
			   "P7 * +FU+KI *  * +FU *  *  * \n"
			   "P8+KY+GI *  *  *  *  *  *  * \n"
			   "P9+OU+KE+KI *  * -HI *  * +KY\n"
			   "P+00HI00FU00FU\n"
			   "P-00KA00GI00KE00FU\n"
			   "-\n").getInitialState());
    MoveLogProbVector moves;
    const Move m64ka(Position(4,6), Position(6,4), BISHOP, SILVER, false, BLACK);
    BreakThreatmate::generate(500, state, m64ka, moves);
    const Move m71gi(Position(7,1), SILVER, WHITE);
    CPPUNIT_ASSERT(moves.isMember(MoveLogProb(m71gi, 100)));
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
