/* square.h
 */
#ifndef OSL_SQUARE_H
#define OSL_SQUARE_H

#include "osl/misc/loki.h"
#include "osl/offset.h"
#include "osl/directionTraits.h"
#include <boost/utility/enable_if.hpp>

namespace osl 
{
  /**
   * 座標.
   *        盤面のインデックス
   * X, Yも1-9の範囲で表す 
   * Xは右から数える．Yは上から数える
   * なお駒台は0
   * <pre>
   * (A0)  ......................... (00)
   * (A1)  ......................... (01)
   * (A2) 92 82 72 62 52 42 32 22 12 (02)
   * (A3) 93 83 73 63 53 43 33 23 13 (03)
   * (A4) 94 84 74 64 54 44 34 24 14 (04)
   * (A5) 95 85 75 65 55 45 35 25 15 (05)
   * (A6) 96 86 76 66 56 46 36 26 16 (06)
   * (A7) 97 87 77 67 57 47 37 27 17 (07)
   * (A8) 98 88 78 68 58 48 38 28 18 (08)
   * (A9) 99 89 79 69 59 49 39 29 19 (09)
   * (AA) 9A 8A 7A 6A 5A 4A 3A 2A 1A (0A)
   * (AB) ...........................(0B)
   * (AC) ...........................(0C)
   * (AD) ...........................(0D)
   * (AE) ...........................(0E)
   * (AF) ...........................(0F) 
   * </pre>
   */
  class Square;
  bool operator==(Square l, Square r);

  class Square
  {
    unsigned int square;
    explicit Square(int p) : square(p)
    {
    }
  public:
    static const Square makeDirect(int value) { return Square(value); }
    unsigned int uintValue() const { return square; }
    enum {
      PIECE_STAND=0,
      MIN=0,
      SIZE=0x100
    };
    Square() : square(PIECE_STAND)
    {
    }
    static const Square STAND() { return Square(PIECE_STAND); }
    Square(int x, int y) : square((x*Offset::BOARD_HEIGHT)+y+1)
    {
      assert(square < SIZE);
    }
    /**
     * assertなしに作る
     */
    static const Square makeNoCheck(int x, int y) { 
      return Square((x*Offset::BOARD_HEIGHT)+y+1); 
    }
    static const Square nth(unsigned int i) { return Square(i+MIN); }
    /**
     * 将棋としてのX座標を返す. Squareの内部表現に依存しない．
     */
    int x() const { return square >> 4; }
    /**
     * 将棋としてのY座標を返す. Squareの内部表現に依存しない．
     */
    int y() const { return (square&0xf)-1; }
    /**
     * y+1を返す
     */
    int y1() const { return square&0xf; }
    unsigned int index() const { return square - MIN; }
    static unsigned int indexMax() { return SIZE - MIN; }
    int indexForOffset32() const { return square + (square&0xf0); }

    bool isPieceStand() const { return square == PIECE_STAND; }
    bool isOnBoardSlow() const;
    /**
     * 盤面上を表すかどうかの判定．
     * 1<=x() && x()<=9 && 1<=y() && y()<=9
     * Squareの内部表現に依存する．
     */
    bool isOnBoard() const { 
      return (0xffffff88&(square-0x12)&
	      ((unsigned int)((square&0x77)^0x12)+0xffffff77))==0;
    }
    /**
     * onBoardから8近傍のオフセットを足した点がedgeかどうかの判定
     * そこそこ速くなった．
     */
    bool isEdge() const { 
      assert(!isPieceStand() && 0<=x() && x()<=10 && 0<=y() && y()<=10);
      return (0x88&(square-0x12)&((square&0x11)+0xf7))!=0;
    }
    bool isValid() const;


    const Square squareForBlackSlow(Player player) const;

    const Square squareForBlack(Int2Type<BLACK>) const{
      assert(isOnBoard());
      Square ret=*this;
      return ret;
    }

    const Square squareForBlack(Int2Type<WHITE>) const{
      assert(isOnBoard());
      Square ret=makeDirect(Square(9,9).uintValue()+Square(1,1).uintValue()-uintValue());
      return ret;
    }
    /** 
     * 後手の場合は盤面を引っくり返す.
     * PIECE_STANDの場合は扱えない．
     */
    template<Player P>
    const Square squareForBlack() const{
      return squareForBlack(Int2Type<P>());
    }
    const Square squareForBlack(Player player) const{
      if(player==BLACK)
	return squareForBlack<BLACK>();
      else
	return squareForBlack<WHITE>();
    }

    const Square rotate180() const 
    {
      return squareForBlack<WHITE>();
    }
    const Square rotate180EdgeOK() const 
    {
      Square ret=makeDirect(Square(9,9).uintValue()+Square(1,1).uintValue()-uintValue());
      return ret;
    }
    const Square rotate180Safe() const 
    {
      if (isPieceStand())
	return *this;
      return squareForBlack<WHITE>();
    }
    const Square flipHorizontal() const
    {
      if (isPieceStand())
	return *this;
      return Square(10-x(), y());
    }

    static const Square onBoardMax(){ return Square(9,9); }
    static const Square onBoardMin(){ return Square(1,1); }
  
    /**
     * squareがONBOARD_MINとONBOARD_MAXの間にある
     */
    bool isOnBoardRegion() const {
      return static_cast<unsigned int>(index()-onBoardMin().index()) 
	<= static_cast<unsigned int>(onBoardMax().index()-onBoardMin().index());
    }

    Square& operator++() {
      square += 1;
      return *this;
    }

    static int reverseX(int x) { return 10-x; }
    static int reverseY(int y) { return 10-y; }
  private:
    template<Player P>
    static bool canPromoteY(int y,Int2Type<P>);
    static bool canPromoteY(int y,Int2Type<BLACK>)
    {
      return y <= 3;
    }
    static bool canPromoteY(int y,Int2Type<WHITE>)
    {
      return y >= 7;
    }
    bool canPromote(Int2Type<BLACK>) const{
      return (uintValue()&0xf)<=4;
    }
    bool canPromote(Int2Type<WHITE>) const{
      return (uintValue()&0x8)!=0;
    }
  public:
    template <Player P>
    static bool canPromoteY(int y) { return canPromoteY(y,Int2Type<P>()); }
    template <Player P>
    bool canPromote() const{
      return canPromote(Int2Type<P>());
    }
    bool canPromote(Player player) const 
    {
      if (player==BLACK) 
	return canPromote<BLACK>();
      else 
	return canPromote<WHITE>();
    }
    /**
     * 2つのSquare(onBoardであることが前提)が，
     * xが等しいかyが等しい
     */
    bool isULRD(Square sq) const{
      assert(isOnBoard() && sq.isOnBoard());
      unsigned int v=uintValue() ^ sq.uintValue();
      return (((v+0xefull)^v)&0x110ull)!=0x110ull;
    }
    /**
     * 2つのSquare(onBoardであることが前提)のxが等しい
     */
    bool isUD(Square sq) const{
      assert(isOnBoard() && sq.isOnBoard());
      unsigned int v=uintValue() ^ sq.uintValue();
      return (v&0xf0)==0;
    }
    /**
     * sqがPlayer Pにとって上
     */
    template<Player P>
    bool isU(Square sq) const{
      assert(isOnBoard() && sq.isOnBoard());
      unsigned int v=uintValue() ^ sq.uintValue();
      if(P==BLACK)
	return ((v|(uintValue()-sq.uintValue()))&0xf0)==0;
      else
	return ((v|(sq.uintValue()-uintValue()))&0xf0)==0;
    }
    /**
     * 2つのSquare(onBoardであることが前提)のyが等しい
     */
    bool isLR(Square sq) const{
      assert(isOnBoard() && sq.isOnBoard());
      unsigned int v=uintValue() ^ sq.uintValue();
      return (v&0xf)==0;
    }
    Square& operator+=(Offset offset) {
      square += offset.intValue();
      return *this;
    }
    Square& operator-=(Offset offset) {
      square -= offset.intValue();
      return *this;
    }
    const Square operator+(Offset offset) const {
      Square result(*this);
      return result+=offset;
    }
    const Square operator-(Offset offset) const {
      Square result(*this);
      return result-=offset;
    }
    const Offset operator-(Square other) const {
      return Offset::makeDirect(square - other.square);
    }
    template<int Y>
    bool yEq() {
      return (uintValue()&0xf)==(Y+1);
    }
    template<int Y>
    bool yLe(typename boost::enable_if_c<Y != 2>::type * =0) {
      return (uintValue()&0xf)<=(Y+1);
    }
    template<int Y>
    bool yLe(typename boost::enable_if_c<Y == 2>::type * =0) {
      return (uintValue()&0xc)==0;
    }
    template<int Y>
    bool yGe(typename boost::enable_if_c<Y!=7>::type * =0) {
      return (uintValue()&0xf)>=(Y+1);
    }
    template<int Y>
    bool yGe(typename boost::enable_if_c<Y==7>::type * =0) {
      return (uintValue()&0x8)!=0;
    }
    template <Player P, Direction D>
    const Square neighbor() const {
      return *this + DirectionPlayerTraits<D,P>::offset();
    }
    template <Player P, Direction D>
    const Square back() const {
      return neighbor<PlayerTraits<P>::opponent,D>();
    }
  };

  inline bool operator==(Square l, Square r)
  {
    return l.uintValue() == r.uintValue();
  }
  inline bool operator!=(Square l, Square r)
  {
    return ! (l == r);
  }
  inline bool operator<(Square l, Square r)
  {
    return l.uintValue() < r.uintValue();
  }
  inline bool operator>(Square l, Square r)
  {
    return l.uintValue() > r.uintValue();
  }
  std::ostream& operator<<(std::ostream&, Square);

} // namespace osl

#endif /* OSL_POSITION_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
