#include "osl/move_generator/escape_.h"
#include "osl/move_generator/allMoves.h"
#include "osl/move_action/store.h"
#include "osl/move_classifier/classifier.h"
#include "osl/move_classifier/safeMove.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <boost/foreach.hpp>
#include <boost/progress.hpp>
#include <algorithm>
#include <iostream>
#include <fstream>

typedef osl::NumEffectState test_state_t;
class EscapeMovesTest : public CppUnit::TestFixture 
#if OSL_WORDSIZE == 32
		      , public osl::misc::Align16New
#endif
{
  CPPUNIT_TEST_SUITE( EscapeMovesTest );
  CPPUNIT_TEST(testBug);
  CPPUNIT_TEST(testValid);
  CPPUNIT_TEST(testCapture);
  CPPUNIT_TEST(testEscape);
  CPPUNIT_TEST(testBlocking);
  CPPUNIT_TEST(testMoveMember);
  CPPUNIT_TEST_SUITE_END();
private:
  osl::NumEffectState state;
public:
  EscapeMovesTest() : state((osl::SimpleState(osl::HIRATE)))
  {
  }
  void setUp();
  void testBug();
  void testValid();
  void testCapture();
  void testEscape();
  void testBlocking();
  void testMoveMember();
};

CPPUNIT_TEST_SUITE_REGISTRATION(EscapeMovesTest);

using namespace osl;
using namespace osl::move_action;
using namespace osl::move_generator;

void assertSafe(const NumEffectState& state, const MoveVector& moves)
{
  for (size_t i=0; i<moves.size(); ++i)
  {
    CPPUNIT_ASSERT(move_classifier::SafeMove<BLACK>::isMember
		   (state, moves[i].ptype(), moves[i].from(),
		     moves[i].to()));
  }
}

void EscapeMovesTest::testBug()
{
  NumEffectState state(CsaString(
			 "P1-KY *  *  *  *  *  *  * +UM\n"
			 "P2 * +KI *  * +HI *  *  * -KY\n"
			 "P3 *  * -KE *  *  *  *  *  * \n"
			 "P4-FU-OU *  * -KI *  * -FU-FU\n"
			 "P5 * +KE-GI-FU *  *  * +KE * \n"
			 "P6+FU+OU *  * +FU+FU *  * +FU\n"
			 "P7 *  *  *  * +GI *  *  *  * \n"
			 "P8 *  *  *  *  *  *  * +HI * \n"
			 "P9+KY *  * -UM *  *  *  * +KY\n"
			 "P+00FU00FU00FU00FU00FU00FU00FU00FU00KI\n"
			 "P-00FU00FU00KE00GI00GI00KI\n"
			 "+\n").getInitialState());
  test_state_t eState(state);
  const Square kingSquare = Square(8,6);
  {
    effect::Liberty8<BLACK> liberty(eState, kingSquare);
    CPPUNIT_ASSERT_EQUAL(2, liberty.count());
  }
  {
    effect::Liberty8<BLACK> liberty(state, kingSquare);
    CPPUNIT_ASSERT_EQUAL(2, liberty.count());
  }
  
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store>::generateCaptureKing<BLACK>(eState,state.pieceAt(kingSquare),
			  Square(7,5), store);
  }
  assertSafe(eState, moves);
  moves.clear();
  {
    Store store(moves);
    Escape<Store>::
      generateEscape<BLACK,KING>(eState,state.pieceAt(kingSquare),store);
  }
  assertSafe(eState, moves);
  CPPUNIT_ASSERT_EQUAL((size_t)2, moves.size());

  moves.clear();
  {
    Store store(moves);
    Escape<Store>::
      generateKingEscape<BLACK,false>(eState, store);
  }
  assertSafe(eState, moves);
  CPPUNIT_ASSERT_EQUAL((size_t)2, moves.size());
}


void EscapeMovesTest::setUp(){
  state=CsaString(
"P1+NY *  *  *  *  * -OU * -KY\n"
"P2 * +TO *  *  * -GI-KI *  *\n"
"P3 * -RY *  * +UM * -KI-FU-FU\n"
"P4 *  * +FU-FU *  *  *  *  *\n"
"P5+KE * -KE * +FU *  * +FU *\n"
"P6 *  *  * +FU+GI-FU *  * +FU\n"
"P7+KE * -UM *  *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n"
).getInitialState();
}


static void testValid(NumEffectState& state){
  {
    NumEffectState state1=CsaString(
"P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
"P2 *  *  *  *  *  * -KI *  * \n"
"P3-FU * -FU-FU-FU-FU *  * -FU\n"
"P4 *  * +HI *  *  *  *  *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  *  *  *  *  *  *  *  * \n"
"P7+FU * +FU+FU+FU+FU+FU * +FU\n"
"P8 * +GI+KI *  *  *  *  *  * \n"
"P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P+00KA\n"
"P-00KA\n"
"P+00HI\n"
"-\n"
).getInitialState();
    test_state_t eState(state1);
    MoveVector moves;
    {
      Store store(moves);
      Escape<Store >::
	generateMoves<WHITE,false>(eState,eState.pieceAt(Square(7,3)),eState.pieceAt(Square(7,4)),store);
    }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
    
    CPPUNIT_ASSERT(moves.isMember(Move(Square(7,3),Square(7,4),PAWN,ROOK,false,WHITE)));
  }
  {
    NumEffectState state1=CsaString(
"P1-KY+RY *  * -FU-OU * -KE-KY\n"
"P2 *  *  *  * +GI-GI-KI *  * \n"
"P3-FU * -FU-FU *  *  *  *  * \n"
"P4 * -FU *  *  * -FU+FU * -FU\n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 *  * +FU+FU *  *  *  *  * \n"
"P7+FU+FU+GI *  * +FU+GI * +FU\n"
"P8 *  * +KI * +KI *  *  *  * \n"
"P9+KY+KE * +OU+KE * -HI+KE+KY\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00FU\n"
"P-00KI\n"
"P-00KA\n"
"P-00KA\n"
"+\n"
).getInitialState();
    test_state_t eState(state1);
    MoveVector moves;
    {
      Store store(moves);
      Escape<Store >::
	generateMoves<WHITE,false>(eState,eState.pieceAt(Square(4,1)),eState.pieceAt(Square(5,2)),store);
    }
    CPPUNIT_ASSERT(moves.isMember(Move(Square(4,1),Square(5,2),KING,SILVER,false,WHITE)));
    CPPUNIT_ASSERT(moves.isMember(Move(Square(4,1),Square(3,1),KING,PTYPE_EMPTY,false,WHITE)));
    // 開き王手になっている
    CPPUNIT_ASSERT(!moves.isMember(Move(Square(5,1),Square(5,2),PAWN,SILVER,false,WHITE)));
  }

  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.pieceAt(Square(8,9)),eState.pieceAt(Square(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  //  std::cerr << moves << std::endl;
  for (size_t i=0;i<moves.size();i++)
    CPPUNIT_ASSERT(eState.isValidMove(moves[i]) ||
		   (std::cerr << eState << moves[i] << std::endl,0));
  // 本当に escape になっているかはチェックしない
}

void EscapeMovesTest::testValid(){
  ::testValid(state);
}

static void testCapture(NumEffectState& state){

  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.pieceAt(Square(8,9)),
		    eState.pieceAt(Square(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  
  CPPUNIT_ASSERT(moves.isMember(Move(Square(9,5),Square(8,3),PKNIGHT,PROOK,true,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(9,5),Square(8,3),KNIGHT,PROOK,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,2),Square(8,3),PPAWN,PROOK,false,BLACK)));
}

void EscapeMovesTest::testCapture(){
  ::testCapture(state);
}

/**
 * 当たりをつけられている駒を逃げる手を生成する
 */
static void testEscape(NumEffectState& state){
  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.pieceAt(Square(8,9)),
		    eState.pieceAt(Square(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  // std::cerr << moves << std::endl;
  // valid escape
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,9),Square(9,8),KING,PTYPE_EMPTY,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,9),Square(7,9),KING,PTYPE_EMPTY,false,BLACK)));
  // 
  CPPUNIT_ASSERT(!moves.isMember(Move(Square(8,9),Square(9,9),KING,PTYPE_EMPTY,false,BLACK)));
  //
  CPPUNIT_ASSERT(eState.hasEffectAt(WHITE,Square(8,8)));
  CPPUNIT_ASSERT(!moves.isMember(Move(Square(8,9),Square(8,8),KING,PTYPE_EMPTY,false,BLACK)));
  CPPUNIT_ASSERT(!moves.isMember(Move(Square(8,9),Square(7,8),KING,PTYPE_EMPTY,false,BLACK)));
}

void EscapeMovesTest::testEscape(){
  ::testEscape(state);
}


static void testBlocking(NumEffectState& state){
  test_state_t eState(state);
  MoveVector moves;
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<BLACK,false>(eState,eState.pieceAt(Square(8,9)),
		    eState.pieceAt(Square(8,3)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  
  CPPUNIT_ASSERT(moves.isMember(Move(Square(9,7),Square(8,5),KNIGHT,PTYPE_EMPTY,false,BLACK)));
  //
  CPPUNIT_ASSERT(!moves.isMember(Move(Square(9,8),Square(8,8),KING,PTYPE_EMPTY,false,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,4),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,5),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,6),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,7),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,8),PAWN,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,4),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,5),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,6),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,7),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,8),LANCE,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,4),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,5),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,6),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,7),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,8),KNIGHT,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,4),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,5),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,6),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,7),GOLD,BLACK)));
  CPPUNIT_ASSERT(moves.isMember(Move(Square(8,8),GOLD,BLACK)));
  {
  NumEffectState state1=CsaString(
"P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
"P2 *  *  *  *  *  * -KI-KA * \n"
"P3-FU * -FU-FU-FU-FU *  * -FU\n"
"P4 *  *  *  *  *  * +HI *  * \n"
"P5 *  *  *  *  *  *  *  *  * \n"
"P6 * -HI+FU *  *  *  *  *  * \n"
"P7+FU *  * +FU+FU+FU+FU * +FU\n"
"P8 * +KA+KI *  *  *  *  *  * \n"
"P9+KY+KE+GI * +OU+KI+GI+KE+KY\n"
"P+00FU\n"
"P+00FU\n"
"P+00FU\n"
"P-00FU\n"
"P-00FU\n"
"-\n"
).getInitialState();
  moves.clear();
  test_state_t eState1(state1);
  {
    Store store(moves);
    Escape<Store >::
      generateMoves<WHITE,false>(eState1,eState1.pieceAt(Square(3,2)),
		    eState1.pieceAt(Square(3,4)),store);
  }
    {
      size_t origSize=moves.size();
      moves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,moves.size());
    }
  CPPUNIT_ASSERT(moves.isMember(Move(Square(3,3),PAWN,WHITE)) ||
		 (std::cerr << eState1 << moves << std::endl,0));
  }
}
// 
void EscapeMovesTest::testBlocking(){
  ::testBlocking(state);
}

static void testMoveFile(const std::string& filename){
  Record rec=CsaFile(filename).getRecord();
  test_state_t state(rec.getInitialState());
  vector<osl::Move> moves=rec.getMoves();
  for (unsigned int i=0;i<moves.size();i++){
    MoveVector allMoves;
    {
    Store store(allMoves);
    //    std::cerr << "Before AllGene: " << state.pawnMask[0] << "," << state.pawnMask[1] << std::endl;
    AllMoves<Store>::
      generate(state.turn(),state,store);
    }
    //    std::cerr << "After AllGene: " << state.pawnMask[0] << "," << state.pawnMask[1] << std::endl;
    {
      size_t origSize=allMoves.size();
      allMoves.unique();
      CPPUNIT_ASSERT_EQUAL(origSize,allMoves.size());
    }
    for (int num=0;num<40;num++){
      Piece p=state.pieceOf(num);
      const Player pl=state.turn();
      if (p.isOnBoard() && p.owner()==pl){
	Square pos=p.square();
	if (state.hasEffectAt(alt(pl),pos)){
	  if(p.ptype()!=KING &&
	     state.hasEffectAt(alt(pl),state.kingSquare(pl)))
	    continue;
	  //
	  Piece attacker=state.findCheapAttack(alt(pl),pos);
	  if (p.ptype() == KING)
	    pl == BLACK 
	      ? state.findCheckPiece<BLACK>(attacker)
	      : state.findCheckPiece<WHITE>(attacker);
	  //
	  MoveVector escapeMoves, escape_cheap;
	  {
	    Store storeEscape(escapeMoves), store_cheap(escape_cheap);
	    if(pl==BLACK) {
	      Escape<Store >::
		generateMoves<BLACK,false>(state,p,attacker,storeEscape);
	      Escape<Store>::
		generateMoves<BLACK,true>(state,p,attacker,store_cheap);
	    }
	    else
	    {
	      Escape<Store >::
		generateMoves<WHITE,false>(state,p,attacker,storeEscape);
	      Escape<Store>::
		generateMoves<WHITE,true>(state,p,attacker,store_cheap);
	    }
	  }
	  // 逃げる手は全部生成する
	  BOOST_FOREACH (Move move, allMoves) {
	    if(!Classifier::isSafeMove(state,move)) continue;
	    assert(state.isConsistent(true));
	    NumEffectState next_state = state;
	    next_state.makeMove(move);
	    assert(next_state.isConsistent(true));
	    if (!next_state.hasEffectAt(next_state.turn(),
					next_state.pieceOf(num).square()))
	    {
	      CPPUNIT_ASSERT(escapeMoves.isMember(move)
			     || (std::cerr << "state=\n" << next_state << "\nlast move=" << allMoves[i] << "\n" << escapeMoves << std::endl,0)
		);
	    }
	  }
	  // 王手の場合は生成した手は全部逃げる手
	  if(p.ptype()==KING){
	    BOOST_FOREACH (Move move, escapeMoves) { 
	      assert(state.isConsistent(true));
	      CPPUNIT_ASSERT(! move.ignoreUnpromote(state.turn()));
	      
	      NumEffectState next_state = state;
	      next_state.makeMove(move);
	      CPPUNIT_ASSERT(!next_state.inCheck(pl)
			     || (std::cerr << "next_state=\n" << next_state << "\nlast move=" << move << std::endl << state << p << ' ' << attacker << "\n",0)
		);
	    }
	    BOOST_FOREACH (Move move, escape_cheap) { 
	      CPPUNIT_ASSERT(! move.ignoreUnpromote(state.turn()));	      
	      NumEffectState next_state = state;
	      next_state.makeMove(move);
	      CPPUNIT_ASSERT(! next_state.inCheck(pl));
	    }
	  }
	}
      }
    }
    assert(state.isConsistent(true));
    Move move=moves[i];
    state.makeMove(move);
    assert(state.isConsistent(true) ||
	   (std::cerr << move << std::endl << state << std::endl,0)
	   );
  }
}

void EscapeMovesTest::testMoveMember(){
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=500;
  if (OslConfig::inUnitTestShort()) 
      count=10;
  boost::scoped_ptr<boost::progress_display> progress;
  if (OslConfig::inUnitTestLong())
    progress.reset(new boost::progress_display(count, std::cerr));
  std::string filename;
  while((ifs >> filename) && filename != "" && ++i<count){
    if (progress)
      ++(*progress);
    testMoveFile(OslConfig::testCsaFile(filename));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
