#ifndef OSL_MOVE_H
#define OSL_MOVE_H
#include "osl/player.h"
#include "osl/ptype.h"
#include "osl/square.h"
#include "osl/piece.h"
#include <iosfwd>

/** move 関係でつかまえ所のないエラーがでるときに定義する */
// #define MOVE_DEBUG
#ifdef MOVE_DEBUG
#  include <cassert>
#  define move_assert(x) assert(x)
#else
#  define move_assert(x) 
#endif
// 2009/12/10 以前のfromが下位にあるパターンと
// operator< を同じにしたい時に定義する．
// #define PRESERVE_MOVE_ORDER

namespace osl
{
  /**
   * 圧縮していない moveの表現 .
   * - invalid: isInvalid 以外の演算はできない
   * - declare_win: isInvalid 以外の演算はできない
   * - pass: from, to, ptype, oldPtype はとれる．player()はとれない．
   * 
   * Pieceとpromotepをそろえる  -> 変える． 
   * 下位から 
   * 2009/12/10から
   * - to       : 8 bit 
   * - from     : 8 bit 
   * - capture ptype    : 4 bit 
   * - dummy    : 3 bit 
   * - promote? : 1 bit  
   * - ptype    : 4 bit - promote moveの場合はpromote後のもの
   * - owner    : signed 
   * 2009/12/10以前
   * - from     : 8 bit 
   * - to       : 8 bit 
   * - dummy    : 3 bit 
   * - promote? : 1 bit  
   * - capture ptype    : 4 bit 
   * - ptype    : 4 bit - promote moveの場合はpromote後のもの
   * - owner    : signed 
   */
  class Move
  {
  public:
    static const int BitOffsetPromote=Piece::BitOffsetMovePromote;  // 23
  private:
    int move;
    explicit Move(int value) : move(value)
    {
    }
    enum { 
      INVALID_VALUE = (1<<8), DECLARE_WIN = (2<<8),
      BLACK_PASS = 0, WHITE_PASS = (-1)<<28, 
    };
  public:
    int intValue() const { return move; }
    /** 駒を取らない手を [0, 16305] にmap */
    unsigned int hash() const;
    /** 一局面辺りの合法手の最大値 
     * 重複して手を生成することがある場合は，600では不足かもしれない
     */
    static const unsigned int MaxUniqMoves=600;
  private:
    void init(Square from, Square to, Ptype ptype,
	      Ptype capture_ptype, bool is_promote, Player player)
    {
      move =  (to.uintValue()
 	       + (from.uintValue()<<8)
	       + (static_cast<unsigned int>(capture_ptype)<<16)
	       + (static_cast<unsigned int>(is_promote)<<BitOffsetPromote)
	       + (static_cast<unsigned int>(ptype)<<24)
	       + (static_cast<int>(player)<<28));
    }
  public:
    Move() : move(INVALID_VALUE)
    {
    }
    /** INVALID でも PASS でもない. isValid()かどうかは分からない．*/
    bool isNormal() const { 
      // PASS や INVALID は to() が 00
      return move & 0x00ff; 
    }
    bool isPass() const { return (move & 0xffff) == 0; }
    static const Move makeDirect(int value) { return Move(value); }
    static const Move PASS(Player P) { return Move(P<<28); }
    static const Move INVALID() { return Move(INVALID_VALUE); }
    static const Move DeclareWin() { return Move(DECLARE_WIN); }
    /**
     * 移動
     */
    Move(Square from, Square to, Ptype ptype,
	 Ptype capture_ptype, bool is_promote, Player player)
    {
      move_assert(from.isValid());
      move_assert(to.isOnBoard());
      move_assert(isValid(ptype));
      move_assert(isValid(capture_ptype));
      move_assert(isValid(player));
      init(from, to, ptype, capture_ptype, is_promote, player);
      move_assert(isValid());
    }
    /**
     * drop
     */
    Move(Square to, Ptype ptype, Player player)
    {
      move_assert(to.isOnBoard());
      move_assert(isValid(ptype));
      move_assert(isValid(player));
      init(Square::STAND(), to, ptype, PTYPE_EMPTY, false, player);
      move_assert(isValid());
    }

    const Square from() const 
    {
      assert(! isInvalid());
      move_assert(isValidOrPass());
      const Square result = Square::makeDirect((move>>8) & 0xff);
      return result;
    }
    const Square to() const {
      assert(! isInvalid());
      move_assert(isValidOrPass());
      const Square result = Square::makeDirect(move & 0xff);
      return result;
    }
    /** fromとtoをまとめて同一性の判定など */
    unsigned int fromTo() const { return move & 0xffff; }
    /**
     * pieceに使うためのmaskなので
     */
    int promoteMask() const {
      assert(isNormal());
      return (static_cast<int>(move)&(1<<BitOffsetPromote));
    }
    bool isPromotion() const { assert(isNormal()); return (move & (1<<BitOffsetPromote))!=0; }
    bool isCapture() const { assert(isNormal()); return capturePtype() != PTYPE_EMPTY; }
    bool isCaptureOrPromotion() const { return isCapture() || isPromotion(); }
    bool isDrop() const { assert(isNormal()); return from().isPieceStand(); }
      
    Ptype ptype() const {
      assert(! isInvalid());
      move_assert(isValidOrPass());
      const Ptype result = static_cast<Ptype>((move >> 24) & 0xf);
      return result;
    }
    /** 移動後のPtype, i.e., 成る手だった場合成った後 */
    PtypeO ptypeO() const {
      assert(! isInvalid());
      const PtypeO result = static_cast<PtypeO>(move >> 24);
      return result;
    }
    /** 移動前のPtypeO, i.e., 成る手だった場合成る前 */
    PtypeO oldPtypeO() const {
      assert(! isInvalid());
      const PtypeO result = static_cast<PtypeO>((move>>24)+((move >> (BitOffsetPromote-3))&8));
      return result;
    }
    /** 移動前のPtype, i.e., 成る手だった場合成る前 */
    Ptype oldPtype() const { 
      assert(! isInvalid());
      move_assert(isValidOrPass());
      const PtypeO old_ptypeo = static_cast<PtypeO>((move>>24)+((move >> (BitOffsetPromote-3))&8));
      return getPtype(old_ptypeo); 
    }
    Ptype capturePtype() const {
      assert(isNormal());
      const Ptype result = static_cast<Ptype>((move>>16)&0xf);
      return result;
    }
    PtypeO capturePtypeO() const {
      assert(isCapture());
      return newPtypeO(alt(player()), capturePtype());
    }
    PtypeO capturePtypeOSafe() const {
      if (! isCapture())
	return PTYPEO_EMPTY;
      return capturePtypeO();
    }

    Player player() const {
      assert(! isInvalid());
      const Player result = static_cast<Player>(move>>28);
      return result;
    }
    bool isValid() const;
    /** state に apply 可能でない場合にtrue */
    bool isInvalid() const { 
      return static_cast<unsigned int>(move-1) < DECLARE_WIN; 
    }
    bool isValidOrPass() const { return isPass() || isValid(); }

    Move newFrom(Square new_from) const
    {
      assert(isNormal());
      int result = static_cast<int>(intValue());
      result &= ~(0xff00);
      result += (new_from.uintValue()<<8);
      return makeDirect(result);
    }
    Move newAddFrom(Square new_from) const
    {
      assert(isNormal());
      assert(from().uintValue()==0);
      int result = static_cast<int>(intValue());
      result += (new_from.uintValue()<<8);
      return makeDirect(result);
    }
    /**
     * no capture moveからcapture moveを作る
     */
    const Move newAddCapture(Piece capture) const
    {
      assert(! isCapture());
      return makeDirect(intValue()+(capture.intValue()&0xf0000));
    }
    const Move newCapture(Piece capture) const
    {
      return makeDirect((intValue()&0xfff0ffff)+(capture.intValue()&0xf0000));
    }
    const Move newCapture(Ptype capture) const
    {
      return makeDirect((intValue()&0xfff0ffff)
			+(static_cast<int>(capture)<<Piece::BitOffsetPtype));
    }
    /**
     * promote moveからunpromote moveを作る
     */
    const Move unpromote() const {
      assert(isNormal());
      move_assert(isPromotion() && isPromoted(ptype()));
      return makeDirect(intValue()^((1<<BitOffsetPromote)^(1<<27)));
    }
    /**
     * unpromote moveからpromote moveを作る
     */
    const Move promote() const {
      assert(isNormal());
      move_assert(!isPromotion() && canPromote(ptype()));
      return makeDirect(intValue()^((1<<BitOffsetPromote)^(1<<27)));
    }
    /**
     * moveのtoをoffsetだけ変える．
     * 元のtoが0以外でも使える
     */
    inline Move newAddTo(Offset o) const{
      return makeDirect(intValue()+o.intValue());
    }
    /**
     * つくってあったmoveの雛形のsquareをsetする．
     * mのtoは0
     */
    inline Move newAddTo(Square sq) const{
      assert((intValue()&0xff)==0);
      return Move::makeDirect(intValue()+sq.uintValue());
    }
    /**
     * 作ってあったPTYPE_EMPTYのひな形のPTYPEをsetする 
     */
    inline Move newAddPtype(Ptype newPtype) const{
      assert(ptype()==PTYPE_EMPTY);
      return Move::makeDirect(intValue()
			      + (static_cast<unsigned int>(newPtype)<<24));
    }
    template<Player P>
    static bool ignoreUnpromote(Ptype ptype,Square from,Square to){
      switch(ptype){
      case PAWN: 
	return to.canPromote<P>();
      case BISHOP: case ROOK: 
	return to.canPromote<P>() || from.canPromote<P>();
      case LANCE:
	return (P==BLACK ? to.y()==2 : to.y()==8);
      default: return false;
      }
    }
    /**
     * 合法手ではあるが，打歩詰め絡み以外では有利にはならない手.  
     * TODO 遅い
     */
    template<Player P>
    bool ignoreUnpromote() const{
      assert(player()==P);
      if(isDrop()) return false;
      return ignoreUnpromote<P>(ptype(),from(),to());
    }
    bool ignoreUnpromote() const{
      if(player()==BLACK) return ignoreUnpromote<BLACK>();
      else return ignoreUnpromote<WHITE>();
    }
    /**
     * MoveをunpromoteするとcutUnpromoteなMoveになる
     */
    template<Player P>
    bool hasIgnoredUnpromote() const{
      assert(player()==P);
      if(!isPromotion()) return false;
      switch(ptype()){
      case PPAWN: 
	return (P==BLACK ? to().y()!=1 : to().y()!=9);
      case PLANCE:
	return (P==BLACK ? to().y()==2 : to().y()==8);
      case PBISHOP: case PROOK: 
	return true;
      default: return false;
      }
    }
    bool hasIgnoredUnpromote() const{
      if(player()==BLACK) return hasIgnoredUnpromote<BLACK>();
      else return hasIgnoredUnpromote<WHITE>();
    }
    const Move rotate180() const;
  };
  inline bool operator<(Move lhs, Move rhs)
  {
#ifdef PRESERVE_MOVE_ORDER
    int l=lhs.intValue();
    l=(l&0xffff0000)+((l>>8)&0xff)+((l<<8)&0xff00);
    int r=rhs.intValue();
    r=(r&0xffff0000)+((r>>8)&0xff)+((r<<8)&0xff00);
    return l<r;
#else
    return lhs.intValue() < rhs.intValue();
#endif
  }
  inline bool operator==(Move lhs, Move rhs)
  {
    return lhs.intValue() == rhs.intValue();
  }
  inline bool operator!=(Move lhs, Move rhs)
  {
    return ! (lhs == rhs);
  }

  std::ostream& operator<<(std::ostream& os, Move move);
  namespace stl
  {
    template <typename T> struct hash;
    template <> struct hash<Move>
    {
      unsigned long operator()(Move m) const { return m.intValue(); }
    };
  } // namespace stl
}
#endif /* OSL_MOVE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
