#include "osl/record/csaRecord.h"
#include "osl/record/csa.h"
#include "osl/record/csaIOError.h"
#include "osl/stl/vector.h"
#include "osl/record/csaString.h"
#include "osl/state/simpleState.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

#include <fstream>

class CsaTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( CsaTest );
  CPPUNIT_TEST( testShow );
  CPPUNIT_TEST( testLoadKatagyoku );
  CPPUNIT_TEST( testLoadKatagyokuFile );
  CPPUNIT_TEST( testSpeed );
  CPPUNIT_TEST( testSpeedGame );

  CPPUNIT_TEST( testSearchInfo );
  CPPUNIT_TEST( testResult );
  CPPUNIT_TEST( testStr2Move );
  CPPUNIT_TEST( testError );
  CPPUNIT_TEST_SUITE_END();
 private:
  osl::Record rec;
  osl::SimpleState state;
  osl::vector<osl::Move> moves;

 public:
  void setUp();
  void testShow();
  void testLoad();
  void testLoadKatagyoku();
  void testLoadKatagyokuFile();
  void testSpeed();
  void testSpeedGame();

  void testSearchInfo();
  void testResult();
  void testStr2Move();
  void testError();
};

CPPUNIT_TEST_SUITE_REGISTRATION(CsaTest);

using namespace osl;
using namespace osl::record;

void CsaTest::testSearchInfo()
{
  // XXX
  const int old_unit_test_level = OslConfig::inUnitTest();
  OslConfig::setInUnitTest(0);

  CsaString csa(
    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
    "P2 * -HI *  *  *  *  * -KA * \n"
    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
    "P8 * +KA *  *  *  *  * +HI * \n"
    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
    "+\n"
    "+7776FU\n"
    "'** 3\n"
    "-8384FU\n"
    "'** -5\n");
  const Record& record = csa.getRecord();
  vector<Move> moves;
  vector<int> times;
  vector<std::string> comments;
  vector<SearchInfo> info;
  record.getMoves(moves, times, comments, info);

  CPPUNIT_ASSERT_EQUAL((size_t)2, info.size());
  CPPUNIT_ASSERT_EQUAL( 3, info[0].value);
  CPPUNIT_ASSERT_EQUAL(-5, info[1].value);

  OslConfig::setInUnitTest(old_unit_test_level);
}

void CsaTest::setUp(){
  rec=CsaFile(OslConfig::testPublicFile("hirate.csa")).getRecord();
  state=rec.getInitialState();
  moves=rec.getMoves();
}

void CsaTest::testShow(){
  vector<Move> ms;
  NumEffectState s;

  CPPUNIT_ASSERT_EQUAL(std::string(""),
                       csa::show(&*ms.begin(), &*ms.end()));

  ms.push_back(csa::strToMove("+7776FU", s));
  s.makeMove(ms.back());
  CPPUNIT_ASSERT_EQUAL(std::string("+7776FU"),
                       csa::show(&*ms.begin(), &*ms.end()));

  ms.push_back(csa::strToMove("-3334FU", s));
  s.makeMove(ms.back());

  CPPUNIT_ASSERT_EQUAL(std::string("+7776FU-3334FU"),
                       csa::show(&*ms.begin(), &*ms.end()));
}

void CsaTest::testLoad(){
  //  std::cout << rec << std::endl;
  // st/d::cout << irs.getState() << std::endl;
}

void CsaTest::testLoadKatagyoku(){
  CsaString csa(
    "P1 *  *  *  *  *  *  *  * -OU\n"
    "P2 *  *  *  *  *  *  *  *  * \n"
    "P3 *  *  *  *  *  *  *  * +FU\n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  *  *  *  *  *  *  *  * \n"
    "P+00KI\n"
    "P-00AL\n"
    "+\n");
  SimpleState sstate=csa.getInitialState();
  CPPUNIT_ASSERT_EQUAL(BLACK, sstate.kingPiece(BLACK).owner());
  CPPUNIT_ASSERT_EQUAL(Square::STAND(),
		       sstate.kingPiece(BLACK).square());
}

void CsaTest::testLoadKatagyokuFile(){
  const char* test_file_name = "osl_record_csa_testLoadKatagyokuFile.csa";
  std::ofstream os(test_file_name);
  os <<
    "P1 *  *  *  *  *  *  *  * -OU\n"
    "P2 *  *  *  *  *  *  *  *  * \n"
    "P3 *  *  *  *  *  *  *  * +FU\n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  *  *  *  *  *  *  *  * \n"
    "P+00KI\n"
    "P-00AL\n"
    "+\n";
  os.close();

  osl::Record myrec=CsaFile(test_file_name).getRecord();
  osl::SimpleState mystate=myrec.getInitialState();
  CPPUNIT_ASSERT_EQUAL(BLACK, mystate.kingPiece(BLACK).owner());
  CPPUNIT_ASSERT_EQUAL(Square::STAND(),
		       mystate.kingPiece(BLACK).square());
  unlink(test_file_name);
}

struct posToInt {
  int operator()(){
    return 1;
  }
};


void CsaTest::testSpeed(){
  Record rec=CsaString(
"P1+NY+TO *  *  *  * -OU-KE-KY\n"
"P2 *  *  *  *  * -GI-KI *  *\n"
"P3 * +RY *  * +UM * -KI-FU-FU\n"
"P4 *  * +FU-FU *  *  *  *  *\n"
"P5 *  * -KE * +FU *  * +FU *\n"
"P6-KE *  * +FU+GI-FU *  * +FU\n"
"P7 *  * -UM *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n").getRecord();
  SimpleState state0=rec.getInitialState();
  //  std::cout << state0;
  // std::cout << "BitXmask" << std::endl;
  {
    const Square pos0=csa::strToPos("55"),pos1=csa::strToPos("54");
    const Move move=Move(pos0,pos1,
			    PAWN,PTYPE_EMPTY,false,BLACK);
    const SimpleState stateCopy(state0);
    CPPUNIT_ASSERT_EQUAL(stateCopy, state0);
  }
  {
    const Square pos1=csa::strToPos("99");
    const Move move=Move(pos1,PAWN,BLACK);
    const SimpleState stateCopy(state0);
    CPPUNIT_ASSERT_EQUAL(stateCopy, state0);
  }
  {
    const Square pos0=csa::strToPos("53"),pos1=csa::strToPos("42");
    const Move move=Move(pos0,pos1,
			     PBISHOP,SILVER,false,BLACK);
    const SimpleState stateCopy(state0);
    CPPUNIT_ASSERT_EQUAL(stateCopy, state0);
  }
}

void CsaTest::testSpeedGame(){
}

void CsaTest::testResult()
{
  {
    const Record record(CsaString("P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
				  "P2 * -HI *  *  *  *  * -KA * \n"
				  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				  "P4 *  *  *  *  *  *  *  *  * \n"
				  "P5 *  *  *  *  *  *  *  *  * \n"
				  "P6 *  *  *  *  *  *  *  *  * \n"
				  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
				  "P8 * +KA *  *  *  *  * +HI * \n"
				  "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
				  "+\n"
				  "+7776FU\n"
				  "-8384FU\n").getRecord());
    CPPUNIT_ASSERT_EQUAL(Record::UNKNOWN, record.getResult());
  }
  {
    const Record record(CsaString("P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
				  "P2 * -HI *  *  *  *  * -KA * \n"
				  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				  "P4 *  *  *  *  *  *  *  *  * \n"
				  "P5 *  *  *  *  *  *  *  *  * \n"
				  "P6 *  *  *  *  *  *  *  *  * \n"
				  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
				  "P8 * +KA *  *  *  *  * +HI * \n"
				  "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
				  "+\n"
				  "+7776FU\n"
				  "%TORYO\n").getRecord());
    CPPUNIT_ASSERT_EQUAL(Record::BLACK_WIN, record.getResult());
  }
  {
    const Record record(CsaString("P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
				  "P2 * -HI *  *  *  *  * -KA * \n"
				  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				  "P4 *  *  *  *  *  *  *  *  * \n"
				  "P5 *  *  *  *  *  *  *  *  * \n"
				  "P6 *  *  *  *  *  *  *  *  * \n"
				  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
				  "P8 * +KA *  *  *  *  * +HI * \n"
				  "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
				  "+\n"
				  "%TORYO\n").getRecord());
    CPPUNIT_ASSERT_EQUAL(Record::WHITE_WIN, record.getResult());
  }
}

void CsaTest::testStr2Move()
{
  SimpleState state;
  CPPUNIT_ASSERT_EQUAL(Move(), csa::strToMove("%TORYO", state));
  CPPUNIT_ASSERT_EQUAL(Move::DeclareWin(), csa::strToMove("%KACHI", state));
  CPPUNIT_ASSERT_EQUAL(Move::PASS(BLACK), csa::strToMove("%PASS", state));
}

void CsaTest::testError()
{
  try
  {
    // missing P1
    const Record record(CsaString("P2 * -HI *  *  *  *  * -KA * \n"
				  "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
				  "P4 *  *  *  *  *  *  *  *  * \n"
				  "P5 *  *  *  *  *  *  *  *  * \n"
				  "P6 *  *  *  *  *  *  *  *  * \n"
				  "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
				  "P8 * +KA *  *  *  *  * +HI * \n"
				  "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
				  "+\n").getRecord());
    CPPUNIT_FAIL("failed to detect errors");
  }
  catch (CsaIOError& e) {
    CPPUNIT_ASSERT(e.what()
		   == std::string("incomplete position description in csaParseLine"));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
