#ifndef OSMIUM_GEOM_TILE_HPP
#define OSMIUM_GEOM_TILE_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <cstdint>

#include <osmium/geom/mercator_projection.hpp>

namespace osmium {

    namespace geom {

        namespace detail {

            template <typename T>
            inline T restrict_to_range(T value, T min, T max) {
                if (value < min) return min;
                if (value > max) return max;
                return value;
            }

        } // namespace detail

        /**
         * A tile in the usual Mercator projection.
         */
        struct Tile {

            uint32_t x;
            uint32_t y;
            uint32_t z;

            explicit Tile(uint32_t zoom, uint32_t tx, uint32_t ty) noexcept : x(tx), y(ty), z(zoom) {
            }

            explicit Tile(uint32_t zoom, const osmium::Location& location) :
                z(zoom) {
                osmium::geom::Coordinates c = lonlat_to_mercator(location);
                const int32_t n = 1LL << zoom;
                const double scale = detail::max_coordinate_epsg3857 * 2 / n;
                x = detail::restrict_to_range<int32_t>((c.x + detail::max_coordinate_epsg3857) / scale, 0, n-1);
                y = detail::restrict_to_range<int32_t>((detail::max_coordinate_epsg3857 - c.y) / scale, 0, n-1);
            }

        }; // struct Tile

        inline bool operator==(const Tile& a, const Tile& b) {
            return a.z == b.z && a.x == b.x && a.y == b.y;
        }

        inline bool operator!=(const Tile& a, const Tile& b) {
            return ! (a == b);
        }

        /**
         * This defines an arbitrary order on tiles for use in std::map etc.
         */
        inline bool operator<(const Tile& a, const Tile& b) {
            if (a.z < b.z) return true;
            if (a.z > b.z) return false;
            if (a.x < b.x) return true;
            if (a.x > b.x) return false;
            return a.y < b.y;
        }

    } // namespace geom

} // namespace osmium

#endif // OSMIUM_GEOM_TILE_HPP
