/*
*   pam_abl - a PAM module and program for automatic blacklisting of hosts and users
*
*   Copyright (C) 2009 Chris Tasma pam-abl@deksai.com
*
*   This program is free software: you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation, either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <limits.h>
#include "pam_abl.h"

void die(const char *msg, ...) {
    va_list ap;
    va_start(ap, msg);
    fprintf(stderr, "Fatal: ");
    vfprintf(stderr, msg, ap);
    fprintf(stderr, "\n");
    va_end(ap);
    exit(1);
}

void make_key(DBT *dbt, const char *key) {
    memset(dbt, 0, sizeof(*dbt));
    dbt->data = (void *) key;
    dbt->size = strlen(key) + 1;
    dbt->ulen = dbt->size + 1;
}

/* Grow the buffer attached to a DBT if necessary.
 */
int grow_buffer(DBT *dbt, u_int32_t minsize) {
    /*log_debug(args, "Current size %ld, desired size %ld", dbt->ulen, minsize);*/
    if (dbt->ulen < minsize) {
        void *nd;
        if (nd = realloc(dbt->data, minsize), NULL == nd) {
            log_sys_error(ENOMEM, "allocating record buffer");
            return ENOMEM;
        }
        dbt->data = nd;
        dbt->ulen = minsize;
        /*log_debug(args, "Buffer grown to %ld", dbt->ulen);*/
    }
    return 0;
}

int dbopen(const abl_args *args, const char *dbfile, const char *dbname, DB **db) {
    int err;

    if (err = db_create(db, NULL, 0), 0 != err) {
        log_db_error(err, "creating database object");
        return err;
    }

    if (err = (*db)->open(*db, NULL, dbfile, dbname, DB_BTREE,
                       DB_CREATE, DBPERM), 0 != err) {
        log_db_error(err, "opening or creating database");
        return err;
    }

    log_debug(args, "%s opened", dbname);

    return 0;
} 

/* db is an already opened database.  */
int dbget(DB *db, DBT *key, DBT *dbtdata) {
    int err; 

    memset(dbtdata, 0, sizeof(*dbtdata));
    dbtdata->flags = DB_DBT_USERMEM;

    err = db->get(db, NULL, key, dbtdata, 0);

    /*Called with DB_DBT_USERMEM?  What was there wasn't enough*/
    if (TOOSMALL == err) {
        if (err = grow_buffer(dbtdata, dbtdata->size), 0 != err) {
            return err;
        }
        dbtdata->size = 0;
        /* ...and try again. */
        err = db->get(db, NULL, key, dbtdata, 0);
    }

    if (0 != err && DB_NOTFOUND != err) {
        return err;
    }

    if (DB_NOTFOUND == err) {
        dbtdata->size = 0;
    }
    return err;
}

const char * get_kv(const abl_info *info) {
    return info->subject == HOST ? info->host : info->user;
}
    
int update_status(const abl_args *args, abl_info *info) {
    DB *db;
    DBT key, data;
    DBC *dbc;
    int err = 0;

    err = dbopen(args, info->dbfile, DB_STATE, &db);
    if (err) return err;

    make_key(&key, get_kv(info));
    err = dbget(db, &key, &data);
    if (err && err != DB_NOTFOUND) return err;
    if (data.data == NULL || *(int *) data.data != info->state ) {
        log_debug(args,"state changed data %d info %d", data.data ? *(int *) data.data : -1,info->state);
        info->state_change = 1;
        data.data = &info->state;
        data.size = sizeof(info->state);
        err = db->put(db,NULL,&key,&data,0);
    }
    else {
        info->state_change = 0;
    }
    return db->close(db,0);
}

/* Log a login attempt */
int record(const abl_args *args, const char *dbfile, const char *kv, time_t tm, long maxage) {
    int err, err2;
    DB  *db;
    DBT key, data;

    err = dbopen(args, dbfile, DB_TIME, &db);

    make_key(&key, kv);

    if (err != 0) {
        return err;
    }

    err = dbget(db, &key, &data);
    if (0 != err && DB_NOTFOUND != err) {
        return err;
    }

    if (0 == err) {
        rule_purge(&data, maxage, tm);
    } else if (DB_NOTFOUND == err) {
        data.size = 0;
    }

    if (err = grow_buffer(&data, data.size + sizeof(time_t)), 0 != err) {
        return err;
    }

    memcpy((char *) data.data + data.size, &tm, sizeof(time_t));
    data.size += sizeof(time_t);


    if (err = db->put(db, NULL, &key, &data, 0), 0 != err) {
        log_sys_error(err, "updating database");
    }
    if (err2 = db->close(db, 0), 0 != err2) {
        log_sys_error(err2, "closing database");
        if (0 == err) {
            err = err2;
        }
    }

    return err;
}

/* Check whether access should be permitted for the specified user/host.
 */
int check(const abl_args *args, abl_info *info, time_t tm) {
    DB *db;
    DBT key, data;
    int err, err2, sz;
    const char *kv;

    kv = info->subject == HOST ? info->host : info->user;

    err = dbopen(args, info->dbfile, DB_TIME, &db);
    make_key(&key, kv);

    memset(&data, 0, sizeof(data));

    err = dbget(db, &key, &data);
    if (0 != err) {
        if (DB_NOTFOUND == err) {
            info->state = CLEAR;
            err = 0;
        }
        goto check_fail;
    }

    sz = data.size / sizeof(time_t);
    info->state = rule_test(args, info->rule, info->user, info->service, (time_t *) data.data, sz, tm);

    /* Cleanup */
check_fail:
    if (err2 = db->close(db, 0), 0 != err2) {
        log_sys_error(err2, "closing database");
        if (0 == err) {
            err = err2;
        }
    }

    return err;
}

/*XXX This is going to go away*/
int check_host(const abl_args *args, abl_info *info, time_t tm) {
    if (!args->host_rule) {
        log_warning(args,"Cannot check host!  No host_rule in config");
        return 1;
    }
    if (NULL != args->host_db) {
        int err;
        info->subject = HOST;
        info->dbfile = args->host_db;
        info->rule = args->host_rule;

        if (NULL != info->host) {
            log_debug(args, "Checking host %s", info->host);
            check(args, info, tm);
            return 0;
        } 

    } else {
        log_warning(args, "check_host: No host database found in config.");
        return 1;
    }
}

int check_user(const abl_args *args, abl_info *info, time_t tm) {
    if (!args->host_rule) {
        log_warning(args,"Cannot check user!  No user_rule in config");
        return 1;
    }
    if (NULL != args->user_db) {
        info->subject = USER;
        info->dbfile = args->user_db;
        info->rule = args->user_rule;
        if (NULL != info->user) {
            log_debug(args, "Checking user %s", info->user);
            check(args, info, tm);
            return 0;
        }
    } else {
        log_warning(args, "check_user: No host database found in config.");
        return 1;
    }
}

/*Yes, this takes args and then something that is in args....
  Don't ask.  It will be fixed eventually.*/
int prepare_command(const abl_args *args, const char *cmd, const abl_info *info,
        char **string) {
    int i;
    int cmd_sz = strlen(cmd);
    int strstore_sz = 0;
    int host_sz = 0;
    int user_sz = 0;
    int service_sz = 0;
    char subst;
    char *strstore = *string; //Because double pointers get confusing

    if(info->host != NULL) host_sz = strlen(info->host);
    if(info->user != NULL) user_sz = strlen(info->user);
    if(info->service != NULL) service_sz = strlen(info->service);

    strstore = calloc(COMMAND_SIZE,sizeof(char));
    if (strstore == NULL) die("Could not allocate memory for running command");

    for(i=0; i < cmd_sz; i++) {
        if(*(cmd + i)  == '%') {
            subst = *(cmd + i + 1); //grab substitution letter
            i += 2;                 //move index past '%x'
            switch(subst) {
                case 'u':
                    if(strstore_sz + user_sz >= COMMAND_SIZE) {
                        log_warning(args, "command length error: %d > %d.  Adjust COMMAND_SIZE in pam_abl.h\n",strlen(strstore)+user_sz,COMMAND_SIZE);
                        return(1);
                    }
                    else if (!info->user) {
                        log_warning(args, "No user to substitute: %s.",cmd);
                        return(1);
                    }
                    else {
                        strcat(strstore,info->user);
                        strstore_sz += user_sz;
                    }
                    break;
                case 'h':
                    if(strstore_sz + host_sz >= COMMAND_SIZE) {
                        log_warning(args, "command length error: %d > %d.  Adjust COMMAND_SIZE in pam_abl.h\n",strlen(strstore)+host_sz,COMMAND_SIZE);
                        return(1);
                    }
                    else if (!info->host) {
                        log_warning(args, "No host to substitute: %s.",cmd);
                        return(1);
                    }
                    else {
                        strcat(strstore,info->host);
                        strstore_sz += host_sz;
                    }
                    break;
                case 's':
                    if(strstore_sz + service_sz >= COMMAND_SIZE) {
                        log_warning(args, "command length error: %d > %d.  Adjust COMMAND_SIZE in pam_abl.h\n",strlen(strstore)+service_sz,COMMAND_SIZE);
                        return(1);
                    }
                    else if (!info->service) {
                        log_warning(args, "No service to substitute: %s.",cmd);
                        return(1);
                    }
                    else {
                        strcat(strstore,info->service);
                        strstore_sz += service_sz;
                    }
                    break;
            }
        }
        *(strstore + strstore_sz) = *(cmd + i);
        strstore_sz++;
    }
    *string = strstore;
    return(0);
}


int run_command(const abl_args *args, const abl_info *info) {
    int  err = 0;
    char *command = NULL;
    switch (info->subject) {
        case (HOST):
            if(info->state == BLOCKED && args->host_blk_cmd != NULL)
                err = prepare_command(args,args->host_blk_cmd,info,&command);
            else if(info->state == CLEAR && args->host_clr_cmd != NULL)
                err = prepare_command(args,args->host_clr_cmd,info,&command);
           break;
        case (USER):
            if(info->state == BLOCKED && args->user_blk_cmd != NULL)
                err = prepare_command(args,args->user_blk_cmd,info,&command);
            else if(info->state == CLEAR && args->user_clr_cmd != NULL)
                err = prepare_command(args,args->user_clr_cmd,info,&command);
           break;
        default:
           return(0);
    }
    if(err != 0) {
        log_warning(args,"Failed to run command.");
    }
    else if (command) {
        log_debug(args,"running command %s",command);
        err = system(command);
        if (err == -1) log_warning(args,"Failed to run command: %s",command);
        free(command);
    }
    else if (!command)
        log_debug(args,"No command to run for this situation.");
    
    return err;
}

