#include <qapplication.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qstatusbar.h>
#include <qhbox.h>
#include <qprinter.h>
#include <qregexp.h>
#include <qlabel.h>
#include <qwt_counter.h>
#include <qwt_math.h>
#include "pixmaps.h"
#include "bode_plot.h"
#include "bode.h"

class PrintFilter: public QwtPlotPrintFilter
{
public:
	PrintFilter() {};

	virtual QFont font(const QFont &f, Item, int) const
	{
		QFont f2 = f;
		f2.setPointSize(f.pointSize() * 0.75);
		return f2;
	}
};

//-----------------------------------------------------------------
//
//      bode.cpp -- A demo program featuring QwtPlot and QwtCounter
//
//      This example demonstrates the mapping of different curves
//      to different axes in a QwtPlot widget. It also shows how to
//      display the cursor position and how to implement zooming.
//
//-----------------------------------------------------------------

QString zoomInfo("Zoom: Press mouse button and drag");
QString cursorInfo("Cursor Pos: Press mouse button in plot region");

MainWin::MainWin(QWidget *p , const char *name): 
    QMainWindow(p, name)
{
    d_zoom = 0;
    
    d_plot = new BodePlot(this);
    d_plot->setMargin(5);

    setCentralWidget(d_plot);

    QToolBar *toolBar = new QToolBar(this);

    QToolButton *btnZoom = new QToolButton(toolBar);
    btnZoom->setTextLabel("Zoom");
    btnZoom->setPixmap(zoom_xpm);
    btnZoom->setToggleButton(TRUE);
    btnZoom->setUsesTextLabel(TRUE);

    QToolButton *btnPrint = new QToolButton(toolBar);
    btnPrint->setTextLabel("Print");
    btnPrint->setPixmap(print_xpm);
    btnPrint->setUsesTextLabel(TRUE);

    toolBar->setStretchableWidget(new QWidget(toolBar));

    QHBox *dampBox = new QHBox(toolBar);
    dampBox->setSpacing(10);
    (void)new QLabel("Damping Factor", dampBox);
    QwtCounter *cntDamp = new QwtCounter(dampBox);
    cntDamp->setRange(0.0, 5.0, 0.01);
    cntDamp->setValue(0.0);
    
    addToolBar(toolBar);
    (void)statusBar();

    showInfo(cursorInfo);

    connect(cntDamp, SIGNAL(valueChanged(double)), 
        d_plot, SLOT(setDamp(double))); 
    connect(btnPrint, SIGNAL(clicked()), SLOT(print()));
    connect(btnZoom, SIGNAL(toggled(bool)), SLOT(zoom(bool)));
    connect(d_plot, SIGNAL(plotMouseMoved(const QMouseEvent&)),
            SLOT(plotMouseMoved( const QMouseEvent&)));
    connect(d_plot, SIGNAL(plotMousePressed(const QMouseEvent &)),
            SLOT(plotMousePressed( const QMouseEvent&)));
    connect(d_plot, SIGNAL(plotMouseReleased(const QMouseEvent &)),
            SLOT(plotMouseReleased( const QMouseEvent&)));
}

void MainWin::print()
{
    QPrinter printer;

    QString docName = d_plot->title();
    if ( docName.isEmpty() )
    {
        docName.replace (QRegExp (QString::fromLatin1 ("\n")), tr (" -- "));
        printer.setDocName (docName);
    }

    printer.setCreator("Bode example");
    printer.setOrientation(QPrinter::Landscape);

    if (printer.setup())
        d_plot->print(printer, PrintFilter());
}

void MainWin::zoom(bool on)
{
    if (on)
    {
        d_zoom = TRUE;
    }
    else
    {
        // Disable Zooming.
        d_plot->setAxisAutoScale(QwtPlot::yLeft);
        d_plot->setAxisAutoScale(QwtPlot::yRight);
        d_plot->setAxisAutoScale(QwtPlot::xBottom);
        d_plot->replot();
        d_zoom = FALSE;
    }
    
    if (d_zoom)
        showInfo(zoomInfo);
    else
        showInfo(cursorInfo);
}

void MainWin::showInfo(const QString &text)
{
    statusBar()->message(text);
}

void MainWin::plotMouseMoved(const QMouseEvent &e)
{
    QString info;
    info.sprintf("Freq=%g, Ampl=%g, Phase=%g",
        d_plot->invTransform(QwtPlot::xBottom, e.pos().x()),
        d_plot->invTransform(QwtPlot::yLeft, e.pos().y()),
        d_plot->invTransform(QwtPlot::yRight, e.pos().y())
    );
    showInfo(info);
}

void MainWin::plotMousePressed(const QMouseEvent &e)
{
    // store position
    d_p1 = e.pos();
    
    // update cursor pos display
    plotMouseMoved(e);
    
    if (d_zoom)
        d_plot->setOutlineStyle(Qwt::Rect); 
    else
        d_plot->setOutlineStyle(Qwt::Cross);
}

void MainWin::plotMouseReleased(const QMouseEvent &e)
{
    // some shortcuts
    int axl= QwtPlot::yLeft, axr = QwtPlot::yRight, axb= QwtPlot::xBottom;
    
    if (d_zoom)
    {
        // Don't invert any scales which aren't inverted
        int x1 = qwtMin(d_p1.x(), e.pos().x());
        int x2 = qwtMax(d_p1.x(), e.pos().x());
        int y1 = qwtMin(d_p1.y(), e.pos().y());
        int y2 = qwtMax(d_p1.y(), e.pos().y());
        
        // limit selected area to a minimum of 11x11 points
        int lim = 5 - (y2 - y1) / 2;
        if (lim > 0)
        {
            y1 -= lim;
            y2 += lim;
        }
        lim = 5 - (x2 - x1 + 1) / 2;
        if (lim > 0)
        {
            x1 -= lim;
            x2 += lim;
        }
        
        // Set fixed scales
        d_plot->setAxisScale(axl, 
            d_plot->invTransform(axl,y1), d_plot->invTransform(axl,y2));
        d_plot->setAxisScale(axr, 
            d_plot->invTransform(axr,y1), d_plot->invTransform(axr,y2));
        d_plot->setAxisScale(axb, 
            d_plot->invTransform(axb,x1), d_plot->invTransform(axb,x2));
        d_plot->replot();
        
        showInfo(cursorInfo);
        d_plot->setOutlineStyle(Qwt::Triangle);
        d_zoom = FALSE;
    }
}

int main (int argc, char **argv)
{
    QApplication a(argc, argv);

    MainWin w;
    a.setMainWidget(&w);
    w.resize(540,400);
    w.show();

    int rv = a.exec();
    return rv;
}
