#include "cpuload.h"
#include <qwt_math.h>

Lowpass::Lowpass()
{
    tsampl = 1.0;
    tconst = 0.0;
    recalc();
    reset(0.0);
}

double Lowpass::input(double v)
{
    val = c2 * v + c1 * val;
    return val;
}

void Lowpass::recalc()
{
    if (tconst > 0.00001)
       c1 = exp(-tsampl / tconst);
    else
       c1 = 0.0;

    c2 = 1.0 - c1;
}


void MainWin::setTSampl(double sec)
{
    int i; 
    int ms = int(sec * 1000.0);

    if(tmrID != 0) 
       killTimer(tmrID);

    for(i=0;i<ThermoCnt;i++)
       lp[i].setTSampl(sec);

    tmrID = startTimer(ms);
}

void MainWin::setTConst(double sec)
{
    int i;
    for(i=0;i<ThermoCnt;i++)
       lp[i].setTConst(sec);
}

void MainWin::read()
{
    static char buffer[20];
    int i;

    ifstream inp("/proc/stat");
    inp >> buffer;
    for(i=0;i<ThermoCnt;i++)
    {
	old[i] =  val[i];
	inp >> val[i];
    }

}


//---------------------------------------
// MainWin::update()
//
//	read values from /proc/loadavg
//  and display them. Adjust thermometer scales
//  if necessary.
//---------------------------------------
void MainWin::update()
{
    unsigned long delta[ThermoCnt];
    unsigned long sum = 0;
    double factor;
    double newval,maxval;
    int i;

    read();

    for(i=0;i<ThermoCnt;i++)
    {
	delta[i] = val[i] - old[i];
	sum += delta[i];
    }

    if (sum > 0)
       factor = 100.0 / double(sum);
    else 
       factor = 0.0;

    if (dynscale)
    {
	for(i=0;i<ThermoCnt;i++)
	{
	    newval = lp[i].input(double(delta[i]) * factor);
	    maxval = qwtMax(qwtCeil125(newval), 3.0);
	    if ( (maxval > th[i]->maxValue()) || (maxval < 0.35 * th[i]->maxValue()) )
	       th[i]->setRange(0.0,qwtCeil125(maxval));
	    th[i] -> setValue(newval);
	}
    }
    else
    {			
	for(i=0;i<ThermoCnt;i++)
	   th[i]->setValue(lp[i].input(double(delta[i]) * factor));		
    }
}

//---------------------------------------
// MainWin::timerEvent
//
// 	update thermometers
//---------------------------------------
void MainWin::timerEvent(QTimerEvent *)
{
    update();
}

MainWin::MainWin(QApplication &a)
: QWidget()
{
    int i;
    QColor cFill("MidnightBlue");

    for(i=0;i<ThermoCnt;i++)
       lp[i].reset(0);
    
    for(i=0;i<ThermoCnt;i++)
    {
	th[i] = new QwtThermo(this,"");
	lb[i] = new QLabel(this);
    }

    puProg = new QPopupMenu;
    puProg->insertItem("Quit", &a, SLOT(quit()));

    puScale = new QPopupMenu;
    puScale->insertItem("Dynamic", this, SLOT(setDynScale()));
    puScale->insertItem("Fixed (0-100)", this, SLOT(setFixedScale()));


    puConf = new QPopupMenu;
    puConf->insertItem("Update Mode...", this, SLOT(showDialog()));
    puConf->insertItem("Scale", puScale);

    menu = new QMenuBar(this);
    menu->insertItem("Program", puProg);
    menu->insertItem("Options", puConf);
    
    cfg = new ConfigDiag;

    lb[CpuUser]->setText("% User");
    lb[CpuNice]->setText("% Nice");
    lb[CpuSystem]->setText("% System");
    lb[CpuIdle]->setText(" % Idle");

    for(i=0;i<ThermoCnt;i++)
    {
	th[i]->setGeometry(10 + i*60 ,30,50,100);
	lb[i]->setGeometry(10 + i*60, 130,50,20);
    }

    for(i=0;i<ThermoCnt;i++)
    {
	th[i]->setOrientation(QwtThermo::Vertical, QwtThermo::Left);
	th[i]->setRange(0.0,100.0);
	th[i]->setValue(0.0);
	th[i]->setBorderWidth(1);
	th[i]->setPipeWidth(4);
	th[i]->setFont(QFont("Helvetica",10));
	th[i]->setScaleMaxMajor(6);
	th[i]->setScaleMaxMinor(5);
	th[i]->setFillColor(cFill);

	lb[i]->setAlignment(AlignRight|AlignTop);
    }
    
    dynscale = 0;
    tmrID = 0;

    connect(cfg, SIGNAL(tConstChg(double)), SLOT(setTConst(double)));
    connect(cfg, SIGNAL(tSamplChg(double)), SLOT(setTSampl(double)));

    read();
    setTConst(1.0);
    setTSampl(0.2);
    cfg->setTConst(1.0);
    cfg->setTSampl(0.2);

    setFixedSize(10 + ThermoCnt*60 ,150);

}

void MainWin::setFixedScale()
{
    int i;
    dynscale = 0;
    for(i=0;i<ThermoCnt;i++)
       th[i]->setRange(0.0,100.0);
}

MainWin::~MainWin()
{
    int i;
    for(i=0;i<ThermoCnt;i++)
    {
	delete th[i];
	delete lb[i];
    }
    delete cfg;
    delete menu;
    delete puProg;

    if (tmrID != 0) killTimer(tmrID);
}

void MainWin::showDialog()
{
    cfg->show();
}

ConfigDiag::ConfigDiag(QWidget *p = 0, const char *name = 0)
: QDialog(p, name)
{
    
    ctConst = new QwtCounter(this);
    ctSampl = new QwtCounter(this);
    lbSampl = new QLabel("Update Rate [s]", this);
    lbConst = new QLabel("Average Time [s]", this);
    btDismiss = new QPushButton("Dismiss",this);

    lbSampl->setGeometry(10,10,120,20);
    lbConst->setGeometry(10,40,120,20);
    ctSampl->setGeometry(130,10,120,20);
    ctConst->setGeometry(130,40,120,20);
    btDismiss->setGeometry(95,75,70,25);

    ctConst->setRange(0.0,60,0.5);
    ctSampl->setRange(0.1,5.0,0.1);

    setCaption("CPU Load - Configuration");
    setFixedSize(260,110);

    connect(btDismiss, SIGNAL(clicked()), SLOT(accept()));
    connect(ctConst, SIGNAL(valueChanged(double)), SLOT(chgTConst(double)));
    connect(ctSampl, SIGNAL(valueChanged(double)), SLOT(chgTSampl(double)));
}

ConfigDiag::~ConfigDiag()
{
}

void ConfigDiag::setTSampl(double t)
{
    ctSampl->setValue(t);
}

void ConfigDiag::setTConst(double t)
{
    ctConst->setValue(t);
}


int main (int argc, char **argv)
{
    QApplication a(argc, argv);

    MainWin w(a);
    
    a.setMainWidget(&w);

    w.setCaption("CPU Load");

    w.show();
    
    return a.exec();
}









