/*-*- c++ -*-******************************************************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#include <qlayout.h>
#include <qlineedit.h>
#include "qwt_counter.h"
#include "qwt_arrbtn.h"

/*!
  \brief ctor

  The default number of buttons is set to 2. The default increments are:
  \li Button 1: 1 step
  \li Button 2: 10 steps
  \li Button 3: 100 steps

  \param parent
  \param name Forwarded to QWidget's ctor.
 */
QwtCounter::QwtCounter(QWidget *parent, char *name ):
    QWidget(parent,name)
{
    d_increment[Button1] = 1;
    d_increment[Button2] = 10;
    d_increment[Button3] = 100;

    QHBoxLayout *layout = new QHBoxLayout(this);
    layout->setAutoAdd(TRUE);

    int i;
    for(i=0;i<ButtonCnt;i++)
    {
        QwtArrowButton *btn =
            new QwtArrowButton(i+1, QwtArrowButton::Down,this);
        btn->setAutoRepeat(TRUE);
        connect(btn, SIGNAL(released()), SLOT(btnReleased()));
        connect(btn, SIGNAL(clicked()), SLOT(btnClicked()));

        d_buttonDown[i] = btn;
    }

    d_valueEdit = new QLineEdit(this);
    d_valueEdit->setReadOnly(TRUE);

    layout->setStretchFactor(d_valueEdit, 10);

    for(i=0;i<ButtonCnt;i++)
    {
        QwtArrowButton *btn =
            new QwtArrowButton(i+1, QwtArrowButton::Up, this);
        btn->setAutoRepeat(TRUE);
        connect(btn, SIGNAL(released()), SLOT(btnReleased()));
        connect(btn, SIGNAL(clicked()), SLOT(btnClicked()));
    
        d_buttonUp[i] = btn;
    }

    setNumButtons(2);
    setRange(0.0,1.0,0.001);
    setValue(0.0);
}

/*!
  Specify the number of steps by which the value
  is incremented or decremented when a specified button
  is pushed.

  \param btn One of \c QwtCounter::Button1, \c QwtCounter::Button2,
             \c QwtCounter::Button3
  \param nSteps Number of steps
*/
void QwtCounter::setIncSteps(QwtCounter::Button btn, int nSteps)
{
    if (( btn >= 0) && (btn < ButtonCnt))
       d_increment[btn] = nSteps;
}

/*!
  \return the number of steps by which a specified button increments the value
  or 0 if the button is invalid.
  \param btn One of \c QwtCounter::Button1, \c QwtCounter::Button2,
  \c QwtCounter::Button3
*/
int QwtCounter::incSteps(QwtCounter::Button btn) const
{
    if (( btn >= 0) && (btn < ButtonCnt))
       return d_increment[btn];

    return 0;
}

/*!
  \brief Set a new value
  \param v new value
  Calls QwtDblRange::setValue and does all visual updates.
  \sa QwtDblRange::setValue
*/

void QwtCounter::setValue(double v)
{
    QwtDblRange::setValue(v);

    showNum(value());
    updateButtons();
}

/*!
  \brief Notify a change of value
*/
void QwtCounter::valueChange()
{
    showNum(value());
    updateButtons();

    emit valueChanged(value());
}

/*!
  \brief Update buttons according to the current value
*/
void QwtCounter::updateButtons()
{
    for ( int i = 0; i < ButtonCnt; i++ )
    {
        d_buttonDown[i]->setEnabled(value() > minValue());
        d_buttonUp[i]->setEnabled(value() < maxValue());
    }
}

/*!
  \brief Specify the number of buttons on each side of the label
  \param n Number of buttons
*/
void QwtCounter::setNumButtons(int n)
{
    for ( int i = 0; i < ButtonCnt; i++ )
    {
        if ( i < n )
        {
            d_buttonDown[i]->show();
            d_buttonUp[i]->show();
        }
        else
        {
            d_buttonDown[i]->hide();
            d_buttonUp[i]->hide();
        }
    }

    d_nButtons = n;
}

/*!
    \return The number of buttons on each side of the widget.
*/
int QwtCounter::numButtons() const 
{ 
    return d_nButtons; 
}

//!  Display number string
void QwtCounter::showNum(double d)
{
    QString v;
    v.setNum(d);
    d_valueEdit->setText(v);
}

//!  Button clicked
void QwtCounter::btnClicked()
{
    for ( int i = 0; i < ButtonCnt; i++ )
    {
        if ( d_buttonUp[i] == sender() )
            incValue(d_increment[i]);

        if ( d_buttonDown[i] == sender() )
            incValue(-d_increment[i]);
    }
}

//!  Button released
void QwtCounter::btnReleased()
{
    emit buttonReleased(value());
}

/*!
  \brief Notify change of font

  This function updates the fonts of all widgets
  contained in QwtCounter.
  \param f new font
*/
void QwtCounter::fontChange(const QFont &f)
{
    QWidget::fontChange( f );
    d_valueEdit->setFont(font());
}

//! A size hint
QSize QwtCounter::sizeHint() const
{
    QString tmp;
    QFontMetrics fm(d_valueEdit->font());

    int w = fm.width(tmp.setNum(minValue()));
    int w1 = fm.width(tmp.setNum(maxValue()));
    if ( w1 > w )
        w = w1;
    w1 = fm.width(tmp.setNum(minValue() + step()));
    if ( w1 > w )
        w = w1;
    w1 = fm.width(tmp.setNum(maxValue() - step()));
    if ( w1 > w )
        w = w1;

    // QLineEdit::minimumSizeHint is for one char. Subtracting
    // the size for the char we get all the margins, frames ...

    w += d_valueEdit->minimumSizeHint().width() - fm.maxWidth();

    // Ok, now we replace default sizeHint of lblValue by
    // what we need really need.

    w += QWidget::sizeHint().width() - d_valueEdit->sizeHint().width();

    return QSize(w, QWidget::sizeHint().height());
}

//! \return Preferred/Fixed
QSizePolicy QwtCounter::sizePolicy() const
{
    QSizePolicy sp;
    sp.setHorData( QSizePolicy::Preferred );
    sp.setVerData( QSizePolicy::Fixed );
    return sp;
}
