/*-*- c++ -*-******************************************************************
 * Qwt Widget Library
 * Copyright (C) 1997   Josef Wilgen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#include <qdrawutil.h>
#include <qpainter.h>
#include "qwt_slider.h"

/*!
  \brief Constructor
  \param parent parent widget
  \param name The Widget's name. Default = 0.
  \param Orient Orientation of the slider. Can be Qt::Horizontal
         or Qt::Vertical. Defaults to Horizontal.
  \param scalePos Position of the scale.  Can be QwtSlider::None,
         QwtSlider::Left, QwtSlider::Right, QwtSlider::Top,
         or QwtSlider::Bottom. Defaults to QwtSlider::None.
  \param bgStyle Background style. QwtSlider::BgTrough draws the
         slider button in a trough, QwtSlider::BgSlot draws
         a slot underneath the button. An or-combination of both
         may also be used. The default is QwtSlider::BgTrough.
*/
QwtSlider::QwtSlider(QWidget *parent, const char *name,
        Orientation orient, ScalePos scalePos, int bgStyle): 
    QwtSliderBase(parent,name, WRepaintNoErase|WResizeNoErase)
{
    d_borderWidth = 2;
    d_scaleDist = 4;
    d_orient = orient;
    d_scalePos = scalePos;
    d_xMargin = 0;
    d_yMargin = 0;
    d_bgStyle = bgStyle;

    if (bgStyle == BgSlot)
    {
        d_thumbLength = 16;
        d_thumbWidth = 30;
    }
    else
    {
        d_thumbLength = 31;
        d_thumbWidth = 16;
    }

    d_sliderRect.setRect(0,0,8,8);

    QwtScaleDraw::Orientation so;
    if ( d_orient == Qt::Vertical )
    {
        if (d_scalePos == Right)
           so = QwtScaleDraw::Right;
        else
           so = QwtScaleDraw::Left;
    }
    else
    {
        if (d_scalePos == Bottom)
           so = QwtScaleDraw::Bottom;
        else
           so = QwtScaleDraw::Top;
    }

    d_scale.setGeometry(0,0,100,so);
}

/*!
  \brief Change the slider's border width
  \param bd border width
*/
void QwtSlider::setBorderWidth(int bd)
{
    if ( bd < 0 )
        bd = 0;

    if ( bd != d_borderWidth )
    {
        d_borderWidth = bd;
        layoutSlider();
    }
}

/*!
  \brief Set the slider's thumb length
  \param l new length
*/
void QwtSlider::setThumbLength(int thumbLength)
{
    if ( thumbLength < 8 )
        thumbLength = 8;

    if ( thumbLength != d_thumbLength )
    {
        d_thumbLength = thumbLength;
        layoutSlider();
    }
}

/*!
  \brief Change the width of the thumb
  \param w new width
*/
void QwtSlider::setThumbWidth(int w)
{
    if ( w < 4 )
        w = 4;

    if ( d_thumbWidth != w )
    {
        d_thumbWidth = w;
        layoutSlider();
    }
}


//! Notify changed scale
void QwtSlider::scaleChange()
{
    if (!hasUserScale())
    {
        d_scale.setScale(minValue(), maxValue(), 
            d_maxMajor, d_maxMinor, d_scale.scaleDiv().logScale());
    }

    layoutSlider();
}


//! Notify change in font
void QwtSlider::fontChange(const QFont &f)
{
    QwtSliderBase::fontChange( f );
    layoutSlider();
}

//! Draw the slider into the specified rectangle.
void QwtSlider::drawSlider(QPainter *p, const QRect &r)
{
    QRect cr(r);

    if (d_bgStyle & BgTrough)
    {
        qDrawShadePanel(p, r.x(), r.y(),
            r.width(), r.height(),
            colorGroup(), TRUE, d_borderWidth,0);

        cr.setRect(r.x() + d_borderWidth,
            r.y() + d_borderWidth,
            r.width() - 2 * d_borderWidth,
            r.height() - 2 * d_borderWidth);

        p->fillRect(cr.x(), cr.y(), cr.width(), cr.height(), 
            colorGroup().brush(QColorGroup::Mid));
    }

    if ( d_bgStyle & BgSlot)
    {
        int ws = 4;
        int ds = d_thumbLength / 2 - 4;
        if ( ds < 1 )
            ds = 1;

        QRect rSlot;
        if (d_orient == Qt::Horizontal)
        {
            if ( cr.height() & 1 )
                ws++;
            rSlot = QRect(cr.x() + ds, 
                    cr.y() + (cr.height() - ws) / 2,
                    cr.width() - 2 * ds, ws);
        }
        else
        {
            if ( cr.width() & 1 )
                ws++;
            rSlot = QRect(cr.x() + (cr.width() - ws) / 2, 
                    cr.y() + ds,
                    ws, cr.height() - 2 * ds);
        }
        p->fillRect(rSlot.x(), rSlot.y(), rSlot.width(), rSlot.height(),
            colorGroup().brush(QColorGroup::Dark));
        qDrawShadePanel(p, rSlot.x(), rSlot.y(),
            rSlot.width(), rSlot.height(), colorGroup(), TRUE, 1 ,0);

    }

    drawThumb(p, cr, xyPosition(value()));
}

//! Draw the thumb at a position
void QwtSlider::drawThumb(QPainter *p, const QRect &sliderRect, int pos)
{
    pos++; // shade line points one pixel below
    if (d_orient == Qt::Horizontal)
    {
        qDrawShadePanel(p, pos - d_thumbLength / 2, 
            sliderRect.y(), d_thumbLength, sliderRect.height(),
            colorGroup(), FALSE, d_borderWidth, 
            &colorGroup().brush(QColorGroup::Button));

        qDrawShadeLine(p, pos, sliderRect.y(), 
            pos, sliderRect.y() + sliderRect.height() - 2, 
            colorGroup(), TRUE, 1);
    }
    else // Vertical
    {
        qDrawShadePanel(p,sliderRect.x(), pos - d_thumbLength / 2, 
            sliderRect.width(), d_thumbLength,
            colorGroup(),FALSE, d_borderWidth, 
            &colorGroup().brush(QColorGroup::Button));

        qDrawShadeLine(p, sliderRect.x(), pos,
            sliderRect.x() + sliderRect.width() - 2, pos, 
            colorGroup(), TRUE, 1);
    }
}

//! Find the x/y position for a given value v
int QwtSlider::xyPosition(double v) const
{
    int pos;
    if ( minValue() == d_scale.d1() && maxValue() == d_scale.d2() )
    {
        // We prefer to use the transformation of d_scale.
        // So ticks and marker are always in sync and we don't have
        // to take care of rounding problems.

        pos = d_scale.transform(v);
    }
    else
    {
        // range and d_scale differ ? Sounds strange but
        // might happen with logarithmic scales

        const double f = (v - minValue()) / (maxValue() - minValue());
        if ( d_orient == Qt::Horizontal )
            pos = (int)(d_scale.i1() + f * (d_scale.i2() - d_scale.i1()));
        else
            pos = (int)(d_scale.i1() - f * (d_scale.i1() - d_scale.i2()));
    }
    return pos;
}

//! Determine the value corresponding to a specified mouse location.
double QwtSlider::getValue(const QPoint &p)
{
    double rv;

    if ( minValue() == d_scale.d1() && maxValue() == d_scale.d2() )
    {
        rv = d_scale.invTransform(
            d_orient == Qt::Horizontal ? p.x() : p.y());
    }
    else
    {
        double pos;
        double range;
        if ( d_orient == Qt::Horizontal )
        {
            pos = p.x() - d_scale.i1();
            range = d_scale.i2() - d_scale.i1();
        }
        else
        {
            pos = d_scale.i1() - p.y();
            range = d_scale.i1() - d_scale.i2();
        }
            
        rv = minValue() + pos / range * (maxValue() - minValue()); 
    }

    return(rv);
}


/*!
  \brief Determine scrolling mode and direction
  \param p point
*/
void QwtSlider::getScrollMode(const QPoint &p, 
    int &scrollMode, int &direction )
{
    if (!d_sliderRect.contains(p))
    {
        scrollMode = ScrNone;
        direction = 0;
        return;
    }

    const int pos = ( d_orient == Qt::Horizontal ) ? p.x() : p.y();
    const int markerPos = xyPosition(value());

    if ((pos > markerPos - d_thumbLength / 2)
        && (pos < markerPos + d_thumbLength / 2))
    {
        scrollMode = ScrMouse;
        direction = 0;
        return;
    }

    scrollMode = ScrPage;
    direction = (pos > markerPos) ? 1 : -1;
    if ( d_scale.i1() > d_scale.i2() )
        direction = -direction;
}

//! Qt paint event
void QwtSlider::paintEvent(QPaintEvent *e)
{
    // Use double-buffering
    QRect ur = e->rect();
    if ( ur.isValid() )
    {
        QPixmap pix( ur.size() );
        pix.fill( this, ur.topLeft() );
        QPainter p( &pix, this );
        p.translate( -ur.x(), -ur.y() );
        draw(&p, ur);
        p.end();
        bitBlt( this, ur.topLeft(), &pix );
    }
}

//! redraw the slider
void QwtSlider::draw(QPainter *p, const QRect&)
{
    if (d_scalePos != None)
        d_scale.draw(p);

    drawSlider(p, d_sliderRect);
}

//! Qt resize event
void QwtSlider::resizeEvent(QResizeEvent *)
{
    layoutSlider( FALSE );
}

//! Recalculate the slider's geometry and layout based on
//  the current rect and fonts.
//  \param update_geometry  notify the layout system and call update
//         to redraw the scale

void QwtSlider::layoutSlider( bool update_geometry )
{
    int sliderWidth = d_thumbWidth;
    int sld1 = d_thumbLength / 2 - 1;
    int sld2 = d_thumbLength / 2 + d_thumbLength % 2;
    if ( d_bgStyle & BgTrough )
    {
        sliderWidth += 2 * d_borderWidth;
        sld1 += d_borderWidth;
        sld2 += d_borderWidth;
    }

    int scd = 0;
    if ( d_scalePos != None )
        scd = d_scale.minBorderDist( fontMetrics() );

    int slo = scd - sld1;
    if ( slo < 0 )
        slo = 0;

    const QRect r = rect();
    switch ( d_scalePos )
    {
        case Top:
        {
            d_sliderRect.setRect(r.x() + d_xMargin + slo,
                r.y() + r.height() -
                d_yMargin - sliderWidth,
                r.width() - 2 * d_xMargin - 2 * slo,
                sliderWidth);
            d_scale.setGeometry(d_sliderRect.x() + sld1,
                d_sliderRect.y() - d_scaleDist,
                d_sliderRect.width() - sld1 - sld2,
                QwtScaleDraw::Top);
            break;
        }
        case Bottom:
        {
            d_sliderRect.setRect(r.x() + d_xMargin + slo,
                r.y() + d_yMargin,
                r.width() - 2 * d_xMargin - 2 * slo,
                sliderWidth);
            d_scale.setGeometry(d_sliderRect.x() + sld1,
                d_sliderRect.y() + d_sliderRect.height() +
                d_scaleDist,
                d_sliderRect.width() - sld1 - sld2,
                QwtScaleDraw::Bottom);
            break;
        }
        case Left:
        {
            d_sliderRect.setRect(r.x() + r.width() -
                sliderWidth - d_xMargin,
                r.y() + d_yMargin + slo,
                sliderWidth,
                r.height() - 2 * d_yMargin - 2 * slo);
            d_scale.setGeometry(d_sliderRect.x() - d_scaleDist,
                d_sliderRect.y() + sld1,
                d_sliderRect.height() - sld1 - sld2,
                QwtScaleDraw::Left);
            break;
        }
        case Right:
        {
            d_sliderRect.setRect(r.x() + d_xMargin,
                r.y() + d_yMargin + slo,
                sliderWidth,
                r.height() - 2 * d_yMargin - 2 * slo);
            d_scale.setGeometry(d_sliderRect.x() + d_sliderRect.width() +
                d_scaleDist,
                d_sliderRect.y() + sld1,
                d_sliderRect.height() - sld1 - sld2,
                QwtScaleDraw::Right);
            break;
        }
        default:
        {
            d_sliderRect.setRect(r.x() + d_xMargin,
                r.y() + d_yMargin,
                r.width() - 2 * d_xMargin,
                r.height() - 2 * d_yMargin);

            // QwtScaleDraw is derived from QwtDiMap. Even if
            // no scale is required we keep the map alive.

            if ( d_orient == Qt::Horizontal )
            {
                d_scale.setIntRange(
                    d_sliderRect.x() + sld1,
                    d_sliderRect.x() + d_sliderRect.width() - sld2 - 1);
            }
            else
            {
                d_scale.setIntRange(
                    d_sliderRect.y() + d_sliderRect.height() - sld2 - 1,
                    d_sliderRect.y() + sld1);
            }
        }
    }

    if ( update_geometry )
    {
        updateGeometry();
        update();
    }
}

//! Notify change of value
void QwtSlider::valueChange()
{
    QwtSliderBase::valueChange();
    update();
}


//! Notify change of range
void QwtSlider::rangeChange()
{
    if (!hasUserScale())
    {
        d_scale.setScale(minValue(), maxValue(), 
            d_maxMajor, d_maxMinor, d_scale.scaleDiv().logScale());
    }

    QwtSliderBase::rangeChange();
    layoutSlider();
}

/*!
  \brief Set distances between the widget's border and internals.
  \param xMargin Horizontal margin
  \param yMargin Vertical margin
*/
void QwtSlider::setMargins(int xMargin, int yMargin)
{
    if ( xMargin < 0 )
        xMargin = 0;
    if ( yMargin < 0 )
        yMargin = 0;

    if ( xMargin != d_xMargin || yMargin != d_yMargin )
    {
        d_xMargin = xMargin;
        d_yMargin = yMargin;
        layoutSlider();
    }
}

/*!
  \return Horizontal: MinimumExpanding/Fixed,
        Vertical: Fixed/MinimumExpanding
*/
QSizePolicy QwtSlider::sizePolicy() const
{
    QSizePolicy sp;
    if ( d_orient == Qt::Horizontal )
    {
        sp.setHorData( QSizePolicy::MinimumExpanding );
        sp.setVerData( QSizePolicy::Fixed );
    }
    else
    {
        sp.setHorData( QSizePolicy::Fixed );
        sp.setVerData( QSizePolicy::MinimumExpanding );
    }
    return sp;
}

/*!
  \return QwtSlider::minimumSizeHint()
*/
QSize QwtSlider::sizeHint() const
{
    return minimumSizeHint();
}

/*!
  \brief Return a minimum size hint
  \warning The return value of QwtSlider::minimumSizeHint() depends on 
           the font and the scale.
*/
QSize QwtSlider::minimumSizeHint() const
{
    int w = 0, h = 0;

    int sliderWidth = d_thumbWidth;
    if (d_bgStyle & BgTrough)
        sliderWidth += 2 * d_borderWidth;

    if (d_scalePos != None)
    {
        int msWidth = d_scale.minWidth( QPen(), fontMetrics() );
        int msHeight = d_scale.minHeight( QPen(), fontMetrics() );
        int msMbd = d_scale.minBorderDist( fontMetrics() );

        int mbd = d_thumbLength / 2;
        if (d_bgStyle & BgTrough)
            mbd += d_borderWidth;

        if ( mbd < msMbd )
            mbd = msMbd;

        if (d_orient == Qt::Vertical)
        {
            w = 2 * d_xMargin + sliderWidth + msWidth + d_scaleDist;
            h = msHeight - 2 * msMbd + 2 * mbd + 2 * d_yMargin;
        }
        else
        {
            w = msWidth - 2 * msMbd + 2 * mbd + 2 * d_xMargin;
            h = 2 * d_yMargin + sliderWidth  + msHeight + d_scaleDist;
        }
    }
    else  // no scale
    {
        if (d_orient == Qt::Vertical)
        {
            w = 2 * d_xMargin + sliderWidth;
            h = 200 + 2 * d_yMargin;
        }
        else
        {
            w = 200 + 2 * d_xMargin;
            h = 2 * d_yMargin + sliderWidth;
        }
    }
    return QSize(w,h);
}
