/* SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only */
/* Copyright (c) 2022 Brett Sheffield <bacs@librecast.net> */

#include "test.h"
#include <errno.h>
#include <librecast.h>
#include <librecast/if.h>
#include <limits.h>
#include "../src/mld_pvt.h"
#include <net/if.h>
#include <sys/types.h>
#include <ifaddrs.h>

unsigned int get_invalid_ifx(void)
{
	char ifname[IF_NAMESIZE];
	for (unsigned int ifx = 1; ifx < UINT_MAX; ifx++) {
		if (!if_indextoname(ifx, ifname)) return ifx;
	}
	return 0;
}

/* find an interface that supports multicast */
unsigned get_multicast_if(void)
{
	unsigned ifidx = 0;
	/* create tap interface and bring it up */
	char ifname[IFNAMSIZ];
#ifdef __linux__
	lc_tuntap_create(ifname, IFF_TAP | IFF_NO_PI);
	lc_ctx_t *lctx = lc_ctx_new();
	lc_link_set(lctx, ifname, 1);
	lc_ctx_free(lctx);
	ifidx = if_nametoindex(ifname);
#endif
	if (!ifidx) {
		test_log("unable to create tap device: %s\n", strerror(errno));
		/* failed to create tap, find multicast capable interface */
		struct ifaddrs *ifa = NULL, *ifap = NULL;
		test_assert(getifaddrs(&ifa) != -1, "getifaddrs(): %s", strerror(errno));
		for (ifap = ifa; ifap; ifap = ifap->ifa_next) {
			if (!(ifap->ifa_flags & IFF_MULTICAST)) continue;
			/* we don't want the loopback interface */
			if (ifap->ifa_flags & IFF_LOOPBACK) continue;
			if (ifap->ifa_addr == NULL) continue;
			if (ifap->ifa_addr->sa_family != AF_INET6) continue;
			ifidx = if_nametoindex(ifap->ifa_name);
			test_log("found multicast interface %s [%u]\n", ifap->ifa_name, ifidx);
			break;
		}
		freeifaddrs(ifa);
	}
	else {
		test_log("tap %s created\n", ifname);
	}
	return ifidx;
}
