/* This is not really -*-c++-*-, but it tricks Emacs into believing so anyway. */

//! TODO:
//!
//! - intercept <ol>/<li>:ification of lists
//! - set different link colours for seen/unseen; <a style="color: foo" ...>
//! - try avoiding mapping overusage and the useless caching of diary entries
//! - add the inode creation time to the cache, allowing for coolness in the alldays tag
//! - make things work even before any diary entries exist.

#include <module.h>
inherit "module";
inherit "roxenlib";

constant cvs_version="$Id: diary.pike,v 0.51 1999/04/24 05:26:07 johan Exp $";

//! Use a different name for the module in development versions
//#ifndef ALPHA
//#define ALPHA
//#endif

/* Date entry constants, sort of ugly */
constant NONEXISTENT	= -1;
constant INVALID	= -2;

//! Constants used by prev() and next()
constant MOST_RECENT = 0; //!
constant FIRST_EVER  = 0; //! Not synonyms; analogous, though. :)
constant YEAR        = 1;
constant MONTH       = 2;
constant DAY         = 3;

//! The following comment borrowed from the countdown.pike module.
//! This means that the module just might stop working in future
//! (>1.2) Roxen distributions.

// :-) This code is not exactly conforming to the Roxen API, since it
// uses a rather private mapping the language object (which you are
// not even supposed to know the existence of). But I wanted some nice
// month->number code that did not depend on a static mapping.
// Currently, this means that you can enter the name of the month or day in
// your native language, if it is supported by roxen.
//constant roxenlang = roxen->language;
string month_name(int n, string language) // 1<=n<=12
{ return roxen->language(language || query( "language" ), "month" )( n ); }
string week_day(int n, string language) // 1<=n<=7
{ return roxen->language(language || query( "language" ), "day" )( n ); }
string ordered_number(int n, string language)
{
  return number2string( n,
			([ "type":"string" ]),
			roxen->language(language || query( "language" ), "ordered"));
}
#ifdef 0
string roman_number(int n)
{ return number2string(n, ([ "type":"roman" ]), ([])); }
#endif

/************************************************************************
                    All sorts of convenient functions
/************************************************************************/

mixed recursive_add(mixed a, mixed b, int depth)
{
  if( !depth ) return a + b;
  depth--;
  mixed ret;
  switch(sprintf("%t-%t",a,b))
    {
    case "array-array":
      ret = allocate(sizeof(a));
      for(int e = 0 ; e<sizeof( a ) ; e++)
	ret[e] = recursive_add( a[e], b[e], depth );
      return ret;

    case "mapping-mapping":
      ret = a ^ b;
      foreach( indices(a & b), mixed ind )
	ret[ind] = recursive_add( a[ind], b[ind], depth );
      return ret;
    }
  return a + b;
}

//! Returns an array of integer indexes denoting case
//! insensitive occurences of the needle in the haystack,
//! in rising order.
int *multi_search_nocase(string haystack, string needle)
{
  string data = lower_case( haystack ),
        match = lower_case( needle );
  int idx, offs = 0;
  int *occurences = ({ });

  while((idx = search( data[offs..], match ))!=-1) {
    idx += offs;
    occurences += ({ idx });
    offs = idx + 1;
  }

  return occurences;
}

//! A simple case-insensitive replace function, which returns
//! its data in the interlieved unchanged { changed array format
array replace_nocase(string text, string find, string replace)
{
  int *hits = multi_search_nocase( text, find );
  int delta = sizeof( find ),
  positions = sizeof( hits );
  array unchanged = allocate( positions+1 ),
          changed = allocate( positions );
  int current, previous = 0;
  for(int i = 0 ; i<positions ; i++ )
  {
    current = hits[i];
    unchanged[i] = text[previous..current-1];
      changed[i] = replace;
      previous = current + delta;
  }
  unchanged[-1] = text[previous..];
  return ({ unchanged, changed });
}

//! A multi-value case-insensitive replace function, which
//! both takes and returns its data in the interlieved
//! unchanged { changed array format:
//! unchanged is always one element larger than changed,
//! since all changed sections are enclosed with unchanged
//! data. A totally changed string is hence represented as
//! unchanged=({ "", "" }), changed=({ "changed string" })
//! Replacement is naturally done in the same order as occur
//! the value pairs fetched from the the find/replace arrays
array multi_replace_nocase(array unchanged, array changed,
			   array find, array replace)
{
  string f, r;
  array tmp_unc, tmp_chg, result;
  int j = 0, last;
  while( j<sizeof( find ) )
  {
    f = find[j];
    r = replace[j++];
    tmp_unc = ({ });
    tmp_chg = ({ });
    last = sizeof( unchanged )-1;
    for(int i = 0 ; i<=last ; i++)
    {
      result = replace_nocase( unchanged[i], f, r );
      tmp_unc += result[0];
      tmp_chg += result[1];
      if(i < last)
	tmp_chg += ({ changed[i] });
    }
    unchanged = tmp_unc;
      changed = tmp_chg;
  }

  return ({ unchanged, changed });
}

//! Does a regexp search-and-replace on the text and returns the
//! result in the interleaved unchanged { changed array format.
//! Unfortunately, neither ^, $, \< nor \> have any effect in these
//! regexps. :-( Thay may well be used, but are not respected.

//! ?! -- It seems they might actually work in practice, though I
//! don't quite understand why. It *should* be possible to generate
//! a situation where ^, $, \< or \> *don't* work, but I have not
//! yet managed to. And even if so, why spoil the fun? :-)
array replace_regexp(string text, string regexp, string replacement)
{
  object r;
  array hit, changed = ({ }), temp,
           unchanged = ({ text });
  int i = 0;
  string original;
  if(catch { r = Regexp( "("+regexp+")" ); } )
    return ({ unchanged, changed }); //! Illegal regexp given!

  while(i<sizeof( unchanged ))
  {
    original = unchanged[i];
    if(hit = r->split( original ))
    {
      temp = original / hit[0];
      unchanged[i] = temp[0];
      unchanged += ({ temp[1..]*hit[0] });
      changed += ({ replace(replacement,
			    lambda(int n)
			    {
			      array a = allocate(n);
			      for(int i = 0 ; i<n ; i++)
				a[i] = sprintf("\\%d", i+1);
			      return a;
			    }( sizeof( hit )-1 ),
			    hit[1..] )
		  });
    }
    i++;
  }
  return ({ unchanged, changed });
}

//! A definete power function -- replaces regexp matches from
//! the "from" array with "replacement regexps" fetched from
//! the "to" array, in the same order as the arrays themselves,
//! in the interleaved unchanged { changed array format.
array multi_replace_regexp(array unchanged, array changed,
			   array from, array to)
{
  string regexp, replacement;
  array tmp_unc, tmp_chg, result;
  int j = 0, last;
  while( j<sizeof( from ) )
  {
    regexp = from[j];
    replacement = to[j++];
    tmp_unc = ({ });
    tmp_chg = ({ });
    last = sizeof( unchanged )-1;
    for(int i = 0 ; i<=last ; i++)
    {
      result = replace_regexp( unchanged[i], regexp, replacement );
      tmp_unc += result[0];
      tmp_chg += result[1];
      if(i < last)
	tmp_chg += ({ changed[i] });
    }
    unchanged = tmp_unc;
      changed = tmp_chg;
  }

  return ({ unchanged, changed });
}

//! interleave a string given in the unchanged { changed
//! array format, rendering a plain string
string interleave(array unchanged, array changed)
{
  string result = unchanged[0];
  for(int i = 0 ; i<sizeof( changed ) ;)
    result += changed[i++] + unchanged[i];
  return result;
}

string my_root_address(object request_id)
{
  if(request_id->misc->host)
    return "http://"+request_id->misc->host;
  else
    return request_id->conf->query( "MyWorldLocation" );
}

void create()
{
  set_module_creator("Johan Sundstrm");
  set_module_url("http://a205.ryd.student.liu.se/(Diary)/docs/");

  defvar( "mountpoint", "/diary/", "Mount point", TYPE_LOCATION,
	  "This is where the module will be inserted in the "
	  "namespace of your server." );

  //! All language files are supposed to be in this directory,
  //! and, imitating language.pike, we fetch the filenames
  //! minus extension and hope this is the english language
  //! representation for this language is that string...
  array files = Array.map(get_dir("languages"),
			  lambda(string s)
			  { return s[0..search(s,".")-1]; }
			  );

  //! Some rather ugly code that returns an array of
  //! arrays of aliases for languages spoken by roxen
  //! (since we are not certain all files in the files
  //! array were actually successfully loaded)
  array l = Array.uniq(Array.map(indices(roxen->languages),
				 lambda(string l)
  { return roxen->languages[l]["\000"]; }
				 ))->aliases();

  //! Now, we just do a cross-section of these arrays
  array languages = ({ });
  array temp;
  foreach( l, array aliases )
  {
    if(sizeof(temp = aliases & files))
      languages += temp;
    else //! bad filename; go for the last alias instead:
      languages += ({ aliases[-1] });
  }

  defvar( "language", "english", "Language", TYPE_STRING_LIST,
	  "This is the language utilized by the module when "
	  "reporting dates etc. I'm peeking around private "
	  "Roxen internal structures for language support, "
	  "which might stop working in future (>1.2) Roxen "
	  "versions.", sort( languages ) );

  defvar( "searchpath", "", "Filesystem path", TYPE_DIR,
	  "The path to your diary directory in the "
	  "real filesystem." );

  defvar( "templatefile", "", "Layout templatefile", TYPE_FILE,
	  "The path to the template used to define "
	  "the look of your diary pages. Most "
	  "people would want this file to include "
	  "the tags <tt>&lt;calendar&gt;</tt> and "
	  "<tt>&lt;diary&gt;</tt>." );

  defvar( "template",
	  "<html><head><title></title></head>\n"
	  "<body>"
	  "<table><tr><td><navbar></td><td><diary></td></tr></table>"
	  "</body></html>",
	  "Default template", TYPE_TEXT_FIELD|VAR_EXPERT,
	  "If no templatefile was found, this template will "
	  "be used instead. An empty title element will be "
	  "filled with the date of the diary entry." );

  defvar( "future?", 0, "Show dates in the future?", TYPE_FLAG|VAR_MORE,
	  "If set, the &lt;monthmap&gt; will always display a whole "
	  "month. Otherwise, it will not list dates that have not "
	  "yet happened, which the module author personally "
	  "prefers -- hence it is the default setting. :-)" );

  //! ...and all Text -> HTML magic:

  defvar( "paragraph?", 1,
	  "Text2HTML magic:Paragraph breaking", TYPE_FLAG|VAR_MORE,
	  "If set, a paragraph break element (<tt>&lt;p&gt;</tt>) "
	  "will be inserted where the text file contains two "
	  "consecutive newline characters. For example, this text:<p>"
	  "<pre>Finally I've figured out how to document this function!"
	  "\n\n"
	  "I'll just give an example of how it's _used_,\n"
	  "instead of just explaining how it's implemented!</pre>"
	  "will be turned to this HTML code:<br>"
	  "<pre>Finally I've figured out how to document this function!"
	  "\n&lt;p&gt;\n"
	  "I'll just give an example of how it's _used_,\n"
	  "instead of just explaining how it's implemented!</pre>"
	  );

  defvar( "separator?", 1,
	  "Text2HTML magic:Separator breaking", TYPE_FLAG|VAR_MORE,
	  "If set, a separator image will be be inserted where the text "
	  "file contains three consecutive newline characters. For "
	  "example, this text:<p>"
	  "<pre>And that was that."
	  "\n\n\n"
	  "And now for something completely different!</pre>"
	  "will be turned to this HTML code:<br>"
	  "<pre>And that was that."
	  "\n&lt;p align=center&gt;&lt;img alt=\"---\" align=center "
	  "width=\"147\" height=\"17\" src=\"images/snirkel.gif\"&gt;"
	  "&lt;/p&gt;\n"
	  "And now for something completely different!</pre>"
	  );

  defvar( "line break?", 1,
	  "Text2HTML magic:Line breaking", TYPE_FLAG|VAR_MORE,
	  "If set, all occurences of ^M (the carriage return "
	  "character, ASCII 13, \"Ctrl-M\") will be be substituted "
	  "for a line break, <tt>&lt;br&gt;</tt>. For example, "
	  "this poem, Copyright &copy; <a href=\"http://www.lucas"
	  "arts.com/\">LucasArts</a>, each line ending in the ^M "
	  "character:<p>"
	  "<pre>It shone, pale as bone, as I stood there alone
\n"
	  "And I thought to myself how the moon
\n"
	  "that night, cast its light, on my hearts true delight
\n"
	  "and the reef where her body was strewn
</pre>"
	  "would be turned to this HTML code:<br>"
	  "<pre>It shone, pale as bone, as I stood there alone&lt;br&gt;\n"
	  "And I thought to myself how the moon&lt;br&gt;\n"
	  "that night, cast its light, on my hearts true delight&lt;br&gt;\n"
	  "and the reef where her body was strewn&lt;br&gt;</pre>"
	  );

  defvar( "excerpts?", 1,
	  "Text2HTML magic:Preformatted excerpts", TYPE_FLAG|VAR_MORE,
	  "<pre>--------8&lt;--------8&lt;-------8&lt;--------8&lt;--------\n"
	  "If set, whenever a section marked as \"cut-out\",\n"
	  "in this fashion, will be marked as pre-\n"
	  "formatted, just like this text (enclosed in a\n"
	  "&lt;pre&gt; container tag).\n"
	  "\n"
	  "How many  dashes or how many scissors there are\n"
	  "in the enclosing \"cut here\" markers is really\n"
	  "unimportant, as long as there are at least\n"
	  "three dashes before the first pair of scissors\n"
	  "and three dashes after the last pair of\n"
	  "scissors. These lines, themselves, disappear.\n"
	  "\n"
	  "Note: this only works for one section per file.\n"
	  "--------8&lt;--------8&lt;-------8&lt;--------8&lt;--------</pre>"
	  );

  defvar( "urlify?", 1,
	  "Text2HTML magic:URLify URL:s", TYPE_FLAG|VAR_MORE,
	  "When set, links (such as http://a205.ryd.student.liu.se/"
	  "(Diary)/docs/) words will turn into real links (such as "
	  "<a href=\"http://a205.ryd.student.liu.se/(Diary)/docs/\""
	  ">http://a205.ryd.student.liu.se/(Diary)/docs/</a>)." );

  defvar( "magic urlify?", 1,
	  "Text2HTML magic:URLify &quot;named&quot; URL:s", TYPE_FLAG|VAR_MORE,
	  "When set, links (such as \"My docs\" (\"http://a205.ryd."
	  "student.liu.se/(Diary)/docs/\") will turn into real links "
	  "(such as <a href=\"http://a205.ryd.student.liu.se/(Diary)/"
	  "docs/\">My docs</a>)." );

  defvar( "dateify?", 1,
	  "Text2HTML magic:URLify dates", TYPE_FLAG|VAR_MORE,
	  "When set, plain-text references to other dates present "
	  "in the diary (such as 1999-02-05) will turn into real links "
	  "(such as <a href=\"1999-02-05.html\">1999-02-05</a>)." );

  defvar( "boldify?", 1,
	  "Text2HTML magic:Make _bold_", TYPE_FLAG|VAR_MORE,
	  "When set, _important words_ will turn <b>bold</b>." );

  defvar( "emphasize?", 1,
	  "Text2HTML magic:Emphasize", TYPE_FLAG|VAR_MORE,
	  "When set, /emphasized words/ will really turn out "
	  "<em>emphasized</em>." );

  defvar( "quote?", 1,
	  "Text2HTML magic:Quote possible HTML", TYPE_FLAG|VAR_MORE,
	  "When turned on, &lt;, &gt;, &amp;, &#34;, &#39; and ASCII 0 "
	  "will be turned into <tt>&amp;lt;</tt>, <tt>&amp;gt;</tt>, "
	  "<tt>&amp;amp;</tt>, <tt>&amp;#34;</tt>, <tt>&amp;#39;</tt> "
	  "and <tt>&amp;#0;</tt>, in effect making them visible to "
	  "the reader, as they were probably meant to be." );

  defvar( "tm?", 1,
	  "Text2HTML magic:Substitute &lt;TM&gt;", TYPE_FLAG|VAR_MORE,
	  "When turned on, <tt>&lt;TM&gt;</tt> will be turned into "
	  "<sup><small>TM</small></sup>. (<tt>&lt;sup&gt;&lt;small&gt;"
	  "TM&lt;/small&gt;&lt;/sup&gt;</tt>)" );

  defvar( "(C|R)?", 1,
	  "Text2HTML magic:Substitute (C) and (R) for symbols",
	  TYPE_FLAG|VAR_MORE,
	  "When turned on, <tt>(C)</tt> will be turned into &copy;, "
	  "and <tt>(R)</tt> into &reg;." );

#if 0
  defvar( "enumerate?", 1,
	  "Text2HTML magic:Substitute enumerations",
	  TYPE_FLAG|VAR_MORE|VAR_EXPERT,
	  "When turned on, sections like this one:<p><pre>"
	  "1. I wanted the feature.\n"
	  "2. It felt right.\n"
	  "3. It added hack value.\n\n"
	  "Naturally, I had to implement it.</pre>"
	  "will be turned to this HTML code:<pre><br>"
	  "&lt;ol&gt;&lt;li&gt;I wanted the feature.\n"
	  "&lt;li&gt;It felt right.\n"
	  "&lt;li&gt;It added hack value.&lt;/ol&gt;\n\n"
	  "Naturally, I had to implement it.</pre>" );
#endif
}

string|void check_variable(string name, string value)
{
  switch(name)
  {
    case "mountpoint":
      if(sizeof(value))
      {
	if(value[-1] != '/')
	  call_out(set, 0, "mountpoint", value+"/");
        return 0;
      }
      else
	return "You must not leave this field empty.";
    case "searchpath":
      return 0; //! Should perhaps check that the directory is readable... Stdio.File()->open(value);
  }
}

class Date {
  int year;
  int month;
  int day;

  string filename()
  {
    return sprintf("%04d-%02d-%02d",year,month,day);
  }

  int weekday_number()
  {
    return localtime(mktime(([ "year":year-1900,
			       "mon":month-1,
			       "mday":day])))->wday+1;
  }

  int `==(object date)
  {
    return objectp(date) && day==date->day && month==date->month && year==date->year;
  }

  int validp()
  {
    return year  > 1899
	&& month > 0
	&& day   > 0;
  }

  object create(string s, int|void Y, int|void M, int|void D)
  { //! given a human-formatted date string, return a date
    switch(s)
    {
      case "":
	year = Y;
	month = M;
	day = D;
	break;
      case "INVALID":
        year = month = day = -1;
        break;
      case "NONEXISTENT":
	year = month = day = -2;
	break;
      default:
	foreach(" .html"/"", string c) s-=c;
        if(Regexp("^[0-9]+-[0-9]+-[0-9]+$")->match(s))     //! Y-M-D
          sscanf(s,"%d-%d-%d",year,month,day);
        else if(Regexp("^[0-9]+/[0-9]+'[0-9]+$")->match(s))//! D/M'Y
          sscanf(s,"%d/%d'%d",day,month,year);
        else
          year=month=day=-1;
    }
  } //! create()
} ////! class Date

string date2url(object date, object|void id, string|void linktext, string|void host)
{
  if(objectp( date ) && date->validp())
  {
    host = host ? host : "/"; //! *Harkel*
    if(host[-1]=='/')
      host = host[0..sizeof( host )-2];
    string file = date->filename();
    int seen = id && id->config[date];
    return sprintf("<a href=\"%s%s%s.html%s\"%s>%s</a>",
		   host, query( "mountpoint" ), file,
		   (id && id->query ? "?"+id->query : ""),
		   (seen ? " color=black" : ""),
		   (linktext || file));
  } else
    return linktext;
} //! date2url()

//! Global variable section:
int lastupdate = 0;         //! modify timestamp of the diary directory the last time we read it
object *dates = ({ });      //! date[number] (array of dates for which we have diary entries)
array modify_time = ({ });  //! modify[number] (array of inode modify times, corresponding to dates)
mapping number = ([ ]);     //! entry number[year][month][day] (number in the 'dates' array)
mapping update_time = ([ ]);//! modifytime[year][month][day] (when corresponding file was last read in)
mapping diary_text = ([ ]); //! diary entry text[year][month][day]
//! End of global variable section

void refresh_dir()
{ //! Update the global diary entry database if needed
  if(query( "searchpath" ) == "")
    return; //! First time around since installing the module
  int modifytime = file_stat(query( "searchpath" ))[3];
  if(lastupdate!=modifytime)
  {
    lastupdate = modifytime;
    string *entries;
    entries = sort(Array.filter(get_dir(query( "searchpath" )),
				lambda(string filename) //! match filenames "<1000-9999>-<01-12>-<01-31>"
    { return Regexp("^[1-9][0-9][0-9][0-9]-(0[1-9]|1[0-2])-(0[1-9]|[12][0-9]|30|31)$")->match(filename); }
				));
    dates = Array.map(entries, Date);
    int index = 0;
    foreach(dates, object d)
    {
      number = recursive_add(number,
			     ([ d->year :
			      ([ d->month :
			       ([ d->day : index++ ]) ]) ]),
			     2 );
    }
  }
} //! refresh_dir()

int refresh_file(object d)
{ //! will return false if the file does not exist
  int Y = d->year,
      M = d->month,
      D = d->day;
  string path = query( "searchpath" )+d->filename();
  int *stat = file_stat( path );
  if(!arrayp( stat ))
  { //! This diary file has been deleted! (we really ought to flush it totally; FIXME?)
    refresh_dir();
    return 0;
  } else
  {
    int modifytime = stat[3];
    if(((update_time[Y] && update_time[Y][M] && update_time[Y][M][D]) != modifytime )
      || !diary_text[Y]
      || !diary_text[Y][M]
      || !diary_text[Y][M][D])
    {
      update_time = recursive_add(update_time, ([ Y : ([ M : ([ D : modifytime ]) ]) ]), 2);
      object diaryfile = Stdio.File();
      if(!diaryfile->open(path,"r"))
	diary_text = recursive_add(diary_text, ([ Y : ([ M : ([ D : "Failed to open file." ]) ]) ]), 2);
      else
      {
	diary_text = recursive_add(diary_text, ([ Y : ([ M : ([ D : diaryfile->read() ]) ]) ]), 2);
	diaryfile->close();
      }
    }
    return 1;
  }
} //! refresh_file()

int date_exists(object date)
{ //! Does date exist in the dates array?
  return search(dates, date)!=-1;
}

int month_exists(int Y, int M)
{ //! Are there any entries this month?
  return number[Y] && number[Y][M];
}

string get_text(object date)
{ //! Give the text from the diary entry of the given date, or 0 if nonexistent
  refresh_file( date );
  int Y = date->year;
  int M = date->month;
  int D = date->day;
  return diary_text[Y] && diary_text[Y][M] && diary_text[Y][M][D];
}

//! Returns an array of strings showing the values of all queried
//! variables. Only works for string and integer variables for now.
array multi_query(string ... queries)
{
  return Array.map(Array.transpose(({ queries,
				      Array.map(queries, query)
				     })),
		   lambda(array temp)
  { return sprintf("%s:%s",temp[0], (string)temp[1]); } );
}

string get_info(object d)
{
  refresh_file( d );
  int Y = d->year, M = d->month, D = d->day;
  mapping dir = localtime( lastupdate ), file = 0;
  if(update_time[Y] && update_time[Y][M] && update_time[Y][M][D])
    file = localtime( update_time[Y][M][D] );
  array cvsinfo = cvs_version / " ", temp,
    modinfo = ({ cvsinfo[2],
		 replace(cvsinfo[3], "/", "-"),
		 @cvsinfo[4..5] }),
    config = multi_query( "searchpath", "templatefile",
			  "language", "future?", "paragraph?",
			  "separator?", "line break?", "urlify?",
			  "magic urlify?", "dateify?", "boldify?",
			  "emphasize?", "quote?", "tm?", "(C|R)?",
			  "enumerate?" ),
    global = ({ (string)sizeof( dates ),
		(string)lastupdate,
		sprintf("(%d-%02d-%02d %02d:%02d:%02d)",
			dir->year+1900, dir->mon+1, dir->mday,
			dir->hour, dir->min, dir->sec ) }),
    date = ({ d->filename(),
	      file ? (string)update_time[Y][M][D] : "0",
	      file ? sprintf("(%d-%02d-%02d %02d:%02d:%02d)",
			     file->year+1900, file->mon+1, file->mday,
			     file->hour, file->min, file->sec )
		   : "(unavailable)" });
  return Array.map(({ modinfo, config, global, date }),
		   lambda(array i){ return i * "\t"; }) * "\n" + "\n";
}

array(object) get_dates()
{ //! Returns a chronologically sorted array of all dates in the database
  refresh_dir();
  return dates;
}

object nth_date(int n)
{ //! will return false or the nth entry's date
  if(n>0) n--;
  refresh_dir();
  int max = sizeof(dates);
  if (n<-max) return Date("INVALID");
  if (n>=max) return Date("NONEXISTENT");
  return dates[n];
}

object prev(object|void date, int|void depth)
{
  if(!objectp( date ))
    return sizeof( dates ) ? dates[0] : 0;
  int Y = date->year,
      M = date->month,
      D = date->day;
  mapping level;
  array previous = ({ });
  switch( depth )
  {
    case DAY: //! Find the latest day (in this month, if possible)
      if(level = number[Y] && number[Y][M])
	previous = sort(Array.filter(indices(level), `<, D));
      if(sizeof( previous ))
      {
	D = previous[-1];
	return Date("", Y, M, D);
      } //! If we didn't find a day, keep looking backwards (no break;)
    case MONTH: //! Find the last day the preceeding month (if any)
      if(level = number[Y])
	previous = sort(Array.filter(indices(level), `<, M));
      if(sizeof(previous))
      {
	M = previous[-1];
	D = sort(indices(number[Y][M]))[-1];
	return Date("", Y, M, D);
      }
    case YEAR: //! Find the last day the preceeding year (if any)
      previous = sort(Array.filter(indices(number), `<, Y));
      if(sizeof( previous ))
      {
	Y = previous[-1];
	M = sort(indices(number[Y]))[-1];
	D = sort(indices(number[Y][M]))[-1];
	return Date("", Y, M, D);
      }
      return 0; //! There are no earlier dates available.
    default: //! Return the last date, or, if we were there, 0.
      return (sizeof(dates))
	     && date != dates[0]
	     && dates[0];
  }
}//! prev()

object next(object|void date, int|void depth)
{
  if(!objectp( date ))
    return sizeof( dates ) ? dates[-1] : 0;
  int Y = date->year,
      M = date->month,
      D = date->day;
  mapping level;
  array previous = ({ });
  switch( depth )
  {
    case DAY: //! Find the next day (in this month, if possible)
      if(level = number[Y] && number[Y][M])
	previous = sort(Array.filter(indices(level), `>, D));
      if(sizeof(previous))
      {
	D = previous[0];
	return Date("", Y, M, D);
      } //! If we didn't find a day, keep looking forward (no break;)
    case MONTH: //! Find the first day next month (if any)
      if(level = number[Y])
	previous = sort(Array.filter(indices(level), `>, M));
      if(sizeof(previous))
      {
	M = previous[0];
	D = sort(indices(number[Y][M]))[0];
	return Date("", Y, M, D);
      }
    case YEAR: //! Find the first day next year (if any)
      previous = sort(Array.filter(indices(number), `>, Y));
      if(sizeof(previous))
      {
	Y = previous[0];
	M = sort(indices(number[Y]))[0];
	D = sort(indices(number[Y][M]))[0];
	return Date("", Y, M, D);
      }
      return 0; //! There are no later dates available.
    default: //! Return the first date, or, if we were there, 0.
      return (sizeof( dates ))
	     && date != dates[-1]
	     && dates[-1];
  }
}//! next()

object first_date(int|void Y, int|void M)
{ return next(Date("", Y, M-1, 0), MONTH); }

object last_date(int|void Y, int|void M)
{ return prev(Date("", Y, M+1, 0), MONTH); }

//! Roxen API functions

array register_module()
{
  return ({
	   MODULE_LOCATION,
#ifdef ALPHA
	   " Diary",
#else
	   "Diary",
#endif
	   "<p>This module that turns a directory of pure diary textfiles "
	   "into a fullblown HTML diary, providing stacks of navigation "
	   "features, great page design freedom without having to mess up "
	   "the diary text with HTML goo."
	   "</p><p>"
	   "All you have to do is to make a template into which the diary "
	   "entries will be put, set up a roxen-readable directory and put "
	   "your textfiles in this directory, naming each entry YYYY-MM-DD; "
	   "thus telling the diary module what date should be tied to each "
	   "file."
	   "</p>\n<p>"
	   "In-depth documentation as well as information about the latest "
	   "version of the module can be found at the <a href=\"http://a205"
	   ".ryd.student.liu.se/(Diary)/docs/\">module documentation</a> "
	   "page at the developer's site. The module author would be only "
	   "too happy about hearing from you if you use the module. Have "
	   "fun!</p>\n"
	   "<p align=right>/<a href=\"mailto:johan@id3.org\">Johan "
	   "Sundstrm</a></p>",
	   0,
	   0, });
}

string status()
{
  return "<pre>"
    + "Latest update:  " + ctime(lastupdate)
    + "Number of days: " + sizeof(dates) + "\n"
//    + "Update times:   " + sprintf("<pre>%O</pre>", update_time)
    + "Diary entries:  " + Array.map(get_dates(), date2url, 0, 0, my_configuration()->query("MyWorldLocation")) *
    "\n                " + "\n"
    + "</pre>";
}

array find_dir(string dir_name, object request_id)
{
  report_notice(sprintf("Diary: find_dir(%s) was invoked.", dir_name));
  refresh_dir();
  return Array.map(dates, lambda( object d ) { return d->filename() + ".html"; });
}

object dateFromFilename( string filename, object|void id )
{
  object d;
  if(Regexp("^[-0-9][0-9]*$")->match(filename))
  { // "date" given as a numbered index -n ... n
    int n;
    sscanf(filename, "%d", n);
    refresh_dir();
    d = nth_date(n);
    if( id )
      id->misc->redirect = "yes";
  }
  else if(filename=="")
  {
    refresh_dir();
    d = next( MOST_RECENT );
    if( id )
      id->misc->redirect = "yes";
  }
  else
  {
    d = Date( filename );
    if( id )
      id->misc->redirect = 0;
  }

  return d;
}

array stat_file( string filename, object id )
{
  array stat;
  switch( filename )
  {
    case ".":
      stat = file_stat(query( "searchpath" ));
      stat = 0;
      break;
    default:
      stat = file_stat(query( "searchpath" )+dateFromFilename( filename )->filename());
  }
  report_notice(sprintf("Diary: stat_file(%s) returned %O.", filename, stat));
  return stat;
}

string fix_case(string text, mapping arg)
{ //! Generic function for all those tags who print out text nicely
  if(arg->capitalize)
    text = String.capitalize( text );
  if(arg->uppercase)
    text = upper_case( text );
  return text;
}

string tag_month(string name, mapping arg, object id, object date)
{ //! Return a link to a month or just the name of the month
  if(arg->help)
    return sprintf("<tt>&lt;%s [alt=text] [capitalize] "
		   "[uppercase]&gt;</tt> - "
		   "Creates a link to the given month, "
		   "should it contain any diary entry, "
		   "otherwise just prints the month name "
		   "in full, in the language selected in "
		   "the diary module configuration interface. "
		   "The name may be overridden using the alt "
		   "argument to the tag. The capitalize "
		   "and uppercase arguments, when given, "
		   "modify the case of the link text.", name);
  int m = search(({ "","jan","feb","mar",
		       "apr","may","jun",
		       "jul","aug","sep",
		       "oct","nov","dec" }), name),
   year = date->year,
  month = date->month,
   urlp = month_exists(year, m),
   this = m==month;
  string text = arg->alt || month_name(m, id->variables->language); //! Language aware and friendly. Thanks, Roxen!
  text = fix_case( text, arg );
  string url = (this ? "<b>"
	             : (urlp ? "<a href=\""
			       + query( "mountpoint" )
	                       + first_date(year, m)->filename()
	                       + ".html"
			       + (id->query ? "?" + id->query: "")
			       + "\">"
			     : "" ))
             + text
             + (this ? "</b>"
		     : (urlp ? "</a>"
	                     : "" ));
  return url;
}

string tag_relative(string name, mapping arg, object id, object d)
{ //! Return a link to a reference relative to something
  if(arg->help)
    return sprintf("<tt>&lt;%s alt=text [capitalize]"
		   "[uppercase] &gt;</tt> - "
		   "Creates a link to %s, if %s, "
		   "otherwise just prints the alt argument."
		   "If you want the link to be an image, "
		   "better make the alt argument a valid "
		   "<tt>&lt;img&gt;</tt> tag. The capitalize "
		   "and uppercase tags, when given, modify "
		   "the case of the link text.", name,
		   ([ "first":"the first diary entry",
		    "last":"the last diary entry",
		    "lastyear":"the final diary entry the previous year",
		    "nextyear":"the first diary entry the next year",
		    "lastmonth":"the final diary entry the previous month",
		    "prev":"the previous diary entry",
		    "next":"the next diary entry",
		    "this":"the current diary entry"
		    ])[name],
		   ([ "first":"one exists",
		    "last":"one exists",
		    "lastyear":"there were any present",
		    "nextyear":"there were any present",
		    "lastmonth":"there were any present",
		    "nextmonth":"there were any present",
		    "prev":"any exist",
		    "next":"any exist",
		    "this":"it exists (it should! :-)"
		    ])[name]);
  object date;
  string text;
  switch( name )
  {
    case "first":
      date = prev(d);
      text = arg->alt || "first";
      break;
    case "last":
      date = next(d);
      text = arg->alt || "last";
      break;
    case "lastyear":
      date = prev(d, YEAR);
      text = arg->alt || date ? (string)(date->year)
	                      : (string)(d->year-1);
      break;
    case "nextyear":
      date = next(d, YEAR);
      text = arg->alt || date ? (string)(date->year)
			      : (string)(d->year+1);
      break;
    case "lastmonth":
      date = prev(d, MONTH);
      text = arg->alt || "&lt;=";
      break;
    case "nextmonth":
      date = next(d, MONTH);
      text = arg->alt || "=&gt;";
      break;
    case "prev":
      date = prev(d, DAY);
      text = arg->alt || "previous";
      break;
    case "next":
      date = next(d, DAY);
      text = arg->alt || "next";
      break;
    case "this":
      date = d;
      text = arg->alt || "this";
      break;
    default:
      return sprintf("The %s tag was invoked. How did you do that?", name);
  }
  text = fix_case( text, arg );
  return date2url( date, id, text );
}

string tag_info(string name, mapping arg, object id, object d)
{
  if(arg->help)
    return sprintf("<tt>&lt;%s alt=text [capitalize]"
		   "[uppercase]&gt;</tt> - "
		   "Inserts a string formatted according "
		   "to the case options given, naming the "
		   "%s in the language configured in the "
		   "diary module configuration interface.",
		   name, name);
  string text;
  switch( name )
  {
    case "weekday":
      text = week_day(d->weekday_number(), id->variables->language);
      break;
    case "monthname":
      text = month_name(d->month, id->variables->language);
      break;
  }
  return fix_case( text, arg );
}

string tag_number(string name, mapping arg, object id, object d)
{
  if(arg->help)
    return sprintf("<tt>&lt;%s&gt; - Inserts the %s "
		   "of the current diary entry, as a "
                   "rather plain number.",name,
      name != "ordered" ? name
                        : "ordered number of the day "
    "this month, (given in the language as configured "
    "in the diary module configuration interface)");

  switch( name )
  {
    case "year"   : return (string)d->year;
    case "month"  : return sprintf("%02d",d->month);
    case "day"    : return sprintf("%02d",d->day);
    case "ordered": return ordered_number( d->day, id->variables->language );
  }
}

string do_magic(string text, object id)
{
  array unchanged = ({ text }),
        changed = ({ }), temp,
        from = ({ }),
        to = ({ });
  int act = 0;
  if(query( "excerpts?" ))
  {
    temp = multi_replace_regexp(unchanged, changed,
				({ "\n---+8<[8<-]*---\n(.+)\n---+8<[8<-]*---\n" }),
				({ "\n<pre>\\1</pre>\n" }));
    unchanged = temp[0];
      changed = temp[1];
  }
  if(query( "separator?" ))
  { from += ({ "\n\n\n" });
    to += ({ "\n<p align=center><img alt=\"---\" align=center width="
	     "\"147\" height=\"17\" src=\"images/snirkel.gif\"></p>\n<p>" });
    act = 1;
  }
  if(query( "paragraph?" ))
  { from += ({  "\n\n" });
    to += ({ "\n<p>\n" });
    act = 1;
  }

  if( act )
  {
    temp = multi_replace_nocase( unchanged, changed, from, to );
    unchanged = temp[0]; from = ({ });
      changed = temp[1];   to = ({ });
    act = 0;
  }

  if(query( "magic urlify?" ))
  { //! Turn "named" URL-lookalikes into URL:s -- supported variants:
    //! "description of URL" ("this://is the url/which may contain all but double quotes/")
    //! "description of URL" (this://is the url/which may contain all but right parenthesises/)
    //! descriptive-word ("same://URL as first case/")
    //! descriptive-word (same://URL as second case/)
    temp = multi_replace_regexp(unchanged, changed,
				({ "(^|[ \n])\"([^\"]+)\".\\(\"((http:/|ftp:/|gopher:/|telnet:/|)/[^\"]+)\"\\)",
				   "(^|[ \n])\"([^\"]+)\".\\(((http:/|ftp:/|gopher:/|telnet:/|)/[^)]+)\\)",
				   "(^|[ \n])([^ \n]+).\\(\"((http:/|ftp:/|gopher:/|telnet:/|)/[^\"]+)\"\\)",
				   "(^|[ \n])([^ \n]+).\\(((http:/|ftp:/|gopher:/|telnet:/)/[^)]+)\\)" }),
				({ "\\1<a href=\"\\3\">\\2</a>",
				   "\\1<a href=\"\\3\">\\2</a>",
				   "\\1<a href=\"\\3\">\\2</a>",
				   "\\1<a href=\"\\3\">\\2</a>" }));
    unchanged = temp[0];
      changed = temp[1];
  }
  if(query( "urlify?" ))
  { //! Turn URL-lookalikes into URL:s -- supported variants:
    //! "protocol://this method is/preferred/, since it may use anything but double quotes/"
    //! [protocol://this method can use all but right brackets in the URL/]
    //! (protocol://this method can't use right parenthesises)
    //! protocol://whis/method/can't/even/use/spaces/
    temp = multi_replace_regexp(unchanged, changed,
				({ "\"((http|ftp|gopher|telnet)://[^\"]+)\"",
				   "(\\[)((http|ftp|gopher|telnet)://[^]]+)(\\])",
				   "(\\()((http|ftp|gopher|telnet)://[^)]+)(\\))",
				   "([^\"])((http|ftp|gopher|telnet)://[^ \n]+)" }),
				({ "<a href=\"\\1\">\\1</a>",
				   "\\1<a href=\"\\2\">\\2</a>\\4",
				   "\\1<a href=\"\\2\">\\2</a>\\4",
				   "\\1<a href=\"\\2\">\\2</a>" }));
    unchanged = temp[0];
      changed = temp[1];
  }
  if(query( "dateify?" ) && sizeof( dates ))
  { //! Fix possible date references into real URL:s
    temp = multi_replace_regexp(unchanged, changed,
				({ "(" + Array.map(dates,
						   lambda( object d )
						   {return d->filename();}
						  ) * "|" + ")" }),
				({ "<a href=\"\\1.html"
				 + (id->query ? "?" + id->query : "")
				 + "\">\\1</a>" }));
    unchanged = temp[0];
      changed = temp[1];
  }
  if(query( "tm?" ))
  { from += ({  "<tm>" });
    to += ({ "<sup><small>TM</small></sup>" });
  }
  if(query( "(C|R)?" ))
  { from += ({  "(C)", "(R)" });
    to += ({ "&copy;", "&reg;" });
  }
  if(query( "quote?" ))
  { from += ({ "&",      "<", ">",     "\"",    "\'",   "\000" });
    to += ({ "&amp;", "&lt;", "&gt;", "&#34;", "&#39;", "&#0;" });
  }
  if(query( "line break?" ))
  { from += ({  "\r" });
    to += ({ "<br>" });
  }

  temp = multi_replace_nocase( unchanged, changed, from, to );
  unchanged = temp[0];
    changed = temp[1];

  if(query( "boldify?" ))
  {
    temp = multi_replace_regexp(unchanged, changed,
				({ "(^|[ (\n])_([^ _)][^_)]+)_([ .,-;!?)\n]|$)" }),
				({ "\\1<b>\\2</b>\\3" }));
    unchanged = temp[0];
      changed = temp[1];
  }
  if(query( "emphasize?" ))
  {
    temp = multi_replace_regexp(unchanged, changed,
				({ "(^|[ (\n])/([^ /)][^/)]+)/([ .,-;!?)\n]|$)" }),
				({ "\\1<em>\\2</em>\\3" }));
    unchanged = temp[0];
      changed = temp[1];
  }

#ifdef 0
  if(query( "enumerate?" ))
  {
  }
#endif

  return interleave(@temp);
}

string tag_diary(string name, mapping arg, object id, object d)
{ //! FIXME: This tag wants some more coolness, like overriding
  //!        options given in the config interface, for example
  if(arg->help)
    return sprintf("<tt>&lt;%s&gt;</tt> - "
		   "Inserts the diary entry corresponding to the date "
		   "stipulated by the URL. Depending on the settings "
		   "given in the configuration interface below "
		   , name);

  return do_magic(get_text( d ), id);
}

string tag_month_map(string name, mapping arg, object id, object date)
{ //! FIXME: could do with more cool options
  if(arg->help)
    return sprintf("<tt>&lt;%s [wholemonth] [capitalize] "
		   "[uppercase]&gt;</tt> - Inserts a month navigation "
		   "map of the current month. The weekday headers are "
		   "formatted according to the case options given, and "
		   "are in the language configured in the diary module "
		   "configuration interface. When the wholemonth "
		   "argument is given, the whole month will always be "
		   "shown, even when viewing the present month and some "
		   "dates are still in the future.", name);
  int wholemonth = !zero_type( arg->wholemonth )
		   || query( "future?" );
  int Y = date->year, M = date->month, D = date->day;
  refresh_dir();

  string nr, ret = "<pre>" +
    Array.map(({ 2, 3, 4, 5, 6, 7, 1 }),
	      lambda(int day, mapping arg, string language)
              {
		return fix_case(week_day(day, language)[0..1],arg);
	      }, arg, id->variables->language)*" ";
  //               M Ti On To Fr L S
  //                1  2  3  4  5  6  0
  // (x+6)%7 -> 0..6 (y+1)%8 -> 1..6, 0
  int m = M-1;
  mapping timemap = ([ "year":Y-1900, "mon":m ]);
  int w = (localtime(mktime(timemap+([ "mday":1 ])))["wday"]+6)%7;
  //ret += "   " * w; //! Pike 0.6-ism for next line. :-(
  ret += sprintf("\n%"+w*3+"s","");
  int d = 1, t = time(),
     vt = mktime(timemap + ([ "mday":d ])),
   left = (wholemonth || vt<t) && m == localtime( vt )->mon;
  do
  {
    for(;w<7;w++)
    {
      nr = sprintf("%2s", left ? (string)d : "");
      if(d == D)
	nr = sprintf("<b>%s</b>", nr);
      else if(!zero_type( number[Y][M][d] )) //! URLize?
	nr = date2url(Date("", Y, M, d), id, nr);
      if( w<6 ) //! Mid-week?
	nr += " ";
      else //! Last day of week.
	nr += "\n";
      ret += nr;

      vt = mktime(timemap + ([ "mday":++d ]));
      //! Any date left to write out this month?
      left = (wholemonth || vt<t) && m == localtime( vt )->mon;
    }
    w = 0;
  } while ( left );
  return ret + "</pre>\n";
}

string tag_all(string name, mapping arg, object id, object d)
{ //! FIXME: add sorting in other ways et cetera
  if(arg->help)
    return sprintf("<tt>&lt;%s [module=number]&gt;</tt> - "
		   "Inserts links to all existing date entries, in order. "
		   "a modulo argument may be given to set the number of columns "
		   "used in the list.", name);
  string mp = query( "mountpoint" ), result = "<nobr>";
  array dates;
  dates = Array.map(get_dates(),
		    lambda(object d, string mp, object id)
                    { string date = d->filename();
		     return "<a href=\"" + mp + date
		          + ".html" + (id->query ? "?" + id->query : "")
		          + "\">" + date + "</a>";
		    }, mp, id);
  for(int i=0 ; i<sizeof(dates) ; i++)
    result += dates[i] +
              ((i % (int)arg->modulo) == (int)arg->modulo-1 ?
	       "</nobr><br>\n<nobr>" : " ");
  return result + "</nobr>";
}

string tag_last_modified(string name, mapping arg, object id, object d)
{
  if(!objectp( d ) || !d->validp())
    return "[Invalid date]";
  if( arg->help )
    return sprintf("<tt>&lt;%s [mostrecent] [href=linktext] "
		   "[date options]&gt;</tt> "
		   "- tells the last modification time of the currently "
		   "shown day, or of the most recently changed diary "
		   "entry, if the mostrecent argument was given. When the "
		   "href argument is given, a link to that day will be "
		   "shown, otherwise, the output will be that date instead. "
		   "All other options are forwarded to the date tag, e g \"brief\", "
		   "\"lang\" (set by default to the configured language "
		   "when not given explicitly).", name);
  int href = !zero_type( arg->href ), t = id->misc->diary_time, Y, M, D;
  mapping time = localtime( t );
  Y = time->year  + 1900;
  M = time->mon + 1;
  D = time->mday;
  if(!zero_type( arg->mostrecent ))
  {
    refresh_dir();
    foreach(indices( update_time ), int y)
      foreach(indices( update_time[y] ), int m)
        foreach(indices( update_time[y][m] ), int d)
          if( update_time[y][m][d] >= t )
	  {
	    t = update_time[y][m][d];
	    Y = y ; M = m ; D = d;
	  }
    // d = Date("", Y, M, D); -- really no use, until we use d for something
    m_delete(arg, "mostrecent");
  }

  if( href )
  {
    string date = sprintf("%d-%02d-%02d", Y, M, D);
    return make_container("a",
			  ([ "href" : date + ".html"
			            + (id->query ? "?" + id->query : "") ]),
			  arg->href=="href"
			  ? (Y||M||D
			     ? date
			     : "?")
			  : arg->href );
  }
  else
  {
    arg->unix_time = (string)t;
    m_delete(arg, "href");
    if(zero_type( arg->lang ))
      arg->lang = id->variables->language || query( "language" );
    if(zero_type( arg->type ))
      arg->type = "string";
    return parse_rxml(make_tag("date", arg), id);
  }
}

object|mapping find_file(string file_name, object id)
{
  string template, //! The RXML framework for our diary page
         text,    // ! The raw text diary page
         result; //  ! The finished page

  if(!(< "GET", "HEAD" >)[id->method])
    return 0; //! Other methods not supported

  object d = dateFromFilename( file_name, id );
  if(id->prestate->expert)
    result = get_info( d );
  else if(!d->validp()
       || !date_exists(d))
    return 0;                  //! Fall through to other modules
  else if(id->prestate->raw)
    result = get_text( d );
  else if(!(template = Stdio.read_bytes(query( "templatefile" ))))
    result = "The diary module's template file was improperly set up.";
  else
  {
    refresh_file( d );
    if( id->misc->redirect ) //! Transform /mp/ to /mp/date
      return http_redirect(query( "mountpoint" )
			   + d->filename() + ".html"
			   + (id->query ? "?" + id->query : ""),
			   id);
    int Y = d->year,
        M = d->month,
        D = d->day;
    if(update_time[Y] &&
       update_time[Y][M] &&
       update_time[Y][M][D])
      id->misc->diary_time = update_time[Y][M][D];

    result = parse_html(template,
	  ([
	     "jan":tag_month, "feb":tag_month, "mar":tag_month,
	     "apr":tag_month, "may":tag_month, "jun":tag_month,
	     "jul":tag_month, "aug":tag_month, "sep":tag_month,
	     "oct":tag_month, "nov":tag_month, "dec":tag_month,
	     "first"    :tag_relative,        "last":tag_relative,
	     "lastyear" :tag_relative,    "nextyear":tag_relative, 
	     "lastmonth":tag_relative,   "nextmonth":tag_relative,
	     "prev"     :tag_relative,        "next":tag_relative,
	     "this"     :tag_relative,        "year":tag_number,
	     "month"    :tag_number,           "day":tag_number,
	     "weekday"  :tag_info,       "monthname":tag_info,
	     "ordered"  :tag_number,  "lastmodified":tag_last_modified,
	     "server":lambda(string name, mapping arg, object id)
		      {
			if(id && id->misc && id->misc->host)
			  return "http://" + id->misc->host;
			else
			  return id->conf->query( "MyWorldLocation" );
		      },
	     "diary"   :tag_diary,     "alldays":tag_all,
	     "monthmap":tag_month_map,
	   ]), ([ ]), id, d );

    result = parse_rxml(result, id, 0,
			([ " _extra_heads" :
			 ([ "Last-Modified" :
			  http_date(id->misc->diary_time) ]) ]));
  }

  return http_string_answer(result);
}

void start()		{ refresh_dir(); }
string query_location()	{ return query( "mountpoint" ); }
