function [defaultValues] = getDefaultValues(varargin)
% [defaultValues] = getDefaultValues(typecode, level, version, pkgversion(optional))
%
% Takes
%
% 1. typecode; a string representing the type of object being queried
% 2. level, an integer representing an SBML level
% 3. version, an integer representing an SBML version
% 4. pkgversion, an integer representing the SBML package version 
%       (defaults to 1)
%
% Returns
%
% 1. an array of default values for an SBML structure of the given typecode, level and version
%
% *NOTE:* The corresponding fields present in an SBML  structure can be found using
%   the function `getStructureFieldnames`

%<!---------------------------------------------------------------------------
% This file is part of libSBML.  Please visit http://sbml.org for more
% information about SBML, and the latest version of libSBML.
%
% Copyright (C) 2013-2017 jointly by the following organizations:
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%     3. University of Heidelberg, Heidelberg, Germany
%
% Copyright (C) 2009-2013 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%  
% Copyright (C) 2006-2008 by the California Institute of Technology,
%     Pasadena, CA, USA 
%  
% Copyright (C) 2002-2005 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. Japan Science and Technology Agency, Japan
% 
% This library is free software; you can redistribute it and/or modify it
% under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation.  A copy of the license agreement is provided
% in the file named "LICENSE.txt" included with this software distribution
% and also available online as http://sbml.org/software/libsbml/license.html
%----------------------------------------------------------------------- -->

typecode = varargin{1};
level = varargin{2};
version = varargin{3};

pkgCount = 0;

if (nargin > 3)
    if (nargin < 5)
        error('not enough arguments');
    end;
    pkgCount = length(varargin{4});
 %   packages = cell(1, pkgCount);
  %  pkgVersion = zeros(1, pkgCount);
    packages = varargin{4};
    pkgVersion = varargin{5};
end;


    defaultValues = [];
done = 1;


switch (typecode)
  case {'SBML_ALGEBRAIC_RULE', 'AlgebraicRule', 'algebraicRule', 'rule'}
    fhandle = str2func('getAlgebraicRuleDefaultValues');
  case {'SBML_ASSIGNMENT_RULE', 'AssignmentRule', 'assignmentRule'}
    fhandle = str2func('getAssignmentRuleDefaultValues');
  case {'SBML_COMPARTMENT', 'Compartment', 'compartment'}
    fhandle = str2func('getCompartmentDefaultValues');
  case {'SBML_COMPARTMENT_TYPE', 'CompartmentType', 'compartmentType'}
    fhandle = str2func('getCompartmentTypeDefaultValues');
  case {'SBML_COMPARTMENT_VOLUME_RULE', 'CompartmentVolumeRule', 'compartmentVolumeRule'}
    fhandle = str2func('getCompartmentVolumeRuleDefaultValues');
  case {'SBML_CONSTRAINT', 'Constraint', 'constraint'}
    fhandle = str2func('getConstraintDefaultValues');
  case {'SBML_DELAY', 'Delay', 'delay'}
    fhandle = str2func('getDelayDefaultValues');
  case {'SBML_EVENT', 'Event', 'event'}
    fhandle = str2func('getEventDefaultValues');
  case {'SBML_EVENT_ASSIGNMENT', 'EventAssignment', 'eventAssignment'}
    fhandle = str2func('getEventAssignmentDefaultValues');
  case {'SBML_FUNCTION_DEFINITION', 'FunctionDefinition', 'functionDefinition'}
    fhandle = str2func('getFunctionDefinitionDefaultValues');
  case {'SBML_INITIAL_ASSIGNMENT', 'InitialAssignment', 'initialAssignment'}
    fhandle = str2func('getInitialAssignmentDefaultValues');
  case {'SBML_KINETIC_LAW', 'KineticLaw', 'kineticLaw'}
    fhandle = str2func('getKineticLawDefaultValues');
  case {'SBML_LOCAL_PARAMETER', 'LocalParameter', 'localParameter'}
    fhandle = str2func('getLocalParameterDefaultValues');
  case {'SBML_MODEL', 'Model', 'model'}
    fhandle = str2func('getModelDefaultValues');
  case {'SBML_MODIFIER_SPECIES_REFERENCE', 'ModifierSpeciesReference', 'modifierSpeciesReference', 'modifier'}
    fhandle = str2func('getModifierSpeciesReferenceDefaultValues');
  case {'SBML_PARAMETER', 'Parameter', 'parameter'}
    fhandle = str2func('getParameterDefaultValues');
  case {'SBML_PARAMETER_RULE', 'ParameterRule', 'parameterRule'}
    fhandle = str2func('getParameterRuleDefaultValues');
  case {'SBML_PRIORITY', 'Priority', 'priority'}
    fhandle = str2func('getPriorityDefaultValues');
  case {'SBML_RATE_RULE', 'RateRule', 'ruleRule'}
    fhandle = str2func('getRateRuleDefaultValues');
  case {'SBML_REACTION', 'Reaction', 'reaction'}
    fhandle = str2func('getReactionDefaultValues');
  case {'SBML_SPECIES', 'Species', 'species'}
    fhandle = str2func('getSpeciesDefaultValues');
  case {'SBML_SPECIES_CONCENTRATION_RULE', 'SpeciesConcentrationRule', 'speciesConcentrationRule'}
    fhandle = str2func('getSpeciesConcentrationRuleDefaultValues');
  case {'SBML_SPECIES_REFERENCE', 'SpeciesReference', 'speciesReference', 'reactant', 'Reactant', 'product', 'Product'}
    fhandle = str2func('getSpeciesReferenceDefaultValues');
  case {'SBML_SPECIES_TYPE', 'SpeciesType', 'speciesType'}
    fhandle = str2func('getSpeciesTypeDefaultValues');
  case {'SBML_STOICHIOMETRY_MATH', 'StoichiometryMath', 'stoichiometryMath'}
    fhandle = str2func('getStoichiometryMathDefaultValues');
  case {'SBML_TRIGGER', 'Trigger', 'trigger'}
    fhandle = str2func('getTriggerDefaultValues');
  case {'SBML_UNIT', 'Unit', 'unit'}
    fhandle = str2func('getUnitDefaultValues');
  case {'SBML_UNIT_DEFINITION', 'UnitDefinition', 'unitDefinition'}
    fhandle = str2func('getUnitDefinitionDefaultValues');
  otherwise
    done = 0; 
end;

if done == 1
  [defaultValues] = feval(fhandle, level, version);
else
    i = 1;
    found = 0;
    while (i < pkgCount+1)
        [found, fhandle] = getFields(typecode, packages{i}, 0);
        if (found == 1)
            break;
        end;
        i = i + 1;
    end;
    if (found == 1)
        [defaultValues] = feval(fhandle, level, version, pkgVersion(i));
        done = 1;
    end;
end;
 
if (done == 1 && pkgCount > 0)
    % we may need to extend the fields 
    i = 1;
    while (i < pkgCount+1)
        [found, fhandle] = getFields(typecode, packages{i}, 1);
        if (found == 1)
            copySBML = defaultValues;
            [newdefaultValues] = feval(fhandle, level, version, pkgVersion(i));
            defaultValues = [copySBML, newdefaultValues];
        end;
        i = i + 1;
    end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [found, fhandle] = getFields(typecode, pkg, extension)
    fhandle = str2func('disp');
    found = 0;
    if (strcmp(pkg, 'fbc'))
        if (extension)
            if (isFBCExtension(typecode))
        [found, fhandle] = getFBCFieldnameFunction(typecode);
            end;
        else
        [found, fhandle] = getFBCFieldnameFunction(typecode);
        end;
    end;

function extend = isFBCExtension(typecode)
   extend = 0;
   switch (typecode)
        case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
             extend = 1;
       case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
            extend = 1;
        case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
            extend = 1;
   end;

function [found, fhandle] = getFBCFieldnameFunction(typecode)
    found = 1;
  switch (typecode)
        case {'SBML_FBC_FLUXBOUND', 'FluxBound', 'fluxBound', 'fbc_fluxBound'}
      fhandle = str2func('getFluxBoundDefaultValues');
        case {'SBML_FBC_FLUXOBJECTIVE', 'FluxObjective', 'fluxObjective', 'fbc_fluxObjective'}
      fhandle = str2func('getFluxObjectiveDefaultValues');
        case {'SBML_FBC_OBJECTIVE', 'Objective', 'objective', 'fbc_objective'}
      fhandle = str2func('getObjectiveDefaultValues');
        case {'SBML_FBC_GENE_PRODUCT', 'GeneProduct', 'geneProduct', 'fbc_geneProduct'}
      fhandle = str2func('getGeneProductDefaultValues');
        case {'SBML_FBC_GENE_PRODUCT_ASSOCIATION', 'GeneProductAssociation', 'geneProductAssociation', 'fbc_geneProductAssociation'}
      fhandle = str2func('getGeneProductAssociationDefaultValues');
        case {'SBML_FBC_ASSOCIATION', 'Association', 'association', 'fbc_association'}
      fhandle = str2func('getAssociationDefaultValues');
    case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
        fhandle = str2func('getFBCModelDefaultValues');    
    case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
      fhandle = str2func('getFBCSpeciesDefaultValues');
    case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
      fhandle = str2func('getFBCReactionDefaultValues');
    otherwise
      fhandle = str2func('disp');
      found = 0;
    end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getAlgebraicRuleDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_ALGEBRAIC_RULE', ...
		               '', ...
		               '', ...
		               'scalar', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_ALGEBRAIC_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_ALGEBRAIC_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_ALGEBRAIC_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_ALGEBRAIC_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_ALGEBRAIC_RULE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_ALGEBRAIC_RULE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getAssignmentRuleDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_ASSIGNMENT_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_ASSIGNMENT_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_ASSIGNMENT_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_ASSIGNMENT_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_ASSIGNMENT_RULE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_ASSIGNMENT_RULE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getCompartmentDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_COMPARTMENT', ...
		               '', ...
		               '', ...
		               '', ...
		               1.0, ...
		               '', ...
		               '', ...
		               int32(0), ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_COMPARTMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(3), ...
			               0/0, ...
			               '', ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_COMPARTMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(3), ...
			               0/0, ...
			               '', ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_COMPARTMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               int32(3), ...
			               0/0, ...
			               '', ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_COMPARTMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               int32(3), ...
			               0/0, ...
			               '', ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_COMPARTMENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_COMPARTMENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getCompartmentTypeDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = {
			               'SBML_COMPARTMENT_TYPE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_COMPARTMENT_TYPE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_COMPARTMENT_TYPE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = [];
        elseif (version == 2)
		defaultValues = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getCompartmentVolumeRuleDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_COMPARTMENT_VOLUME_RULE', ...
		               '', ...
		               '', ...
		               'scalar', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = [];
    elseif (version == 4 || version == 5)
		defaultValues = [];
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = [];
        elseif (version == 2)
		defaultValues = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getConstraintDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = {
			               'SBML_CONSTRAINT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_CONSTRAINT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_CONSTRAINT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_CONSTRAINT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_CONSTRAINT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getDelayDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = {
			               'SBML_DELAY', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_DELAY', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_DELAY', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_DELAY', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getEventAssignmentDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_EVENT_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getEventDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_EVENT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_EVENT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               [], ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_EVENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_EVENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               int32(0), ...
			               [], ...
			               [], ...
			               [], ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_EVENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               int32(0), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_EVENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               int32(0), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getFunctionDefinitionDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_FUNCTION_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_FUNCTION_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_FUNCTION_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_FUNCTION_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_FUNCTION_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_FUNCTION_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getInitialAssignmentDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_INITIAL_ASSIGNMENT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getKineticLawDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_KINETIC_LAW', ...
		               '', ...
		               '', ...
		               '', ...
		               [], ...
		               '', ...
		               '', ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_KINETIC_LAW', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_KINETIC_LAW', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
			               int32(-1), ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_KINETIC_LAW', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_KINETIC_LAW', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_KINETIC_LAW', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               [], ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_KINETIC_LAW', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               [], ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getLocalParameterDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = [];
    elseif (version == 4 || version == 5)
		defaultValues = [];
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_LOCAL_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_LOCAL_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getModelDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_MODEL', ...
		               '', ...
		               '', ...
		               level, ...
		               version, ...
		               '', ...
		               [], ...
		               [], ...
		               [], ...
		               [], ...
		               [], ...
		               [], ...
		               [], ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_MODEL', ...
			               '', ...
			               '', ...
			               '', ...
		                 level, ...
		                 version, ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               '', ...
			               '', ...
			               [], ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_MODEL', ...
			               '', ...
			               '', ...
			               '', ...
		                 level, ...
		                 version, ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               '', ...
			               '', ...
			               [], ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_MODEL', ...
			               '', ...
			               '', ...
			               '', ...
		                 level, ...
		                 version, ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               '', ...
			               '', ...
			               [], ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_MODEL', ...
			               '', ...
			               '', ...
			               '', ...
		                 level, ...
		                 version, ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               '', ...
			               '', ...
			               [], ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_MODEL', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
		                 level, ...
		                 version, ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_MODEL', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
		                   level, ...
		                   version, ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               '', ...
			               '', ...
			               '', ...
                           '', ...
			               [], ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getModifierSpeciesReferenceDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_MODIFIER_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getParameterDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_PARAMETER', ...
		               '', ...
		               '', ...
		               '', ...
		               0/0, ...
		               '', ...
		               int32(0), ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(-1), ...
			               int32(0), ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_PARAMETER', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getParameterRuleDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_PARAMETER_RULE', ...
		               '', ...
		               '', ...
		               'scalar', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = [];
    elseif (version == 4 || version == 5)
		defaultValues = [];
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = [];
        elseif (version == 2)
		defaultValues = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getPriorityDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = [];
    elseif (version == 4 || version == 5)
		defaultValues = [];
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_PRIORITY', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_PRIORITY', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getRateRuleDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_RATE_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_RATE_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_RATE_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_RATE_RULE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_RATE_RULE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_RATE_RULE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getReactionDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_REACTION', ...
		               '', ...
		               '', ...
		               '', ...
		               [], ...
		               [], ...
		               [], ...
		               int32(0), ...
		               int32(0), ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_REACTION', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_REACTION', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               int32(0), ...
			               int32(0), ...
			               int32(-1), ...
			               int32(0), ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_REACTION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_REACTION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_REACTION', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_REACTION', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
			               [], ...
			               [], ...
			               [], ...
			               int32(0), ...
			               '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getSpeciesConcentrationRuleDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_SPECIES_CONCENTRATION_RULE', ...
		               '', ...
		               '', ...
		               'scalar', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = [];
    elseif (version == 4 || version == 5)
		defaultValues = [];
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = [];
        elseif (version == 2)
		defaultValues = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getSpeciesDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_SPECIES', ...
		               '', ...
		               '', ...
		               '', ...
		               '', ...
		               0/0, ...
		               '', ...
		               int32(0), ...
		               0, ...
		               int32(0), ...
		               int32(0), ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_SPECIES', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               0, ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_SPECIES', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               0, ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_SPECIES', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               0, ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_SPECIES', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               0, ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_SPECIES', ...
			               '', ...
			               '', ...
			               '', ...                         
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_SPECIES', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               '', ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               int32(0), ...
			               '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getSpeciesReferenceDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_SPECIES_REFERENCE', ...
		               '', ...
		               '', ...
		               '', ...
		               int32(1), ...
		               int32(1), ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               1, ...
			               int32(1), ...
			               '', ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               1, ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               1, ...
			               [], ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               1, ...
			               [], ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               int32(0), ...
			               int32(0), ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_SPECIES_REFERENCE', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               '', ...
			               0/0, ...
			               int32(0), ...
			               int32(0), ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getSpeciesTypeDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = {
			               'SBML_SPECIES_TYPE', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_SPECIES_TYPE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_SPECIES_TYPE', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = [];
        elseif (version == 2)
		defaultValues = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getStoichiometryMathDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = {
			               'SBML_STOICHIOMETRY_MATH', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_STOICHIOMETRY_MATH', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = [];
        elseif (version == 2)
		defaultValues = [];
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getTriggerDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = [];
elseif (level == 2)
	if (version == 1)
		defaultValues = [];
	elseif (version == 2)
		defaultValues = [];
	elseif (version == 3)
		defaultValues = {
			               'SBML_TRIGGER', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_TRIGGER', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_TRIGGER', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               int32(0), ...
			               int32(0), ...
			               '', ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_TRIGGER', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               int32(0), ...
			               int32(0), ...
			               '', ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getUnitDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_UNIT', ...
		               '', ...
		               '', ...
		               '', ...
			       int32(-1), ...
		               int32(0)', ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_UNIT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(1), ...
			               int32(0), ...
			               1.0, ...
			               0.0, ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_UNIT', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(1), ...
			               int32(0), ...
			               1.0, ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_UNIT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               int32(1), ...
			               int32(0), ...
			               1.0, ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_UNIT', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               int32(1), ...
			               int32(0), ...
			               1.0, ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_UNIT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               0/0, ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_UNIT', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               0/0, ...
			               0/0, ...
			               0/0, ...
                                       '', ...
                                       '', ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [defaultValues] = getUnitDefinitionDefaultValues(level, version)












if ~isValidLevelVersionCombination(level, version)
	error('invalid level/version combination');
end;

if (level == 1)
	defaultValues = {
		               'SBML_UNIT_DEFINITION', ...
		               '', ...
		               '', ...
		               '', ...
		               [], ...
	                };
elseif (level == 2)
	if (version == 1)
		defaultValues = {
			               'SBML_UNIT_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
		                };
	elseif (version == 2)
		defaultValues = {
			               'SBML_UNIT_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               '', ...
			               [], ...
		                };
	elseif (version == 3)
		defaultValues = {
			               'SBML_UNIT_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
		                };
    elseif (version == 4 || version == 5)
		defaultValues = {
			               'SBML_UNIT_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
		                };
	end;
elseif (level == 3)
	if (version == 1)
		defaultValues = {
			               'SBML_UNIT_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
		                };
        elseif (version == 2)
		defaultValues = {
			               'SBML_UNIT_DEFINITION', ...
			               '', ...
			               '', ...
			               '', ...
                           [], ...
			               int32(-1), ...
			               '', ...
			               '', ...
			               [], ...
		                };
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCModelDefaultValues(level, ...
                                                         version, pkgVersion)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 1)
          SBMLfieldnames = { int32(1), ...
                             [], ...
                             [], ...
                             '', ...
                           };
          nNumberFields = 4;
    elseif(pkgVersion ==2)
          SBMLfieldnames = { int32(2), ...
                             [], ...
                             [], ...
                             '', ...
                             0, ...
                           };
          nNumberFields = 5;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCSpeciesDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { int32(-1), ...
                         '', ...
                         int32(0), ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
      nNumberFields = 6;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFluxBoundDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version <3)
    if (pkgVersion == 1)
      SBMLfieldnames = { 'SBML_FBC_FLUXBOUND', ...
                         '', ...
                         '', ...
                         '', ...
                           [], ...
                         int32(-1), ...
                         '', ...
                         '', ...
                         '', ...
                         0/0, ...
                         int32(0), ...
                         3 ...
                         int32(version) ...
                         1, ...
                       };
      nNumberFields = 14;
    else
        SBMLfieldnames = [];
        nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFluxObjectiveDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'SBML_FBC_FLUXOBJECTIVE', ...
                         '', ...
                         '', ...
                         '', ...
                           [], ...
                         int32(-1), ...
                         '', ...
                         0/0, ...
                         int32(0), ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
      nNumberFields = 12;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getObjectiveDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'SBML_FBC_OBJECTIVE', ...
                         '', ...
                         '', ...
                         '', ...
                           [], ...
                         int32(-1), ...
                         '', ...
                         '', ...
                         [], ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
      nNumberFields = 12;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getGeneProductDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion  == 2)
      SBMLfieldnames = { 'SBML_FBC_GENE_PRODUCT', ...
                         '', ...
                         '', ...
                         '', ...
                           [], ...
                         int32(-1), ...
                         '', ...
                         '', ...
                         '', ...
                         '', ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
      nNumberFields = 13;
    else
        SBMLfieldnames = [];
        nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCReactionDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)
      SBMLfieldnames = { '', ...
                         '', ...
                         [], ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
      nNumberFields = 6;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getGeneProductAssociationDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)

      SBMLfieldnames = { 'SBML_FBC_GENE_PRODUCT_ASSOCIATION', ...
                         '', ...
                         '', ...
                         '', ...
                           [], ...
                         int32(-1), ...
                         '', ...
                         '', ...
                         [], ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
                   nNumberFields = 12;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getAssociationDefaultValues(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)

      SBMLfieldnames = { 'SBML_FBC_ASSOCIATION', ...
                         '', ...
                         '', ...
                         '', ...
                            [], ...
                        int32(-1), ...
                         '', ...
                         3 ...
                         int32(version) ...
                         int32(pkgVersion), ...
                       };
                   nNumberFields = 10;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;



function valid = isValidLevelVersionCombination(level, version)
valid = 1;

if ~isIntegralNumber(level)
	error('level must be an integer');
elseif ~isIntegralNumber(version)
	error('version must be an integer');
end;

if (level < 1 || level > 3)
	error('current SBML levels are 1, 2 or 3');
end;

if (level == 1)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports versions 1-2 of SBML Level 1');
	end;

elseif (level == 2)
	if (version < 1 || version > 5)
		error('SBMLToolbox supports versions 1-5 of SBML Level 2');
	end;

elseif (level == 3)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports version 1-2 of SBML Level 3');
	end;

end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function value = isIntegralNumber(number)
value = 0;

integerClasses = {'int8', 'uint8', 'int16', 'uint16', 'int32', 'uint32', 'int64', 'uint64'};

% since the function isinteger does not exist in MATLAB Rel 13
% this is not used
%if (isinteger(number))
if (ismember(class(number), integerClasses))
    value = 1;
elseif (isnumeric(number))
    % if it is an integer 
    if (number == fix(number))
        value = 1;
    end;
end;



