# $DUH: Callbacks.pm,v 1.26 2002/12/16 03:25:23 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::Callbacks - multiple-callback container, callback tutorial

=head1 SYNOPSIS

    use PMilter::Callbacks qw(:all);
    use PMilter::Server;

    my $cb = PMilter::Callbacks->new({ helo => \&myhelo, ... }[, ...]);

    PMilter::Server::register(undef, $cb);

    %PMilter::Callbacks::DEFAULT_CALLBACKS;

=head1 USING PMILTER CALLBACKS

Under construction.

See L<Sendmail::Milter> docs (URL in that perldoc manual) for a more
complete description of the callback interface.

The export tag ':all' will export all SMFIF_* and SMFI_* constants into the
caller's namespace.  (The compatibility Sendmail::Milter interface does this
by default.)

=head1 DESCRIPTION

The PMilter::Callbacks object provides a container or chaining mechanism for
encapsulating multiple milter objects into one server.  Each element has its
own getpriv/setpriv private data store accessible from the milter context
object (see L<PMilter::Session>), allowing most milters to be added to a
chain with little or no modification.

When finally used as part of a milter connection, each element is executed
in turn for each callback, with the following behavior for different return
values:

=over 4

=item SMFIS_CONTINUE

Continue to the next element in the list; do continue to call this element
for future callbacks.

=item SMFIS_ACCEPT

Continue to the next element in the list; do not call this element again for
this connection/message.  (If returned in "connect" or "helo", applies to
the entire connection; otherwise applies to only the message in progress.)

=item SMFIS_BREAK

=item SMFIS_REJECT

=item SMFIS_TEMPFAIL

Do not continue to the next element in the list; propagate this value upward
to the root of the element tree.  In essence, these values can be thought as
raising exceptions, which stop all activity up to the top level of
processing.

SMFIS_BREAK is a PMilter extension specifically intended for use with
PMilter::Callbacks.  It is equivalent to SMFIS_ACCEPT, but continues up the
element tree rather than simply excluding the current element from future
calls.  See L<PMilter::Callbacks::Leaf> for the analogue to "catch
SMFIS_BREAK", which will turn it back into SMFIS_ACCEPT at that tree level,
allowing other sibling trees to continue executing.

=back

Note that the "close" callback is always called if supplied, and the "abort"
callback is always called if SMFIS_ACCEPT or SMFIS_BREAK has not been
returned for "connect" or "helo".

=head1 METHODS

=over 4

=cut

package PMilter::Callbacks;
use base Exporter;

use strict;
use warnings;

use Carp;
use PMilter;
use Symbol;

*VERSION = *PMilter::VERSION;

# Exported constants

use constant SMFIS_BREAK	=> -1;
use constant SMFIS_CONTINUE	=> 0;
use constant SMFIS_REJECT	=> 1;
use constant SMFIS_DISCARD	=> 2;
use constant SMFIS_ACCEPT	=> 3;
use constant SMFIS_TEMPFAIL	=> 4;

use constant SMFIF_ADDHDRS	=> 0x01;
use constant SMFIF_CHGBODY	=> 0x02;
use constant SMFIF_ADDRCPT	=> 0x04;
use constant SMFIF_DELRCPT	=> 0x08;
use constant SMFIF_CHGHDRS	=> 0x10;
use constant SMFIF_MODBODY	=> SMFIF_CHGBODY;

use constant SMFI_V1_ACTS	=> SMFIF_ADDHDRS|SMFIF_CHGBODY|SMFIF_ADDRCPT|SMFIF_DELRCPT;
use constant SMFI_V2_ACTS	=> SMFI_V1_ACTS|SMFIF_CHGHDRS;
use constant SMFI_CURR_ACTS	=> SMFI_V2_ACTS;

our @EXPORT_OK = qw(
	SMFIS_BREAK
	SMFIS_CONTINUE
	SMFIS_REJECT
	SMFIS_DISCARD
	SMFIS_ACCEPT
	SMFIS_TEMPFAIL

	SMFIF_ADDHDRS
	SMFIF_CHGBODY
	SMFIF_ADDRCPT
	SMFIF_DELRCPT
	SMFIF_CHGHDRS
	SMFIF_MODBODY

	SMFI_V1_ACTS
	SMFI_V2_ACTS
	SMFI_CURR_ACTS
);
our %EXPORT_TAGS = ( all => \@EXPORT_OK );

# Globals

# these are ordered so that lack of all remaining elements is
# equivalent to SMFIS_ACCEPT for that callback.
my @cbnames = qw(close connect helo abort envfrom envrcpt header eoh body eom);

my %cbindex = map { $cbnames[$_] => $_ } (0 .. $#cbnames);
our %DEFAULT_CALLBACKS = map { $_ => $_.'_callback' } @cbnames;

# Methods

=pod

=item new(CALLBACKS[, ...])

Create a PMilter::Callbacks object with the given contained element(s).  
Each element will be validated, its methods found, and stashed into an
internal list.  The returned reference can be passed to additional Callbacks
constructors, or passed directly to PMilter::Server::register().

If there is an error while validating an item, this will call Carp::confess
for a backtrace and exit.  This requires trapping the error with "eval" if
the program should continue running.

Each element must be either a hashref of callbacks (or a blessed one, such
as another PMilter::Callbacks object), or the filename of a module.  

If a hashref is supplied, it may optionally contain three non-callback keys:

=over 4

=item NAME

The name of this callback object, which is used in debugging or error
messages.

=item FLAGS

The SMFIF_* action flags requested by this callback object.

=item PACKAGE

If specified, this Perl package name is searched, for callbacks not
explicitly named in the hashref, for the subroutine names from
C<%PMilter::Callbacks::DEFAULT_CALLBACKS>.

=back

If a module filename is supplied, it is first searched in the current
directory, and then in the PMilter/modules subdirectories of C<@INC>.  
There are several prefabricated modules included by PMilter; see L<PMilter>
for a list of them.

An arrayref can also be supplied, which must contain the filename of a
module as the first argument; remaining arguments are passed to the module's
initialization code.

=cut

sub new {
	my $list = shift;

	unless (ref($list) eq 'ARRAY') {
		unshift(@_, $list);
		$list = [];
	}

	my $class = shift;

	confess 'invoked PMilter::Callbacks::new as non-class' unless $class;
	confess 'callback list empty' unless scalar @_;

	my $this = bless {
		CBLIST => [],
		FLAGS => 0,
		PACKAGE => ':'
	}, $class;

	my %hascbs = map { $_ => 1 } (@$list, 'connect');
	my $lastcb = -1;

	foreach my $x (@_) {
		my $callbacks = $x; # make modifiable
		local @_ = (); # args for module inclusion, below

		if (UNIVERSAL::isa($callbacks, 'ARRAY')) {
			@_ = @$callbacks;
			$callbacks = shift @_;
		}

		unless (UNIVERSAL::isa($callbacks, 'HASH')) {
			# note @_ is still in scope

			if (! -f $callbacks) {
				foreach my $dir (@INC) {
					if (-f $dir.'/PMilter/modules/'.$callbacks) {
						$callbacks = $dir.'/PMilter/modules/'.$callbacks;
						last;
					}
				}
			}
			# file-not-found will be handled by do() and confess $@

			my $cbref = do($callbacks);

			confess $@ if $@;
			confess "module $callbacks returned invalid value" unless UNIVERSAL::isa($cbref, 'HASH');

			$cbref->{NAME} = $callbacks unless $cbref->{NAME};

			$callbacks = $cbref;
		}

		my $name = $callbacks->{NAME} || '(unnamed callbacks)';
		my $pkg = $callbacks->{PACKAGE} || ':';
		my $cbcopy = { LASTCB => -1, NAME => $name };

		for my $i (0 .. $#cbnames) {
			my $cbname = $cbnames[$i];
			my $cb = $callbacks->{$cbname};

			if (ref($cb)) {
				confess "$name: callback $cbname is a non-code ref" if (ref($cb) ne 'CODE');
			} else {
				if (defined($cb)) { # scalar name
					$cb = qualify_to_ref($cb, $pkg);
					confess "$name: callback $cbname does not exist" unless exists(&$cb);
				} else {
					$cb = qualify_to_ref($DEFAULT_CALLBACKS{$cbname}, $pkg);
					next unless exists(&$cb);
				}
				$cb = \&$cb;
			}

			$cbcopy->{$cbname} = $cb;
			$cbcopy->{LASTCB} = $i;

			$hascbs{$cbname} = 1;
			$lastcb = $i if ($i > $lastcb);
		}

		confess "$name: no callbacks specified and no defaults found" if ($cbcopy->{LASTCB} < 0);

		push(@{$this->{CBLIST}}, $cbcopy);
		$this->{FLAGS} |= ($callbacks->{FLAGS} || 0);
	}

	# We need envfrom if someone is using a later callback.
	$hascbs{envfrom} = 1 if ($lastcb > $cbindex{envfrom});

	foreach my $cbname (keys %hascbs) {
		$this->{$cbname} = sub { $this->call($cbname, @_); };
	}

	$this;
}

# private, but overridable
sub call {
	my $this = shift;
	my $what = shift;
	my $ctx = shift;
	my $priv = $ctx->getpriv;

	my $rc = SMFIS_ACCEPT; # in case everyone is in the skiplists

	if ($what eq 'connect' || $what eq 'close') {
		# First time through: initialize.
		$priv = {} unless $priv;

		# Clear both lists
		$priv->{skipconn} = [];
		$priv->{skipmsg} = [];
	} elsif ($what eq 'envfrom' || $what eq 'abort') {
		# Clear per-message list
		$priv->{skipmsg} = [];
	} else {
		return $rc unless $priv->{skipmsg};
	}

	for my $i (0 .. $#{$this->{CBLIST}}) {
		next if $priv->{skipconn}[$i] || $priv->{skipmsg}[$i];

		$rc = SMFIS_CONTINUE; # someone's alive...

		my $cbref = $this->{CBLIST}[$i];
		my $newrc = SMFIS_ACCEPT;

		if ($cbref->{LASTCB} >= $cbindex{$what}) {
			# Only if we haven't called the last callback for this object.

			next unless $cbref->{$what};

			$ctx->setpriv($priv->{privlist}[$i]);
			$newrc = &{$cbref->{$what}}($ctx, @_);
			$priv->{privlist}[$i] = $ctx->getpriv;
		}

		if ($newrc == SMFIS_ACCEPT || $newrc == SMFIS_BREAK) {
			if ($what eq 'connect' || $what eq 'helo') {
				# Connection-global; skip all messages.
				$priv->{skipconn}[$i] = 1;
			} else {
				# Skip only until next RSET or MAIL FROM:.
				$priv->{skipmsg}[$i] = 1;
			}
		}

		# abort and close should always be run, in spite of "rejections",
		# so don't change $rc for those two.  Also don't propagate
		# SMFIS_ACCEPT, as that case was handled with skiplists above.

		$rc = $newrc unless ($newrc == SMFIS_ACCEPT || $what eq 'abort' || $what eq 'close');

		last unless ($rc == SMFIS_CONTINUE);
	}

	$ctx->setpriv($priv);
	$rc;
}

1;

__END__

=back

=head1 SEE ALSO

L<PMilter::Server> for the main invocation interface

L<PMilter::Session> on how to use the callback context object

L<Sendmail::Milter> for notes on the Sendmail::Milter compatibility
interface
