# $DUH: TXT.pm,v 1.6 2002/12/12 17:14:18 tv Exp $
#
# Copyright (c) 2002 Todd Vierling <tv@pobox.com> <tv@duh.org>.
# All rights reserved.
# Please see the COPYRIGHT file, part of the PMilter distribution,
# for full copyright and license terms.

=pod

=head1 NAME

PMilter::DNSBL::TXT - DNSBL checker that returns a TXT record, if present

=head1 SYNOPSIS

    my $dnsbl = new PMilter::DNSBL::TXT('foo.bar.com');

=head1 DESCRIPTION

This is a special case of B<PMilter::DNSBL> that retrieves a TXT record for
a query if a DNSBL lookup succeeds (and matches, if the returns list is
non-empty).  The TXT record (if present) is appended to the extended status
message (if present) passed to the C<new> constructor.

NOTE:  This module requires that the optional Perl Net::DNS module be
installed.

=cut

package PMilter::DNSBL::TXT;
use base PMilter::DNSBL;

use strict;
use warnings;

use Carp;
use Net::DNS::Resolver;

# single global resolver - might be a threading issue
my $res = new Net::DNS::Resolver;

sub query {
	my $this = shift;
	my $entry = shift;

	my $rv = $this->SUPER::query($entry);

	if ($rv && $rv->[0]) {
		my $resq = $res->query($entry.'.'.$this->{domain}, 'TXT');
		if ($resq) {
			foreach my $rr ($resq->answer) {
				if ($rr->type eq 'TXT') {
					if ($rv->[2]) {
						$rv->[2] .= ' - '.$rr->txtdata;
					} else {
						$rv->[2] = $rr->txtdata;
					}
					last;
				}
			}
		}
	}

	return $rv;
}

1;

__END__

=head1 BUGS

This class is slow, because Net::DNS is slow.

=head1 SEE ALSO

L<PMilter::DNSBL>
