
////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics.hpp>
#include <SFML/Audio.hpp>
#include <cmath>


////////////////////////////////////////////////////////////
/// Entry point of application
///
/// \return Application exit code
///
////////////////////////////////////////////////////////////
int main()
{
	// Defines PI
	const float PI = 3.14159f;

    // Create the window of the application
    sf::RenderWindow App(sf::VideoMode(800, 600, 32), "SFML Pong");

    // Load the sounds used in the game
    sf::SoundBuffer BallSoundBuffer;
    if (!BallSoundBuffer.LoadFromFile("datas/pong/ball.wav"))
    {
        return EXIT_FAILURE;
    }
    sf::Sound BallSound(BallSoundBuffer);

    // Load the images used in the game
    sf::Image BackgroundImage, LeftPaddleImage, RightPaddleImage, BallImage;
    if (!BackgroundImage.LoadFromFile("datas/pong/background.jpg")    ||
        !LeftPaddleImage.LoadFromFile("datas/pong/paddle_left.png")   ||
        !RightPaddleImage.LoadFromFile("datas/pong/paddle_right.png") ||
        !BallImage.LoadFromFile("datas/pong/ball.png"))
    {
        return EXIT_FAILURE;
    }

	// Initialize the end text
	sf::String End;
    End.SetFont("datas/pong/cheeseburger.ttf");
	End.SetSize(60.f);
	End.Move(150.f, 200.f);
    End.SetColor(sf::Color(50, 50, 250));

    // Create the sprites of the background, the paddles and the ball
    sf::Sprite Background(BackgroundImage);
    sf::Sprite LeftPaddle(LeftPaddleImage);
    sf::Sprite RightPaddle(RightPaddleImage);
    sf::Sprite Ball(BallImage);

    LeftPaddle.Move(10, (App.GetViewRect().GetHeight() - LeftPaddle.GetHeight()) / 2);
    RightPaddle.Move(App.GetViewRect().GetWidth() - RightPaddle.GetWidth() - 10, (App.GetViewRect().GetHeight() - RightPaddle.GetHeight()) / 2);
    Ball.Move((App.GetViewRect().GetWidth() - Ball.GetWidth()) / 2, (App.GetViewRect().GetHeight() - Ball.GetHeight()) / 2);

    // Define the paddles properties
    sf::Clock AITimer;
	const float AITime     = 0.1f;
    float LeftPaddleSpeed  = 400.f;
    float RightPaddleSpeed = 400.f;

    // Define the ball properties
    float BallSpeed = 400.f;
    float BallAngle;
    do
    {
        // Make sure the ball initial angle is not too much vertical
        BallAngle = sf::Randomizer::Random(0.f, 2 * PI);
    } while (std::abs(std::cos(BallAngle)) < 0.7f);

	bool IsPlaying = true;
    bool IsRunning = true;
    while(IsRunning)
    {
        // Handle events
        sf::Event Event;
        while (App.GetEvent(Event))
        {
            // Window closed or escape key pressed : exit
            if ((Event.Type == sf::Event::Closed) || 
               ((Event.Type == sf::Event::KeyPressed) && (Event.Key.Code == sf::Key::Escape)))
            {
                IsRunning = false;
                break;
            }
        }

        if (IsPlaying)
		{
			// Move the player's paddle
            if (App.GetInput().IsKeyDown(sf::Key::Up) && (LeftPaddle.GetTop() > 5.f))
				LeftPaddle.Move(0.f, -LeftPaddleSpeed * App.GetFrameTime());
            if (App.GetInput().IsKeyDown(sf::Key::Down) && (LeftPaddle.GetTop() < App.GetViewRect().GetHeight() - LeftPaddle.GetHeight() - 5.f))
				LeftPaddle.Move(0.f, LeftPaddleSpeed * App.GetFrameTime());

			// Move the computer's paddle
            if (((RightPaddleSpeed < 0.f) && (RightPaddle.GetTop() > 5.f)) ||
                ((RightPaddleSpeed > 0.f) && (RightPaddle.GetTop() < App.GetViewRect().GetHeight() - RightPaddle.GetHeight() - 5.f)))
            {
                RightPaddle.Move(0.f, RightPaddleSpeed * App.GetFrameTime());
            }

            // Update the computer's paddle direction according to the ball position
            if (AITimer.GetElapsedTime() > AITime)
            {
                AITimer.Reset();
                if ((RightPaddleSpeed < 0) && (Ball.GetTop() + Ball.GetHeight() > RightPaddle.GetTop() + RightPaddle.GetHeight()))
                    RightPaddleSpeed = -RightPaddleSpeed;
                if ((RightPaddleSpeed > 0) && (Ball.GetTop() < RightPaddle.GetTop()))
                    RightPaddleSpeed = -RightPaddleSpeed;
            }

			// Move the ball
			float Factor = BallSpeed * App.GetFrameTime();
			Ball.Move(std::cos(BallAngle) * Factor, std::sin(BallAngle) * Factor);

			// Check collisions between the ball and the screen
			if (Ball.GetLeft() < 0.f)
			{
				IsPlaying = false;
				End.SetText("You lost !\n(press escape to exit)");
			}
			if (Ball.GetLeft() + Ball.GetWidth() > App.GetViewRect().GetWidth())
			{
				IsPlaying = false;
				End.SetText("You won !\n(press escape to exit)");
			}
			if (Ball.GetTop() < 0.f)
			{
				BallSound.Play();
				BallAngle = -BallAngle;
				Ball.SetTop(0.1f);
			}
			if (Ball.GetTop() + Ball.GetHeight() > App.GetViewRect().GetHeight())
			{
				BallSound.Play();
				BallAngle = -BallAngle;
				Ball.SetTop(App.GetViewRect().GetHeight() - Ball.GetHeight() - 0.1f);
			}

			// Check the collisions between the ball and the paddles
			// Left Paddle
			if (Ball.GetLeft() < LeftPaddle.GetLeft() + LeftPaddle.GetWidth() && 
				Ball.GetLeft() > LeftPaddle.GetLeft() + (LeftPaddle.GetWidth() / 2.0f) &&
				Ball.GetTop() + Ball.GetHeight() >= LeftPaddle.GetTop() &&
				Ball.GetTop() <= LeftPaddle.GetTop() + LeftPaddle.GetHeight())
			{
				BallSound.Play();
				BallAngle = PI - BallAngle;
				Ball.SetLeft(LeftPaddle.GetLeft() + LeftPaddle.GetWidth() + 0.1f);
			}

			// Right Paddle
			if (Ball.GetLeft() + Ball.GetWidth() > RightPaddle.GetLeft() &&
				Ball.GetLeft() + Ball.GetWidth() < RightPaddle.GetLeft() + (RightPaddle.GetWidth() / 2.0f) &&
				Ball.GetTop() + Ball.GetHeight() >= RightPaddle.GetTop() &&
				Ball.GetTop() <= RightPaddle.GetTop() + RightPaddle.GetHeight())
			{
				BallSound.Play();
				BallAngle = PI - BallAngle;
				Ball.SetLeft(RightPaddle.GetLeft() - Ball.GetWidth() - 0.1f);
			}
		}

        // Draw the background, paddles and ball sprites
        App.Draw(Background);
        App.Draw(LeftPaddle);
        App.Draw(RightPaddle);
        App.Draw(Ball);

        // If the game is over, display the end message
        if (!IsPlaying)
            App.Draw(End);

        // Display things on screen
        App.Display();
    }

    return EXIT_SUCCESS;
}
