////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/Image.hpp>
#include <SFML/Graphics/ImageLoader.hpp>
#include <SFML/Graphics/GraphicsDevice.hpp>
#include <SFML/Graphics/OpenGL.hpp>
#include <SFML/Window/OpenGLCaps.hpp>
#include <algorithm>
#include <iostream>
#include <vector>


namespace sf
{
////////////////////////////////////////////////////////////
/// Default constructor
////////////////////////////////////////////////////////////
Image::Image() :
myWidth        (0),
myHeight       (0),
myTextureWidth (0),
myTextureHeight(0),
myGLTexture    (0),
myUpdated      (false)
{

}


////////////////////////////////////////////////////////////
/// Copy constructor
////////////////////////////////////////////////////////////
Image::Image(const Image& Copy) :
VideoResource  (Copy),
myWidth        (Copy.myWidth),
myHeight       (Copy.myHeight),
myTextureWidth (Copy.myTextureWidth),
myTextureHeight(Copy.myTextureHeight),
myPixels       (Copy.myPixels),
myGLTexture    (0),
myUpdated      (false)
{
    CreateTexture();
}


////////////////////////////////////////////////////////////
/// Construct an empty image
////////////////////////////////////////////////////////////
Image::Image(unsigned int Width, unsigned int Height, const Color& Col) :
myWidth        (0),
myHeight       (0),
myTextureWidth (0),
myTextureHeight(0),
myGLTexture    (0),
myUpdated      (false)
{
    Create(Width, Height, Col);
}


////////////////////////////////////////////////////////////
/// Construct the image from pixels in memory
////////////////////////////////////////////////////////////
Image::Image(unsigned int Width, unsigned int Height, const Uint8* Data) :
myWidth        (0),
myHeight       (0),
myTextureWidth (0),
myTextureHeight(0),
myGLTexture    (0),
myUpdated      (false)
{
    LoadFromPixels(Width, Height, Data);
}


////////////////////////////////////////////////////////////
/// Destructor
////////////////////////////////////////////////////////////
Image::~Image()
{
    // Destroy video resources
    DestroyVideoResources();
}


////////////////////////////////////////////////////////////
/// Load the image from a file
////////////////////////////////////////////////////////////
bool Image::LoadFromFile(const std::string& Filename)
{
    // Let the image loader load the image into our pixel array
    bool Success = priv::ImageLoader::GetInstance().LoadImageFromFile(Filename, myPixels, myWidth, myHeight);

    if (Success)
    {
        // Loading succeeded : we can create the texture
        if (CreateTexture())
            return true;
    }

    // Oops... something failed
    Reset();

    return false;
}


////////////////////////////////////////////////////////////
/// Load the surface from a file in memory
////////////////////////////////////////////////////////////
bool Image::LoadFromMemory(const char* Data, std::size_t SizeInBytes)
{
    // Let the image loader load the image into our pixel array
    bool Success = priv::ImageLoader::GetInstance().LoadImageFromMemory(Data, SizeInBytes, myPixels, myWidth, myHeight);

    if (Success)
    {
        // Loading succeeded : we can create the texture
        if (CreateTexture())
            return true;
    }

    // Oops... something failed
    Reset();

    return false;
}


////////////////////////////////////////////////////////////
/// Load the image directly from an array of pixels
////////////////////////////////////////////////////////////
bool Image::LoadFromPixels(unsigned int Width, unsigned int Height, const Uint8* Data)
{
    if (Data)
    {
        // Store the texture dimensions
        myWidth  = Width;
        myHeight = Height;

        // Fill the pixel buffer with the specified raw data
        const Color* Ptr = reinterpret_cast<const Color*>(Data);
        myPixels.assign(Ptr, Ptr + Width * Height);

        // We can create the texture
        if (CreateTexture())
        {
            return true;
        }
        else
        {
            // Oops... something failed
            Reset();
            return false;
        }
    }
    else
    {
        // No data provided : create a white image
        return Create(Width, Height, Color(255, 255, 255, 255));
    }
}


////////////////////////////////////////////////////////////
/// Save the content of the image to a file
////////////////////////////////////////////////////////////
bool Image::SaveToFile(const std::string& Filename) const
{
    // Let the image loader save our pixel array into the image
    return priv::ImageLoader::GetInstance().SaveImageToFile(Filename, myPixels, myWidth, myHeight);
}


////////////////////////////////////////////////////////////
/// Create an empty image
////////////////////////////////////////////////////////////
bool Image::Create(unsigned int Width, unsigned int Height, const Color& Col)
{
    // Store the texture dimensions
    myWidth  = Width;
    myHeight = Height;

    // Recreate the pixel buffer and fill it with the specified color
    myPixels.clear();
    myPixels.resize(Width * Height, Col);

    // We can create the texture
    if (CreateTexture())
    {
        return true;
    }
    else
    {
        // Oops... something failed
        Reset();
        return false;
    }
}


////////////////////////////////////////////////////////////
/// Create transparency mask from a specified colorkey
////////////////////////////////////////////////////////////
void Image::CreateMaskFromColor(const Color& ColorKey, Uint8 Alpha)
{
    // Calculate the new color (old color with no alpha)
    Color NewColor(ColorKey.r, ColorKey.g, ColorKey.b, Alpha);

    // Replace the old color by the new one
    std::replace(myPixels.begin(), myPixels.end(), ColorKey, NewColor);

    // The texture will need to be updated
    myUpdated = false;
}


////////////////////////////////////////////////////////////
/// Resize the image - warning : this function does not scale the image,
/// it just ajdusts size (add padding or remove pixels)
////////////////////////////////////////////////////////////
bool Image::Resize(unsigned int Width, unsigned int Height, const Color& Col)
{
    // Check size
    if ((Width == 0) || (Height == 0))
    {
        std::cerr << "Invalid new size for image (width = " << Width << ", height = " << Height << ")" << std::endl;
        return false;
    }

    // Create a new pixel array with the desired size
    std::vector<Color> Pixels(Width * Height, Col);

    // Copy the old pixel buffer into the new one
    for (unsigned int i = 0; i < std::min(Width, myWidth); ++i)
        for (unsigned int j = 0; j < std::min(Height, myHeight); ++j)
            Pixels[i + j * Width] = myPixels[i + j * myWidth];
    Pixels.swap(myPixels);

    // Store the new texture dimensions
    myWidth  = Width;
    myHeight = Height;

    // We can create the texture
    if (CreateTexture())
    {
        return true;
    }
    else
    {
        // Oops... something failed
        Reset();
        return false;
    }
}


////////////////////////////////////////////////////////////
/// Change the color of a pixel
/// Don't forget to call Update when you end modifying pixels
////////////////////////////////////////////////////////////
void Image::SetPixel(unsigned int X, unsigned int Y, const Color& Col)
{
    // Check if pixel is whithin the image bounds
    if ((X >= myWidth) || (Y >= myHeight))
    {
        std::cerr << "Cannot set pixel (" << X << "," << Y << ") for image "
                  << "(width = " << myWidth << ", height = " << myHeight << ")" << std::endl;
        return;
    }

    myPixels[X + Y * myWidth] = Col;

    // The texture will need to be updated
    myUpdated = false;
}


////////////////////////////////////////////////////////////
/// Get a pixel from the image
////////////////////////////////////////////////////////////
const Color& Image::GetPixel(unsigned int X, unsigned int Y) const
{
    // Check if pixel is whithin the image bounds
    if ((X >= myWidth) || (Y >= myHeight))
    {
        std::cerr << "Cannot get pixel (" << X << "," << Y << ") for image "
                  << "(width = " << myWidth << ", height = " << myHeight << ")" << std::endl;
        return Color::Black;
    }

    return myPixels[X + Y * myWidth];
}


////////////////////////////////////////////////////////////
/// Get a read-only pointer to the array of pixels (8 bits integers RGBA)
/// Array size is GetWidth() x GetHeight() x 4
/// This pointer becomes invalid if you reload or resize the image
////////////////////////////////////////////////////////////
const Uint8* Image::GetPixelsPtr() const
{
    if (!myPixels.empty())
    {
        return reinterpret_cast<const Uint8*>(&myPixels[0]);
    }
    else
    {
        std::cerr << "Trying to access the pixels of an empty image" << std::endl;
        return NULL;
    }
}


////////////////////////////////////////////////////////////
/// Bind the image for rendering
////////////////////////////////////////////////////////////
void Image::Bind() const
{
    if (myGLTexture)
    {
        // Check if the texture needs to be updated
        if (!myUpdated)
            Update();

        // Bind it
        GLCheck(glEnable(GL_TEXTURE_2D));
        GLCheck(glBindTexture(GL_TEXTURE_2D, myGLTexture));
    }
}


////////////////////////////////////////////////////////////
/// Enable or disable image smoothing filter
////////////////////////////////////////////////////////////
void Image::SetSmooth(bool Smooth) const
{
    if (myGLTexture)
    {
        // Change OpenGL texture filter
        GLCheck(glBindTexture(GL_TEXTURE_2D, myGLTexture));
        GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, Smooth ? GL_LINEAR : GL_NEAREST));
        GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, Smooth ? GL_LINEAR : GL_NEAREST));
        GLCheck(glBindTexture(GL_TEXTURE_2D, 0));
    }
}


////////////////////////////////////////////////////////////
/// Enable or disable image repeat mode
/// (ie. how to define pixels outside the texture range)
////////////////////////////////////////////////////////////
void Image::SetRepeat(bool Repeat) const
{
    if (myGLTexture)
    {
        // Change OpenGL texture wrap mode
        GLCheck(glBindTexture(GL_TEXTURE_2D, myGLTexture));
        GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, Repeat ? GL_REPEAT : GL_CLAMP));
        GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, Repeat ? GL_REPEAT : GL_CLAMP));
        GLCheck(glBindTexture(GL_TEXTURE_2D, 0));
    }
}


////////////////////////////////////////////////////////////
/// Return the width of the image
////////////////////////////////////////////////////////////
unsigned int Image::GetWidth() const
{
    return myWidth;
}


////////////////////////////////////////////////////////////
/// Return the height of the image
////////////////////////////////////////////////////////////
unsigned int Image::GetHeight() const
{
    return myHeight;
}


////////////////////////////////////////////////////////////
/// Convert a subrect expressed in pixels, into float
/// texture coordinates
////////////////////////////////////////////////////////////
FloatRect Image::GetTexCoords(const IntRect& Rect, bool Adjust) const
{
    float Offset = Adjust ? 0.5f : 0.f;
    return FloatRect((Rect.Left   + Offset) / myTextureWidth,
                     (Rect.Top    + Offset) / myTextureHeight,
                     (Rect.Right  - Offset) / myTextureWidth,
                     (Rect.Bottom - Offset) / myTextureHeight);
}


////////////////////////////////////////////////////////////
/// Get a valid texture size according to hardware support
////////////////////////////////////////////////////////////
unsigned int Image::GetValidTextureSize(unsigned int Size)
{
    if (OpenGLCaps::CheckExtension("GL_ARB_texture_non_power_of_two"))
    {
        // If hardware supports NPOT textures, then just return the unmodified size
        return Size;
    }
    else
    {
        // If hardware doesn't support NPOT textures, we calculate the nearest power of two
        unsigned int PowerOfTwo = 1;
        while (PowerOfTwo < Size)
            PowerOfTwo *= 2;

        return PowerOfTwo;
    }
}


////////////////////////////////////////////////////////////
/// Assignment operator
////////////////////////////////////////////////////////////
Image& Image::operator =(const Image& Other)
{
    Image Temp(Other);

    std::swap(myWidth,         Temp.myWidth);
    std::swap(myHeight,        Temp.myHeight);
    std::swap(myTextureWidth,  Temp.myTextureWidth);
    std::swap(myTextureHeight, Temp.myTextureHeight);
    std::swap(myGLTexture,     Temp.myGLTexture);
    std::swap(myUpdated,       Temp.myUpdated);
    myPixels.swap(Temp.myPixels);

    return *this;
}


////////////////////////////////////////////////////////////
/// Create the OpenGL texture
////////////////////////////////////////////////////////////
bool Image::CreateTexture()
{
    // Check if texture parameters are valid before creating it
    if (!myWidth || !myHeight || myPixels.empty())
    {
        Reset();
        return false;
    }

    // Destroy previous OpenGL texture if it was loaded
    if (myGLTexture)
    {
        GLCheck(glDeleteTextures(1, (GLuint*)&myGLTexture));
        myGLTexture = 0;
    }

    // Adjust internal texture dimensions depending on NPOT textures support
    myTextureWidth  = GetValidTextureSize(myWidth);
    myTextureHeight = GetValidTextureSize(myHeight);

    // Check maximum texture size
    unsigned int MaxSize = static_cast<unsigned int>(OpenGLCaps::GetMaxTextureSize());
    if ((myTextureWidth >= MaxSize) || (myTextureHeight >= MaxSize))
    {
        std::cerr << "Failed to create image, it's internal size is too high ("
                  << myWidth << "x" << myTextureHeight << ")" << std::endl;
        Reset();
        return false;
    }

    // Create the OpenGL texture
    GLCheck(glGenTextures(1, (GLuint*)&myGLTexture));
    GLCheck(glBindTexture(GL_TEXTURE_2D, myGLTexture));
    GLCheck(glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, myTextureWidth, myTextureHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL));

    // Set texture parameters for 2D rendering
    GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S,     GL_REPEAT));
    GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T,     GL_REPEAT));
    GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
    GLCheck(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));

    // Copy the pixel buffer to the OpenGL texture
    Update();

    return true;
}


////////////////////////////////////////////////////////////
/// Update the whole image in video memory
////////////////////////////////////////////////////////////
void Image::Update() const
{
    if (myGLTexture && myWidth && myHeight && !myPixels.empty())
    {
        // Update texture pixels
        GLCheck(glBindTexture(GL_TEXTURE_2D, myGLTexture));
        GLCheck(glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, myWidth, myHeight, GL_RGBA, GL_UNSIGNED_BYTE, &myPixels[0]));
        GLCheck(glBindTexture(GL_TEXTURE_2D, 0));
    }

    myUpdated = true;
}


////////////////////////////////////////////////////////////
/// Reset the image attributes
////////////////////////////////////////////////////////////
void Image::Reset()
{
    myWidth         = 0;
    myHeight        = 0;
    myTextureWidth  = 0;
    myTextureHeight = 0;
    myGLTexture     = 0;
    myUpdated       = false;
    myPixels.clear();
}


////////////////////////////////////////////////////////////
/// Destroy all video resources that need a valid rendering context
////////////////////////////////////////////////////////////
void Image::DestroyVideoResources()
{
    // Destroy textures
    if (myGLTexture)
    {
        GLCheck(glDeleteTextures(1, (GLuint*)&myGLTexture));
        myGLTexture = 0;
        myUpdated   = false;
    }
}

} // namespace sf
