////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/String.hpp>
#include <SFML/Graphics/FontManager.hpp>
#include <SFML/Graphics/Image.hpp>
#include <SFML/Graphics/OpenGL.hpp>
#include <locale>


namespace sf
{
////////////////////////////////////////////////////////////
/// Construct the string from a multibyte text
////////////////////////////////////////////////////////////
String::String(const std::string& Text, const std::string& Font, float Size) :
mySize(Size)
{
    SetFont(Font);
    SetText(Text);
}


////////////////////////////////////////////////////////////
/// Construct the string from a unicode text
////////////////////////////////////////////////////////////
String::String(const std::wstring& Text, const std::string& Font, float Size) :
mySize(Size)
{
    SetFont(Font);
    SetText(Text);
}


////////////////////////////////////////////////////////////
/// Preload a bitmap font (otherwise, it is done the first time the font is drawn)
////////////////////////////////////////////////////////////
void String::PreloadFont(const std::string& Font, float Char, std::wstring Charset)
{
    // Requesting the font will make the font manager create it
    priv::FontManager::GetInstance().GetBitmapFont(Font, static_cast<unsigned int>(Char), Charset);
}


////////////////////////////////////////////////////////////
/// Set the text (from a multibyte string)
////////////////////////////////////////////////////////////
void String::SetText(const std::string& Text)
{
    if (!Text.empty())
    {
        std::vector<wchar_t> Buffer(Text.size());

        #ifdef __MINGW32__
                // MinGW has a bad support for wstring conversions (and wchar_t based standard classes in general)
                mbstowcs(&Buffer[0], Text.c_str(), Text.size());
        #else
                std::locale Locale("");
                std::use_facet<std::ctype<wchar_t> >(Locale).widen(Text.data(), Text.data() + Text.size(), &Buffer[0]);
        #endif

        myText.assign(&Buffer[0], Buffer.size());
    }
    else
    {
        myText = L"";
    }
}


////////////////////////////////////////////////////////////
/// Set the text (from a unicode string)
////////////////////////////////////////////////////////////
void String::SetText(const std::wstring& Text)
{
    myText = Text;
}


////////////////////////////////////////////////////////////
/// Set the font of the string
////////////////////////////////////////////////////////////
void String::SetFont(const std::string& Font)
{
    myFont = Font;
}


////////////////////////////////////////////////////////////
/// Set the size of the string
////////////////////////////////////////////////////////////
void String::SetSize(float Size)
{
    mySize = Size;
}


////////////////////////////////////////////////////////////
/// Get the text (returns a unicode string)
////////////////////////////////////////////////////////////
const std::wstring& String::GetUnicodeText() const
{
    return myText;
}


////////////////////////////////////////////////////////////
/// Get the text (returns a multibyte string)
////////////////////////////////////////////////////////////
std::string String::GetText() const
{
    if (!myText.empty())
    {
        std::vector<char> Buffer(myText.size());

        #ifdef __MINGW32__
                // MinGW has a bad support for wstring conversions (and wchar_t based standard classes in general)
                wcstombs(&Buffer[0], myText.c_str(), myText.size());
        #else
                std::locale Locale("");
                std::use_facet<std::ctype<wchar_t> >(Locale).narrow(myText.data(), myText.data() + myText.size(), '?', &Buffer[0]);
        #endif

        return std::string(&Buffer[0], Buffer.size());
    }
    else
    {
        return "";
    }
}


////////////////////////////////////////////////////////////
/// Get the font used by the string
////////////////////////////////////////////////////////////
const std::string& String::GetFont() const
{
    return myFont;
}


////////////////////////////////////////////////////////////
/// Get the size of the characters
////////////////////////////////////////////////////////////
float String::GetSize() const
{
    return mySize;
}


////////////////////////////////////////////////////////////
/// Get the string rectangle on screen
////////////////////////////////////////////////////////////
FloatRect String::GetRect() const
{
    // Get the bitmap font from the font manager
    const priv::FontManager::Font& BitmapFont = priv::FontManager::GetInstance().GetBitmapFont(myFont, static_cast<unsigned int>(mySize));

    // Initial values
    float CurWidth = 0;
    float Width    = 0;
    float Height   = mySize;

    // Go through each character
    for (std::size_t i = 0; i < myText.size(); ++i)
    {
        // Get the current character
        wchar_t c = myText[i];

        // Check if the character is in the charset
        std::map<wchar_t, priv::FontManager::Font::Character>::const_iterator It = BitmapFont.Characters.find(c);
        if (It == BitmapFont.Characters.end())
        {
            // No : add a space and continue to the next character
            CurWidth += mySize;
            continue;
        }
        const priv::FontManager::Font::Character& CurChar = It->second;

        // Get the dimensions of the current character from the font description
        const float Advance = CurChar.Advance * mySize / BitmapFont.CharSize;

        // Handle special characters...
        switch (c)
        {
            case L' ' :  CurWidth += Advance;    continue;
            case L'\t' : CurWidth += mySize * 4; continue;
            case L'\v' : Height   += mySize * 4; continue;

            case L'\n' :
                Height += mySize;
                if (CurWidth > Width) Width = CurWidth;
                CurWidth = 0;
                continue;
        }

        // Advance to the next character
        CurWidth += Advance;
    }

    // In case there was no '\n' in the string
    if (Width < 1)
        Width = CurWidth;

    return FloatRect(GetLeft(), GetTop(), GetLeft() + Width, GetTop() + Height);
}


////////////////////////////////////////////////////////////
/// /see sfDrawable::Render
////////////////////////////////////////////////////////////
void String::Render(const RenderWindow&) const
{
    // No text, no rendering :)
    if (myText.empty())
        return;

    // Get the bitmap font from the font manager
    const priv::FontManager::Font& BitmapFont = priv::FontManager::GetInstance().GetBitmapFont(myFont, static_cast<unsigned int>(mySize));

    // Set the scaling factor to get the actual size
    float FactorX = GetScaleX() * mySize / BitmapFont.CharSize;
    float FactorY = GetScaleY() * mySize / BitmapFont.CharSize;
    GLCheck(glScalef(FactorX, FactorY, 1.f));

    // Bind the font texture
    BitmapFont.Texture.Bind();

    float X = 0;
    float Y = 0;

    // Draw the sprite
    glBegin(GL_QUADS);
    for (std::size_t i = 0; i < myText.size(); ++i)
    {
        // Get the current character
        wchar_t c = myText[i];

        // Check if the character is in the charset
        std::map<wchar_t, priv::FontManager::Font::Character>::const_iterator It = BitmapFont.Characters.find(c);
        if (It == BitmapFont.Characters.end())
        {
            // No : add a space and continue to the next character
            X += mySize;
            continue;
        }
        const priv::FontManager::Font::Character& CurChar = It->second;

        // Get the dimensions of the current character from font description
        const IntRect&   Rect       = CurChar.Rect;
        const FloatRect& Coord      = CurChar.Coord;
        const unsigned int AdvanceX = CurChar.Advance;
        const unsigned int AdvanceY = BitmapFont.CharSize;

        // Handle special characters
        switch (c)
        {
            case L' ' :  X += AdvanceX;        continue;
            case L'\n' : Y += AdvanceY; X = 0; continue;
            case L'\t' : X += AdvanceX * 4;    continue;
            case L'\v' : Y += AdvanceY * 4;    continue;
        }

        // Draw a textured quad for the current character
        glTexCoord2f(Coord.Left,  Coord.Top);    glVertex2f(X + Rect.Left,  Y + Rect.Top    + AdvanceY);
        glTexCoord2f(Coord.Left,  Coord.Bottom); glVertex2f(X + Rect.Left,  Y + Rect.Bottom + AdvanceY);
        glTexCoord2f(Coord.Right, Coord.Bottom); glVertex2f(X + Rect.Right, Y + Rect.Bottom + AdvanceY);
        glTexCoord2f(Coord.Right, Coord.Top);    glVertex2f(X + Rect.Right, Y + Rect.Top    + AdvanceY);

        // Advance to the next character
        X += AdvanceX;
    }
    glEnd();
}

} // namespace sf
