////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Window/Window.hpp>
#include <SFML/Window/WindowImpl.hpp>
#include <SFML/System/Sleep.hpp>
#include <iostream>
#include <memory>

namespace
{
    // Create a global dummy window, so that we have a valid OpenGL context at program startup
    //
    // TODO : provide a way to control the dummy creation / destruction, as the order of
    // initialization of globals is completely random across compile units...
    //
    std::auto_ptr<sf::priv::WindowImpl> DummyWindow(sf::priv::WindowImpl::New());
}


namespace sf
{
////////////////////////////////////////////////////////////
/// Default constructor
///
////////////////////////////////////////////////////////////
Window::Window() :
myWindow        (NULL),
myLastFrameTime (0.f),
myIsExternal    (false),
myFramerateLimit(0)
{

}


////////////////////////////////////////////////////////////
/// Construct a new window
////////////////////////////////////////////////////////////
Window::Window(VideoMode Mode, const std::string& Title, unsigned long WindowStyle, int AntialiasingLevel) :
myWindow        (NULL),
myLastFrameTime (0.f),
myIsExternal    (false),
myFramerateLimit(0)
{
    Create(Mode, Title, WindowStyle, AntialiasingLevel);
}


////////////////////////////////////////////////////////////
/// Construct the window from an existing control
////////////////////////////////////////////////////////////
Window::Window(WindowHandle Handle, int AntialiasingLevel) :
myWindow        (NULL),
myLastFrameTime (0.f),
myIsExternal    (true),
myFramerateLimit(0)
{
    Create(Handle, AntialiasingLevel);
}


////////////////////////////////////////////////////////////
/// Destructor
////////////////////////////////////////////////////////////
Window::~Window()
{
    // Destroy the window implementation
    delete myWindow;
}


////////////////////////////////////////////////////////////
/// Create the window
////////////////////////////////////////////////////////////
void Window::Create(VideoMode Mode, const std::string& Title, unsigned long WindowStyle, int AntialiasingLevel)
{
    // Check validity of video mode
    if ((WindowStyle & Style::Fullscreen) && !Mode.IsValid())
    {
        std::cerr << "The requested video mode is not available, switching to a valid mode" << std::endl;
        Mode = VideoMode::GetMode(0);
    }

    Initialize(priv::WindowImpl::New(Mode, Title, WindowStyle, AntialiasingLevel));
}


////////////////////////////////////////////////////////////
/// Create the window from an existing control
////////////////////////////////////////////////////////////
void Window::Create(WindowHandle Handle, int AntialiasingLevel)
{
    Initialize(priv::WindowImpl::New(Handle, AntialiasingLevel));
}


////////////////////////////////////////////////////////////
/// Get the width of the rendering region of the window
////////////////////////////////////////////////////////////
unsigned int Window::GetWidth() const
{
    return myWindow ? myWindow->GetWidth() : 0;
}


////////////////////////////////////////////////////////////
/// Get the height of the rendering region of the window
////////////////////////////////////////////////////////////
unsigned int Window::GetHeight() const
{
    return myWindow ? myWindow->GetHeight() : 0;
}


////////////////////////////////////////////////////////////
/// Get the depth buffer bits
////////////////////////////////////////////////////////////
unsigned int Window::GetDepthBits() const
{
    return myWindow ? myWindow->GetDepthBits() : 0;
}


////////////////////////////////////////////////////////////
/// Get the stencil buffer bits
////////////////////////////////////////////////////////////
unsigned int Window::GetStencilBits() const
{
    return myWindow ? myWindow->GetStencilBits() : 0;
}


////////////////////////////////////////////////////////////
/// Get the event on top of events stack, if any
////////////////////////////////////////////////////////////
bool Window::GetEvent(Event& EventReceived)
{
    // Pop first event of queue, if not empty
    if (!myEvents.empty())
    {
        EventReceived = myEvents.front();
        myEvents.pop();

        return true;
    }

    return false;
}


////////////////////////////////////////////////////////////
/// Enable / disable vertical synchronization
////////////////////////////////////////////////////////////
void Window::UseVerticalSync(bool Enabled)
{
    if (SetCurrent())
        myWindow->UseVerticalSync(Enabled);
}


////////////////////////////////////////////////////////////
/// Show or hide the mouse cursor
////////////////////////////////////////////////////////////
void Window::ShowMouseCursor(bool Show)
{
    if (myWindow)
        myWindow->ShowMouseCursor(Show);
}


////////////////////////////////////////////////////////////
/// Change the position of the mouse cursor
////////////////////////////////////////////////////////////
void Window::SetCursorPosition(unsigned int Left, unsigned int Top)
{
    if (myWindow)
        myWindow->SetCursorPosition(Left, Top);
}


////////////////////////////////////////////////////////////
/// Change the position of the window on screen
////////////////////////////////////////////////////////////
void Window::SetPosition(int Left, int Top)
{
    if (!myIsExternal)
    {
        if (myWindow)
            myWindow->SetPosition(Left, Top);
    }
    else
    {
        std::cerr << "Warning : trying to change the position of an external SFML window, which is not allowed" << std::endl;
    }
}


////////////////////////////////////////////////////////////
/// Show or hide the window
////////////////////////////////////////////////////////////
void Window::Show(bool State)
{
    if (!myIsExternal)
    {
        if (myWindow)
            myWindow->Show(State);
    }
    else
    {
        std::cerr << "Warning : trying to show/hide an external SFML window, which is not allowed" << std::endl;
    }
}


////////////////////////////////////////////////////////////
/// Set the window as the current target for rendering
////////////////////////////////////////////////////////////
bool Window::SetCurrent() const
{
    if (myWindow)
    {
        myWindow->SetCurrent();
        return true;
    }

    return false;
}


////////////////////////////////////////////////////////////
/// Display the window on screen
////////////////////////////////////////////////////////////
void Window::Display()
{
    // Limit the framerate if needed
    if (myFramerateLimit > 0)
    {
        float FrameTime = 1.f / myFramerateLimit;
        if (myClock.GetElapsedTime() < FrameTime)
            Sleep(FrameTime - myClock.GetElapsedTime());
    }

    // Measure the time elapsed since last frame
    myLastFrameTime = myClock.GetElapsedTime();
    myClock.Reset();

    if (SetCurrent())
    {
        // Display the backbuffer on screen
        myWindow->Display();

        // Let the window implementation process incoming events
        myWindow->DoEvents();

        // Notify the derived class
        OnDisplay();
    }
}


////////////////////////////////////////////////////////////
/// Get the input manager of the window
////////////////////////////////////////////////////////////
const Input& Window::GetInput() const
{
    return myInput;
}


////////////////////////////////////////////////////////////
/// Set the framerate at a fixed frequency
////////////////////////////////////////////////////////////
void Window::SetFramerateLimit(unsigned int Limit)
{
    myFramerateLimit = Limit;
}


////////////////////////////////////////////////////////////
/// Get time elapsed since last frame
////////////////////////////////////////////////////////////
float Window::GetFrameTime() const
{
    return myLastFrameTime;
}


////////////////////////////////////////////////////////////
/// Change the joystick threshold, ie. the value below which
/// no move event will be generated
////////////////////////////////////////////////////////////
void Window::SetJoystickThreshold(float Threshold)
{
    if (myWindow)
        myWindow->SetJoystickThreshold(Threshold);
}


////////////////////////////////////////////////////////////
/// Called after the window has been created
////////////////////////////////////////////////////////////
void Window::OnCreate()
{
    // Nothing by default
}


////////////////////////////////////////////////////////////
/// Called when the window displays its content on screen
////////////////////////////////////////////////////////////
void Window::OnDisplay()
{
    // Nothing by default
}


////////////////////////////////////////////////////////////
/// Called after an event has been received
////////////////////////////////////////////////////////////
void Window::OnEventReceived(const Event&)
{
    // Nothing by default
}


////////////////////////////////////////////////////////////
/// Receive an event from window
////////////////////////////////////////////////////////////
void Window::OnEvent(const Event& EventReceived)
{
    myEvents.push(EventReceived);

    SetCurrent();
    OnEventReceived(EventReceived);
}


////////////////////////////////////////////////////////////
/// Initialize internal window
////////////////////////////////////////////////////////////
void Window::Initialize(priv::WindowImpl* Window)
{
    // Destroy window if already created
    if (myWindow)
        delete myWindow;

    // Assign new window and listen to its events
    myWindow = Window;
    myWindow->AddListener(this);

    // Attach input to the window
    myWindow->AddListener(&myInput);

    // Disable vertical synchronization and show mouse cursor by default (to get a consistent behavior)
    UseVerticalSync(false);
    ShowMouseCursor(true);

    // Reset frame time
    myClock.Reset();
    myLastFrameTime = 0.f;

    // Notify the derived class
    SetCurrent();
    OnCreate();
}

} // namespace sf
