/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffData.h
 * Interface of SpiffData.
 */

#ifndef SPIFF_DATA_H
#define SPIFF_DATA_H

#include "SpiffDefines.h"
#include <deque>
#include <utility>

namespace Spiff {



class SpiffExtension;
class SpiffDataPrivate;



/**
 * Provides shared functionality for SpiffTrack and SpiffProps.
 */
class SpiffData {

	friend class SpiffDataPrivate;

private:
	/// @cond DOXYGEN_NON_API
	SpiffDataPrivate * const d; ///< D pointer
	/// @endcond

protected:
	/**
	 * Creates a new SpiffData object.
	 */
	SpiffData();

	/**
	 * Copy constructor.
	 *
	 * @param source  Source to copy from
	 */
	SpiffData(const SpiffData & source);

	/**
	 * Assignment operator.
	 *
	 * @param source  Source to copy from
	 */
	SpiffData & operator=(const SpiffData & source);

	/**
	 * Destroys this SpiffData object and deletes all
	 * memory associated with it that has not been stolen before.
	 */
	virtual ~SpiffData();

public:
	/**
	 * Overwrites the image property. If <c>copy</c> is true
	 * the string will be copied, otherwise just assigned.
	 * In both cases the associated memory will be deleted on
	 * object destruction.
	 *
	 * @param image		Image string to set
	 * @param copy		Copy flag
	 */
	void giveImage(const XML_Char * image, bool copy);

	/**
	 * Overwrites the info property. If <c>copy</c> is true
	 * the string will be copied, otherwise just assigned.
	 * In both cases the associated memory will be deleted on
	 * object destruction.
	 *
	 * @param info		Info string to set
	 * @param copy		Copy flag
	 */
	void giveInfo(const XML_Char * info, bool copy);

	/**
	 * Overwrites the annotation property. If <c>copy</c> is true
	 * the string will be copied, otherwise just assigned.
	 * In both cases the associated memory will be deleted on
	 * object destruction.
	 *
	 * @param annotation	Annotation string to set
	 * @param copy			Copy flag
	 */
	void giveAnnotation(const XML_Char * annotation, bool copy);

	/**
	 * Overwrites the creator property. If <c>copy</c> is true
	 * the string will be copied, otherwise just assigned.
	 * In both cases the associated memory will be deleted on
	 * object destruction.
	 *
	 * @param creator	Creator string to set
	 * @param copy		Copy flag
	 */
	void giveCreator(const XML_Char * creator, bool copy);

	/**
	 * Overwrites the title property. If <c>copy</c> is true
	 * the string will be copied, otherwise just assigned.
	 * In both cases the associated memory will be deleted on
	 * object destruction.
	 *
	 * @param title		Title string to set
	 * @param copy		Copy flag
	 */
	void giveTitle(const XML_Char * title, bool copy);

	/**
	 * Appends a link to the link list.
	 *
	 * @param rel			Link rel to append
	 * @param copyRel		Rel copy flag
	 * @param content		Link content to append
	 * @param copyContent	Content copy flag
	 */
	void giveAppendLink(const XML_Char * rel, bool copyRel, const XML_Char * content, bool copyContent);

	/**
	 * Appends a meta to the meta list.
	 *
	 * @param rel			Meta rel to append
	 * @param copyRel		Rel copy flag
	 * @param content		Meta content to append
	 * @param copyContent	Content copy flag
	 */
	void giveAppendMeta(const XML_Char * rel, bool copyRel, const XML_Char * content, bool copyContent);

	/**
	 * Appends an extension to the extension list.
	 *
	 * @param extension		Extension to append
	 * @param copy			Copy flag
	 */
	void giveAppendExtension(const SpiffExtension * extension, bool copy);

	/**
	 * Overwrites the image property. The string is
	 * only assigned not copied. The ownership is
	 * not transferred.
	 *
	 * @param image		Image string to set
	 */
	void lendImage(const XML_Char * image);

	/**
	 * Overwrites the info property. The string is
	 * only assigned not copied. The ownership is
	 * not transferred.
	 *
	 * @param info		Info string to set
	 */
	void lendInfo(const XML_Char * info);

	/**
	 * Overwrites the annotation property. The string is
	 * only assigned not copied. The ownership is
	 * not transferred.
	 *
	 * @param annotation		Annotation string to set
	 */
	void lendAnnotation(const XML_Char * annotation);

	/**
	 * Overwrites the creator property. The string is
	 * only assigned not copied. The ownership is
	 * not transferred.
	 *
	 * @param creator		Creator string to set
	 */
	void lendCreator(const XML_Char * creator);

	/**
	 * Overwrites the title property. The string is
	 * only assigned not copied. The ownership is
	 * not transferred.
	 *
	 * @param title		Title string to set
	 */
	void lendTitle(const XML_Char * title);

	/**
	 * Appends a link pair to the link list.
	 * The associated memory is neither copied nor
	 * deleted on onject destruction.
	 *
	 * @param rel		Link rel to append
	 * @param content	Link content to append
	 */
	void lendAppendLink(const XML_Char * rel, const XML_Char * content);

	/**
	 * Appends a meta pair to the meta list.
	 * The associated memory is neither copied nor
	 * deleted on onject destruction.
	 *
	 * @param rel		Meta rel to append
	 * @param content	Meta content to append
	 */
	void lendAppendMeta(const XML_Char * rel, const XML_Char * content);

	/**
	 * Appends an extension to the extension list.
	 * The associated memory is neither copied nor
	 * deleted on onject destruction.
	 *
	 * @param extension		Extension to append
	 */
	void lendAppendExtension(SpiffExtension * extension);

	/**
	 * Steals the image property.
	 *
	 * @return	Image URI, can be NULL
	 */
	XML_Char * stealImage();

	/**
	 * Steals the info property.
	 *
	 * @return	Info URI, can be NULL
	 */
	XML_Char * stealInfo();

	/**
	 * Steals the annotation property.
	 *
	 * @return	Annotation, can be NULL
	 */
	XML_Char * stealAnnotation();

	/**
	 * Steals the creator property.
	 *
	 * @return	Creator, can be NULL
	 */
	XML_Char * stealCreator();

	/**
	 * Steals the title property.
	 *
	 * @return	Title, can be NULL
	 */
	XML_Char * stealTitle();

	/**
	 * Steals the first link pair from the list.
	 * If the list is empty <c>NULL</c> is returned.
	 *
	 * NOTE: Do not forget to delete the pair!
	 *
	 * @return		First link pair, can be NULL
	 */
	std::pair<XML_Char *, XML_Char *> * stealFirstLink();

	/**
	 * Steals the first meta pair from the list.
	 * If the list is empty <c>NULL</c> is returned.
	 *
	 * NOTE: Do not forget to delete the pair!
	 *
	 * @return		First meta pair, can be NULL
	 */
	std::pair<XML_Char *, XML_Char *> * stealFirstMeta();

	/**
	 * Steals the extension from the list.
	 * If the list is empty <c>NULL</c> is returned.
	 *
	 * NOTE: Do not forget to delete the extension!
	 *
	 * @return		First extension, can be NULL
	 */
	SpiffExtension * stealFirstExtension();

	/**
	 * Returns the image property.
	 *
	 * @return	Image URI, can be NULL
	 */
	const XML_Char * getImage() const;

	/**
	 * Returns the info property.
	 *
	 * @return	Info URI, can be NULL
	 */
	const XML_Char * getInfo() const;

	/**
	 * Returns the annotation property.
	 *
	 * @return	Annotation, can be NULL
	 */
	const XML_Char * getAnnotation() const;

	/**
	 * Returns the creator property.
	 *
	 * @return	Creator, can be NULL
	 */
	const XML_Char * getCreator() const;

	/**
	 * Returns the title property.
	 *
	 * @return	Title, can be NULL
	 */
	const XML_Char * getTitle() const;

	/**
	 * Gets a specific link pair from the list.
	 * If the list is empty <c>NULL</c> is returned.
	 *
	 * NOTE: The returned pair has to be deleted manually!
	 *
	 * @return		Specified link pair, can be NULL
	 */
	std::pair<const XML_Char *, const XML_Char *> * getLink(int index) const;

	/**
	 * Gets a specific meta pair from the list.
	 * If the list is empty <c>NULL</c> is returned.
	 *
	 * NOTE: The returned pair has to be deleted manually!
	 *
	 * @return		Specified meta pair, can be NULL
	 */
	std::pair<const XML_Char *, const XML_Char *> * getMeta(int index) const;

	/**
	 * Gets a specific extension from the list.
	 * If the list is empty <c>NULL</c> is returned.
	 *
	 * NOTE: The returned extension has to be deleted manually!
	 *
	 * @return		Specified extension, can be NULL
	 */
	const SpiffExtension * getExtension(int index) const;

	/**
	 * Returns the number of link pairs.
	 *
	 * @return	Number of link pairs
	 */
	int getLinkCount() const;

	/**
	 * Returns the number of meta pairs.
	 *
	 * @return	Number of meta pairs
	 */
	int getMetaCount() const;

	/**
	 * Returns the number of extensions.
	 *
	 * @return	Number of extensions
	 */
	int getExtensionCount() const;

private:
	/**
	 * Appends a meta or link pair to a container.
	 *
	 * @param container		Container to work with
	 * @param rel			Rel to append
	 * @param ownRel		Rel ownership flag
	 * @param content		Content to append
	 * @param ownContent	Content ownership flag
	 */
	static void appendHelper(std::deque<std::pair<
			std::pair<const XML_Char *, bool> *,
			std::pair<const XML_Char *, bool> *> *> * & container,
			const XML_Char * rel,
			bool ownRel,
			const XML_Char * content,
			bool ownContent);

	/**
	 * Appends an extension to a container.
	 *
	 * @param container		Container to work with
	 * @param extenstion	Extenstion to append
	 * @param own			Ownership flag
	 */
	static void appendHelper(std::deque<
			std::pair<const SpiffExtension *, bool> *> * & container,
			const SpiffExtension * extenstion,
			bool own);

protected:
	/**
	 * Steals a property. If the property's memory is not owned
	 * a clone is returned. In any case you own the memory
	 * return and have to delete it.
	 *
	 * @param property	Property to steal
	 * @param own		Owner flag
	 * @return			Stolen property value, can be NULL
	 */
	static XML_Char * stealHelper(const XML_Char * & property, bool own);

private:
	/**
	 * Steals the first entry from a container.
	 *
	 * @param container		Container to steal from
	 * @return				First entry, can be NULL
	 */
	static std::pair<XML_Char *, XML_Char *> * stealFirstHelper(
			std::deque<std::pair<std::pair<const XML_Char *, bool> *,
			std::pair<const XML_Char *, bool> *> *> * & container);

	/**
	 * Steals the first entry from a container.
	 *
	 * @param container		Container to steal from
	 * @return				First entry, can be NULL
	 */
	static SpiffExtension * stealFirstHelper(std::deque<
			std::pair<const SpiffExtension *, bool> *> * & container);

	/**
	 * Returns a specific entry from a container
	 * or <c>NULL</c> if the entry does not exist.
	 *
	 * NOTE: The returned pair has to be deleted manually!
	 *
	 * @param container		Container to work with
	 * @param index			Index of the entry to return
	 * @return				Entry content, can be NULL
	 */
	static std::pair<const XML_Char *, const XML_Char *> * getHelper(
			std::deque<std::pair<std::pair<const XML_Char *, bool> *,
			std::pair<const XML_Char *, bool> *> *> * & container,
			int index);

	/**
	 * Returns a specific entry from a container
	 * or <c>NULL</c> if the entry does not exist.
	 *
	 * NOTE: The returned pair has to be deleted manually!
	 *
	 * @param container		Container to work with
	 * @param index			Index of the entry to return
	 * @return				Entry content, can be NULL
	 */
	static const SpiffExtension * getHelper(
		std::deque<std::pair<const SpiffExtension *, bool> *> * & container,
		int index);

};



}

#endif // SPIFF_DATA_H
