/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffExtensionWriter.h
 * Interface of SpiffExtensionWriter.
 */

#ifndef SPIFF_EXTENSION_WRITER_H
#define SPIFF_EXTENSION_WRITER_H



#include "SpiffDefines.h"

namespace Spiff {



class SpiffExtension;
class SpiffXmlFormatter;
class SpiffExtensionWriterPrivate;



class SpiffExtensionWriter {

private:
	/// @cond DOXYGEN_NON_API
	SpiffExtensionWriterPrivate * const d; ///< D pointer
	/// @endcond

protected:
	const SpiffExtension * extension; ///< Extension to write
	SpiffXmlFormatter * output; ///< Output formatter to write to

public:
	/**
	 * Creates a new SpiffExtensionWriter object.
	 *
	 * @param extension		Extension to write
	 * @param output		Output formatter to write to
	 */
	SpiffExtensionWriter(const SpiffExtension * extension,
			SpiffXmlFormatter * output);

	/**
	 * Copy constructor.
	 *
	 * @param source  Source to copy from
	 */
	SpiffExtensionWriter(const SpiffExtensionWriter & source);

	/**
	 * Assignment operator.
	 *
	 * @param source  Source to copy from
	 */
	SpiffExtensionWriter & operator=(const SpiffExtensionWriter & source);

	/**
	 * Destroys this SpiffExtensionWriter object and deletes all
	 * memory associated with it.
	 */
	virtual ~SpiffExtensionWriter();

	/**
	 * Writes the associated extension to the output formater.
	 */
	void write();

private:
	/**
	 * Writes the opening extension tag also registering all
	 * previously unknown namespaces.
	 *
	 * @param nsRegs	NULL-terminated list of namespace registrations (uri/prefix pairs)
	 */
	void writeExtensionStart(const XML_Char ** nsRegs);

	/**
	 * Writes the extension body.
	 */
	virtual void writeExtensionBody() = 0;

	/**
	 * Writes the closing extension tag.
	 */
	void writeExtensionStop();

	/**
	 * Returns a list of the namespaces used inside
	 * the extension body. Moreprecisely this
	 * is an alternating NULL-terminated list
	 * of URI/prefix pairs.
	 *
	 * @return	List of URI/prefix pairs
	 */
	virtual const XML_Char ** getNamespaceRegs();

};



}

#endif // SPIFF_EXTENSION_WRITER_H
