/* MenuBuilder.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.menus;

import be.ugent.caagt.swirl.GenericSelectionModel;
import java.awt.event.ItemListener;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.JMenuBar;
import javax.swing.JPopupMenu;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import org.jdom.JDOMException;

/**
 * Allows menus to be constructed from a configuration file.<p>
 * The configuration file uses an XML-format, the DTD of which can be found
 * <a href="../../../../../../dtds/configuration.dtd">here</a>. The corresponding
 * public identifier is "{@code -//SWIRL//MenuBuilder configuration 1.0//EN}".<p>
 * More information can be found in the <a href="../../../../../../tutorial.html">MenuBuilder tutorial</a>.
 */
public class MenuBuilder {

    
    //
    private final Set<String> predicates;
    
    //
    private final ActionMap actionMap;
    //
    private final Map<String, JToggleButton.ToggleButtonModel> toggleMap;
    //
    private final Map<String, GroupInfo<?>> groupMap;
    
    private final ConfigurationLoader configurationLoader;

    /**
     * Create a new object of this type with the associated action map.
     */
    public MenuBuilder(ActionMap actionMap) {
        this.predicates = new HashSet<String>();
        this.actionMap = actionMap;
        this.toggleMap = new HashMap<String, JToggleButton.ToggleButtonModel>();
        this.groupMap = new HashMap<String, GroupInfo<?>>();
        
        this.configurationLoader = new ConfigurationLoader();
    }

    /**
     * Create a new object of this type with a default action map.
     */
    public MenuBuilder() {
        this(new ActionMap());
    }

    /**
     * The action map used by this builder
     * @return the cation map used by this builder
     */
    public ActionMap getActionMap() {
        return actionMap;
    }

    /**
     * Set a predicate with the given name. Predicates can influence
     * which parts of a menu are built.
     */
    public void setPredicate(String predicate) {
        predicates.add(predicate);
    }

    /**
     * Remove (unset) the predicate with the given name.
     * @see #setPredicate
     */
    public void removePredicate(String predicate) {
        predicates.remove(predicate);
    }
    
    //
    Set<String> getPredicates () {
        return predicates;
    }

    /**
     * Register an action this builder.<p>
     * Has the same effect as registering the action with the action map
     * of this builder.
     * @param id Identifier used in the configuration file for this action
     * @param action Action object to be associated with this identifier
     */
    public void registerAction (String id, Action action) {
        actionMap.put (id, action);
    }
    
    /**
     * Register an item listener for a toggle.
     * @param id Identifier used for this toggle in the configuration file
     * @param itemListener Item listener to be associated with this identifier
     */
    public void registerToggle (String id, ItemListener itemListener) {
        JToggleButton.ToggleButtonModel bm = new JToggleButton.ToggleButtonModel();
        bm.addItemListener(itemListener);
        toggleMap.put(id, bm);
    }
    
    /**
     * Register a selection group
     * @param id Identifier used for this group in the configuration file
     * @param model Selection model to be associated with this group
     * @param namesAndConstants List of names and constants for the buttons
     * in this group
     */
    public<E> void registerGroup (String id, GenericSelectionModel<E> model, Object... namesAndConstants) {
        groupMap.put(id, new GroupInfo<E> (model, namesAndConstants));
    }
    
    /**
     * Return the action which was registered with the given id.
     */
    Action getAction (String id) {
        return actionMap.get(id);
    }
    
    /**
     * Return the button model for the toggle which was registered
     * with the given id.
     */
    JToggleButton.ToggleButtonModel getToggleModel (String id) {
        return toggleMap.get(id);
    }
    
    /**
     * Return the selection model for the group which was registered
     * with the given id.
     */
    GenericSelectionModel getGroup (String id) {
        return groupMap.get(id).getSelectionModel();
    }
    
    /**
     * Return information on the group which was registered
     * with the given id.
     */
    GroupInfo getGroupInfo (String id) {
        return groupMap.get(id);
    }

    /**
     * Load configuration information from the given class path resource.
     * @param bundleName Name of the resource bundle which will be used
     * to resolve the various keys in the cnofiguration file
     */
    public void load(String resource, String bundleName) {
        try {
            configurationLoader.setBundleName(bundleName);
            configurationLoader.load(resource);
        } catch (IOException ex) {
            throw new MenuConfigurationException("Could not read menu configuration file", ex);
        } catch (JDOMException ex) {
            throw new MenuConfigurationException("Could not read menu configuration file", ex);
        }
    }

    /**
     * Load configuration information from the given input stream.
     * @param bundleName Name of the resource bundle which will be used
     * to resolve the various keys in the cnofiguration file
     */
    public void load(InputStream inputStream, String bundleName) throws IOException {
        try {
            configurationLoader.setBundleName(bundleName);
            configurationLoader.load(inputStream);
        } catch (JDOMException ex) {
            throw new MenuConfigurationException("Could not parse menu configuration file", ex);
        }
    }
    
    private Iterable<EntryConfiguration> getExistingRoot (String key) {
        Iterable<EntryConfiguration> list = configurationLoader.getRoot(key);
        if (list == null) {
            throw new MenuConfigurationException("Unknown root key: " + key);
        }
        else
            return list;
    }

    /**
     * Build a menu bar which corresponds to the root configuration
     * element with the given key. All actions, groups and toggles that occur
     * in this menu must have been registered with the builder before this
     * method is called.
     * @see #registerAction
     * @see #registerToggle
     * @see #registerGroup
     */
    public JMenuBar createJMenuBar(String key) {
        JMenuBar menuBar = new JMenuBar();
        ToMenuBar ctmb = new ToMenuBar(menuBar, this);
        for (EntryConfiguration entryConfiguration : getExistingRoot(key)) {
            entryConfiguration.accept(ctmb);
        }
        return menuBar;
    }

    /**
     * Build a popup menu which corresponds to the menu configuration
     * element with the given key.
     * @see #registerAction
     * @see #registerToggle
     * @see #registerGroup
     */
    public JPopupMenu createJPopupMenu(String key) {
        JPopupMenu popupMenu = new JPopupMenu();
        ToPopupMenu ctpm = new ToPopupMenu(popupMenu, this);
        for (EntryConfiguration entryConfiguration : getExistingRoot(key)) {
            entryConfiguration.accept(ctpm);
        }
        return popupMenu;
    }

    /**
     * Build a tool bar which corresponds to the menu configuration
     * element with the given key. The menu element should not have any submenus.
     * @see #registerAction
     * @see #registerToggle
     * @see #registerGroup
     */
    public JToolBar createJToolBar(String key) {
        JToolBar toolBar = new JToolBar();
        ToToolBar cttb = new ToToolBar(toolBar, this);
        for (EntryConfiguration entryConfiguration : getExistingRoot(key)) {
            entryConfiguration.accept(cttb);
        }
        return toolBar;
    }

}
