/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <libsyncml/syncml.h>
#include <libsyncml/syncml_internals.h>
 
#ifdef ENABLE_OBEX
#include <libsyncml/sml_error_internals.h>
#include <libsyncml/sml_transport_internals.h>

#ifdef ENABLE_BLUETOOTH
#include <bluetooth/bluetooth.h>
#endif

#include "obex_client.h"
#include "obex_client_internals.h"

#include <fcntl.h>
#include <sys/poll.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <termios.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <stdio.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netinet/in.h>

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

static void _smlObexEvent(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %i, %i, %i)", __func__, handle, object, mode, event, obex_cmd, obex_rsp);
	SmlTransportObexClientEnv *env = OBEX_GetUserData(handle);
	SmlError *error = NULL;
	
	switch (event)  {
		case OBEX_EV_PROGRESS:
			smlTrace(TRACE_INTERNAL, "Progress");
			break;
		case OBEX_EV_REQDONE:
			smlTrace(TRACE_INTERNAL, "Request Done");
			env->busy = FALSE;
			
			if (obex_rsp != OBEX_RSP_SUCCESS) {
				smlErrorSet(&error, SML_ERROR_GENERIC, "Request not successfull: %i", obex_rsp);
				goto error;
			}
			
			/* Get the connection id if we connected */
			switch (obex_cmd) {
				case OBEX_CMD_CONNECT:;
					uint8_t headertype = 0;
					obex_headerdata_t header;
					uint32_t len;
					char *who = NULL;
					
					while (OBEX_ObjectGetNextHeader(env->obexhandle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p", headertype, header.bq4, header.bs);
						switch (headertype) {
							case OBEX_HDR_CONNECTION:
								smlTrace(TRACE_INTERNAL, "Found connection number: %d", header.bq4);
								env->connection_id = header.bq4;
								break;
							case OBEX_HDR_WHO:
								who = g_strndup((char *)header.bs, len);
								smlTrace(TRACE_INTERNAL, "Found who: %s", who);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!env->connection_id) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing connection id");
						g_free(who);
						goto error;
					}
					
					if (!who || strcmp(who, "SYNCML-SYNC")) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing or wrong who response");
						g_free(who);
						goto error;
					}
					g_free(who);
					
					smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_CONNECT_DONE, NULL, NULL);
					break;
				case OBEX_CMD_DISCONNECT:;
					//smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DISCONNECT_DONE, NULL, NULL);
					break;
				case OBEX_CMD_GET:;
					smlTrace(TRACE_INTERNAL, "Got GET command done");
					uint32_t length = 0;
					char *body = NULL;
					
					while (OBEX_ObjectGetNextHeader(handle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p, len %i", headertype, header.bq4, header.bs, len);
						switch (headertype) {
							case OBEX_HDR_LENGTH:
								smlTrace(TRACE_INTERNAL, "Found length: %d", header.bq4);
								length = header.bq4;
								break;
							case OBEX_HDR_BODY:
								if (!length) {
									smlTrace(TRACE_INTERNAL, "Length not given. Calculating it to: %i", len);
									length = len;
								}
								
								if (!length) {
									smlErrorSet(&error, SML_ERROR_GENERIC, "Got zero length!");
									goto error;
								}
								
								body = smlTryMalloc0(length, &error);
								if (!body)
									goto error;
								
								memcpy(body, header.bs, length);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!length) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing length");
						goto error;
					}
					
					if (!body) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing body");
						goto error;
					}
					
					SmlTransportData *tspdata = smlTransportDataNew(body, length, env->mimetype, TRUE, &error);
					if (!tspdata)
						goto error;
					
					smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DATA, tspdata, NULL);
					break;
			}
			break;
		case OBEX_EV_LINKERR:
			smlTrace(TRACE_INTERNAL, "Link Error: %i", obex_rsp);
			smlErrorSet(&error, SML_ERROR_GENERIC, "Link Error");
			goto error;
			break;
		case OBEX_EV_STREAMEMPTY:
			smlTrace(TRACE_INTERNAL, "Empty Stream");
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
	env->busy = FALSE;
	env->error = TRUE;
	return;
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

static void *smlTransportObexClientInit(SmlTransport *tsp, const void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	SmlTransportObexClientEnv *env = smlTryMalloc0(sizeof(SmlTransportObexClientEnv), error);
	if (!env)
		goto error;
	
	const SmlTransportObexClientConfig *conf = data;
	env->tsp = tsp;
	env->path = g_strdup(conf->url);
	env->type = conf->type;
	env->port = conf->port;
	
	switch (conf->type) {
		case SML_OBEX_TYPE_NET:
			env->obexhandle = OBEX_Init(OBEX_TRANS_FD, _smlObexEvent, 0);
			break;
		case SML_OBEX_TYPE_IRDA:
			env->obexhandle = OBEX_Init(OBEX_TRANS_IRDA, _smlObexEvent, 0);
			break;
		case SML_OBEX_TYPE_BLUETOOTH:
			env->obexhandle = OBEX_Init(OBEX_TRANS_BLUETOOTH, _smlObexEvent, 0);
			break;
		case SML_OBEX_TYPE_SERIAL:
			env->obexhandle = OBEX_Init(OBEX_TRANS_FD, _smlObexEvent, 0);			
			break;
		case SML_OBEX_TYPE_USB:
			env->obexhandle = OBEX_Init(OBEX_TRANS_USB, _smlObexEvent, 0);
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown obex type");
			goto error_free_env;
	}
	
	if (!env->obexhandle) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to open connection");
		goto error_free_env;
	}
	
	OBEX_SetUserData(env->obexhandle, env);

    /* Create a buffer that will hold the stream chunks */
    env->stream_chunk = smlTryMalloc0(STREAM_CHUNK, error);
    if (!env->stream_chunk)
    	goto error_close_handle;
    
	smlTrace(TRACE_EXIT, "%s: %p", __func__, env);
	return env;

error_close_handle:
	OBEX_Cleanup(env->obexhandle);
error_free_env:
	g_free(env->path);
	g_free(env);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool smlTransportObexClientFinalize(void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, error);
	smlAssert(data);
	SmlTransportObexClientEnv *env = data;
	
	smlAssert(env->tsp);
	g_free(env->path);
	g_free(env->stream_chunk);
	
	OBEX_Cleanup(env->obexhandle);
	
	g_free(env);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void smlTransportObexClientConnect(void *data)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, data);
	smlAssert(data);
	SmlTransportObexClientEnv *env = data;
	int fd = 0;
	SmlError *error = NULL;
	int obex_intf_cnt;
	obex_interface_t *obex_intf;
	
	if (env->type == SML_OBEX_TYPE_NET) {
		struct sockaddr_in addr;
		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_port = htons(env->port);
		
		struct hostent *hostinfo = gethostbyname (env->path);
		if (!hostinfo) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown host %s", env->path);
	    	goto error;
		}
		addr.sin_addr = *(struct in_addr *) hostinfo->h_addr_list[0];
		
		fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
		if (fd < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Cannot create socket: %s", strerror(errno));
			goto error;
		}
			
		smlTrace(TRACE_INTERNAL, "socket %i", fd);
		
		const char *addrstr = inet_ntoa(addr.sin_addr);
		smlTrace(TRACE_INTERNAL, "peer addr = %d %s %i", hostinfo->h_addr_list[0], addrstr, env->port);

		if (connect(fd, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Cannot connect socket: %s", strerror(errno));
			goto error_close;
		}
		smlTrace(TRACE_INTERNAL, "connect done");

	} else if (env->type == SML_OBEX_TYPE_USB) {
		smlTrace(TRACE_INTERNAL, "connecting to usb interface %i", env->port);
		
		obex_intf_cnt = OBEX_FindInterfaces(env->obexhandle, &obex_intf);
		smlTrace(TRACE_INTERNAL, "found %i interfaces", obex_intf_cnt);
		
		if (obex_intf_cnt <= 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "There are no valid USB interfaces");
			goto error;
		} else if (env->port >= obex_intf_cnt) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to find the USB interface number %i", env->port);
			goto error;
		} else {
			if (OBEX_InterfaceConnect(env->obexhandle, &obex_intf[env->port]) < 0) {
				smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to connect to the interface");
				goto error;
			}
		}
		smlTrace(TRACE_INTERNAL, "usb connect done");
	} else if (env->type == SML_OBEX_TYPE_BLUETOOTH) {
#ifdef ENABLE_BLUETOOTH
		if (!env->path) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "No bluetooth address given");
			goto error;
		}
		
		smlTrace(TRACE_INTERNAL, "connecting to bluetooth device %s channel %i", env->path, env->port);

		uint8_t channel = env->port;
		bdaddr_t bdaddr;
		str2ba(env->path, &bdaddr);

		if (BtOBEX_TransportConnect(env->obexhandle, BDADDR_ANY, &bdaddr, channel) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Bluetooth connect error");
			goto error;
		}

		smlTrace(TRACE_INTERNAL, "bluetooth connect done");
#else
		smlErrorSet(&error, SML_ERROR_GENERIC, "Bluetooth not enabled");
		goto error;
#endif
 	} else {
		struct termios tio;
	    memset(&tio, 0, sizeof(tio));
	
		/* Open the file descriptor */
		fd = open(env->path, O_RDWR | O_NOCTTY);
	    if (fd == -1) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to open %s: %s", env->path, strerror(errno));
	    	goto error;
	    }
	    
	    /* Lock it*/
	    if (lockf(fd, F_TLOCK, 0)) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to lock %s: %s", env->path, strerror(errno));
			goto error_close;
	    }
	    
	    if (tcgetattr(fd, &tio)) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get attr from %s", env->path);
			goto error_unlock;
	    }
	    
	    /* Make the transmission raw (8 bit clean, no echo etc) */
	    /* Set the speed to a higher value (9600 would be default) */
	    cfmakeraw(&tio);
	    cfsetispeed(&tio, B115200);
	    cfsetospeed(&tio, B115200);
	
	    if (tcsetattr(fd, TCSANOW, &tio)) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to set attr from %s", env->path);
			goto error_unlock;
	    }
	    tcflush(fd, TCIFLUSH);
	}
	
	if (env->type != SML_OBEX_TYPE_USB && env->type != SML_OBEX_TYPE_BLUETOOTH) {
		/* Start the obex transport */
		if (FdOBEX_TransportSetup(env->obexhandle, fd, fd, 4096) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to setup transport");
			goto error_unlock;
		}
	}
	
	/* Make a new connect object */
	obex_object_t *obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_CONNECT);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new connect object");
		goto error_transport_close;
	}

	/* Now add the header for the sync target */
	obex_headerdata_t header;
	header.bs = (unsigned char *)"SYNCML-SYNC";
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TARGET, header, strlen((char *)header.bs), OBEX_FL_FIT_ONE_PACKET);
    
	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
    
	if (env->error) {
		smlTrace(TRACE_EXIT, "Unable to send connect request. Bailing out");
		return;
	}
	
	/* Lets see if it was successfull */
	while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error_free_obj:
	OBEX_ObjectDelete(env->obexhandle, obj);
error_transport_close:
	OBEX_Cleanup(env->obexhandle);
error_unlock:
	if (!lockf(fd, F_ULOCK, 0))
		smlTrace(TRACE_ERROR, "%s: error_unlock failed.", __func__);
error_close:
	close(fd);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void smlTransportObexClientDisconnect(void *data, void *linkdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, linkdata);
	smlAssert(data);
	SmlTransportObexClientEnv *env = data;
	SmlError *error = NULL;
	
	env->error = FALSE;

	/* Make a new disconnect object */
	obex_object_t *obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_DISCONNECT);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new disconnect object");
		goto error;
	}
	
	/* Add the connection id */
	obex_headerdata_t header;
	header.bq4 = env->connection_id;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_CONNECTION, header, sizeof(env->connection_id), OBEX_FL_FIT_ONE_PACKET);

	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
	
	if (env->error) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send disconnect request. Bailing out");
		goto error;
	}
    
	/* Lets see if it was successfull */
	/*while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}*/
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DISCONNECT_DONE, NULL, NULL);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_obj:
	OBEX_ObjectDelete(env->obexhandle, obj);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void smlTransportObexClientSend(void *userdata, void *link, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, userdata, link, data, error);
	smlAssert(data);
	smlAssert(userdata);
	SmlTransportObexClientEnv *env = userdata;
	smlAssert(env);
	
	if (error)
		goto error;
	
	env->error = FALSE;
	env->mimetype = data->type;
	
	/* Make a new put command */
	obex_object_t *obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_PUT);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new put object");
		goto error;
	}

	/* Now add the header for the put target */
	obex_headerdata_t header;
	
	smlTrace(TRACE_INTERNAL, "Adding connection id %i", env->connection_id);
	
	/* Add the connection id */
	header.bq4 = env->connection_id;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_CONNECTION, header, sizeof(env->connection_id), OBEX_FL_FIT_ONE_PACKET);
	
	const char *target = NULL;
	switch (data->type) {
		case SML_MIMETYPE_WBXML:
			target = SML_ELEMENT_WBXML;
			break;
		case SML_MIMETYPE_XML:
			target = SML_ELEMENT_XML;
			break;
		case SML_MIMETYPE_SAN:
			target = SML_ELEMENT_SAN;
			break;
		default:
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mime type");
			goto error_free_obj;
	}
	smlTrace(TRACE_INTERNAL, "Target %s", target);
	
	/* Now convert to unicode. It requires 2 bytes per char + end*/
	/*unsigned char *unicode = g_malloc0(2 * strlen(target) + 2);

	unsigned int unicodesize = OBEX_CharToUnicode(unicode, (const unsigned char *)target, 2 * strlen(target) + 2);
	if (unicodesize == -1) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "unable to convert to unicode");
		goto error_free_obj;
	}*/
	
	/* Now add the target mime type */
	/*header.bs = (unsigned char *)unicode;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, unicodesize, 0);*/
	header.bs = (unsigned char *)target;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, strlen(target) + 1, 0);
	
	/* Now the data and size */
	header.bq4 = (uint32_t)data->size;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_LENGTH, header, sizeof(uint32_t), 0);
	
	header.bs = (unsigned char *)data->data;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_BODY, header, data->size, 0);
	
	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
    
	/* Lets see if it was successfull */
	while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}
	
	if (env->error) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send put request. Bailing out");
		goto error;
	}
	
	smlTrace(TRACE_INTERNAL, "Done sending the put request");
	
	if (!data->needsAnswer) {
		smlTrace(TRACE_EXIT, "%s: No answer is needed", __func__);
		return;
	}
	
	/* Make a new get command */
	obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_GET);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new put object");
		goto error;
	}

	/* Now add the header for the get target */
	smlTrace(TRACE_INTERNAL, "Adding connection id %i", env->connection_id);
	
	/* Add the connection id */
	header.bq4 = env->connection_id;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_CONNECTION, header, sizeof(env->connection_id), OBEX_FL_FIT_ONE_PACKET);
	
	/* Now add the target mime type */
	//header.bs = (unsigned char *)unicode;
	//OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, unicodesize, 0);
	header.bs = (unsigned char *)target;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, strlen(target) + 1, 0);
	
	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
    
	/* Lets see if it was successfull */
	while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_obj:
	OBEX_ObjectDelete(env->obexhandle, obj);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	return;
}

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
SmlBool smlTransportObexClientNew(SmlTransport *tsp, SmlError **error)
{
	tsp->functions.initialize = smlTransportObexClientInit;
	tsp->functions.finalize = smlTransportObexClientFinalize;
	tsp->functions.connect = smlTransportObexClientConnect;
	tsp->functions.disconnect = smlTransportObexClientDisconnect;
	tsp->functions.send = smlTransportObexClientSend;
	return TRUE;
}

#endif //ENABLE_OBEX

/*@}*/
