/*
 * Model a collection of derived tags and a TagcollFilter to add them to a
 * collection
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include <tagcoll/DerivedTags.h>

#include <tagcoll/tagexpr/TagexprParser.h>

#include <tagcoll/stringf.h>

using namespace std;
using namespace Tagcoll;

//#define DEBUG_PARSER 1
//#define DEBUG_EXPAND 1

#if DEBUG_PARSER || DEBUG_EXPAND

#include "debug.inc"

#endif

DerivedTagList::~DerivedTagList() throw ()
{
	for (expressions_t::iterator i = expressions.begin();
			i != expressions.end(); i++)
		delete i->second;
}

void DerivedTagList::add(const string& tag, const Tagexpr* expr) throw ()
{
	expressions.insert(make_pair(tag, expr->clone()));
}
	

// DerivedTag all the tags in a tag set
OpSet<string> DerivedTagList::getDerived(const OpSet<string>& tags) const throw ()
{
	OpSet<string> res;
	for (expressions_t::const_iterator i = expressions.begin();
			i != expressions.end(); i++)
	{
		TagexprContext context(tags, expressions);
#ifdef DEBUG_EXPAND
		string expr = i->second->format();
		fprintf(stderr, " Matching %.*s to ", PFSTR(expr));
		printTagset(stderr, tags);
		bool val = i->second->eval(context);
		fprintf(stderr, ": %s\n", val ? "true" : "false");
		if (val)
			res += i->first;
#else
		if (i->second->eval(context))
			res += i->first;
#endif
	}
#ifdef DEBUG_EXPAND
	fprintf(stderr, "Result of expansion of ");
	printTagset(stderr, tags);
	fprintf(stderr, " is: ");
	printTagset(stderr, res);
	fprintf(stderr, "\n");
#endif
	return res;
}

void DerivedTagList::parse(ParserInput& in) throw (ParserException)
{
	string tag;
	string expr;

	int c;
	enum {TAG, TAGCOLON, SEXPR, EXPR} state = TAG;
	int line = 1;
	while ((c = in.nextChar()) != ParserInput::Eof)
	{
		if (c == '\n')
		{
			if (tag.size() > 0 && expr.size() > 0)
			{
				Tagexpr* tagexpr = TagexprParser::instance()->parse(expr);
				if (tagexpr)
				{
#ifdef DEBUG_PARSER
					string e = tagexpr->format();
					fprintf(stderr, "%d: %.*s -> %.*s\n", line, PFSTR(tag), PFSTR(e));
#endif
					expressions[tag] = tagexpr;
				}
				else
					throw ParserException(in, TagexprParser::instance()->getErrorMessage());
			} else
				fprintf(stderr, "In derived tags file, ignoring incomplete line %d.\n", line);
			tag = string();
			expr = string();
			state = TAG;
			line++;
		} else
			switch (state)
			{
				// Read item
				case TAG:
					switch (c)
					{
						case ':':
							state = TAGCOLON;
							break;
						default:
							tag += c;
							break;
					}
					break;
				// After colon on item
				case TAGCOLON:
					switch (c)
					{
						case ' ':
						case '\t':
							state = SEXPR;
							break;
						case ':':
							tag += c;
							break;
						default:
							tag += ':';
							tag += c;
							state = EXPR;
							break;
					}
					break;
				// Space before tag
				case SEXPR:
					switch (c)
					{
						case ' ':
						case '\t':
							break;
						default:
							expr += c;
							state = EXPR;
							break;
					}
					break;
				// Read tag
				case EXPR:
					expr += c;
					break;
			}
	}
}

// vim:set ts=4 sw=4:
