/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: io.c,v 1.11.2.4 2003/05/27 17:18:56 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/io.h>
#include <zorp/log.h>
#include <zorp/freeq.h>

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <fcntl.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <netinet/tcp.h>
#  include <netinet/in.h>
#endif

/**
 * z_fd_set_nonblock:
 * @fd: file descriptor to change
 * @enable: specifies whether to enable or disable O_NONBLOCK
 *
 * This function enables or disables the non-blocking mode of operation on
 * the given fd.
 *
 * Returns: whether the operation was successful
 **/
gboolean 
z_fd_set_nonblock(int fd, gboolean enable)
{
#ifdef G_OS_WIN32

  /* Note: this assumes the fd is a socket. */
  unsigned long argp;
  argp = enable;
  if (ioctlsocket(fd, FIONBIO, &argp) == SOCKET_ERROR)
#else
  int flags;

  if ((flags = fcntl(fd, F_GETFL)) == -1)
    return FALSE;

  if (enable)
    flags |= O_NONBLOCK;
  else
    flags &= ~O_NONBLOCK;
    
  if (fcntl(fd, F_SETFL, flags) < 0)
#endif
    {
      /*LOG
        Indicates that changing the blocking behaviour for the given
        fd failed.
       */
      z_log(NULL, CORE_ERROR, 4, "Changing blocking mode failed; fd='%d', enable='%d', error='%m'", fd, enable);
      return FALSE;
    }
  return TRUE;
}

/**
 * z_fd_set_keepalive:
 * @fd: file descriptor of a socket
 * @enable: whether to enable or disable TCP keepalive
 *
 * This function enables or disables the TCP keepalive feature for socket
 * specified by @fd.
 *
 * Returns: whether the operation was successful
 **/
gboolean
z_fd_set_keepalive(int fd, gboolean enable)
{
  if (setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (char *)(&enable), sizeof(enable)) == -1)
    {
      /*LOG
        Indicates that changing the keep-alive socket option failed for
        the given reason.
       */
      z_log(NULL, CORE_ERROR, 4, "setsockopt(SOL_SOCKET, SO_KEEPALIVE) failed; fd='%d', enable='%d', error='%m'", fd, enable);
      return FALSE;
    }
  return TRUE;
}

/**
 * z_fd_set_keepalive:
 * @fd: file descriptor of a socket
 * @enable: whether to enable or disable TCP SO_OOBINLINE
 *
 * This function enables or disables the TCP SO_OOBINLINE feature for socket
 * specified by @fd.
 *
 * Returns: whether the operation was successful
 **/
gboolean
z_fd_set_oobinline(int fd, gboolean enable)
{
  if (setsockopt(fd, SOL_SOCKET, SO_OOBINLINE, (char *)(&enable), sizeof(enable)) == -1)
    {
      /*LOG
        This message indicates that changing the SO_OOBINLINE socket option
        failed for the given reason.
       */
      z_log(NULL, CORE_ERROR, 4, "setsockopt(SOL_SOCKET, SO_OOBINLINE) failed; fd='%d', enable='%d', error='%m'", fd, enable);
      return FALSE;
    }
  return TRUE;
}
