/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: streambuf.c,v 1.28.2.3 2003/05/28 12:42:53 sasa Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/streambuf.h>
#include <zorp/stream.h>
#include <zorp/log.h>
#include <zorp/ssl.h>
#include <zorp/zorplib.h>

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <assert.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <sys/socket.h>
#  include <sys/poll.h>
#endif

#define MAX_BUF_LEN 4096

typedef struct _ZStreamBufBuf
{
  gchar *buf;
  guint len, pos;
} ZStreamBufBuf;

typedef struct _ZStreamBuf
{
  ZStream super;

  guint min_threshold;
  guint max_threshold;
  gboolean write_locked;
  
  GList *buffers;
  GMutex *buffer_lock;
  
  stream_buf_writeable writeable_callback;
  stream_buf_error error_callback;
  gpointer user_data;
  
} ZStreamBuf;

static void z_stream_buf_flush(ZStreamBuf *self);

static GIOStatus
z_stream_buf_read_method(ZStream  *stream,
                           gchar  *buf,
                           gsize   count,
                           gsize  *bytes_read,
                          GError **error)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;
  GIOStatus res;

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  self->super.parent->timeout = self->super.timeout;
  res = z_stream_read(self->super.parent, buf, count, bytes_read, error);

  z_leave();
  return res;
}

/* FIXMEE
  Ez most kisse broken, mivel atlatszoan nem lehet
  stack-elni mas stream "ala".
  Streambuf-ot csak a z_stream_write_buf -al szabad irni!!!
  Gondolkozni kellene egy korrektebb megoldason.
static GIOStatus
z_stream_buf_write_method(ZStream  *stream G_GNUC_UNUSED,
                       const gchar  *buf G_GNUC_UNUSED,
                             gsize   count G_GNUC_UNUSED,
                             gsize  *bytes_written G_GNUC_UNUSED,
                            GError **error G_GNUC_UNUSED)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;
  gboolean ret;
  GList *list;
  ZStreamBufBuf *streambuf;

  assert(0);

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);
  
  self->super.parent->timeout = self->super.timeout;
  
  ret = z_stream_write_buf(stream, (gchar *)buf, count, TRUE, FALSE);
  
  if (ret == G_IO_STATUS_NORMAL)
    {
      *bytes_written = count;
      z_leave();
      return G_IO_STATUS_NORMAL;
    }
  
  list = g_list_last(self->buffers);
  
  streambuf = (ZStreamBufBuf *) list->data;
  
  if (streambuf->buf == buf && streambuf->pos == 0)
    {
      self->buffers = g_list_delete_link(self->buffers, list);
    }
  z_leave();
  return G_IO_STATUS_AGAIN;
}
*/

static GIOStatus
z_stream_buf_shutdown_method(ZStream *stream, int i, GError **error)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;
  GIOStatus res;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  z_stream_set_nonblock(self->super.parent, FALSE);
  z_stream_buf_flush(self);
  res = z_stream_shutdown(self->super.parent, i, error);
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_buf_close_method(ZStream *stream, GError **error)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;
  GIOStatus res;

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  res = z_stream_close(self->super.parent, error);
  
  z_leave();
  return res;
}

static void
z_stream_buf_flush(ZStreamBuf *self)
{
  ZStreamBufBuf *buf;
  guint i = 10;
  guint write_len;
  GIOStatus res = G_IO_STATUS_NORMAL;
  z_enter();
  
  g_mutex_lock(self->buffer_lock);
  while (self->buffers && i && res == G_IO_STATUS_NORMAL)
    {
      buf = (ZStreamBufBuf *) self->buffers->data;
      res = z_stream_write(self->super.parent,
                           buf->buf + buf->pos,
                           buf->len - buf->pos,
                           &write_len,
                           NULL);
      if (res == G_IO_STATUS_NORMAL)
        {
          buf->pos += write_len;
          if (buf->pos >= buf->len)
            {
              g_free(buf->buf);
              g_free(buf);
              self->buffers = g_list_delete_link(self->buffers, self->buffers);
            }
        }
      else if (res != G_IO_STATUS_AGAIN)
        {
          z_cp();
          if (self->error_callback)
            {
              z_cp();
              self->error_callback(&self->super, res, self->super.user_data_write);
              z_cp();
            }
        }
      i--;
    }
  
  if (res == G_IO_STATUS_AGAIN ||
      (res == G_IO_STATUS_NORMAL && i == 0))
    {
      z_cp();
      z_stream_set_cond(self->super.parent, Z_STREAM_FLAG_WRITE, TRUE);
    }

  z_cp();  
  if (self->write_locked && g_list_length(self->buffers) < self->min_threshold)
    {
      z_cp();
      if (self->writeable_callback)
        {
          z_cp();
          self->writeable_callback(&self->super, self->super.user_data_write);
        }
      self->write_locked = FALSE;
    }
  
  z_cp();  
  if (!self->write_locked &&
       g_list_length(self->buffers) > self->max_threshold)
    {
      z_cp();
      self->write_locked = TRUE;
    }
  if (!self->buffers)
    z_stream_set_cond(self->super.parent, Z_STREAM_FLAG_WRITE, FALSE);
  
  g_mutex_unlock(self->buffer_lock);
  
  z_leave();
}

GIOStatus
z_stream_write_buf(ZStream *stream,
                     gchar *buf,
                     guint  buflen,
                  gboolean  must_copy,
                  gboolean  urgent)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;
  gchar *tmpbuf;
  ZStreamBufBuf *stream_buf = g_new0(ZStreamBufBuf, 1);
  gboolean res;
  
  z_enter();
  
  while (self && z_stream_get_type((ZStream *) self) != ZST_BUF)
    self = (ZStreamBuf *) self->super.parent;
  
  if (!self)
    {
      /*LOG
        This message indicate by an internal error.
       */
      z_log(NULL, CORE_ERROR, 2, "Internal error; reason='Bad stream type'");
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  
  assert(g_list_length(self->buffers) < MAX_BUF_LEN);
  
  if (must_copy)
    {
      tmpbuf = g_new(char, buflen);
      memcpy(tmpbuf, buf, buflen);
    }
  else
    tmpbuf = buf;
      
  stream_buf->buf = tmpbuf;
  stream_buf->len = buflen;
  
  g_mutex_lock(self->buffer_lock);
  if (urgent)
    self->buffers = g_list_prepend(self->buffers, stream_buf);
  else
    self->buffers = g_list_append(self->buffers, stream_buf);
  g_mutex_unlock(self->buffer_lock);

  z_stream_buf_flush(self);
  
  if (self->write_locked)
    res = G_IO_STATUS_AGAIN;
  else
    res = G_IO_STATUS_NORMAL;
  z_leave();
  return res;
}

static gboolean
z_stream_buf_read_callback(ZStream *stream, GIOCondition poll_cond, gpointer s)
{
  ZStreamBuf *self = (ZStreamBuf *) s;
  gboolean rc;

  z_enter();

  rc = (*self->super.read_cb)(s, poll_cond, self->super.user_data_read);
  
  z_leave();
  return rc;
}

static gboolean
z_stream_buf_write_callback(ZStream *stream, GIOCondition poll_cond, gpointer s)
{
  ZStreamBuf *self = (ZStreamBuf *) s;
  gboolean rc = TRUE;
  
  z_enter();
  
  z_stream_buf_flush(self);
  
  if (self->super.want_write && self->super.write_cb)
    {
      if (!self->write_locked)
        rc = (*self->super.write_cb)(s, poll_cond, self->super.user_data_write);
    }
  
  z_leave();
  return rc;
}

static gboolean
z_stream_buf_pri_callback(ZStream *stream, GIOCondition poll_cond, gpointer s)
{
  ZStreamBuf *self = (ZStreamBuf *) s;
  gboolean rc;

  z_enter();
  
  rc = (*self->super.pri_cb)(s, poll_cond, self->super.user_data_pri);
  
  z_leave();
  return rc;
}

static gboolean
z_stream_buf_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen)
{
  gboolean ret;
  
  assert(s->type == ZST_BUF);
    
  z_enter();
  switch (ZST_CTRL_MSG(function))
    {
    case ZST_CTRL_SET_COND_WRITE:
      z_stream_ctrl(s->parent, function, value, vlen);
      return TRUE;
    case ZST_CTRL_SET_CALLBACK_READ:
    case ZST_CTRL_SET_CALLBACK_WRITE:
    case ZST_CTRL_SET_CALLBACK_PRI:
      ret = z_stream_ctrl_method(s, function, value, vlen);
      break;
    default:
      ret = z_stream_ctrl_method(s, ZST_CTRL_MSG_FORWARD | function, value, vlen);
      break;
    }
                  
  z_leave();
  return ret;
}
                      

static void
z_stream_buf_free(ZStream *stream)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;

  z_enter();
/*
 * FIXMEE felszabaditani a buffereket!
 */

  z_stream_unref(self->super.parent);
  g_free(self);
  z_leave();
}

static void
z_stream_buf_attach_source_method(ZStream *stream, GMainContext *context)
{
  ZStreamBuf *self = (ZStreamBuf *) stream;

  z_enter();
  z_stream_attach_source(self->super.parent, context);
  
  z_leave();
  return;
}

static void
z_stream_buf_detach_source_method(ZStream *s)
{
  ZStreamBuf *self = (ZStreamBuf *) s;
  
  z_stream_detach_source(self->super.parent);
}

static gboolean z_stream_buf_watch_prepare(ZStream *s,  
                                           GSource *src,
                                              gint *timeout);

static gboolean z_stream_buf_watch_check(ZStream *s, GSource *src);

static gboolean z_stream_buf_watch_dispatch (ZStream *s,
                                             GSource *src);


ZStreamFuncs z_stream_buf_funcs =
{
  z_stream_buf_read_method,
  NULL,
  NULL,
  NULL,
  z_stream_buf_shutdown_method,
  z_stream_buf_close_method,
  z_stream_buf_ctrl_method,
  z_stream_buf_attach_source_method,
  z_stream_buf_detach_source_method,
  z_stream_buf_watch_prepare,
  z_stream_buf_watch_check,
  z_stream_buf_watch_dispatch,
  NULL,
  z_stream_buf_free 
};


ZStream *
z_stream_buf_new(ZStream *stream,
                   guint  min_threshold,
                   guint  max_threshold,
    stream_buf_writeable  writeable_callback,
        stream_buf_error  error_callback)
{
  ZStreamBuf *self = g_new0(ZStreamBuf, 1);

  z_enter();
  z_stream_init(&self->super, ZST_BUF, &z_stream_buf_funcs, stream->name);
  if (max_threshold > MAX_BUF_LEN / 2)
    max_threshold = MAX_BUF_LEN / 2;

  if (max_threshold < min_threshold)
    {
      z_leave();
      return NULL;
    }
  self->min_threshold = min_threshold;
  self->max_threshold = max_threshold;
  self->writeable_callback = writeable_callback;
  self->error_callback = error_callback;

  self->super.parent = stream;
  self->super.timeout = self->super.parent->timeout;

  z_stream_set_callback(self->super.parent,
                        Z_STREAM_FLAG_READ,
                        z_stream_buf_read_callback, self, NULL);
  z_stream_set_callback(self->super.parent,
                        Z_STREAM_FLAG_WRITE,
                        z_stream_buf_write_callback, self, NULL);
  z_stream_set_callback(self->super.parent,
                        Z_STREAM_FLAG_PRI,
                        z_stream_buf_pri_callback, self, NULL);

  z_stream_ref(self->super.parent);
  self->buffer_lock = g_mutex_new();

  z_leave();
  return (ZStream *) self;
}

static gboolean 
z_stream_buf_watch_prepare(ZStream *s,  
                           GSource *src,
                           gint *timeout)
{
  /*NOLOG*/
  z_log(NULL, CORE_ERROR, 1, "Internal error, calling buf watch_prepare;");
  return FALSE;
}

static gboolean 
z_stream_buf_watch_check(ZStream *s, GSource *src)
{
  /*NOLOG*/
  z_log(NULL, CORE_ERROR, 1, "Internal error, calling buf check;");
  return FALSE;
}

static gboolean 
z_stream_buf_watch_dispatch(ZStream *s,
                            GSource *src)
{
  /*NOLOG*/
  z_log(NULL, CORE_ERROR, 1, "Internal error, calling buf dispatch;");
  return FALSE;
}
