/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: misc.c,v 1.43 2004/06/01 09:14:24 abi Exp $
 *
 * Author  : Bazsi, SaSa, Chaoron
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/zorplib.h>
#include <zorp/misc.h>
#include <zorp/log.h>

#include <string.h>
#include <stdlib.h>

#include <ctype.h>
/*
 * ZCharSet is a simple representation of a character set. It can be parsed
 * from a string representation where regexp like expressions can be used. 
 * Checking whether a given character is in the character set is quite fast
 * as each character is represented as a bit in a bitstring.
 */

#define PARSE_STATE_START  0
#define PARSE_STATE_DASH   1
#define PARSE_STATE_END    2
#define PARSE_STATE_ESCAPE 3

/**
 * z_charset_init:
 * @self: ZCharSet instance to initialize
 *
 * Initialize a ZCharSet by setting clearing the character set bitstring.
 **/
void 
z_charset_init(ZCharSet *self)
{
  memset(self, 0, sizeof(*self));
}

/**
 * z_charset_parse:
 * @self: ZCharSet instance, previously initialized by z_charset_init
 * @interval_str: string representation of the character set
 *
 * This function parses an character set from its string representation.
 **/
gboolean 
z_charset_parse(ZCharSet *self, gchar *interval_str)
{
  guint i = 0;
  guchar j;
  guint state = PARSE_STATE_START;
  guint old_state = PARSE_STATE_START;
  guchar start_pos = 0;
  guchar end_pos = 0;
  
  z_enter();
  while (interval_str[i])
    {
      switch (state)
        {
        case PARSE_STATE_START:
          if (interval_str[i] == '\\' && old_state != PARSE_STATE_ESCAPE)
            {
              z_cp();
              old_state = state;
              state = PARSE_STATE_ESCAPE;
            }
          else
            {
              z_cp();
              start_pos = interval_str[i];
              state = PARSE_STATE_DASH;
              old_state = PARSE_STATE_START;
              i++;
            }
          break;
        case PARSE_STATE_DASH:
          if (interval_str[i] == '\\' && old_state != PARSE_STATE_ESCAPE)
            {
              z_cp();
              state = PARSE_STATE_END;
              i--;
            }
          else
            {
              z_cp();
              state = PARSE_STATE_END;
              old_state = PARSE_STATE_DASH;
              if (interval_str[i] == '-')
                i++;
              else
                i--;
            }
          break;
        case PARSE_STATE_END:
          if (interval_str[i] == '\\' && old_state != PARSE_STATE_ESCAPE)
            {
              z_cp();
              old_state = state;
              state = PARSE_STATE_ESCAPE;
            }
          else
            {
              z_cp();
              end_pos = interval_str[i];
              for (j = start_pos; j <= end_pos; j++)
                z_charset_enable(self, j);
              
              i++;
              state = PARSE_STATE_START;
              old_state = PARSE_STATE_END;
            }
          break;
        case PARSE_STATE_ESCAPE:
          z_cp();
          i++;
          state = old_state;
          old_state = PARSE_STATE_ESCAPE;
          break;
        default:
          z_leave();
          return FALSE;
        }
    }

  if (state == PARSE_STATE_DASH)
    {
      z_cp();
      z_charset_enable(self, start_pos);
      state = PARSE_STATE_START;
    }
  
  if (state == PARSE_STATE_START)
    {
      z_leave();
      return TRUE;
    }
  
  z_leave();
  return FALSE;
}

/**
 * z_charset_is_string_valid:
 * @self: ZCharSet instance
 * @str: string to check
 * @len: string length
 *
 * This function checks whether the given string contains valid characters only.
 *
 * Returns: when the string contained valid characters only
 **/
gboolean 
z_charset_is_string_valid(ZCharSet *self, gchar *str, gint len)
{
  gint i;
  
  if (len < 0)
    len = strlen(str);
  
  for (i = 0; i < len; i++)
    {
      if (!z_charset_is_enabled(self, str[i]))
        return FALSE;
    }
  return TRUE;
}

/**
 * g_string_assign_len:
 * @s: GString instance
 * @val: string pointer
 * @len: length of string
 *
 * This function assigns the given string/length value to the specified GString.
 * This function should be defined in GLib, however no such function exists.
 *
 * Returns: the GString instance 
 **/
GString *
g_string_assign_len(GString *s, gchar *val, gint len)
{
  g_string_truncate(s, 0);
  if (val && len)
    g_string_append_len(s, val, len);
  return s;
}

/**
 * z_data_dump:
 * @session_id: session id to be used for the log messages
 * @buf: buffer
 * @len: buffer length
 *
 * This function generates hexdumps of the specified buffer to the system
 * log using z_log().
 **/
void 
z_data_dump(char *session_id, const char *buf, guint len)
{
  guint i, j;
  gchar line[1024];
  
  i = 0;
  while (i < len)
    {
      char *end = line;
      
      for (j = 0; j < 16 && (i + j < len); j++)
        {
          g_snprintf(end, sizeof(line) - (end - line), "%02X ", (unsigned char) buf[i+j]);
          end += 3;
        }
      g_snprintf(end, sizeof(line) - (end - line), " ");
      end++;
      for (j = 0; j < 16 && (i + j < len) && sizeof(line) > (guint)(end - line); j++)
        {
          *end = buf[i + j] > 31 ? buf[i + j] : '.';
          end++;
        }
      *end='\0';
          
      i += j;
      
      /*NOLOG*/
      z_log(session_id, CORE_DUMP, 9, "data line: %s", line);
    }
}

/**
 * z_format_data_dump:
 * @session_id: session id to be used for the log messages
 * @buf: buffer
 * @len: buffer length
 *
 * This function generates hexdumps of the specified buffer to the system
 * log using z_log().
 **/
void
z_format_data_dump(gchar *session_id, char *class, gint level, const char *buf, guint len)
{
  guint i, j;
  gchar line[1024];

  i = 0;
  while (i < len)
    {
      char *end = line;

      for (j = 0; j < 16 && (i + j < len); j++)
        {
          g_snprintf(end, sizeof(line) - (end - line), "%02X ", (unsigned char) buf[i+j]);
          end += 3;
        }
      g_snprintf(end, sizeof(line) - (end - line), " ");
      end++;
      for (j = 0; j < 16 && (i + j < len) && sizeof(line) > (guint)(end - line); j++)
        {
          *end = buf[i + j] > 31 ? buf[i + j] : '.';
          end++;
        }
      *end='\0';

      i += j;

      /*NOLOG*/
      z_log(session_id, class, level, "data line: %s", line);
    }
}

gchar *
z_str_escape(const gchar *s, gint len)
{
  gchar *res;
  gint i = 0, j = 0;;
  
  z_enter();
  
  if (len < 0)
    len = strlen(s) + 1;
  
  res = g_new0(gchar, len * 2);
  
  while (i < len && s[i] != '\0')
    {
      switch (s[i])
        {
	  case ' ':
	    res[j++] = '%';
            res[j++] = '_';
	    break;
          
	  case '%':
	    res[j++] = '%';
	    res[j++] = '%';
	    break;
	  
	  default:
	    res[j++] = s[i];
	}
      i++;
    }
  
  z_leave();
  return res;
}

gchar *
z_str_compress(const gchar *s, gint len)
{
  gchar *res;
  gint i = 0, j = 0;;
  
  z_enter();
  
  if (len < 0)
    len = strlen(s) + 1;
  
  res = g_new0(gchar, len);
  
  while (i < len && s[i] != '\0')
    {
      if (s[i] == '%' && s[i+1] == '%')
        {
	  i++;
	  res[j++] = '%';
	}
      else if (s[i] == '%' && s[i+1] == '_')
        {
	  i++;
	  res[j++] = ' ';
	}
      else
        {
	  res[j++] = s[i];
	}
      i++;
    }
  
  z_leave();
  return res;
}

/**
 * z_port_enabled:
 * @port_range: port range specification
 * @port: port number to check
 *
 * This function parses the given port_range and returns TRUE to indicate
 * whether the specified port number is valid. The port specification is in
 * the format: port1[-port2]([,port1[-port2]])*
 *
 * Returns: TRUE when the port number is valid
 **/
gboolean
z_port_enabled(gchar *port_range, guint port)
{
  long int portl, porth;
  gchar *tmp;
  gchar *err;
  
  if (strlen(port_range) == 0 )
    return FALSE;
  
  tmp = port_range;
  while (*tmp)
    {
      portl = strtol(tmp, &err, 10);
      tmp = err;
      if (*tmp == '-')
        {
          porth = strtol(tmp + 1, &err, 10);
          tmp = err;
        }
      else
        {
          porth = portl;
        }

      if (*tmp != 0 &&  *tmp != ',')
        return FALSE;
    
      if (*tmp)
        {
          tmp++;
          if (*tmp <= '0' && *tmp >= '9')
          return FALSE;
        }
        
      if ( portl <= (long int)port && (long int)port <= porth )
        return TRUE;

    }
  return FALSE;
}

/**
 * z_casestr_equal:
 * @k1: key1 to compare
 * @k2: key2 to compare
 * 
 * GLib hashtable equal function for case insensitive hashtables.
 **/
gboolean
z_casestr_equal(gconstpointer k1, gconstpointer k2)
{
  return g_strcasecmp(k1, k2) == 0;
}

/**
 * z_casestr_hash:
 * @key:
 *
 * GLib hashtable hash function for case insensitive hashtables.
 **/
guint
z_casestr_hash(gconstpointer key)
{
  const char *p = key;
  guint h = toupper(*p);

  if (h)
    for (p += 1; *p != '\0'; p++)
      h = (h << 5) - h + toupper(*p);

  return h;
}


#define ON_OFF_STR(x) (x ? "on" : "off")

/**
 * z_zorplib_version_info:
 * 
 * This function returns a static character string which includes version
 * information and compilation settings. This function is _NOT_ reentrant.
 *
 * Returns: a static string of the version information to be displayed to
 * the user
 **/
gchar *
z_zorplib_version_info(void)
{
  static gchar buf[128];
  
  g_snprintf(buf, sizeof(buf),
             "Zorplib %s\n"
             "Revision: %s\n"
             "Compile-Date: %s %s\n"
             "Trace: %s\n"
             "MemTrace: %s\n"
             "Caps: %s\n"
             "Debug: %s\n"
             "StackDump: %s\n",

             ZORPLIBLL_VERSION, ZORPLIBLL_REVISION, __DATE__, __TIME__,
             ON_OFF_STR(ZORPLIB_ENABLE_TRACE),
             ON_OFF_STR(ZORPLIB_ENABLE_MEM_TRACE),
             ON_OFF_STR(ZORPLIB_ENABLE_CAPS),
             ON_OFF_STR(ZORPLIB_ENABLE_DEBUG),
             ON_OFF_STR(ZORPLIB_ENABLE_STACKDUMP));
  return buf;
}

