/*
 *  VME Linux/m68k TFTP Loader
 *
 *  (c) Copyright 1998 by Nick Holgate
 *
 *  This file is subject to the terms and conditions of the GNU General
 *  Public License.  See the file COPYING for more details.
 */

#include <stdarg.h>

/* linux specific include files */
#include <linux/a.out.h>
#include <linux/elf.h>
#include <linux/linkage.h>
#include <asm/page.h>

#include "version.h"
#include "bootinfo.h"
#include "stream.h"

/*--------------------------------------------------------------------------*/
#ifndef NULL
#define NULL				((void *)0)
#endif
#define TRUE				1
#define FALSE				0
#define FAILURE				-1
#define SUCCESS				0

#define NO_TIMEOUT			0

#define CR					13
#define LF					10

#define STACK_SIZE			8192

#define PAGE_BOUNDARY(n)	(((n) + PAGE_SIZE - 1) & -PAGE_SIZE)

#define BCD2BIN(x)			((((x) >> 4) * 10) + ((x) & 0x0f))

#define endof(s)			(&((char *)(s))[strlen((char *)(s)) - 1])

/*--------------------------------------------------------------------------*/

#define SEEK_SET 	0
#define SEEK_CUR 	1
#define SEEK_END 	2

/*--------------------------------------------------------------------------*/

typedef enum {
	IP_CLIENT,
	IP_SERVER,
	IP_GATEWAY,
	IP_BROADCAST,
	IP_NETMASK

} IPTYPE;

/*--------------------------------------------------------------------------*/
/* _tail is a linker defined symbol used to mark the end of memory
 * used by the loader. See ?vme.ld
 */

extern char _tail;
#define	free_mem_start		(((unsigned long)&_tail + 3) & ~3)

/*--------------------------------------------------------------------------*/

typedef struct boot_tag {
	struct boot_tag	*next;				/* next boot configuration			*/
	char			*label;				/* configuration name				*/
	char			*alias;				/* alternative name					*/
	char			*description;		/* description message				*/
	char			*kernel;			/* Linux kernel file name			*/
	char			*ramdisk;			/* ramdisk image file name			*/
	char			*cmdline;			/* kernel command line				*/
	char			*append;			/* always append this to cmdline	*/
	char			*root;				/* device to mount as root			*/
	char			*console;			/* console device					*/
	char			read_only;			/* flag to mount root read only		*/
	char			restricted;			/* if true don't prompt for cmdline	*/
	char			calldbg;			/* call debugger before kernel		*/
	unsigned long	memsize;			/* memory size						*/

} BOOT;

/*--------------------------------------------------------------------------*/

typedef struct crate_tag {
	unsigned long	timeout;			/* boot: prompt input timeout (secs)*/
	unsigned long	delay;				/* pause before auto boot (secs)	*/
	int				prompt;				/* always prompt for input if TRUE	*/
	char			*display;			/* name of display section label	*/
	char			*kernel;			/* Linux kernel file name			*/
	char			*ramdisk;			/* ramdisk image file name			*/
	char			*cmdline;			/* kernel command line				*/
	char			*append;			/* always append this to cmdline	*/
	char			*root;				/* device to mount as root			*/
	char			*console;			/* console device					*/
	char			read_only;			/* flag to mount root read only		*/
	char			restricted;			/* if true don't prompt for cmdline	*/
	char			calldbg;			/* call debugger before kernel		*/
	unsigned long	memsize;			/* memory size						*/
	BOOT			*boot_dflt;			/* default boot configuration		*/ 
	BOOT			*boot_list;			/* list of boot configurations		*/

} CRATE;

/*--------------------------------------------------------------------------*/
/* bootinfo (internal format)
 */

struct internal_bootinfo
{
	unsigned long	machtype;			/* machine type						*/
	unsigned long	cputype;			/* system CPU						*/
	unsigned long	fputype;			/* system FPU						*/
	unsigned long	mmutype;			/* system MMU						*/
	int				num_memory;			/* # of memory blocks found			*/
	struct mem_info memory[NUM_MEMINFO];/* memory description				*/
	struct mem_info ramdisk;			/* ramdisk description				*/
	char command_line[CL_SIZE];			/* kernel command line parameters	*/
};

/*--------------------------------------------------------------------------*/
/* image of machine registers passed in from boot roms
 */

typedef struct {
	unsigned long d[8];
	void		 *a[7];

} CALLREGS;

/*--------------------------------------------------------------------------*/
/* declarations of stream modules
 */

extern MODULE memory_mod;
extern MODULE gunzip_mod;

/*--------------------------------------------------------------------------*/

#ifndef GLOBAL
#define GLOBAL	extern
#endif

GLOBAL void						(*startcode_entry)(void);
GLOBAL char						*config_data;
GLOBAL char						*config_data_end;
GLOBAL char						input_buffer[256];
GLOBAL unsigned long			detected_mem_size;
GLOBAL int						debug_mode;
GLOBAL int						cpu_type;
GLOBAL int						fpu_type;
GLOBAL CALLREGS					callregs;
GLOBAL struct internal_bootinfo	bi;

#ifdef BOOTINFO_COMPAT_1_0
GLOBAL struct compat_bootinfo	compat_bootinfo;
#endif /* BOOTINFO_COMPAT_1_0 */

#define MAX_BI_SIZE				(4096)
GLOBAL	unsigned long			bi_size;
GLOBAL	union {
			struct bi_record	record;
			unsigned char		fake[MAX_BI_SIZE];
		} bi_union;

/*--------------------------------------------------------------------------*/

/* prototypes for file "boot.c" */
int check_bootinfo_version (const char *memptr);
int add_bi_record (unsigned short tag,unsigned short size,const void *data);
int add_bi_string (unsigned short tag,const unsigned char *s);
int create_bootinfo (void);
int create_compat_bootinfo (void);
void start_kernel (unsigned long kernel_dest_addr,char *kernel_load_addr,unsigned long kernel_mem_size,unsigned long ramdisk_dest_addr,char *ramdisk_load_addr,unsigned long ramdisk_mem_size,int calldbg);
char * load_kernel (char *file_name,unsigned long mem_start,unsigned long *kernel_size,unsigned long bootinfo_size);
void boot_linux (BOOT *boot,const char *more,int autoboot);

/* prototypes for file "?vme.c" */
void loader_init (CALLREGS *regs);
void put_char (const int c);
unsigned long get_time (void);
int get_char (unsigned long timeout);
const char * get_ip (IPTYPE type);
const char * config_filename (void);
int tftp_read (const char *filename,unsigned long count,void *buffer);
unsigned long get_compat_booti_version (void);
unsigned long get_booti_version (void);
unsigned long get_compat_machtype (void);
unsigned long get_machtype (void);
void print_model (void);
unsigned long get_vme_type (void);
int add_vme_bootinfo (void);

/* prototypes for file "config.c" */
int keyword (const char *name);
char * goto_line (int line);
int print_display_section (const char *label);
int find_boot_section (const char *label);
int find_crate_section (const char *search);
int get_number (const char *p,unsigned long *result);
int get_bool (const char *text);
void config_error (int line,char *data);
int bool_config_error (int line,char *data,int willuse);
unsigned long number_config_error (int line,char *data,unsigned long willuse);
void ignore_option (int line,char *data);
void duplicate_option (int line,char *data);
BOOT * find_boot_config (BOOT *boot,char *bootname);
BOOT * add_boot (CRATE *crate,BOOT *boot);
void dump_crate (CRATE *crate);
BOOT * parse_boot (int line,CRATE *crate);
void build_boot_from_defaults (CRATE *crate);
void build_boot_list (CRATE *crate);
void parse_globals (CRATE *crate);
void parse_config (CRATE *crate);

/* prototypes for file "main.c" */
int read_line (unsigned long timeout);
void list_records (CRATE *crate);
BOOT * get_boot_record (CRATE *crate,const char **more);
void start_loader (void);
void Main (CALLREGS *regs);

/* prototypes for file "misc.c" */
void mem_clear (void *mem,unsigned long count);
void mem_move (void *dest,const void *srce,unsigned long count);
void *memcpy (void *dest,const void *srce,unsigned long count);
int mem_cmp (void *mem1,void *mem2,unsigned long count);
void malloc_init (void *heap_base,unsigned long heap_size);
int free (void *memaddr);
void * do_malloc (unsigned long size,int high);
void * malloc (unsigned long size);
void * malloc_low (unsigned long size);
unsigned long maxfree (void);
void * move_up (void *memaddr,unsigned long memsize);
void * realloc (void *memaddr,unsigned long size);
unsigned long disable_icache (void);
unsigned long enable_icache (void);
void invalidate_icache (void);
void erase (int n);
int strlen (const char *s);
char * strcpy (char *d,const char *s);
void strcat (char *d,const char *s);
void strcatn (char *d,const char *s,unsigned long n);
void strncpy (char *d,const char *s,unsigned long n);
int vsoutput (void (*output)(const int),const char *fmt,va_list va);
int sprintf (char *buff,const char *fmt,...);
int printf (const char *fmt,...);
int put_estr (const char *str);
int put_str (const char *str);
void panic (const char *fmt,...);
void * xmalloc (int size);
char * skip_white (const char *p);
void trim_white (char *p);
char * despace (char *str);
int equal_strings (const char *s1,const char *s2);
char * extract (char *p1,char *p2,int len,int delim);
int prescan (char *data,int size);
int check_arch (char *text);
char * mkinet (unsigned long ip);
unsigned atou (const char **strp);
int compare_ip (const char *ipstr,const char *pattern);
void substip (char *dest,const char *srce,int maxlen);
char * tftp (char *filename,unsigned long *filesize);
int get_cpu_type (void);
int get_fpu_type (void);
int ram_probe (unsigned long where);

/* prototypes for file "stream.c" */
void stream_init(void);
void stream_push(MODULE *mod);
int sopen(void *data,long size);
long sread(void *buf,long cnt);
int sseek(long offset,int whence);
int sclose(void);

/*-----------------------------< end of file >------------------------------*/
