# Tests for the SyncDaemon interface
#
# Author: Facundo Batista <facundo@taniquetil.com.ar>
#
# Copyright 2010 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the SyncDaemon communication backend."""

import logging
import os
import unittest

from magicicada.dbusiface import PublicFilesData, ShareOperationError
from magicicada.syncdaemon import (
    ASKING_IDLE,
    mandatory_callback,
    State,
    SyncDaemon,
)
from magicicada.tests.helpers import MementoHandler

from twisted.trial.unittest import TestCase as TwistedTestCase
from twisted.internet import defer, reactor


# It's ok to access private data in the test suite
# pylint: disable=W0212

# Lambda may not be necessary
# pylint: disable=W0108


class FakeDBusInterface(object):
    """Fake DBus Interface, for SD to not use dbus at all during tests."""

    fake_sd_started = False
    fake_pf_data = PublicFilesData(volume='v', node='n',
                                   path='p', public_url='u')
    fake_share_response = None

    def __init__(self, sd):
        pass

    def shutdown(self):
        """Fake shutdown."""

    def get_status(self):
        """Fake status."""
        return defer.succeed(('fakename', 'fakedescrip', False, True,
                              False, 'fakequeues', 'fakeconnection'))

    def get_folders(self):
        """Fake folders."""
        return defer.succeed('fakedata')

    get_content_queue = get_meta_queue = get_folders
    start = quit = connect = disconnect = get_folders
    get_shares_to_me = get_shares_to_others = get_folders

    def get_public_files(self):
        """Fake public files."""
        return defer.succeed([self.fake_pf_data])

    def is_sd_started(self):
        """Fake response."""
        return self.fake_sd_started

    def accept_share(self, share_id):
        """Fake accept share."""
        return self.fake_share_response


class BaseTest(TwistedTestCase):
    """Base test with a SD."""

    timeout = 1

    def setUp(self):
        """Set up."""
        self.hdlr = MementoHandler()
        self.hdlr.setLevel(logging.DEBUG)
        logger = logging.getLogger('magicicada.syncdaemon')
        logger.addHandler(self.hdlr)
        logger.setLevel(logging.DEBUG)
        self.sd = SyncDaemon(FakeDBusInterface)

    def tearDown(self):
        """Tear down."""
        self.sd.shutdown()


class MandatoryCallbackTests(BaseTest):
    """Tests for the mandatory callback generic function."""

    def test_log_function_name(self):
        """Log the function name."""
        some_function = mandatory_callback('bar')
        some_function()
        self.assertTrue(self.hdlr.check_warning(
                        "Callback called but was not assigned", "bar"))

    def test_log_args(self):
        """Log the arguments."""
        some_function = mandatory_callback('bar')
        some_function(1, 2, b=45)
        self.hdlr.debug = True
        self.assertTrue(self.hdlr.check_warning(
                        "Callback called but was not assigned",
                        "1", "2", "'b': 45"))


class InitialDataTests(unittest.TestCase):
    """Tests for initial data gathering."""

    def setUp(self):
        """Set up the test."""
        self.sd = SyncDaemon(FakeDBusInterface)

        self.offline_called = False
        self.sd.on_initial_data_ready_callback = lambda: setattr(self,
                                                        'offline_called', True)
        self.online_called = False
        self.sd.on_initial_online_data_ready_callback = lambda: setattr(self,
                                                        'online_called', True)

    def tearDown(self):
        """Tear down the test."""
        self.sd.shutdown()

    def test_called_by_start(self):
        """Check that start calls get initial data."""
        sd = SyncDaemon(FakeDBusInterface)
        called = []
        sd._get_initial_data = lambda: called.append(True)
        sd.start()
        self.assertTrue(called)

    def test_called_by_nameownerchanged_no(self):
        """Check that it is called when discover that sd started."""
        sd = SyncDaemon(FakeDBusInterface)
        called = []
        sd._get_initial_data = lambda: called.append(True)
        sd.on_sd_name_owner_changed(True)
        self.assertTrue(called)

    def test_called_by_nameownerchanged_yes(self):
        """Check that it is not called when discover that sd stopped."""
        sd = SyncDaemon(FakeDBusInterface)
        called = []
        sd._get_initial_data = lambda: called.append(True)
        sd.on_sd_name_owner_changed(False)
        self.assertFalse(called)

    def test_called_beggining_no(self):
        """Check that it should not be called if no SD."""
        called = []
        orig_met = SyncDaemon._get_initial_data
        SyncDaemon._get_initial_data = lambda s: called.append(True)
        SyncDaemon(FakeDBusInterface)
        SyncDaemon._get_initial_data = orig_met
        self.assertFalse(called)

    def test_called_beggining_yes(self):
        """Check that it should be called if SD already started."""
        called = []
        orig_met = SyncDaemon._get_initial_data
        SyncDaemon._get_initial_data = lambda s: called.append(True)
        FakeDBusInterface.fake_sd_started = True
        SyncDaemon(FakeDBusInterface)
        SyncDaemon._get_initial_data = orig_met
        self.assertTrue(called)

    def test_calls_callbacks(self):
        """Check that initial data calls the callbacks for new data."""
        called = []
        sd = SyncDaemon(FakeDBusInterface)
        f = lambda *a, **kw: called.append(True)
        sd.status_changed_callback = f
        sd.content_queue_changed_callback = f
        sd.meta_queue_changed_callback = f

        sd._get_initial_data()
        self.assertEqual(len(called), 3)

    def test_public_files_info(self):
        """Check we get the public files info at start."""
        sd = SyncDaemon(FakeDBusInterface)
        fake_data = FakeDBusInterface.fake_pf_data
        sd._get_initial_data()
        self.assertEqual(sd.public_files[fake_data.node], fake_data)

    def test_all_ready(self):
        """All data is ready."""
        self.sd._get_initial_data()
        self.assertTrue(self.offline_called)
        self.assertTrue(self.online_called)

    def test_no_public_files(self):
        """Initial gathering is stuck in public files."""
        self.sd.dbus.get_public_files = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertTrue(self.offline_called)
        self.assertFalse(self.online_called)

    def test_no_shares_to_others(self):
        """Initial gathering is stuck in shares to others."""
        self.sd.dbus.get_shares_to_others = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertFalse(self.offline_called)
        self.assertFalse(self.online_called)

    def test_no_shares_to_me(self):
        """Initial gathering is stuck in shares to me."""
        self.sd.dbus.get_shares_to_me = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertFalse(self.offline_called)
        self.assertFalse(self.online_called)

    def test_no_folders(self):
        """Initial gathering is stuck in folders."""
        self.sd.dbus.get_folders = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertFalse(self.offline_called)
        self.assertFalse(self.online_called)

    def test_no_meta_queue(self):
        """Initial gathering is stuck in meta queue."""
        self.sd.dbus.get_meta_queue = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertFalse(self.offline_called)
        self.assertFalse(self.online_called)

    def test_no_content_queue(self):
        """Initial gathering is stuck in content queue."""
        self.sd.dbus.get_content_queue = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertFalse(self.offline_called)
        self.assertFalse(self.online_called)

    def test_no_status(self):
        """Initial gathering is stuck in status."""
        self.sd.dbus.get_status = lambda: defer.Deferred()
        self.sd._get_initial_data()
        self.assertFalse(self.offline_called)
        self.assertFalse(self.online_called)


class StatusChangedTests(BaseTest):
    """Simple signals checking."""

    @defer.inlineCallbacks
    def test_initial_value(self):
        """Fill the status info initially."""
        called = []

        def fake():
            """Fake method."""
            called.append(True)
            return defer.succeed(('fakename', 'fakedescrip', False, True,
                              False, 'fakequeues', 'fakeconnection'))

        self.sd.dbus.get_status = fake
        yield self.sd._get_initial_data()
        self.assertTrue(called)

    def test_statuschanged(self):
        """Test StatusChanged signal."""
        deferred = defer.Deferred()

        def callback(name, description, is_error, is_connected, is_online,
                     queues, connection):
            """Check received data."""
            self.assertEqual(name, 'name')
            self.assertEqual(description, 'description')
            self.assertEqual(is_error, False)
            self.assertEqual(is_connected, True)
            self.assertEqual(is_online, False)
            self.assertEqual(queues, 'queues')
            self.assertEqual(connection, 'connection')
            deferred.callback(True)

        self.sd.status_changed_callback = callback
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        return deferred

    def test_status_changed_affects_current_status(self):
        """Make changes to see how status are reflected."""
        # one set of values
        self.sd.on_sd_status_changed('name1', 'description1', False, True,
                                     False, 'queues1', 'connection1')
        self.assertEqual(self.sd.current_state.name, 'name1')
        self.assertEqual(self.sd.current_state.description, 'description1')
        self.assertEqual(self.sd.current_state.is_error, False)
        self.assertEqual(self.sd.current_state.is_connected, True)
        self.assertEqual(self.sd.current_state.is_online, False)
        self.assertEqual(self.sd.current_state.queues, 'queues1')
        self.assertEqual(self.sd.current_state.connection, 'connection1')

        # again, to be sure they actually are updated
        self.sd.on_sd_status_changed('name2', 'description2', True, False,
                                     True, 'queues2', 'connection2')
        self.assertEqual(self.sd.current_state.name, 'name2')
        self.assertEqual(self.sd.current_state.description, 'description2')
        self.assertEqual(self.sd.current_state.is_error, True)
        self.assertEqual(self.sd.current_state.is_connected, False)
        self.assertEqual(self.sd.current_state.is_online, True)
        self.assertEqual(self.sd.current_state.queues, 'queues2')
        self.assertEqual(self.sd.current_state.connection, 'connection2')

    def test_is_started_fixed_at_init_no(self):
        """Status.is_started is set at init time, to no."""
        FakeDBusInterface.fake_sd_started = False
        sd = SyncDaemon(FakeDBusInterface)
        self.assertFalse(sd.current_state.is_started)

    def test_is_started_fixed_at_init_yes(self):
        """Status.is_started is set at init time, to yes."""
        FakeDBusInterface.fake_sd_started = True
        sd = SyncDaemon(FakeDBusInterface)
        self.assertTrue(sd.current_state.is_started)

    def test_on_stopped(self):
        """Stopped affects the status."""
        self.sd.on_sd_name_owner_changed(False)
        self.assertEqual(self.sd.current_state.name, 'STOPPED')
        self.assertEqual(self.sd.current_state.description,
                         'ubuntuone-client is stopped')
        self.assertEqual(self.sd.current_state.is_error, False)
        self.assertEqual(self.sd.current_state.is_connected, False)
        self.assertEqual(self.sd.current_state.is_online, False)
        self.assertEqual(self.sd.current_state.queues, '')
        self.assertEqual(self.sd.current_state.connection, '')

    def test_on_started(self):
        """Started affects the status."""
        # make _get_initial_data dummy to not affect test
        self.sd._get_initial_data = lambda: None

        self.sd.on_sd_name_owner_changed(True)
        self.assertEqual(self.sd.current_state.name, 'STARTED')
        self.assertEqual(self.sd.current_state.description,
                         'ubuntuone-client just started')
        self.assertEqual(self.sd.current_state.is_error, False)
        self.assertEqual(self.sd.current_state.is_connected, False)
        self.assertEqual(self.sd.current_state.is_online, False)
        self.assertEqual(self.sd.current_state.queues, '')
        self.assertEqual(self.sd.current_state.connection, '')

    def test_processing_meta_if_working_on_meta_or_both(self):
        """Status.processing_meta is True when WORKING_ON_{METADATA,BOTH}."""

        msg = 'processing_meta must be False when %s.'
        for state in ('WORKING_ON_CONTENT', ''):
            self.sd.current_state.set(queues=state)
            self.assertFalse(self.sd.current_state.processing_meta,
                             msg % state)

        msg = 'processing_meta must be True when %s.'
        for state in ('WORKING_ON_METADATA', 'WORKING_ON_BOTH'):
            self.sd.current_state.set(queues=state)
            self.assertTrue(self.sd.current_state.processing_meta, msg % state)

    def test_processing_content_if_working_on_content_or_both(self):
        """Status.processing_content is True when WORKING_ON_{CONTENT,BOTH}."""

        msg = 'processing_content must be False when %s.'
        for state in ('WORKING_ON_METADATA', ''):
            self.sd.current_state.set(queues=state)
            self.assertFalse(self.sd.current_state.processing_content,
                             msg % state)

        msg = 'processing_content must be True when %s.'
        for state in ('WORKING_ON_CONTENT', 'WORKING_ON_BOTH'):
            self.sd.current_state.set(queues=state)
            self.assertTrue(self.sd.current_state.processing_content,
                            msg % state)


class ContentQueueChangedTests(BaseTest):
    """Check the ContenQueueChanged handling."""

    @defer.inlineCallbacks
    def test_initial_value(self):
        """Fill the content queue info initially."""
        called = []
        self.sd.dbus.get_content_queue = lambda: called.append(True)
        yield self.sd._get_initial_data()
        self.assertTrue(called)

    def test_without_setting_callback(self):
        """It should work even if not hooking into the callback."""
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()

    def test_callback_call(self):
        """Call the callback."""
        called = []
        self.sd.content_queue_changed_callback = lambda cq: called.append(cq)
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()
        self.assertTrue(called)

    def test_callback_call_twice_different(self):
        """Call the callback twice for different info."""
        called = []
        self.sd.content_queue_changed_callback = lambda cq: called.append(cq)

        # first call
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(called, [['foo']])

        # second call, different info
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['bar'])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(called, [['foo'], ['bar']])

    def test_callback_call_twice_same(self):
        """Call the callback once, even getting twice the same info."""
        called = []
        self.sd.content_queue_changed_callback = lambda cq: called.append(cq)

        # first call
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(called, [['foo']])

        # second call, same info
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(called, [['foo']])

    def test_cq_state_nothing(self):
        """Check the ContentQueue info, being nothing."""
        self.sd.dbus.get_content_queue = lambda: defer.succeed([])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(self.sd.content_queue, [])

    def test_cq_state_one(self):
        """Check the ContentQueue info, being one."""
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(self.sd.content_queue, ['foo'])

    def test_cq_state_two(self):
        """Check the ContentQueue info, two."""
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo', 'bar'])
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(self.sd.content_queue, ['foo', 'bar'])

    def test_cq_multiple_without_response_having_two(self):
        """Behave correctly on two-changes burst."""
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        called = []

        def fake():
            """Fake."""
            d = d2 if called else d1
            called.append(None)
            return d
        self.sd.dbus.get_content_queue = fake

        # call twice
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()

        # dbus function should be called once
        self.assertEqual(len(called), 1)

        # first call returns
        d1.callback(['foo'])
        d2.callback(['bar'])

        # it should be called once more only
        self.assertEqual(len(called), 2)

    def test_cq_multiple_without_response_having_three(self):
        """Behave correctly on three-changes burst."""
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        called = []

        def fake():
            """Fake."""
            d = d2 if called else d1
            called.append(None)
            return d
        self.sd.dbus.get_content_queue = fake

        # call thrice
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()

        # dbus function should be called once
        self.assertEqual(len(called), 1)

        # first call returns
        d1.callback(['foo'])
        d2.callback(['bar'])

        # it should be called once more only
        self.assertEqual(len(called), 2)

    def test_cq_multiple_without_response_having_more_later(self):
        """Behave correctly on changes burst delayed."""
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        d3 = defer.Deferred()
        defs = [d1, d2, d3]
        called = []

        def fake():
            """Fake."""
            d = defs[len(called)]
            called.append(None)
            return d
        self.sd.dbus.get_content_queue = fake

        # call some times, dbus function should be called once
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()
        self.assertEqual(len(called), 1)

        # first call returns
        d1.callback(['foo'])

        # while calling second time, generate more requests
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()
        self.sd.on_sd_content_queue_changed()

        # second call returns
        d2.callback(['foo'])
        d3.callback(['foo'])

        # it should be called three times total
        self.assertEqual(len(called), 3)


class MetaQueueChangedTests(BaseTest):
    """Check the MetaQueueChanged handling."""

    timeout = 3

    def setUp(self):
        """Set up."""
        BaseTest.setUp(self)
        self.sd.current_state.set(queues='WORKING_ON_METADATA')

    @defer.inlineCallbacks
    def test_initial_value(self):
        """Fill the meta queue info initially."""
        called = []

        def f():
            """Helper function."""
            called.append(True)
            return []

        self.sd.dbus.get_meta_queue = f
        yield self.sd._get_initial_data()
        self.assertTrue(called)

    def test_without_setting_callback(self):
        """It should work even if not hooking into the callback."""
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_meta_queue_changed()

    def test_must_poll_flag(self):
        """Check the flag behaviour."""
        # in True when started
        self.assertTrue(self.sd._must_poll_mq)

        # send a signal, and check it went to False
        self.sd.on_sd_meta_queue_changed()
        self.assertFalse(self.sd._must_poll_mq)

    def test_callback_call(self):
        """Call the callback."""
        called = []
        self.sd.meta_queue_changed_callback = lambda mq: called.append(mq)
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd._check_mq()
        self.assertTrue(called)

    def test_callback_call_twice_different(self):
        """Call the callback twice for different info."""
        called = []
        self.sd.meta_queue_changed_callback = lambda mq: called.append(mq)

        # first call
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd._check_mq()
        self.assertEqual(called, [['foo']])

        # second call, different info
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['bar'])
        self.sd._check_mq()
        self.assertEqual(called, [['foo'], ['bar']])

    def test_callback_call_twice_same(self):
        """Call the callback once, even getting twice the same info."""
        called = []
        self.sd.meta_queue_changed_callback = lambda mq: called.append(mq)

        # first call
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd._check_mq()
        self.assertEqual(called, [['foo']])

        # second call, same info
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd._check_mq()
        self.assertEqual(called, [['foo']])

    def test_polling_initiated_state_changed_polling(self):
        """The polling initiates when state changes and polling."""
        called = []
        self.sd._check_mq = lambda: called.append(True)

        # send some status changed
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        self.assertTrue(called)

    def test_polling_initiated_state_changed_notpolling(self):
        """The polling initiates when state changes and not polling."""
        called = []
        self.sd._check_mq = lambda: called.append(True)
        self.sd._must_poll_mq = False

        # send some status changed
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        self.assertFalse(called)

    def test_polling_aborted_after_signal(self):
        """The polling is aborted if no longer must poll."""
        self.sd._must_poll_mq = False
        self.sd._check_mq()
        self.assertTrue(self.sd._mqcaller is None)

    def test_mq_polling_workinginmetadata_notinqueuemanager(self):
        """Check that it polls mq while working in metadata not being in QM."""
        # set the callback
        deferred = defer.Deferred()

        def fake():
            """Fake."""
            deferred.callback(True)
            return defer.succeed("foo")

        self.sd.dbus.get_meta_queue = fake

        # send status changed to working in metadata
        self.sd.on_sd_status_changed('AUTHENTICATE', 'description', False,
                                     True, False, 'WORKING_ON_METADATA',
                                     'connection')
        return deferred

    def test_mq_polling_workinginmetadata_queuemanager(self):
        """Check that it polls mq while working in metadata being in QM."""
        # set the callback
        deferred = defer.Deferred()

        def fake():
            """Fake."""
            deferred.callback(True)
            return defer.succeed("foo")

        self.sd.dbus.get_meta_queue = fake

        # send status changed to working in metadata
        self.sd.on_sd_status_changed('QUEUE_MANAGER', 'description', False,
                                     True, False, 'WORKING_ON_METADATA',
                                     'connection')
        return deferred

    def test_mq_polling_workinginboth(self):
        """Check that it polls mq while working in both."""
        # set the callback
        deferred = defer.Deferred()

        def fake():
            """Fake."""
            deferred.callback(True)
            return defer.succeed("foo")

        self.sd.dbus.get_meta_queue = fake

        # send status changed to working in metadata
        self.sd.on_sd_status_changed('QUEUE_MANAGER', 'description', False,
                                     True, False, 'WORKING_ON_BOTH',
                                     'connection')
        return deferred

    def test_mq_polls_last_time(self):
        """Was polling, state changed, it needs to poll a last time."""
        # set the callback
        deferred = defer.Deferred()
        changed = self.sd.on_sd_status_changed
        called = []

        def fake():
            """Fake."""
            called.append(None)
            if len(called) == 1:
                changed('QUEUE_MANAGER', 'description', False, True, False,
                        'WORKING_ON_CONTENT', 'connection')
            elif len(called) == 2:
                # check that the caller is set back to None
                self.assertTrue(self.sd._mqcaller is None)
                deferred.callback(True)
            return defer.succeed("foo")

        self.sd.dbus.get_meta_queue = fake

        # send status changed to working in metadata
        changed('QUEUE_MANAGER', 'description', False, True, False,
                'WORKING_ON_BOTH', 'connection')
        return deferred

    def test_mq_caller_is_reset_last_time(self):
        """When MQ is polled last time, the caller should be back to None."""
        # Module 'twisted.internet.reactor' has no 'callLater' member
        # pylint: disable=E1101
        self.sd._mqcaller = reactor.callLater(100, lambda: None)
        self.sd.current_state.set(name='QUEUE_MANAGER',
                                  queues='WORKING_ON_CONTENT')

        # call the method and check
        self.sd._check_mq()
        self.assertTrue(self.sd._mqcaller is None)

    def test_mq_polling_untilfinish(self):
        """Check that it polls mq until no more is needed."""
        # set the callback, and adjust the polling time to faster
        calls = []

        def fake_get(*a):
            """Fake get."""
            calls.append(None)
            if len(calls) < 3:
                pass  # no changes, should keep calling
            elif len(calls) == 3:
                self.sd.on_sd_status_changed('QUEUE_MANAGER', 'description',
                                             False, True, False,
                                             'WORKING_ON_CONTENT', 'connect')

                # allow time to see if a mistaken call happens
                # Module 'twisted.internet.reactor' has no 'callLater' member
                # pylint: disable=E1101
                reactor.callLater(.5, deferred.callback, True)
            elif len(calls) == 4:
                pass    # last call after state changed
            else:
                deferred.errback(ValueError("Too many calls"))
            return defer.succeed("foo")

        self.sd.dbus.get_meta_queue = fake_get
        self.sd._mq_poll_time = .1

        self.sd.on_sd_status_changed('QUEUE_MANAGER', 'description', False,
                                     True, False, 'WORKING_ON_BOTH',
                                     'connection')
        deferred = defer.Deferred()
        return deferred

    def test_mq_state_nothing(self):
        """Check the MetaQueue info, being nothing."""
        self.sd.dbus.get_meta_queue = lambda: defer.succeed([])
        self.sd._check_mq()
        self.assertEqual(self.sd.meta_queue, [])

    def test_mq_state_one(self):
        """Check the MetaQueue info, being one."""
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd._check_mq()
        self.assertEqual(self.sd.meta_queue, ['foo'])

    def test_mq_state_two(self):
        """Check the MetaQueue info, two."""
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo', 'bar'])
        self.sd._check_mq()
        self.assertEqual(self.sd.meta_queue, ['foo', 'bar'])

    def test_mq_multiple_without_response_having_two(self):
        """Behave correctly on two-changes burst."""
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        called = []

        def fake():
            """Fake."""
            d = d2 if called else d1
            called.append(None)
            return d
        self.sd.dbus.get_meta_queue = fake

        # call twice
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()

        # dbus function should be called once
        self.assertEqual(len(called), 1)

        # first call returns
        d1.callback(['foo'])
        d2.callback(['bar'])

        # it should be called once more only
        self.assertEqual(len(called), 2)

    def test_mq_multiple_without_response_having_three(self):
        """Behave correctly on three-changes burst."""
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        called = []

        def fake():
            """Fake."""
            d = d2 if called else d1
            called.append(None)
            return d
        self.sd.dbus.get_meta_queue = fake

        # call thrice
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()

        # dbus function should be called once
        self.assertEqual(len(called), 1)

        # first call returns
        d1.callback(['foo'])
        d2.callback(['bar'])

        # it should be called once more only
        self.assertEqual(len(called), 2)

    def test_mq_multiple_without_response_having_more_later(self):
        """Behave correctly on changes burst delayed."""
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        d3 = defer.Deferred()
        defs = [d1, d2, d3]
        called = []

        def fake():
            """Fake."""
            d = defs[len(called)]
            called.append(None)
            return d
        self.sd.dbus.get_meta_queue = fake

        # call some times, dbus function should be called once
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()
        self.assertEqual(len(called), 1)

        # first call returns
        d1.callback(['foo'])

        # while calling second time, generate more requests
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()
        self.sd.on_sd_meta_queue_changed()

        # second call returns
        d2.callback(['foo'])
        d3.callback(['foo'])

        # it should be called three times total
        self.assertEqual(len(called), 3)

    @defer.inlineCallbacks
    def test_mq_multiple_support_error(self):
        """Leave the state ok even with an error in the underlying call."""
        def fake():
            """Fake."""
            raise ValueError('foo')
        self.sd.dbus.get_meta_queue = fake

        # call it, and "absorb" the error we just generated
        try:
            yield self.sd.on_sd_meta_queue_changed()
        except ValueError:
            pass

        # it should be in IDLE
        self.assertEqual(self.sd._mq_asking, ASKING_IDLE)


class StateTests(unittest.TestCase):
    """Test State class."""

    def test_initial(self):
        """Initial state for vals."""
        st = State()
        self.assertEqual(st.name, '')

    def test_set_one_value(self):
        """Set one value."""
        st = State()
        st.set(name=55)

        # check the one is set, the rest not
        self.assertEqual(st.name, 55)
        self.assertEqual(st.description, '')

    def test_set_two_values(self):
        """Set two values."""
        st = State()
        st.set(name=55, description=77)

        # check those two are set, the rest not
        self.assertEqual(st.name, 55)
        self.assertEqual(st.description, 77)
        self.assertFalse(st.is_error)

    def test_bad_value(self):
        """Set a value that should not."""
        st = State()
        self.assertRaises(AttributeError, st.set, not_really_allowed=44)


class APITests(unittest.TestCase):
    """Check exposed methods and attributes."""

    def setUp(self):
        """Set up the test."""
        self.sd = SyncDaemon(FakeDBusInterface)

        self._replaced = None
        self.called = False

    def tearDown(self):
        """Tear down the test."""
        if self._replaced is not None:
            setattr(*self._replaced)
        self.sd.shutdown()

    def mpatch_called(self, obj, method_name):
        """Monkeypatch to flag called."""
        self._replaced = (obj, method_name, getattr(obj, method_name))
        f = lambda *a, **k: setattr(self, 'called', True)
        setattr(obj, method_name, f)

    def flag_called(self, obj, method_name):
        """Replace callback to flag called."""
        f = lambda *a, **k: setattr(self, 'called', True)
        setattr(obj, method_name, f)

    def test_is_started_yes(self):
        """Check is_started, True."""
        # simulate the signal that indicates the name was registered
        self.sd.on_sd_name_owner_changed(True)
        self.assertTrue(self.sd.current_state.is_started)

    def test_is_started_no(self):
        """Check is_started, False."""
        self.sd.on_sd_name_owner_changed(False)
        self.assertFalse(self.sd.current_state.is_started)

    def test_start(self):
        """Test start calls SD."""
        self.mpatch_called(self.sd.dbus, 'start')
        self.sd.start()
        self.assertTrue(self.called)

    def test_quit(self):
        """Test quit calls SD."""
        self.mpatch_called(self.sd.dbus, 'quit')
        self.sd.quit()
        self.assertTrue(self.called)

    def test_connect(self):
        """Test connect calls SD."""
        self.mpatch_called(self.sd.dbus, 'connect')
        self.sd.connect()
        self.assertTrue(self.called)

    def test_disconnect(self):
        """Test disconnect calls SD."""
        self.mpatch_called(self.sd.dbus, 'disconnect')
        self.sd.disconnect()
        self.assertTrue(self.called)

    def test_on_started(self):
        """Called when SD started."""
        self.flag_called(self.sd, 'on_started_callback')
        self.sd.on_sd_name_owner_changed(True)
        self.assertTrue(self.called)

    def test_on_stopped(self):
        """Called when SD stopped."""
        self.flag_called(self.sd, 'on_stopped_callback')
        self.sd.on_sd_name_owner_changed(False)
        self.assertTrue(self.called)

    def test_on_connected(self):
        """Called when SD connected."""
        self.flag_called(self.sd, 'on_connected_callback')

        # first signal with connected in True
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        self.assertTrue(self.called)

    def test_on_disconnected(self):
        """Called when SD disconnected."""
        self.flag_called(self.sd, 'on_disconnected_callback')

        # connect and disconnect
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        self.sd.on_sd_status_changed('name', 'description', False, False,
                                     False, 'queues', 'connection')
        self.assertTrue(self.called)

    def test_on_online(self):
        """Called when SD goes online."""
        self.flag_called(self.sd, 'on_online_callback')

        # first signal with online in True
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     True, 'queues', 'connection')
        self.assertTrue(self.called)

    def test_on_offline(self):
        """Called when SD goes offline."""
        self.flag_called(self.sd, 'on_offline_callback')

        # go online and then offline
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     True, 'queues', 'connection')
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        self.assertTrue(self.called)


class TestLogs(unittest.TestCase):
    """Test logging."""

    def setUp(self):
        """Set up."""
        self.hdlr = MementoHandler()
        self.hdlr.setLevel(logging.DEBUG)
        logger = logging.getLogger('magicicada.syncdaemon')
        logger.addHandler(self.hdlr)
        logger.setLevel(logging.DEBUG)
        self.sd = SyncDaemon(FakeDBusInterface)

    def tearDown(self):
        """Shut down!"""
        self.sd.shutdown()

    def test_instancing(self):
        """Just logged SD instancing."""
        self.assertTrue(self.hdlr.check_info("SyncDaemon interface started!"))

    def test_shutdown(self):
        """Log when SD shutdowns."""
        self.sd.shutdown()
        msg = "SyncDaemon interface going down"
        self.assertTrue(self.hdlr.check_info(msg))

    @defer.inlineCallbacks
    def test_initial_value(self):
        """Log the initial filling."""
        yield self.sd._get_initial_data()
        self.assertTrue(self.hdlr.check_info("Getting offline initial data"))
        self.assertTrue(self.hdlr.check_info(
                        "All initial offline data is ready"))
        self.assertTrue(self.hdlr.check_info("Getting online initial data"))
        self.assertTrue(self.hdlr.check_info(
                        "All initial online data is ready"))

    def test_start(self):
        """Log the call to start."""
        self.sd.start()
        self.assertTrue(self.hdlr.check_info("Starting u1.SD"))

    def test_quit(self):
        """Log the call to quit."""
        self.sd.quit()
        self.assertTrue(self.hdlr.check_info("Stopping u1.SD"))

    def test_connect(self):
        """Log the call to connect."""
        self.sd.connect()
        self.assertTrue(self.hdlr.check_info("Telling u1.SD to connect"))

    def test_disconnect(self):
        """Log the call to disconnect."""
        self.sd.disconnect()
        self.assertTrue(self.hdlr.check_info("Telling u1.SD to disconnect"))

    def test_check_mq_true(self):
        """Log the MQ check when it asks for info."""
        self.sd.current_state.set(name='QUEUE_MANAGER',
                                   queues='WORKING_ON_METADATA')
        self.sd._check_mq()
        self.assertTrue(self.hdlr.check_info("Asking for MQ information"))

    def test_check_mq_noreally(self):
        """Log the MQ check when it should not work."""
        self.sd.current_state.set(name='QUEUE_MANAGER',
                                   queues='WORKING_ON_CONTENT')
        self.sd._check_mq()
        self.assertTrue(self.hdlr.check_info(
                        "Check MQ called, States not in MQ, call a last time"))

    def test_meta_queue_changed_by_polling(self):
        """Log that MQ has new data, polled."""
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd.current_state.set(name='QUEUE_MANAGER',
                                   queues='WORKING_ON_METADATA')
        self.sd._check_mq()
        self.assertTrue(self.hdlr.check_info(
                        "SD Meta Queue info is new! 1 items"))

    def test_meta_queue_changed_by_signal(self):
        """Log that MQ has new data, alerted by a signal."""
        self.sd.dbus.get_meta_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_meta_queue_changed()
        self.assertTrue(self.hdlr.check_info("SD Meta Queue changed"))
        self.assertTrue(self.hdlr.check_info(
                        "SD Meta Queue info is new! 1 items"))

    def test_meta_queue_changed(self):
        """Log that process_cq has new data."""
        self.sd.dbus.get_content_queue = lambda: defer.succeed(['foo'])
        self.sd.on_sd_content_queue_changed()
        self.assertTrue(self.hdlr.check_info("SD Content Queue changed"))
        self.assertTrue(self.hdlr.check_info(
                        "Content Queue info is new! 1 items"))

    def test_on_status_changed(self):
        """Log status changed."""
        self.sd.on_sd_status_changed('name', 'description', False, True,
                                     False, 'queues', 'connection')
        self.assertTrue(self.hdlr.check_info("SD Status changed"))
        expected = u"    new status: connection='connection', " \
                    "description='description', is_connected=True, " \
                    "is_error=False, is_online=False, name='name', " \
                    "queues='queues'"
        self.assertTrue(self.hdlr.check_debug(expected))

    def test_folders_changed(self):
        """Log when folders changed."""
        self.sd.on_sd_folders_changed()
        self.assertTrue(self.hdlr.check_info("SD Folders changed"))

    def test_shares_changed(self):
        """Log when shares changed."""
        self.sd.on_sd_shares_changed()
        self.assertTrue(self.hdlr.check_info("SD Shares changed"))

    def test_on_name_owner_changed(self):
        """Log name owner changed."""
        self.sd.on_sd_name_owner_changed(True)
        self.assertTrue(self.hdlr.check_info("SD Name Owner changed: True"))

    def test_on_public_files_list(self):
        """Log we got a new public files list."""
        pf1 = PublicFilesData(volume='v', node='n1', path='p', public_url='u')
        pf2 = PublicFilesData(volume='v', node='n2', path='p', public_url='u')
        self.sd.on_sd_public_files_list([pf1, pf2])
        self.assertTrue(self.hdlr.check_info(
                        "Got new Public Files list (2 items)"))

    def test_on_public_files_changed(self):
        """Log we got a change in the public files list."""
        self.sd.public_files = {}
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url')
        self.sd.on_sd_public_files_changed(pf, True)
        self.assertTrue(self.hdlr.check_info(
                        "Change in Public Files list! is_public=True",
                        "node=node", "path=u'path'"))


class MetadataTests(BaseTest):
    """Get Metadata info."""

    def test_get_metadata_no_callback_set(self):
        """It's mandatory to set the callback for this response."""
        self.sd.on_metadata_ready_callback()
        self.assertTrue(self.hdlr.check_warning('on_metadata_ready_callback'))

    def test_get_metadata_ok(self):
        """Get the metadata for given path."""
        self.patch(os.path, 'realpath', lambda p: p)
        called = []
        self.sd.dbus.get_metadata = lambda p: defer.succeed('foo')
        self.sd.on_metadata_ready_callback = lambda *a: called.extend(a)
        self.sd.get_metadata('path')
        self.assertEqual(called, ['path', 'foo'])

    def test_get_metadata_double(self):
        """Get the metadata twice."""
        self.patch(os.path, 'realpath', lambda p: p)
        called = []
        fake_md = {'path1': 'foo', 'path2': 'bar'}
        self.sd.dbus.get_metadata = lambda p: defer.succeed(fake_md[p])
        self.sd.on_metadata_ready_callback = lambda *a: called.append(a)
        self.sd.get_metadata('path1')
        self.sd.get_metadata('path2')
        self.assertEqual(called[0], ('path1', 'foo'))
        self.assertEqual(called[1], ('path2', 'bar'))

    def test_get_metadata_uses_realpath(self):
        """Ask for metadata using the realpath (LP: #612191)."""
        self.patch(os.path, 'realpath', lambda p: '/a/realpath')
        called = []
        self.sd.dbus.get_metadata = lambda p: defer.succeed(p)
        self.sd.on_metadata_ready_callback = lambda *a: called.extend(a)

        self.sd.get_metadata('/a/symlink/path')

        self.assertEqual(called, ['/a/symlink/path', '/a/realpath'])


class FoldersTests(BaseTest):
    """Folders checking."""

    def test_foldercreated_callback(self):
        """Get the new data after the folders changed."""
        # set the callback
        called = []
        self.sd.dbus.get_folders = lambda: called.append(True)

        # they changed!
        self.sd.on_sd_folders_changed()

        # test
        self.assertTrue(called)

    @defer.inlineCallbacks
    def test_initial_value(self):
        """Fill the folder info initially."""
        called = []
        self.sd.dbus.get_folders = lambda: called.append(True)
        yield self.sd._get_initial_data()
        self.assertTrue(called)

    def test_folder_changed_callback(self):
        """Test that the GUI callback is called."""
        # set the callback
        called = []
        self.sd.on_folders_changed_callback = lambda *a: called.append(True)

        # they changed!
        self.sd.on_sd_folders_changed()

        # test
        self.assertTrue(called)


class SharesTests(BaseTest):
    """Shares checking."""

    def test_shares_changed_callback(self):
        """Get the new data after the shares changed."""
        # set the callback
        called = []
        self.sd.dbus.get_shares_to_me = lambda: called.append(True)
        self.sd.dbus.get_shares_to_others = lambda: called.append(True)

        # they changed!
        self.sd.on_sd_shares_changed()

        # test
        self.assertEqual(len(called), 2)

    @defer.inlineCallbacks
    def test_initial_value(self):
        """Fill the folder info initially."""
        called = []
        self.sd.dbus.get_shares_to_me = lambda: called.append(True)
        self.sd.dbus.get_shares_to_others = lambda: called.append(True)
        yield self.sd._get_initial_data()
        self.assertEqual(len(called), 2)

    def test_shares_to_me_changed_callback(self):
        """Test that the GUI callback is called."""
        # set the callback
        cal = []
        self.sd.on_shares_to_me_changed_callback = lambda *a: cal.append(1)
        self.sd.on_shares_to_others_changed_callback = lambda *a: cal.append(2)

        # they changed!
        self.sd.shares_to_me = 'foo'
        self.sd.shares_to_others = 'fakedata'  # what fake dbus will return
        self.sd.on_sd_shares_changed()

        # test
        self.assertEqual(cal, [1])

    def test_shares_to_others_changed_callback(self):
        """Test that the GUI callback is called."""
        # set the callback
        cal = []
        self.sd.on_shares_to_me_changed_callback = lambda *a: cal.append(1)
        self.sd.on_shares_to_others_changed_callback = lambda *a: cal.append(2)

        # they changed!
        self.sd.shares_to_others = 'foo'
        self.sd.shares_to_me = 'fakedata'  # what fake dbus will return
        self.sd.on_sd_shares_changed()

        # test
        self.assertEqual(cal, [2])


class PublicFilesTests(BaseTest):
    """PublicFiles checking."""

    def test_signal_updates_value(self):
        """Test that when signal arrives it updates the internal attribute."""
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url')
        self.sd.on_sd_public_files_list([pf])
        self.assertEqual(len(self.sd.public_files), 1)
        pf = self.sd.public_files['node']
        self.assertEqual(pf.volume, 'volume')
        self.assertEqual(pf.node, 'node')
        self.assertEqual(pf.public_url, 'url')
        self.assertEqual(pf.path, 'path')

    def test_public_files_changed_added_noprevious(self):
        """Test that it adds the new public file."""
        # reset and add
        self.sd.public_files = {}
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url')
        self.sd.on_sd_public_files_changed(pf, True)

        # check is there
        pf = self.sd.public_files['node']
        self.assertEqual(pf.volume, 'volume')
        self.assertEqual(pf.node, 'node')
        self.assertEqual(pf.public_url, 'url')
        self.assertEqual(pf.path, 'path')

    def test_public_files_changed_added_previous(self):
        """Test that it updates the public file."""
        # add one
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url1')
        self.sd.on_sd_public_files_changed(pf, True)

        # update it
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url2')
        self.sd.on_sd_public_files_changed(pf, True)

        # check is updated
        pf = self.sd.public_files['node']
        self.assertEqual(pf.public_url, 'url2')

    def test_public_files_changed_removed_previous(self):
        """Test that it deletes the public file."""
        # add one
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url')
        self.sd.on_sd_public_files_changed(pf, True)

        # remove it
        self.sd.on_sd_public_files_changed(pf, False)

        # check is no longer there
        self.assertFalse('node' in self.sd.public_files)

    def test_public_files_changed_removed_noprevious(self):
        """Test that support the deletion of anything."""
        # reset and remove
        self.sd.public_files = {}
        pf = PublicFilesData(volume='volume', node='node',
                             path='path', public_url='url')
        self.sd.on_sd_public_files_changed(pf, False)

        # check that still is not there
        self.assertFalse('node' in self.sd.public_files)


class HandlingSharesTests(BaseTest):
    """Handling shares checking."""

    def test_on_share_op_success_callback_set(self):
        """It's mandatory to set the callback for this response."""
        self.sd.on_share_op_success_callback()
        self.assertTrue(self.hdlr.check_warning(
                                            'on_share_op_success_callback'))

    def test_on_share_op_error_callback_set(self):
        """It's mandatory to set the callback for this response."""
        self.sd.on_share_op_error_callback()
        self.assertTrue(self.hdlr.check_warning('on_share_op_error_callback'))

    def test_accept_share_ok(self):
        """Accepting a share finishes ok."""
        # monkeypatch
        called = []
        self.sd.dbus.fake_share_response = defer.succeed(None)
        self.sd.on_share_op_success_callback = lambda sid: called.append(sid)
        self.sd.on_share_op_error_callback = lambda *a: None

        # execute and test
        self.sd.accept_share('share_id')
        self.assertEqual(called, ['share_id'])
        self.assertTrue(self.hdlr.check_info("Accepting share", "share_id",
                                             "started"))
        self.assertTrue(self.hdlr.check_info("Accepting share", "share_id",
                                             "finished successfully"))

    def test_accept_share_failure(self):
        """Accepting a share finishes bad."""
        # monkeypatch
        called = []
        e = ShareOperationError(share_id='foo', error='bar')
        self.sd.dbus.fake_share_response = defer.fail(e)
        self.sd.on_share_op_error_callback = \
                                        lambda sid, e: called.append((sid, e))
        self.sd.on_share_op_success_callback = lambda *a: None

        # execute and test
        self.sd.accept_share('share_id')
        self.assertEqual(called, [('share_id', 'bar')])
        self.assertTrue(self.hdlr.check_info("Accepting share", "share_id",
                                             "started"))
        self.assertTrue(self.hdlr.check_info("Accepting share", "share_id",
                                             "finished with error", "bar"))

    def test_accept_share_error(self):
        """Really bad error when accepting a share."""
        # monkeypatch
        e = ValueError('unexpected failure')
        self.sd.dbus.fake_share_response = defer.fail(e)
        self.sd.on_share_op_error_callback = lambda *a: None
        self.sd.on_share_op_success_callback = lambda *a: None

        # execute and test
        self.sd.accept_share('share_id')
        self.assertTrue(self.hdlr.check_info("Accepting share", "share_id",
                                             "started"))
        self.assertTrue(self.hdlr.check_error(
                        "Unexpected error when accepting share", "share_id",
                        "ValueError", "unexpected failure"))
