/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/


/*! \file MagicsSingleton.h

    \brief Definition of ...

    Magics Team - ECMWF 2006

    Started: Wed 27-Oct-2006

*/

#ifndef MagicsSingleton_H
#define MagicsSingleton_H

#include "magics.h"
#include "Data.h"
#include "GeoPoint.h"
#include "Axis.h"
#include "Timer.h"
#include "XYList.h"
#include "GraphPlotting.h"
#include "EpsGraph.h"
#include "GeoPointsDecoder.h"
#include "GribDecoder.h"
#include "OutputHandler.h"
#include "InputMatrix.h"

#include "MagicsSingletonAttributes.h"

namespace magics {

class MagicsSingleton : public MagicsSingletonAttributes
{
	MagicsSingleton() : magics_(0), 
		page_(false), 
		geoTask_(0), 
		xyTask_(0),
		timer_("Magics processing", Log::info()), 
		x_("x"), y_("y")
	{
		if (singleton_) return;
		singleton_ = this;
		magics_ = new MagicsManager();
		output_.set(*magics_);
		
	}    
	~MagicsSingleton() { Log::debug() << "Delete MagicsSingleton\n"; }
	static MagicsSingleton* singleton_;

public:
	static PlotManager& manager() 
	{
		if (!singleton_) new MagicsSingleton();
		return *(singleton_->layout_);
	}

	static void checkPage()
	{
		if (!singleton_) new MagicsSingleton();
		singleton_->layout_->check(*(singleton_->magics_));
	}

	static void execute() 
	{
		assert(singleton_);
		Log::dev() << "Execute --->" << *(singleton_->magics_->root()) << "\n";
		singleton_->magics_->execute();
	}

	static VisualTask<GeoPoint>* getGeoTask()
	{
		assert(singleton_);
		return singleton_->geoTask();
	}

	static VisualTask<UserPoint>* getXYTask()
	{
		assert(singleton_);
		return singleton_->xyTask();
	}

	static void close() 
	{
		if (!singleton_) new MagicsSingleton();
		Log::dev() << "Close --->" << *(singleton_->magics_->root()) << "\n";
		singleton_->magics_->execute();
		singleton_->magics_->close();
		delete singleton_;
		singleton_ = 0;
	}

	static void createTask(Data<GeoPoint>* data) 
	{
		if (!singleton_) new MagicsSingleton();
		checkPage();
		singleton_->geoTask_ = new VisualTask<GeoPoint>();
		singleton_->layout_->add(singleton_->geoTask_);
		singleton_->geoTask_->set(data);
	}
    
	static void createTask(Data<UserPoint>* data) 
	{
		if (!singleton_) new MagicsSingleton();
		checkPage();
		singleton_->xyTask_ = new VisualTask<UserPoint>();
		singleton_->layout_->add(singleton_->xyTask_);
		singleton_->xyTask_->set(data);
	}
	static void superpage()
	{
		if (!singleton_) new MagicsSingleton();
		singleton_->layout_->superpage(*(singleton_->magics_));
		singleton_->page_ = false;
//		singleton_->magics_->resetRoot();
	}

	static void page()
	{
		if (!singleton_) new MagicsSingleton();
		singleton_->layout_->page(*(singleton_->magics_));
		singleton_->page_ = true;
//		singleton_->magics_->resetRoot();
	}
    
	static void subpage()
	{
		if (!singleton_) new MagicsSingleton();
		singleton_->layout_->subpage(*(singleton_->magics_));
		singleton_->page_ = true;
//		singleton_->magics_->resetRoot();
	}
    
	static void add(BaseSceneObject* object) 
	{ 
		singleton_->layout_->add(object); 
	}

	static void addVisualiser(Visualiser<UserPoint>* visdef) 
	{
		if (!singleton_) new MagicsSingleton();
		if (!singleton_->xyTask_)
		{ 
			XYList* input = new XYList();        
			MagicsSingleton::createTask(input);
			Log::dev() << *input << "\n";
			//throw MagicsException("Plotting routine called prior any Data definition--> The call is ignored");
		}
		singleton_->xyTask_->set(visdef);
		singleton_->xyTask_ = 0;
	}

	static void addVisualiser(Visualiser<GeoPoint>* visdef) 
	{
		if (!singleton_) new MagicsSingleton();
		if (!singleton_->geoTask_)
		{ 
			InputMatrix* input = new InputMatrix();     
			if (input->defined() ) {
				MagicsSingleton::createTask(input);
				Log::dev() << *input << "\n";
			}
			else {
				delete input;
				GribDecoder* grib = new GribDecoder();        
				MagicsSingleton::createTask(grib);
				Log::dev() << *grib << "\n";
			}
			
			
			//throw MagicsException("Plotting routine called prior any Data definition--> The call is ignored");
		}
		singleton_->geoTask_->set(visdef);
		singleton_->geoTask_ = 0;
	}
	VisualTask<GeoPoint>* geoTask() { return geoTask_; }
	VisualTask<UserPoint>* xyTask() { return xyTask_; }
	static bool xIsDate() { 
		if (!singleton_) new MagicsSingleton(); 
		return (singleton_->x_ == "date"); 
	}
	static bool yIsDate() { 
		if (!singleton_) new MagicsSingleton(); 
		return (singleton_->y_ == "date"); 
	}
	static void xIsDate(bool date) { 
		if (!singleton_) new MagicsSingleton(); 
		singleton_->x_ = date  ? "date" : "x"; 
	}
	static void yIsDate(bool date) { 
		if (!singleton_) new MagicsSingleton(); 
		singleton_->y_ = date  ? "date" : "y"; 
	}
	static void setHorizontalAxis(HorizontalAxis* axis) { 
		if (!singleton_) new MagicsSingleton(); 
		singleton_->haxis_ = axis; 
	}
	static void setVerticalAxis(VerticalAxis* axis) { 
		if (!singleton_) new MagicsSingleton(); 
		singleton_->vaxis_ = axis; 
	}
	
	static bool isCartesianSystem() {
		
		return singleton_->haxis_ && singleton_->vaxis_; 
		
	}
	
	static void setCartesianSystem() {
		checkPage();
		if (singleton_->haxis_) {
			add(singleton_->haxis_);
			singleton_->haxis_ = 0;
		}
		if (singleton_->vaxis_) {
			add(singleton_->vaxis_);
			singleton_->vaxis_ = 0;
		}
		
	}
protected :
	MagicsManager*  magics_;
	bool 			page_;
	bool            superpage_;
	VisualTask<GeoPoint>* geoTask_;
	VisualTask<UserPoint>* xyTask_;
	Timer timer_;
	string x_;
	string y_;
	HorizontalAxis* haxis_;
	VerticalAxis*   vaxis_;
	OutputHandler   output_;
};
} // namespace magics
#endif

