/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file Proj4Projection.h
    \brief Definition of Proj4Projection.
    \author Meteorological Visualisation Section, ECMWF

    Started: Tue May 18 17:39:58 2010
*/

#ifndef _Proj4Projection_H
#define _Proj4Projection_H

#include <Transformation.h>
#include <Proj4ProjectionAttributes.h>
#include <XmlNode.h>
#include <XmlNode.h>
#include <proj_api.h>

namespace magics
{

class MetaDataVisistor;
class Epsg;
/*! \class Proj4Projection
    \brief Implements a new projection
    \ingroup projection

    This projection ...
*/


class Proj4Projection: public Transformation, public Proj4ProjectionAttributes
{

public:
	Proj4Projection();
	~Proj4Projection();

	/*!
	  \brief sets  from an XML node
	*/
	void set(const XmlNode& node)
	{
        Transformation::set(node);
        Proj4ProjectionAttributes::set(node);
        init();
	}
   /*!
	  \brief sets  from a map
	*/
	void set(const map<string, string>& map)
	{
        Transformation::set(map);
        Proj4ProjectionAttributes::set(map);
        init();
	}
    
    virtual Transformation* clone() const {
		Proj4Projection* transformation = new Proj4Projection();
        transformation->copy(*this);
		return transformation;
	}
	
	/*!
	\\brief Initialise the projection
	*/
	virtual void init() ;
	/*!
	\\brief 
	*/
	virtual PaperPoint operator()(const UserPoint&) const;
	/*!
	\\brief 
	*/
	virtual PaperPoint operator()(const GeoPoint&) const;
	/*!
	\\brief 
	*/
	virtual PaperPoint operator()(const PaperPoint&) const;
	/*!
	\\brief 
	*/
	virtual void revert(const PaperPoint&, GeoPoint&) const;
	/*!
	\\brief 
	*/
	virtual void revert(const PaperPoint&, UserPoint&) const;
	
	void revert(const vector<pair<double, double> > & in, vector<pair<double, double> > & out) const;
	
	/*!
	\\brief Does the projection needs the coastalines to be shifted!
	*/
	virtual bool needShiftedCoastlines() const;
	/*!
	\\brief set the aspect ratio!
	*/
	virtual void aspectRatio(double&, double&) ;
	/*!
	\\brief set the bounding box!
	*/
	virtual void boundingBox(double&, double&, double&, double&) const;
	/*!
	\\brief return the xmin in user coordinates!
	*/
	virtual double getMinX() const;
	/*!
	\\brief return the ymin in user coordinates!
	*/
	virtual double getMinY() const;
	/*!
	\\brief return the xmax in user coordinates!
	*/
	virtual double getMaxX() const;
	/*!
	\\brief return the ymax in user coordinates!
	*/
	virtual double getMaxY() const;
	/*!
	\\brief set the xmin in user coordinates!
	*/
	virtual void setMinX(double) ;
	/*!
	\\brief return the ymin in user coordinates!
	*/
	virtual void setMinY(double) ;
	/*!
	\\brief return the xmax in user coordinates!
	*/
	virtual void setMaxX(double) ;
	/*!
	\\brief return the ymax in user coordinates!
	*/
	virtual void setMaxY(double) ;
	/*!
	\\brief return the xmin in projection coordinates!
	*/
	virtual double getMinPCX() const;
	/*!
	\\brief return the ymin in projection coordinates!
	*/
	virtual double getMinPCY() const;
	/*!
	\\brief return the xmax in projection coordinates!
	*/
	virtual double getMaxPCX() const;
	/*!
	\\brief return the ymax in projection coordinates!
	*/
	virtual double getMaxPCY() const;
	/*!
	\\brief create the grid for the longitudes!!
	*/
	virtual void gridLongitudes(const GridPlotting&) const;
	/*!
	\\brief create the grid for the latitudes!!
	*/
	virtual void gridLatitudes(const GridPlotting&) const;
	/*!
	\\brief calculate the labels
	*/
	virtual void labels(const LabelPlotting&, DrawingVisitor&) const;
	/*!
	\\brief calculate the left labels
	*/
	virtual void labels(const LabelPlotting&, LeftAxisVisitor&) const;
	/*!
	\\brief calculate the right labels
	*/
	virtual void labels(const LabelPlotting&, RightAxisVisitor&) const;
	/*!
	\\brief calculate the bottom labels
	*/
	virtual void labels(const LabelPlotting&, BottomAxisVisitor&) const;
	/*!
	\\brief calculate the top labels
	*/
	virtual void labels(const LabelPlotting&, TopAxisVisitor&) const;
	virtual void coastSetting(map<string, string>&, double, double) const;
	void visit(MetaDataVisitor& visitor, double left, double top, double width, double height);
	void setNewPCBox(double minx, double miny, double maxx, double maxy);
	void verticalLabels(const LabelPlotting& label, double x, double pos, Justification justif)  const;
	void horizontalLabels(const LabelPlotting& label, double y, double pos, VerticalAlign align)  const;
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const; 
	 projPJ from_;
	 projPJ to_;
	 double min_pcx_;
	 double max_pcx_;
	 double min_pcy_;
	 double max_pcy_;
	 Epsg*   projection_;

private:
    //! Copy constructor - No copy allowed
	Proj4Projection(const Proj4Projection&);
    //! Overloaded << operator to copy - No copy allowed
	Proj4Projection& operator=(const Proj4Projection&);

// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const Proj4Projection& p)
		{ p.print(s); return s; }

};
    

} // namespace magics
#endif
