/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file VisualTask.h
    \brief Definition of the Template class VisualTask.
    
    Magics Team - ECMWF 2004
    
    Started: Mon 19-Jan-2004
    
    Changes:
    
*/

#ifndef VisualTask_H
#define VisualTask_H

#include "magics.h"

#include "Task.h"
#include "Data.h"
#include "Visualiser.h"
#include "Log.h"

namespace magics {

template <class P> class Data;
template <class P> class Visualiser;

template <class P>
class VisualTask: public Task {

public:
	VisualTask() : data_(0), visualiser_(0) { Log::debug() << "New VisualTask" << "\n";  };
	virtual ~VisualTask() { 
		
	}
    
	void visit(TitleBase& title) { 
		for ( BaseSceneObjectList::const_iterator child = children_.begin(); child != children_.end(); ++child) {
			(*child)->specialise(*this);
		}
		if (data_) (*data_).visit(title);
		if (visualiser_) (*visualiser_).visit(title); 
	}
    
	void visit(LegendBase& legend) {
		if (data_) (*data_).visit(legend);
		if (visualiser_) (*visualiser_).visit(legend); 
	}
   
    void visit(MetaDataVisitor& meta) {
		if (data_) (*data_).visit(meta);
		if (visualiser_) (*visualiser_).visit(meta); 
	}
    virtual void set(const XmlNode&) { }
	
	//! Implements the BaseSceneObject
	void prepareGraphics() { 
		for ( BaseSceneObjectList::const_iterator child = children_.begin(); child != children_.end(); ++child) {
			(*child)->specialise(*this);
		}
        if (visualiser_ && data_)  
        	(*visualiser_).preparePlot(*data_, *this);        	 
        if ( !data_) 
        	Log::warning() << "the plot is not valid : no data defined" << "\n";
        if ( !visualiser_) 
        	Log::warning() << "the plot is not valid : no action routine defined" << "\n";
        current_ = begin();
	} 
    
	void set(Data<P>* data)       { data_ = data;  }
	void set(Visualiser<P>* visu) { visualiser_ = visu; visualiser_->setParent(this); }
    
protected:
	//! Method to print string about this class on to a stream of type ostream (virtual).
	virtual void print(ostream& out) const 
	{
		if ( data_ && visualiser_) 
		out << "Visualise " << *data_ << " using " << *visualiser_ << "[" << size() << " objects]";
		else out << "VisualTask ";
	}

	Data<P>*       data_;
	Visualiser<P>* visualiser_; 

private:
	//! Copy constructor - No copy allowed
	VisualTask(const VisualTask<P>&);
	//! Overloaded << operator to copy - No copy allowed
	VisualTask<P>& operator=(const VisualTask<P>&);

// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const VisualTask<P>& p)
		{ p.print(s); return s; }
};

} // namespace magics
#endif
