/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file OpenGLMagnifier.cc
    \brief Implementation of OpenGLMagnifier
    \author Graphics Section, ECMWF

    Started: November 2007

*/

#include <OpenGLMagnifier.h>
#include <OpenGLDriver.h>
#include <OpenGLFontHandler.h>
#include <OpenGLMagnifierScale.h>
#include <OpenGLMagnifierButton.h>
#include <OpenGLLayoutNode.h>
#include <OpenGLNode.h>
#include <OpenGLStepNode.h>
#include <OpenGLAreaGroup.h>
#include <OpenGLAnimationControl.h>
#include <OpenGLTextureItem.h>
#include <OpenGLFboItem.h>
#include <OpenGLCore.h>

#include <MtInputEvent.h>

using namespace magics;

OpenGLRingData::OpenGLRingData(float w, float minr, float maxr, int n) :
			width_(w), minRadius_(minr), maxRadius_(maxr), ptNum_(n)
{	
	float dcol;
	float an=0.;
	float an_step=2.*PI/ptNum_;
	for(int i=0; i< ptNum_; i++)
	{		
		coeffCos_.push_back(cos(an));
		coeffSin_.push_back(sin(an));		
		an+=an_step;					
		
	}		
}
OpenGLMagnifier::OpenGLMagnifier(OpenGLDriver *driver,string name) : OpenGLGui(driver,name)
{		
	pointInMagnifier_=false;

	minRadius_=50.;
	
	//Default geometry 
	radius_=100.;	
	centreX_=radius_+20.;
	centreY_=radius_+20.;
	
	//Set text magnification factors	
	float valst[] = {1.5,2.,2.5,3.,3.5,4.};	
	vector<float> valst_vec(valst,valst+6);
	scale_[MGM_TEXT] = new OpenGLMagnifierScale(valst_vec,1,this,Colour(1.,0.,0.), Colour(1.,1.,1.));
	scale_[MGM_TEXT]->set(radius_);	
	
	//Set data magnification factors
	float valsd[] = {2.,5.,10.,15.,25.,50.,75.};	
	vector<float> valsd_vec(valsd,valsd+7);
	scale_[MGM_DATA]= new OpenGLMagnifierScale(valsd_vec,1,this,Colour(0.,0.,1.), Colour(1.,1.,1.));
	scale_[MGM_DATA]->set(radius_);	

	magnifierMode_=MGM_TEXT;
	actScale_=scale_[magnifierMode_];

	magnifierModeChanged_=false;

	//Set frame data
	frames_.push_back(OpenGLRingData(6.,20.,150.,60));
	frames_.push_back(OpenGLRingData(6.,150.,300.,80));
	frames_.push_back(OpenGLRingData(6.,300.,1000.,100));
	
	actFrameIndex_=0;
		
	fontHandler_= new OpenGLFontHandler;

	//button_.push_back(new OpenGLMagnifierButton(this));
	//button_[0]->set(radius_);

	gridValPlotEnabled_=false;
	gridValArea_=new OpenGLGridValArea;

	std::set<int> areaIndex;
	areaIndexStack_.push(areaIndex);

	fbo_=0;
	tex_=0;

	build(); 
	
}	 
	
 
OpenGLMagnifier::~OpenGLMagnifier()
{
	for(std::map<OpenGLMagnifierMode,OpenGLMagnifierScale*>::iterator it= scale_.begin(); it != scale_.end(); it++)
	{
		delete it->second;
	}

        clear();


	if(fbo_)
	{
		delete fbo_;
		delete tex_;
	}
}

//==========================================================
//
// Build, show and hide the magnifier 
//
//==========================================================


void OpenGLMagnifier::build() 
{			
	//We want to call it once
	if(built_==true) return;

	pointInMagnifier_=false; 

	init(0,0,driver_->dimensionX(),driver_->dimensionY());
	
	actScale_->set(radius_);	

	if(magnifierMode_ == MGM_DATA)
	{		
		calcPowerScaleForData(); 
		updateGridValArea(true);
	}	

	if(fbo_==0 && OpenGLCore::instance()->renderMode()  == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->initMag();
		
		fbo_=OpenGLCore::instance()->fboMag();
		tex_=OpenGLCore::instance()->texMag();
	}
	
	built_=true;
	enabled_=true;		
}	

void OpenGLMagnifier::show()
{
	//Build the gui
	if(built_==false)
	{
		build();
	}
	render(true);
}

void OpenGLMagnifier::hide()
{	
	pointInMagnifier_=false;
	
	dragType_=OpenGLWidget::NoDrag;
	
	//Delete the display lists
	clear();

	//Delete fonts
	fontHandler_->print(cout);
	fontHandler_->clear();
	
	//Map the saved bg to fb
	//driver_->mapBgImageToFb(1);
		
	//Copy fb to display	
        //driver->swapFb();  	

	driver_->checkError("Magnifier");	
}	


void OpenGLMagnifier::init(const int x0, const int y0, const int x1, const int y1)
{
	minX_=x0;
	maxX_=x1;
	minY_=y0;
	maxY_=y1;
}
	
void OpenGLMagnifier::clear()
{
	//radius_=100;
		
	//centreX_=radius_+20.;
	//centreY_=radius_+20.;	
	
	glDeleteLists(dpyListFrame_,1);
	dpyListFrame_=0;

        glDeleteLists(dpyListScale_,1);
	dpyListScale_=0;

	clearTextDpl();

	driver_->clearMagnifierLayout();
}

bool OpenGLMagnifier::checkCircleInWindow(const float x,const float y,const float r) const
{	
	if(x+r < maxX_ && x-r > minX_ && y+r < maxY_ && y-r > minY_)
	{
		return true;
	}
	else	
	{
		return false;	
	}
}


bool OpenGLMagnifier::checkPointInWindow(const float x,const float y) const
{
	if((x-minX_)*(maxX_-x) < 0 || (y-minY_)*(maxY_-y) < 0)
	{
		return false;
	}
	else
	{
		return true;	
	}
}

bool OpenGLMagnifier::checkPointInMagnifier(const int x, const int y)
{
	float dx=x-centreX_;
	float dy=y-centreY_;		
	float dist=sqrt(dx*dx+dy*dy);

	if(dist < radius_)
	{  
		return true;
	}
	return false;
}


OpenGLMagnifier::OpenGLMagnifierPointType OpenGLMagnifier::identifyPoint(const int x, const int y)
{	
	//If it is the first time
	if(centreX_ < -99999.) 
	{
		centreX_=x;
		centreY_=y;
	}	
	
	float dx=x-centreX_;
	float dy=y-centreY_;		
	float dist=sqrt(dx*dx+dy*dy);
	
	int i;
	/*if((i=scaleText_->identifyPointInScale(dist,dx,dy)) != -1)
	{
		return MGP_CELL;
	}*/
	
	if(dist < radius_-10.)
	{  
		return MGP_LENS;
	}
	else if(dist >= radius_-10 && dist <= radius_+ 10)			
	{
		return MGP_FRAME;
	}
				
	return MGP_OUTSIDE;
		
}
	

void OpenGLMagnifier::replace(const int x, const int y)
{	
	float r=radius_*0.6;
	
	if(checkPointInWindow(x+r,y) == true || checkPointInWindow(x-r,y) == true ||
	   checkPointInWindow(x,y+r) == true || checkPointInWindow(x,y-r) == true)
	{
		centreX_=x;
		centreY_=y;
		updateGridValArea();		
	}		
}	



void OpenGLMagnifier::resize(const int x, const int y)
{
	float dist=sqrt(static_cast<float>((centreX_-x)*(centreX_-x)+(centreY_-y)*(centreY_-y)));
	
	//Check if magnifying glass is inside window and size is not too small
	if(radius_ >= minRadius_ && checkCircleInWindow(centreX_,centreY_,dist) == true)				
	{		
		radius_=dist;		
		if(radius_<minRadius_) radius_=minRadius_;				

		setFrameData();	
		updateGridValArea();							
	}					
}	



void OpenGLMagnifier::remagnify()
{
	//Clear the text rendering dpl
	clearTextDpl();
	
	//Delete fonts	
	//fontHandler_->clear();
	updateGridValArea();	

	//Re-render
	render(true);
}


void OpenGLMagnifier::setFrameData()
{
	//Set scale according to new radius
	actScale_->set(radius_);

	//Frame 
	for(int i=0; i< frames_.size(); i++)
	{
		if(frames_[i].checkRadius(radius_) == true)
		{
				actFrameIndex_=i;
				return;
		}
	}
	actFrameIndex_=0;
			
}		
		


void OpenGLMagnifier::renderScale()
{
	render(true);
}        


void OpenGLMagnifier::render(const bool sizeChanged)
{			
	int i,nn;
	float sqrt2=0.707106;
	float zoom=actScale_->actValue();						
	
	//Define the frame data
	OpenGLRingData *fr=&frames_[actFrameIndex_];		
	
	float xt[fr->ptNum_],yt[fr->ptNum_],xs[fr->ptNum_],ys[fr->ptNum_];
	
	for(i=0; i< fr->ptNum_; i++)
	{		
		xt[i]=centreX_+fr->coeffCos_[i]*radius_;
		yt[i]=centreY_+fr->coeffSin_[i]*radius_;
	
		xs[i]=radius_+fr->coeffCos_[i]*radius_;
		ys[i]=radius_+fr->coeffSin_[i]*radius_;								
	}	
		
	//Save the colour and polygon
	glPushAttrib(GL_CURRENT_BIT);

	//----------------------------------------------------				
	// Render the lens area by traversing the opengl tree
	// (it is a rectangular area)
	//----------------------------------------------------

	//Bind fbo
	if(fbo_)
	{
		fbo_->bind();
		glClear(GL_COLOR_BUFFER_BIT);
	}

	//Set viewport
	glMatrixMode(GL_MODELVIEW);
        glPushAttrib(GL_VIEWPORT_BIT);
	glViewport(centreX_-radius_,centreY_-radius_,2.*radius_,2.*radius_);

	//Set projection matrix for magnification
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	
	glOrtho(centreX_-radius_/zoom,centreX_+radius_/zoom,
	        centreY_-radius_/zoom,centreY_+radius_/zoom,
				-1.,1.);	

	//Render the magnified plot
	glMatrixMode(GL_MODELVIEW);			
	renderLens();

	//Reset projection matrix
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	
	//Reset viewport
	glPopAttrib();

	//Unbind fbo
	if(fbo_)
	{
		fbo_->unBind();
	}

	//------------------------------------------
	// Now render the frame + lens area
	//------------------------------------------		

	//Bind fbo
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->fboFb()->bind();	
	}
	
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();				

	//Map the magnified plot (in a texture) into the
	//lens area ( in the frambuffer texture)
	vector<PaperPoint> sp, tp;
	for(i=0; i< fr->ptNum_; i++)
	{
		float x=static_cast<float>(xt[i])/driver_->dimensionX();
		float y=static_cast<float>(yt[i])/driver_->dimensionY();
		sp.push_back(PaperPoint(x,y));
		tp.push_back(PaperPoint(xt[i],yt[i]));	
	}		

	tex_->mapTexture(sp,tp);
	
	//Render the frame			

	glTranslatef(centreX_,centreY_,0.);

	//If the size has not changed and the display list is defined
	//just call the display list 	
	if(!sizeChanged && glIsList(dpyListFrame_) == GL_TRUE)
	{
		glCallList(dpyListFrame_);
	}		
	//Else redraw the frame and put it into the display list as well
	else
	{				
		if(glIsList(dpyListFrame_) == GL_FALSE)
		{
			dpyListFrame_=glGenLists(1);
		}	
	
		glNewList(dpyListFrame_,GL_COMPILE_AND_EXECUTE);
		
		renderFrame(fr);
		
		glEndList();	
	}

        //If the size has not changed and the display list is defined
	//just call the display list 				
	if(!sizeChanged && glIsList(dpyListScale_) == GL_TRUE)
	{
		glCallList(dpyListScale_);
	}		
	//Else redraw the frame and put it into the display list as well
	else
	{						
		if(glIsList(dpyListScale_) == GL_FALSE)
		{
			dpyListScale_=glGenLists(1);
		}	
		
		glNewList(dpyListScale_,GL_COMPILE_AND_EXECUTE);

                actScale_->render();

		glEndList();
        }		

	//Modelview
	glPopMatrix();

	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->fboFb()->unBind();		
	}

	//Colour and polygon
	glPopAttrib();	
}

void OpenGLMagnifier::renderFrame(OpenGLRingData *fr)
{		
	float dcol,xx,yy,rad1,rad2;
	int i, i_next;		

	//Save state
	glPushAttrib(GL_CURRENT_BIT | GL_POLYGON_BIT | GL_ENABLE_BIT | GL_LINE_BIT);

	//Only front sides are rendered
	glFrontFace(GL_CCW);
	glEnable(GL_CULL_FACE);	
	glCullFace(GL_BACK);	
	glPolygonMode(GL_FRONT,GL_FILL);
				
	rad1=radius_;
	rad2=rad1+fr->width_;
		
	//------------------------------
	// Render the frame area
	//------------------------------

	//Polygons have to be CCW!
		
	float fnum=fr->ptNum_;
	for(i=0; i< fr->ptNum_; i++)
	{		
		i_next=(i< fr->ptNum_-1)? i+1 : 0;
			
		glBegin( GL_QUADS);
		
		if(i< 0.375*fnum)
		{
			dcol=0.8*2.*fabs(i+0.125*fnum)/fnum;	
		}
		else
		{
			dcol=0.8*2.*fabs(i-0.875*fnum)/fnum;
		}

		glColor3f(0.2+dcol,0.2+dcol,0.2+dcol);		
		
		//glColor3f(fr->colour_.red(),fr->colour_.green(),fr->colour_.blue());
		
		xx=fr->coeffCos_[i]*rad1;
		yy=fr->coeffSin_[i]*rad1;		
		glVertex2f(xx,yy);
			
		xx=fr->coeffCos_[i]*rad2;
		yy=fr->coeffSin_[i]*rad2;					
		glVertex2f(xx,yy);
			
		xx=fr->coeffCos_[i_next]*rad2;
		yy=fr->coeffSin_[i_next]*rad2;		
		glVertex2f(xx,yy);
			
		xx=fr->coeffCos_[i_next]*rad1;
		yy=fr->coeffSin_[i_next]*rad1;		
		glVertex2f(xx,yy);		
			
		glEnd();		
	}
				
	glPolygonMode(GL_FRONT,GL_LINE);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);		

	//------------------------------
	//Draw the frame inner outline
	//------------------------------

	glLineWidth(3);
	rad1=radius_;
	//glColor3f(0.,0.,0.);
	for(i=0; i< fr->ptNum_; i++)
	{		
			i_next=(i< fr->ptNum_-1)? i+1 : 0;
			
			if(i< 0.375*fnum)
			{
				dcol=0.8*2.*fabs(i+0.125*fnum)/fnum;	
			}
			else
			{
				dcol=0.8*2.*fabs(i-0.875*fnum)/fnum;
			}
			
			glColor3f(0.8-dcol,0.8-dcol,0.8-dcol);
			
			glBegin(GL_LINES);			
		
			xx=fr->coeffCos_[i]*rad1;
			yy=fr->coeffSin_[i]*rad1;		
			glVertex2f(xx,yy);
						
			xx=fr->coeffCos_[i_next]*rad1;
			yy=fr->coeffSin_[i_next]*rad1;		
			glVertex2f(xx,yy);		
			
			glEnd();		
	}
		
	//------------------------------				
	//Draw the frame outer outline	
	//------------------------------	

	glLineWidth(1);
	rad1=radius_+fr->width_;			
	for(i=0; i< fr->ptNum_; i++)
	{		
			i_next=(i< fr->ptNum_-1)? i+1 : 0;
			
			glBegin(GL_LINES);
		
			dcol=0.5*2.*static_cast<float>(fr->ptNum_/2-abs(fr->ptNum_/2-i))/static_cast<float>(fr->ptNum_);			
			//glColor3f(0.9-dcol,0.9-dcol,0.9-dcol);
			glColor3f(0.3,0.3,0.3);
			
			xx=fr->coeffCos_[i]*rad1;
			yy=fr->coeffSin_[i]*rad1;		
			glVertex2f(xx,yy);
						
			xx=fr->coeffCos_[i_next]*rad1;
			yy=fr->coeffSin_[i_next]*rad1;		
			glVertex2f(xx,yy);		

			glEnd();		
	}	
		
	glPopAttrib();
		
}




//==========================================================
//
// Visit the tree nodes
//
//==========================================================

void OpenGLMagnifier::visit(OpenGLNode& node)
{	
	(this->*visitFunc_)(node);
}

bool OpenGLMagnifier::visitEnter(OpenGLNode& node)
{	
	return (this->*visitEnterFunc_)(node);
}

bool OpenGLMagnifier::visitLeave(OpenGLNode& node)
{	
	return (this->*visitLeaveFunc_)(node);
}

//==========================================================
//
// Visitors: rendering the lens area
//
//==========================================================

void OpenGLMagnifier::renderLens()
{
	visitFunc_=&OpenGLMagnifier::visit_renderLens;
	visitEnterFunc_=&OpenGLMagnifier::visitEnter_renderLens;
	visitLeaveFunc_=&OpenGLMagnifier::visitLeave_renderLens;

	/*glPushAttrib(GL_COLOR_BUFFER_BIT | GL_LINE_BIT | GL_HINT_BIT );
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);	*/

	//driver_->checkError("renderLens1");

	//glPushAttrib(GL_SCISSOR_BIT);
	//glEnable(GL_SCISSOR_TEST);
	//glScissor(centreX_-radius_,centreY_-radius_,2.*radius_,2.*radius_);

	driver_->glTree()->accept(*this);


	//driver_->checkError("renderLens2");

	//glPopAttrib();

	//driver_->checkError("renderLens3");

	assert(areaIndexStack_.size() == 1);

	//glPopAttrib();
}


bool OpenGLMagnifier::visitEnter_renderLens(OpenGLNode& node)
{	
	float zoom=actScale_->actValue();	

	if(node.classType() == "OpenGLStepNode")
	{
		OpenGLStepNode *step = (OpenGLStepNode*)&node;
		if(driver_->stepToRender_ != step->id())
		{
			return false;					
		}
		
		return true;

	}	
	else if(node.classType() == "OpenGLPreviewLayoutNode")
	{		
		return false;
	}					
	else if(magnifierMode_ == MGM_TEXT && 
	        node.classType() == "OpenGLMagnifierLayoutNode")
	{
		return false;
	}

	//Render preTransform
	
	/*glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	
	glOrtho(centreX_-radius_/zoom,centreX_+radius_/zoom,
	        centreY_-radius_/zoom,centreY_+radius_/zoom,
				-1.,1.);*/	
									
	glMatrixMode(GL_MODELVIEW);				
	
	node.preTransformDpl().call();
	
	/*glMatrixMode(GL_PROJECTION);
	glPopMatrix();*/

	if(node.classType().find("LayoutNode") != string::npos)
	{				
		std::set<int> areaIndex;

		//currentAreaIndex_.clear();
		OpenGLLayoutNode *gll= (OpenGLLayoutNode *) &node;
		OpenGLAreaGroup *grp=gll->areaGroup();	
		if(grp)
		{
			grp->findAreas(centreX_-radius_/zoom, centreY_-radius_/zoom,
				       2.*radius_/zoom,2.*radius_/zoom,	        		      
				       areaIndex);

			//for(std::set<int>::iterator it=areaIndex.begin(); it != areaIndex.end(); it++)
			//{
			//	Log::dev() << "set: " << *it << endl;
			//}				
		}

		areaIndexStack_.push(areaIndex);
	}

	return true;
}

bool OpenGLMagnifier::visitLeave_renderLens(OpenGLNode& node)
{			
	if(node.classType() == "OpenGLStepNode")
	{
		OpenGLStepNode *step = (OpenGLStepNode*)&node;
		if(driver_->stepToRender_ != step->id())
		{
			return true;					
		}
	}	

	else if(node.classType() == "OpenGLPreviewLayoutNode")
	{		
		return true;
	}
	else if(magnifierMode_ == MGM_TEXT && 
		node.classType() == "OpenGLMagnifierLayoutNode")
	{
		return true;
	}

	float zoom=actScale_->actValue();	
	
	//Render postTransform	
	/*glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	
	glOrtho(centreX_-radius_/zoom,centreX_+radius_/zoom,
	        centreY_-radius_/zoom,centreY_+radius_/zoom,
				-1.,1.);	
	*/								
	glMatrixMode(GL_MODELVIEW);			
	
	node.postTransformDpl().call();

	/*glMatrixMode(GL_PROJECTION);
	glPopMatrix();*/

	if(node.classType().find("LayoutNode") != string::npos)
	{				
		areaIndexStack_.pop();
	}

	return true;	
}


void OpenGLMagnifier::visit_renderLens(OpenGLNode& node)
{
	//draw node
	float zoom=actScale_->actValue();	
	
	//Log::dev() <<  "Executing: " <<  node.name() << " (objects: " << node.dpl().size() << ")" << endl;
	
	assert(areaIndexStack_.size() >= 1);

	//for(std::set<int>::iterator it=areaIndexStack_.top().begin(); it != areaIndexStack_.top().end(); it++)
	//{
	//			Log::dev() << "set: " << *it << endl;
	//}
	
	//Render displaylist	
	for(OpenGLDplManager::DplIterator it=node.dpl().firstItem(); it != node.dpl().lastItem(); it++)
	{
		if((*it)->type() == M_DPL_NORMAL && 
		   ( (*it)->areaIndex() == -1 ||
		      areaIndexStack_.top().count((*it)->areaIndex()) != 0) )
		{				
			//Log::dev() << "index: " <<  (*it)->areaIndex() << endl;

			/*glMatrixMode(GL_PROJECTION);
			glPushMatrix();
			glLoadIdentity();
			
			glOrtho(centreX_-radius_/zoom,centreX_+radius_/zoom,
	        		centreY_-radius_/zoom,centreY_+radius_/zoom,
				-1.,1.);*/	
									
			glMatrixMode(GL_MODELVIEW);			
			
			if((*it)->clipping())
			{
				glPushAttrib(GL_SCISSOR_BIT);

				OpenGLLayoutNode *gll=(*it)->parentLayout();
				
				glEnable(GL_SCISSOR_TEST);
				
				OpenGLBox lbox(gll->winX()-(zoom-1)*(centreX_-gll->winX()),
				     gll->winY()-(zoom-1)*(centreY_-gll->winY()),
				     gll->winWidth()*zoom,gll->winHeight()*zoom);
	
				OpenGLBox mbox(centreX_-radius_,centreY_-radius_,2.*radius_,2.*radius_);
				OpenGLBox sbox;

				if(lbox.computeBoxSection(&mbox,&sbox))
				{
					//glScissor(gll->winX(),gll->winY(),gll->winWidth(),gll->winHeight());
					glScissor(sbox.x(),sbox.y(),sbox.width(),sbox.height());
				}
			}

			(*it)->call();				
			
			if((*it)->clipping())
			{
				glDisable(GL_SCISSOR_TEST);
				glPopAttrib();
			}

			/*glMatrixMode(GL_PROJECTION);
			glPopMatrix();	*/						
		}
		

		else if(magnifierMode_== MGM_DATA && ((*it)->type() == M_DPL_TEXT_NOMAGNIFY || 
 			(*it)->type() == M_DPL_TEXT_MAGNIFY))
		//else if(magnifierMode_== MGM_DATA && (*it)->type() == M_DPL_TEXT_NOMAGNIFY) 		
		{				
			glMatrixMode(GL_PROJECTION);
			glPushMatrix();
			glLoadIdentity();
			
			glOrtho(zoom*centreX_-radius_,zoom*centreX_+radius_,
	        		zoom*centreY_-radius_,zoom*centreY_+radius_,
				-1.,1.);	
			
			glMatrixMode(GL_MODELVIEW);				
						
			if((*it)->compiled() == false)
			{
				(*it)->compileStart();
				driver_->renderTextForMagnifier((*it)->parentLayout(),
					    (*it)->text(),1,zoom);
				(*it)->compileEnd();				
			}
			
			(*it)->call();

			glMatrixMode(GL_PROJECTION);
			glPopMatrix();
		}

		else if(magnifierMode_== MGM_TEXT && (*it)->type() == M_DPL_TEXT_MAGNIFY)
		//else if((*it)->type() == M_DPL_TEXT_MAGNIFY)
		{		
			glMatrixMode(GL_PROJECTION);
			glPushMatrix();
			glLoadIdentity();
				
			glOrtho(zoom*centreX_-radius_,zoom*centreX_+radius_,
	        		zoom*centreY_-radius_,zoom*centreY_+radius_,
				-1.,1.);	
			
			glMatrixMode(GL_MODELVIEW);				
						
			if((*it)->compiled() == false)
			{
				(*it)->compileStart();
				driver_->renderTextForMagnifier((*it)->parentLayout(),
					    (*it)->text(),zoom,zoom);
				(*it)->compileEnd();				
			}
			
			(*it)->call();
			
			glMatrixMode(GL_PROJECTION);
			glPopMatrix();
			
		}

	}
	glMatrixMode(GL_MODELVIEW);	
}	

//==========================================================
//
// Visitors: clearing the text rendering display list
//
//==========================================================

void OpenGLMagnifier::clearTextDpl()
{
	visitFunc_=&OpenGLMagnifier::visit_clearTextDpl;
	visitEnterFunc_=&OpenGLMagnifier::visitEnter_clearTextDpl;
	visitLeaveFunc_=&OpenGLMagnifier::visitLeave_clearTextDpl;

	driver_->glTree()->accept(*this);
}

bool OpenGLMagnifier::visitEnter_clearTextDpl(OpenGLNode& node)
{	
	return true;	
}

bool OpenGLMagnifier::visitLeave_clearTextDpl(OpenGLNode& node)
{	
	return true;	
}

void OpenGLMagnifier::visit_clearTextDpl(OpenGLNode& node)
{		
	std::set<mglDplType> s;
	s.insert(M_DPL_TEXT_MAGNIFY);
	s.insert(M_DPL_TEXT_NOMAGNIFY);
	node.dpl().clearDplContent(s);
}	
	
			

//==========================================================
//
// Select and unselect magnifier
//
//==========================================================

void OpenGLMagnifier::drag_start(const int x, const int y) 
{		
	switch(identifyPoint(x,y))
	{				
		/*case MGP_CELL:
			break;	*/
		case MGP_LENS:
			dragType_=OpenGLWidget::ReplaceDrag;
			break;
		case MGP_FRAME:
			dragType_=OpenGLWidget::ResizeDrag; 
			break;
		case MGP_OUTSIDE:
			break;	
		default:
			break;
	}		
}


void OpenGLMagnifier::unselect() 
{	
	dragType_=OpenGLWidget::NoDrag; 
}


//==========================================================
//
// Callback functions
//
//==========================================================


void OpenGLMagnifier::event(MtInputEvent *event)
{
 	int x, y ,button, motion_state; 	

	//focused_=false;
	
	actScale_->event(event,centreX_,centreY_);
	//button_[0]->event(event,centreX_,centreY_);

	if(actScale_->active()) 
	{
		focused_=true;
		return;
	}

	//Check focus and cursor position
	MtMouseEvent* mev;	
	OpenGLMagnifierPointType mgp;
	if(event->type() == Mt::MouseMoveEvent)
	{ 		
		mev = (MtMouseEvent*) event;
		x = mev->x();
 		y = mev->y();
		mgp = identifyPoint(x,y);

		if(mgp == MGP_LENS || mgp == MGP_FRAME || dragType_ != OpenGLWidget::NoDrag)
		{
			focused_=true;
		}
		else
		{
			focused_=false;
			return;
		}
	}

	//if(mgp == MGP_OUTSIDE)
	//{
	//	return;
	//}	

	switch(event->type())
 	{
	
	case Mt::MousePressEvent:
		mev = (MtMouseEvent*) event;		
		x = mev->x();
 		y = mev->y();				

		mgp = identifyPoint(x,y);

		if(mev->button() & Mt::LeftButton)
		{
			if(dragType_ == OpenGLWidget::NoDrag)
 			{
				if(mgp == MGP_LENS)
				{
					dragType_=OpenGLWidget::ReplaceDrag;
				}
				else if(mgp == MGP_FRAME)
				{
					dragType_=OpenGLWidget::ResizeDrag; 
				}
			}					
		}
		
		else if(mev->button() & Mt::RightButton)
		{		
			if(mgp == MGP_LENS)
			{
				changeMagnifierMode();
			}	
		}			
 		break;	
		
 	case Mt::MouseMoveEvent: 		
		mev = (MtMouseEvent*) event;
		x = mev->x();
 		y = mev->y();

		if(mev->button()  & Mt::LeftButton)
		{					
			if(dragType_ == OpenGLWidget::ReplaceDrag)
 			{	
				replace(x,y);
			}
			else if(dragType_ == OpenGLWidget::ResizeDrag)
			{
				resize(x,y);
			}
		}
					
		
		break;	
		
	case Mt::MouseReleaseEvent:
		mev = (MtMouseEvent*) event;		
		x = mev->x();
 		y = mev->y();				
			
		unselect();		
		break;			
	}
}


bool OpenGLMagnifier::active()
{
	if(dragType_ != OpenGLWidget::NoDrag || actScale_->active() || magnifierModeChanged_==true)
	{
		if(magnifierModeChanged_)
		{
			magnifierModeChanged_=false;
		}

		return true;
	}
	else
	{
		return false;
	}


/*	if(pickMode_== PM_REPLACE || pickMode_== PM_RESIZE || pickMode_== PM_REMAGNIFY ||
	   pointInMagnifier_==true)
	{
		return true;
	}
	else
	{
		return false;
	}*/		
}

bool OpenGLMagnifier::updateGridValArea(bool forceUpdate)
{
	//if(gridValPlotEnabled_==false) return false;

	if(magnifierMode_ != MGM_DATA) return false;

 	bool needUpdate =gridValArea_->update(centreX_,centreY_,radius_,actScale_->actValue());

	if ( needUpdate == true || forceUpdate == true )
          
	{
		float x1=gridValArea_->x();
		float y1=gridValArea_->y();
		float x2=gridValArea_->x()+gridValArea_->width();
		float y2=gridValArea_->y()+gridValArea_->height();
		float dx=gridValArea_->dx();
		float dy=gridValArea_->dy();

		driver_->magnifierLayoutUpdate(x1,y1,x2,y2,dx,dy,actScale_->actValue()); 
		
		return true;
	}

	return false;

}

bool OpenGLGridValArea::update(float cx,float cy, float radius, float zoom)
{	
	float x1=cx-radius/zoom; 
	float x2=cx+radius/zoom;
	float y1=cy-radius/zoom;
	float y2=cy+radius/zoom;


	Log::debug() << "area: " << x1 << " " << x2 << " " << y1 << " " << y2 << endl;
	Log::debug() << "      " << x_ << " " << x_+width_ << " " << y_ << " " << y_+height_ << endl; 


	if(zoom_ != zoom || checkArea(x1,y1,x2,y2) == false)
	{
		Log::debug() << "      " << "Magnifier data area has CHANGED!" << endl;

		zoom_=zoom;

		x_=cx-sizeFactor_*radius/zoom; 
		width_=cx+sizeFactor_*radius/zoom-x_;
		y_=cy-sizeFactor_*radius/zoom;
		height_=cy+sizeFactor_*radius/zoom-y_;

		float textWidth = 70.+20;
		float textheight = 40.;
		
		dx_=textWidth/zoom;
		dy_=textheight/zoom;
		
		return true;
	}
	
	return false;
}

bool OpenGLGridValArea::checkArea(float x1,float y1, float x2, float y2)
{
	return x1 >= x_ && x2 <= x_ + width_ && y1 >= y_ && y2 <= y_+height_;
}	

bool OpenGLMagnifier::changeMagnifierMode()
{
	if(magnifierMode_ == MGM_TEXT)
	{
		magnifierMode_ = MGM_DATA;
	}
	else
	{
		magnifierMode_ = MGM_TEXT;
	}

	actScale_=scale_[magnifierMode_];
	actScale_->set(radius_);

	glDeleteLists(dpyListScale_,1);
	dpyListScale_=0;
	clearTextDpl();	

	if(magnifierMode_ == MGM_DATA)
	{
		driver_->clearMagnifierLayout();
		calcPowerScaleForData(); 
		updateGridValArea(true);
	}
	else
	{
		actScale_->set(radius_);
	}

	magnifierModeChanged_=true;
	
}

void OpenGLMagnifier::currentFrameChanged()
{
	clearTextDpl();
	
	if(magnifierMode_ == MGM_DATA)
	{
		driver_->clearMagnifierLayout();

		float orival=actScale_->maxValue();
		float orinum=actScale_->valueNum();

		bool changed=calcPowerScaleForData(); 

		if(changed)
		{
			glDeleteLists(dpyListScale_,1);
			dpyListScale_=0;
		}
		updateGridValArea(true);
	}		
}


bool OpenGLMagnifier::calcPowerScaleForData()
{
	if(magnifierMode_ != MGM_DATA) return false;

	float maxp=driver_->calcMagnifierMaxPower();

	vector<float> fv;

	if(maxp <=5  )
	{
		float fa[] = {2.,3.,4.,5.};	
		fv.insert(fv.end(),fa,fa+4);	
	}
	else if(maxp < 15)
	{
		float fa[] = {2.,5.,10.,15.};
		fv.insert(fv.end(),fa,fa+4);		
	}
	else
	{
		float fa[] = {2.,5.,10.,15.,25.};
		fv.insert(fv.end(),fa,fa+5);			
	}
	if(actScale_->maxValue() != fv.back() ||
           actScale_->valueNum() != fv.size())
	{
		actScale_->setFactors(fv,0); 
		actScale_->set(radius_); 
		return true;
	}
	else
	{
		return false;
	}
}