/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file OpenGLPickObject.cc
    \brief Implementation of OpenGLPickObject.
    \author Graphics Section, ECMWF

    Started: October 2007

*/
#include <GL/gl.h>
//#include <GL/glu.h>

#include <OpenGLPickObject.h>

using namespace magics;

 
OpenGLPickObject::OpenGLPickObject(OpenGLBox* box, double x, double y, double w, double h, OpenGLBox* pbox) : bBox_(x,y,w,h)
{
	//Set bounding box
	//bBox_ = new Box(box->name(),x,y,w,h);
/*	const float xmin = box->xmin();
	const float xmax = box->xmax();
	const float ymin = box->ymin();
	const float ymax = box->ymax();	
	bBox_.setCoordinates(xmin,xmax,ymin,ymax);
	bBox_.info(box->info());
	 
 	//Set pointer to the original magics box
	magBox_=box;
	
	//Set parent
	parentBox_ = pbox;
	
	//Set type
	type_=POT_MAGBOX;
	
	initBox();
	
	//
	modifyStatus_=false;*/

}	 
	
 
OpenGLPickObject::OpenGLPickObject(OpenGLBox* pbox, PickObjectType type) : bBox_(0.,0.,0.,0.)
{	
	//This pickobject is assigned to a user created  object
	magBox_=0;
	
	//Set parent
	parentBox_ = pbox;
	
	//Set type
	type_=type;
	
	//
	modifyStatus_=false;		
}	 


OpenGLPickObject::~OpenGLPickObject()
{			
	for(int i=0; i< objBox_.size();i++)
	{
		delete objBox_[i];
	}
	
	OpenGLBox* box;
	
	for(int i=0; i< cornerBox_.size();i++)
	{
		box=cornerBox_[i];
		delete box;
	}
	
	for(int i=0; i< sideBox_.size();i++)
	{
		box=sideBox_[i];
		delete box; 		
	}	
	 
	//Set boxinfo to 0 without deleting it to prevent its
	//deletion when the Box desctructor is called. Our boxinfo in 
	//bBox is just a copy of the oriuoginal pointer!!! 
	//bBox_.info(0,false); 
}

void OpenGLPickObject::initBox()
{
	createPickBox();
	//setBoundingBox();
	updatePickBox();
}		
	
void OpenGLPickObject::createPickBox()
{	
	int i;
	
	//Create pick boxes around object points if only there are more
	//than two points. Otherwise only the bounding box is
	//used.
	if(obj_.size()>2)
	{			
		for(i=0; i< obj_.size(); i++)
		{		
			string s("object_");
			stringstream sbuff("", ios_base::out);
			sbuff << i;
			s.append(sbuff.str());
			//magics::OpenGLBox *pbox = new magics::OpenGLBox(s,0,0,4,4);
			magics::OpenGLBox *pbox = new magics::OpenGLBox(0,0,4,4);
			objBox_.push_back(pbox);
		}
	}	
	
	//Create pick boxes at bounding box corners
	for(i=0;i< 4; i++)
	{
		string s("corner_");
		stringstream sbuff("", ios_base::out);
		sbuff << i;
		s.append(sbuff.str());
		magics::OpenGLBox *pbox = new magics::OpenGLBox(0,0,7,7);
		cornerBox_.push_back(pbox);
    	}  
	
	//Create pick boxes at bounding box sides 
	for(i=0; i< 4; i++)
    	{      		
		string s("side_");
		stringstream sbuff("", ios_base::out);
		sbuff << i;
		s.append(sbuff.str());
		magics::OpenGLBox *pbox = new magics::OpenGLBox(0,0,4,4);
      		sideBox_.push_back(pbox);
    	}  
		
} 
 

bool OpenGLPickObject::addPoint(PaperPoint &pp)
{
	if(parentBox_->checkPointInBox(pp.x(),pp.y()))
	{
		obj_.push_back(pp);
		setBoundingBox();
		return true;
	}	
	
	return false;
}

bool OpenGLPickObject::setLastPoint(PaperPoint &pp)
{
	/*Log::dev()<< "parent: " << parentBox_->x() << " " << parentBox_->y() << " " 
	     << parentBox_->x()+parentBox_->width() << "  " 
	     << parentBox_->y()+parentBox_->height() << endl;
	     
	Log::dev()<< "point: " <<  pp.x() << " " <<  pp.y()  << endl;  */  
 				
	if(type_ == POT_ZOOMRECT || type_ == POT_AREA)
	{		
	 	double x=pp.x();
		double y=pp.y();
		double x_ori=obj_[2].x();
		double y_ori=obj_[2].y();
						
		//For the zoom rectangle there is a restriction for the point positions:
		// x0<x1 && y0>y1
		if(type_ == POT_ZOOMRECT && 
		   (x <= obj_[0].x() || y >= obj_[0].y()))
		{
			return false;
		}
		
		if(checkNewPositionInParent(x,y,x_ori,y_ori))
		{
			obj_[2].x(x);
			obj_[2].y(y);
			setBoundingBox();		

			//set the other two points
			obj_[1].x(obj_[2].x());
			obj_[1].y(obj_[0].y());			
			obj_[3].x(obj_[0].x());
			obj_[3].y(obj_[2].y());
			
			return true;
		}	
							
	}
	else
	{
		double x=pp.x();
		double y=pp.y();
		double x_ori=obj_.back().x();
		double y_ori=obj_.back().y(); 
	
		if(checkNewPositionInParent(x,y,x_ori,y_ori))
		{
			obj_.back().x(x);
			obj_.back().y(y);
			setBoundingBox();		
			return true;
		}		
	}
	
	return false;
}

void OpenGLPickObject::deleteLastPoint()
{
	obj_.pop_back();
	setBoundingBox();
}	

void OpenGLPickObject::setBoundingBox()
{
	double xmin=10000., xmax=-1.;
	double ymin=10000., ymax=-1.;
	for(int i=0; i< obj_.size(); i++)
	{
		if(xmin > obj_[i].x()) xmin=obj_[i].x();
		if(xmax < obj_[i].x()) xmax=obj_[i].x();
		if(ymin > obj_[i].y()) ymin=obj_[i].y();
		if(ymax < obj_[i].y()) ymax=obj_[i].y();		 
	}

	bBox_.set(xmin,ymin,xmax-xmin,ymax-ymin);
}	

void OpenGLPickObject::updateObject(float xn, float yn, float nw, float nh)
{
	float xp,yp;
	float x=bBox_.x();
	float y=bBox_.y();
	float dx=xn-bBox_.x();
	float dy=yn-bBox_.y();
	float sx=(nw-bBox_.width())/bBox_.width();
	float sy=(nh-bBox_.height())/bBox_.height();
		
	for(int i=0; i< obj_.size(); i++)
	{	     
		xp=obj_[i].x()+dx+(obj_[i].x()-x)*sx;
	     	yp=obj_[i].y()+dy+(obj_[i].y()-y)*sy;
 		
		obj_[i].x(xp);
	        obj_[i].y(yp);	      	      
	} 
}

void OpenGLPickObject::updatePickBox()
{
	double x1=bBox_.x();
	double y1=bBox_.y();
	double x2=bBox_.x()+bBox_.width();   
	double y2=bBox_.y()+bBox_.height();   
	double xh=(x1+x2)/2.;
	double yh=(y1+y2)/2.;
	
	//Object points
	for(int i=0; i< objBox_.size();i++)
	{
		objBox_[i]->x(x1+obj_[i].x()-objBox_[i]->width()/2.);
		objBox_[i]->y(y1+obj_[i].y()-objBox_[i]->height()/2.);
	}
	
	//Corners	
 	cornerBox_[0]->x(x1-cornerBox_[0]->width()/2);
 	cornerBox_[0]->y(y1-cornerBox_[0]->height()/2);  	
  
  	cornerBox_[1]->x(x2-cornerBox_[1]->width()/2);
 	cornerBox_[1]->y(y1-cornerBox_[1]->height()/2);  
  
  	cornerBox_[2]->x(x2-cornerBox_[2]->width()/2);
 	cornerBox_[2]->y(y2-cornerBox_[2]->height()/2); 
  
  	cornerBox_[3]->x(x1-cornerBox_[3]->width()/2);
 	cornerBox_[3]->y(y2-cornerBox_[3]->height()/2); 
    			
 	//Sides 
 	sideBox_[0]->x(xh-sideBox_[0]->width()/2);
 	sideBox_[0]->y(y1-sideBox_[0]->height()/2);
  
 	sideBox_[1]->x(x2-sideBox_[1]->width()/2);
 	sideBox_[1]->y(yh-sideBox_[1]->height()/2);
 
  	sideBox_[2]->x(xh-sideBox_[2]->width()/2);
 	sideBox_[2]->y(y2-sideBox_[2]->height()/2);
 
 	sideBox_[3]->x(x1-sideBox_[3]->width()/2);
 	sideBox_[3]->y(yh-sideBox_[3]->height()/2);
 
}


bool OpenGLPickObject::checkPointInBoundingBox(const double x, const double y)
{		
	if(bBox_.checkPointInBox(x,y))   
	{			
		return true;
	}
	
	return false;		
} 

bool OpenGLPickObject::checkPointInCornerBox(const double x, const double y)
{
	
	for(int i=0; i< cornerBox_.size(); i++)
    	{
      		if(cornerBox_[i]->checkPointInBox(x,y))   
		{			
			currentBox_=cornerBox_[i];
			return true;
		}
	}
	
	return false;
} 

bool OpenGLPickObject::checkPointInSideBox(const double x, const double y)
{
	
	for(int i=0; i< sideBox_.size(); i++)
    	{
      		if(sideBox_[i]->checkPointInBox(x,y))   
		{			
			currentBox_=sideBox_[i];
			return true;
		}
	}	
		
	return false;
} 



bool OpenGLPickObject::checkPointInObjectBox(const double x, const double y)
{

	for(int i=0; i< objBox_.size(); i++)
    	{
      		if(objBox_[i]->checkPointInBox(x,y))
		{			
			currentBox_=objBox_[i];
			return true;
		}
	}
	
	return false;
} 


bool OpenGLPickObject::checkNewPositionInParent(double &x,double &y,double x_ori,double y_ori)
{	
	if(parentBox_->checkPointInBox(x,y))
	{
		return true;
	}	 
	else
	{
		bool rv=false;
		
		const float xp0 = parentBox_->x();
		const float yp0 = parentBox_->y();
		const float xp1 = parentBox_->x()+parentBox_->width();
		const float yp1 = parentBox_->y()+parentBox_->height();
		
		/*if(x < xp0 && x_ori > xp0)
		{
		 	x=xp0;
			rv=true;
		}			
		else if(x > xp1 && x_ori < xp1)
		{
			 x=xp1;
			 rv=true;
		}			  
			
		if(y < yp0 && y_ori > yp0)
		{
			 y=yp0;
			 rv=true;
		}	 
		else if(y > yp1 && y_ori < yp1)
		{
			 y=yp1;
			 rv=true;
		}*/	 	
		
		//It semms to be more logical	
		if(x < xp0)
		{
		 	x=xp0;
			rv=true;
		}			
		else if(x > xp1)
		{
			 x=xp1;
			 rv=true;
		}			  
			
		if(y < yp0)
		{
			 y=yp0;
			 rv=true;
		}	 
		else if(y > yp1)
		{
			 y=yp1;
			 rv=true;
		}	 	

		return rv;
	}
	
	return false;	
}
	       

void OpenGLPickObject::getBoundingBoxCover(float &x0,float &y0, float &x1, float &y1, 
				const float xmin, const float ymin, const float xmax, const float ymax) 
{
	x0 =bBox_.x()-5;
	y0 =bBox_.y()-5;
	x1 =x0+bBox_.width()+10;
	y1 =y0+bBox_.height()+10;
	
	if(x0 <xmin) x0=0.;
	if(y0 <ymin) y0=0.;
	if(x1> xmax) x1=xmax;
	if(y1> ymax) y1=ymax;
}			

bool OpenGLPickObject::replaceObject(const double dx,const double dy)
{
	bool rv=false;
			
	double x0=bBox_.x()+dx;
	double y0=bBox_.y()+dy;
	double x1=x0+bBox_.width();
	double y1=y0+bBox_.height();
	
	rv=true;
	
	if(parentBox_->checkPointInBox(x0,y0) == true &&
	   parentBox_->checkPointInBox(x1,y1) == true)
	{
		rv=true;
	}	 
	else
	{
		double d;
		
		const double xb0=bBox_.x();
		const double yb0=bBox_.y();
		const double xb1=xb0+bBox_.width();
		const double yb1=yb0+bBox_.height();
	
		const double xp0 = parentBox_->x();
		const double yp0 = parentBox_->y();
		const double xp1 = parentBox_->x()+parentBox_->width();
		const double yp1 = parentBox_->y()+parentBox_->height();
	
		if(parentBox_->checkPointInBox(x0,y0) == false)
		{
			if(x0 < xp0 && xb0 > xp0)
			{	
				d=xp0-x0;
				x0=xp0;
				x1=x1+d;
				rv=true;
			}
			if(y0 < yp0 && yb0 > yp0)
			{	
				d=yp0-y0;
				y0=yp0;
				y1=y1+d;
				rv=true;
			}
		}
	
		if(parentBox_->checkPointInBox(x1,y1) == false)
		{
			if(x1 > xp1 && xb1 < xp1)
			{	
				d=x1-xp1;
				x1=xp1;
				x0=x0-d;
				rv=true;
			}
			if(y1 > yp1 && yb1 < yp1)
			{	
				d=y1-yp1;
				y1=yp1;
				y0=y0-d;
				rv=true;
			}
		}	
	}
			
	if(rv==false) return false;
	
	updateObject(x0,y0,x1-x0,y1-y0);
        
	bBox_.x(x0);
	bBox_.y(y0);
	//bBox_.setWidth(w);
	//bBox_.setHeight(h);
	
	updatePickBox();
	
	modifyStatus_=true;
	
	return true;		
}

bool OpenGLPickObject::resizeObject(const double dx,const double dy)
{
	if(currentBox_ == NULL) return false;
	
/*	double x,y,w,h;
	
	int id;		
	string::size_type pos=0;
	pos=currentBox_->name().find("_",pos);
	string s;
	s.assign(currentBox_->name(),pos+1,currentBox_->name().size()-pos-1);	
	istringstream inp(s);   	
   	inp >> id;
	
	if(currentBox_->name().find("corner_") != string::npos)
  	{		
		switch(id)
		{
				
		case 0:			
			x=bBox_.x()+dx;
			y=bBox_.y()+dy;
			w=bBox_.width()-dx;
			h=bBox_.height()-dy;
			break;
		case 1:			
			x=bBox_.x();
			y=bBox_.y()+dy;
			w=bBox_.width()+dx;
			h=bBox_.height()-dy;	
			break;
		case 2:			
			x=bBox_.x();
			y=bBox_.y();
			w=bBox_.width()+dx;
			h=bBox_.height()+dy;	
			break;				
		case 3:			
			x=bBox_.x()+dx;
			y=bBox_.y();
			w=bBox_.width()-dx;
			h=bBox_.height()+dy;
			break;			
		}
				
		double x0=x;
		double y0=y;
		double x1=x+w;
		double y1=y+h;
		
		double xb0=bBox_.x();
		double yb0=bBox_.y();
		double xb1=xb0+bBox_.width();
		double yb1=yb0+bBox_.height();
		
		if(checkNewPositionInParent(x0,y0,xb0,yb0) == false) return false;
		if(checkNewPositionInParent(x1,y1,xb1,yb1) == false) return false;
		
		x=x0;
		y=y0;
		w=x1-x0;
		h=y1-y0;
		
		updateObject(x,y,w,h);
		
		bBox_.x(x);
		bBox_.y(y);
		bBox_.setWidth(w);
		bBox_.setHeight(h);	
				
		updatePickBox();
		modifyStatus_=true;
	}

	else if(currentBox_->name().find("side_") != string::npos)
  	{		
		switch(id)
		{
				
		case 0:			
			x=bBox_.x();
			y=bBox_.y()+dy;
			w=bBox_.width();
			h=bBox_.height()-dy;			
			break;
		case 1:						
			x=bBox_.x();
			y=bBox_.y();
			w=bBox_.width()+dx;
			h=bBox_.height();	
			break;
		case 2:			
			x=bBox_.x();
			y=bBox_.y();
			w=bBox_.width();
			h=bBox_.height()+dy;
			break;				
		case 3:			
			x=bBox_.x()+dx;
			y=bBox_.y();
			w=bBox_.width()-dx;
			h=bBox_.height();
			break;			
		}
		
		double x0=x;
		double y0=y;
		double x1=x+w;
		double y1=y+h;
		
		double xb0=bBox_.x();
		double yb0=bBox_.y();
		double xb1=xb0+bBox_.width();
		double yb1=yb0+bBox_.height();
		
		if(checkNewPositionInParent(x0,y0,xb0,yb0) == false) return false;
		if(checkNewPositionInParent(x1,y1,xb1,yb1) == false) return false;
		updateObject(x,y,w,h);
		
		bBox_.x(x);
		bBox_.y(y);
		bBox_.setWidth(w);
		bBox_.setHeight(h);
			
		updatePickBox();
		
		modifyStatus_=true;
	}		
*/
	return true;
}


void OpenGLPickObject::renderObject(bool tmp_outline)
{	
	vector<PaperPoint> *vp=getObject();
		
	if(vp->size()<1) return;
			
	//Save colour and line attribs
	glPushAttrib(GL_CURRENT_BIT | GL_LINE_BIT);
	
	//Get object points
	float rad=3.;
	int i;
	const int nn=vp->size();
	float xx[nn];
	float yy[nn];
	for(i=0; i< nn; i++)
	{
		xx[i]=(*vp)[i].x();
		yy[i]=(*vp)[i].y();
	}	

	Colour bgc=driver_->pickObjectAttr_.getBackground_colour();
	//Colour lnc=pickObjectAttr_.getLine_colour();
	Colour lnc(0.,0.,0.);
	
	switch(getType())
	{

	case POT_ZOOMRECT :
		if(tmp_outline) break;		
		//Fill the area bounded by the object		
		glColor4f(bgc.red(),bgc.green(),bgc.blue(),
			driver_->pickObjectAttr_.getBackground_opacity()/100.);
		
		//glColor4f(1.,0.,0.,0.5);
			
		glEnable(GL_BLEND); 
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);	
		driver_->setNewPolygonMode(GL_FILL); 
		glRectf(xx[0],yy[0],xx[2],yy[2]);			
		glDisable(GL_BLEND);
						
		//Draw the object outline
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
			
		driver_->setNewPolygonMode(GL_LINE); 
		driver_->setLineParameters(M_SOLID, 2.);		
		glRectf(xx[0],yy[0],xx[2],yy[2]);	
		
		
		//OpenGLPainter->


		break;


	case POT_AREA:
		if(tmp_outline) break;
		
		//Fill the area bounded by the object		
		glColor4f(bgc.red(),bgc.green(),bgc.blue(),
			driver_->pickObjectAttr_.getBackground_opacity()/100.);							
		
		glEnable(GL_BLEND); 
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);	
		driver_->setNewPolygonMode(GL_FILL); 
		glRectf(xx[0],yy[0],xx[2],yy[2]);			
		glDisable(GL_BLEND);
						
		//Draw the object outline
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
				
		driver_->setNewPolygonMode(GL_LINE); 
		driver_->setLineParameters(M_SOLID, 2.);		
		glRectf(xx[0],yy[0],xx[2],yy[2]);
		
		break;

	case POT_LINE:	
		
		//Draw the object outline 
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
				
		if(!tmp_outline)
		{
			driver_->setLineParameters(M_SOLID, 2.);		
		}
		else
		{
			driver_->setLineParameters(M_DASH, 1.);
		}
		driver_->renderPolyline2(nn, xx, yy);
		
		//Fill the object points	  	
		if(!tmp_outline)
		{
			glColor3f(bgc.red(),bgc.green(),bgc.blue());
					
			for(i=0; i< nn; i++)
			{		
				driver_->circle(xx[i],yy[i],rad,8);
			}
		}
		
		//Draw the object points border 
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
				
		driver_->setLineParameters(M_SOLID, 1.);		
		
		for(i=0; i< nn; i++)
		{		
		  driver_->circle(xx[i],yy[i],rad,0);
		}
	
		break;
		
		
	case POT_POLYLINE:			
		//Draw the object outline 		
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
						
		if(!tmp_outline)
		{
			driver_->setLineParameters(M_SOLID, 2.);		
		}
		else
		{
			driver_->setLineParameters(M_DASH, 1.);
		}	
		driver_->renderPolyline(nn, xx, yy);
		
		//Fill the object points	  	
		if(!tmp_outline)
		{
			glColor3f(bgc.red(),bgc.green(),bgc.blue());
			
			for(i=0; i< nn; i++)
			{		
				driver_->circle(xx[i],yy[i],rad,8);
			}
		}
		
		//Draw the object points border 
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
		
		driver_->setLineParameters(M_SOLID, 1.);		
		
		for(i=0; i< nn; i++)
		{		
			driver_->circle(xx[i],yy[i],rad,0);
		}
	
		break;
			
	case POT_POLYGON:			
		//Fill the area bounded by the object		
		if(!tmp_outline && nn >2)
		{
			glColor4f(bgc.red(),bgc.green(),bgc.blue(),
			           driver_->pickObjectAttr_.getBackground_opacity()/100.);			
				
			glEnable(GL_BLEND); 
			glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);	
			driver_->setNewPolygonMode(GL_FILL); 		
			driver_->renderFilledPolygon(nn,xx,yy);							
			glDisable(GL_BLEND);
		}
		
		//Draw the object outline 
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
				
		if(!tmp_outline)
		{
			driver_->setLineParameters(M_SOLID, 2.);		
		}
		else
		{
			driver_->setLineParameters(M_DASH, 1.);
		}
		if(nn==2)
		{
			driver_->renderPolyline2(nn, xx, yy);
			
		}
		else if(nn>2)
		{
			driver_->renderSimplePolygon(nn, xx, yy);
		}
		
		//Fill the object points	  	
		if(!tmp_outline)
		{
			glColor3f(bgc.red(),bgc.green(),bgc.blue());
								
			for(i=0; i< nn; i++)
			{		
				driver_->circle(xx[i],yy[i],rad,8);
			}
		}
		
		//Draw the object points border 
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
				
		driver_->setLineParameters(M_SOLID, 1.);		
		
		for(i=0; i< nn; i++)
		{		
			driver_->circle(xx[i],yy[i],rad,0);
		}
	
		break;				
	
	case POT_POINT:
		
		//Draw the object outline 
		glColor3f(lnc.red(),lnc.green(),lnc.blue());
						
		if(!tmp_outline)
		{
			driver_->setLineParameters(M_SOLID, 1.);		
		}
		else
		{
			driver_->setLineParameters(M_DASH, 1.);
		}
		
		int pnn=2;
		float pxx[2], pyy[2];
		
		pxx[0]=xx[0]-3.;
		pyy[0]=yy[0];
		pxx[1]=xx[0]+3.;
		pyy[1]=yy[0];
		
		driver_->renderPolyline2(pnn, pxx, pyy);
		
		pxx[0]=xx[0];
		pyy[0]=yy[0]-3.;
		pxx[1]=xx[0];
		pyy[1]=yy[0]+3.;
		
		driver_->renderPolyline2(pnn, pxx, pyy);
		
		break;
			
	}

	glPopAttrib();

}
void OpenGLPickObject::renderPickBox(bool p_box,bool bb_frame)
{  		
	/*glPushAttrib(GL_CURRENT_BIT | GL_LINE_BIT);
	
	//Colour bgc=pickObjectAttr_.getBackground_colour();
	//Colour lnc=pickObjectAttr_.getLine_colour();
	Colour lnc(0.,0.,0.);
	
	glColor3f(lnc.red(),lnc.green(),lnc.blue());
				  		  
	//Draw pick boxes onto the selected object		
  	if(p_box)		
  	{
		float x,y,w,h;
		int i;
		
		driver_->setNewPolygonMode(GL_FILL);
	   	vector<Box*> *bo=getCornerBox();
		for(i=0;i<bo->size(); i++)
		{										
			x=(*bo)[i]->x(); 		
			y=(*bo)[i]->y(); 
			w=(*bo)[i]->width();
			h=(*bo)[i]->height();
				
			glRectf(x,y,x+w,y+h);					  
	  	}
	   	
		bo=getSideBox();
		for(i=0;i<bo->size(); i++)
		{					
			x=(*bo)[i]->x(); 		
			y=(*bo)[i]->y(); 
			w=(*bo)[i]->width();
			h=(*bo)[i]->height();
				
			glRectf(x,y,x+w,y+h);					  
	  	}
					
	}
	
	//Draw bounding box 	
  	if(bb_frame)		
  	{		
		driver_->setNewPolygonMode(GL_LINE);
	   	driver_->setLineParameters(M_DASH, 1.);
		
		Box *bo=getBoundingBox();
		float x,y,w,h;
		x=bo->x(); 		
		y=bo->y(); 
		w=bo->width();
		h=bo->height();				
		glRectf(x,y,x+w,y+h);						  	  		
	}
	
	glPopAttrib();	
   */
}


void OpenGLPickObject::renderTmpPickObject(bool bb_frame,bool obj_frame)
{  	
	//Coordinates here must be local to bounding box LL. The GL coordinate
	//system has already translated and scaled properly.	
	
	//Draw selection object bounding box 
	if(bb_frame) renderPickBox(false,true);
	
	//Draw selection object outline
	if(obj_frame) renderObject(true);		
}

