/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file OpenGLDriver.cc
    \brief Implementation of OpenGLDriver.
    \author Graphics Section, ECMWF

    Started: May 2004

*/
#include <sys/stat.h>
#include <ctime>

#include <OpenGLDriver.h>
#include <Polyline.h>
#include <Text.h>
#include <Image.h>
#include <Layout.h>
#include <Layer.h>
#include <AnimationRules.h>
#include <Text.h>
#include <Symbol.h>

#include <OpenGLDriverText.h> //modified version of the OGLFT library

#include <OpenGLDriverManager.h>
#include <OpenGLLayoutNode.h>
#include <OpenGLStepNode.h>
#include <OpenGLLayerNode.h>
#include <OpenGLFontHandler.h>
#include <OpenGLTextureItem.h>
#include <OpenGLAnimationControl.h>
#include <OpenGLPainter.h>
#include <OpenGLSymbolManager.h>
#include <OpenGLAreaGroup.h>
#include <OpenGLFboItem.h>
#include <OpenGLCore.h>

#include <MtInputEvent.h>
#include <MagicsEvent.h>

//#define OPENGL_11


using namespace magics;

static bool supportsOpenGLVersion(int atLeastMajor, int atLeastMinor);


static void parsePoints(vector<PaperPoint> &vPP, string &points)
{
	string::size_type pos = points.find_first_of(",");

	while(pos != string::npos)
	{
		points.replace(pos,1," ",1);
		pos = points.find_first_of(",",pos);
	}

	istringstream totalString(points);
	float x,y;

	while(!totalString.eof())
	{
		totalString >> x >> y;
		PaperPoint pp;
		pp.x(x);
		pp.y(y);
		vPP.push_back(pp);
	}
}

/*!
  \brief Constructor
*/
OpenGLDriver::OpenGLDriver() : currentFont_(""),currentFontSize_(0),ready_(false)
{
    	//lineWidthFactor_=0.67;
	lineWidthFactor_=0.6;
    
	initialized_=false;	

	antialiasing_=false;
    
    	glTree_= new OpenGLNode; 

	currentGlNode_=glTree_;

	if ( observer_ ) registerObserver(observer_); 
 
    	fontHandler_ = new OpenGLFontHandler;

	stepToRender_=0;

	alphaEnabled_=true;

	stepObjectToRender_=0;
	
	magnifierZoom_=1.;

	symbols_=new OpenGLSymbolManager;


	layerTree_=0;
}

/*!
  \brief Destructor
*/
OpenGLDriver::~OpenGLDriver() 
{
	glDeleteTextures(2,tex_bg_id_);	
	gluDeleteTess(tobj_);
	
	for(list<OpenGLTextureItem*>::iterator it=texList_.begin();
	    it != texList_.end(); it++)
	{
			delete (*it);
	}
	texList_.clear();
	
	//Delete the OGLFT font objects		
	fontHandler_->clear();
	
	delete glTree_;

	glLineWidth(1.);

	delete symbols_;

}

/*!
  \brief Opening the driver
*/
void OpenGLDriver::open()
{		
	//If the driver is already initialized then delete the
	// pick objects and skip initialization
	if(initialized_) 
	{										
		for(map<OpenGLLayoutNode*,OpenGLAnimationControl*>::iterator it=animationStepMap_.begin(); 
	                    it != animationStepMap_.end(); it++)
		{
			delete it->second;
		}
		animationStepMap_.clear();
		//framesToRender_.clear();

		//Re- the tree
		delete glTree_;
		glTree_= new OpenGLNode; 
		
		//pageList_.clear();
		//drawableList_.clear();	

		for(list<OpenGLTextureItem*>::iterator it=texList_.begin();
		    it != texList_.end(); it++)
		{
			delete (*it);
		}
		texList_.clear();

		Log::debug() << "---> open: glLayoutStack size: " << 
				glLayoutStack_.size() << endl;

		while(!glLayoutStack_.empty())
		{
			glLayoutStack_.pop();
		}
		
		while(!glLayerStack_.empty())
		{
			glLayerStack_.pop();
		}

		currentGlNode_=glTree_;

		//Delete the OGLFT font objects						
	        fontHandler_->clear();

		//Clear the display
		glClear(GL_COLOR_BUFFER_BIT);
		
		actDpl_=0;

		currentPolygonMode_ = GL_LINE;
		glPolygonMode(GL_FRONT_AND_BACK,currentPolygonMode_);		

		return;
	}
	
	actDpl_=0;

	//We do it for 75 dpi
	setCMscale(75.*1./2.54);// cm -> pixel

	dimensionX_ = floorf(convertCM(getXDeviceLength()));
	dimensionY_ = floorf(convertCM(getYDeviceLength()));

	//Create the Motif widget 
	//setupWidget();
	
	glViewport(0,0,dimensionX_,dimensionY_);			


	//Set the projection matrix
	glMatrixMode(GL_PROJECTION);
	glOrtho(0,dimensionX_ , 0, dimensionY_, -1, 1);  // left, right, bottom, top, near, far

	glMatrixMode(GL_MODELVIEW);      
	
	//! Pattern for Polylines 
	linePatterns_[0] = 0xFFFF; // solid
	linePatterns_[1] = 0x00FF; // dash
	linePatterns_[2] = 0x0101; // dot
	linePatterns_[3] = 0x01FF; // chain_dash
	linePatterns_[4] = 0x1C47; // chain_dot

	//Check antialiased line width range	
	GLfloat fbuf[2];
	glGetFloatv(GL_SMOOTH_LINE_WIDTH_RANGE,fbuf);
	Log::debug() << "GL_SMOOTH_LINE_WIDTH_RANGE: " << fbuf[0] << " " << fbuf[1] << endl;

/*
	GLint ibuf1,ibuf2;
	glGetIntegerv(GL_SAMPLE_BUFFERS,&ibuf1);	
	glGetIntegerv(GL_SAMPLES,&ibuf2);

	//Multisampling (if available)
	if(ibuf1 >=1 && ibuf2 > 1)
	{
		Log::debug() << "GL_MILTISAMPLE: available" << endl;
		glEnable(GL_MULTISAMPLE);
        }
	//else antialias through blending ..
	else
	{*/
		//glEnable(GL_BLEND);
		//glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		//glEnable(GL_LINE_SMOOTH);
		//glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
	//}
			
	//Polygon shading. Since magics++ polygons can have both CW and CCW 
	//orientation both sides have to be rendered. 
	glShadeModel(GL_FLAT);
	glFrontFace(GL_CCW);	// front of Polygons is CounterClockWise (CCW)
        glDisable(GL_CULL_FACE);  // disable culling: both polygon faces will be shaded			        
	glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
	currentPolygonMode_ = GL_LINE;
	
	//Scaling factors have to be stored for 
	//symbol plotting and for rotations in an unscaled co-ordinate system
	coordRatioX_=1.;
	coordRatioY_=1;
		
	// Define tesselation 	
	typedef GLvoid (*GTC)();
	
	tobj_ = gluNewTess();

	gluTessCallback(tobj_, (GLenum) GLU_TESS_VERTEX, (GTC) glVertex2dv );
	gluTessCallback(tobj_, (GLenum) GLU_TESS_BEGIN,  (GTC) glBegin );
	gluTessCallback(tobj_, (GLenum) GLU_TESS_END,    (GTC) glEnd );
	gluTessCallback(tobj_, (GLenum) GLU_TESS_ERROR,  (GTC) TessErrorCB );
	gluTessCallback(tobj_, (GLenum) GLU_TESS_COMBINE,(GTC) TessCombineCB );
	gluTessProperty(tobj_, (GLenum) GLU_TESS_TOLERANCE, 0.00001);
	gluTessNormal(tobj_, 0,0,1); // defines tess. normal -> speed up!
	
	// Define clear color and clear the display 	
	glClearColor(1.0, 1.0, 1.0, 1.0);
	glClear(GL_COLOR_BUFFER_BIT);
	
	//Check version
	//supportsOpenGLVersion(2,0);
	
	//Search for extensions	
	/*const GLubyte* extstr=glGetString(GL_EXTENSIONS);	
	Log::dev()<< "Available OpenGL extensions:" << endl << extstr << endl;	
	
	if(gluCheckExtension(reinterpret_cast<const GLubyte*>("GL_EXT_framebuffer_object")
	                     ,extstr) == GL_TRUE)
	{
	    Log::dev()<< "Extension: GL_EXT_framebuffer_object found!!!"  << endl; 
	}
	*/


	glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_FALSE);  

	//Textures
	glGenTextures( 2, tex_bg_id_);	

	//Fbo setup!!
	OpenGLCore::instance()->init(static_cast<int>(dimensionX_),static_cast<int>(dimensionY_));

	//Set initialization
	initialized_=true;
	
	//Create manager
	OpenGLDriverManager *ogmg = new OpenGLDriverManager(this);    	
    	registerObserver(ogmg);	
	
	painter_=OpenGLPainter::instance();

	Log::debug() << "Drawing starts!" << endl;	
}

/*!
  \brief Closing the driver
*/
void OpenGLDriver::close()
{
	glTree_->print();

	if(actDpl_) actDpl_->compileEnd();
	actDpl_=0;
	
	currentGlNode_=glTree_;

	//Notify observers that the new rendering 
	//has finished so that they could work on
	//the new tree

	//Render the whole graphical tree
	renderTree(stepToRender_);

        
	if(animationStepMap_.size() != 0)
	{
		OpenGLAnimationControl *ac=animationStepMap_.begin()->second;  
	
		vector<MagicsAnimationStepData> steps;
		ac->steps(steps);
		MagicsAnimationStepsEvent event(steps);
		notifyObservers(event);
	}

	if(1)
	{
		getLayerTree();
		MagicsLayerTreeEvent event(layerTree_);
		notifyObservers(event);
	}

	notifyObservers(&OpenGLDriverObserver::update);	

	Log::debug() << "---> close: glLayoutStack size: " << 
				glLayoutStack_.size() << endl;
	
	fontHandler_->print(cout);
	
	//endPage();
		
	//Notify observers that the new rendering 
	//has finished so that they could work on
	//the new tree

	//notifyObservers(&OpenGLDriverObserver::update);	
	
	currentPage_ = 0; 
}

/*!
  \brief starting a new page

  This method has to take care that previous pages are closed and that
  for formats with multiple output files a new file is set up.
*/
MAGICS_NO_EXPORT void OpenGLDriver::startPage() const
{
}

/*!
  \brief ending a page
 
  This method has to take care that for formats with multiple output 
  files are closed.
*/
MAGICS_NO_EXPORT void OpenGLDriver::endPage() const
{
	//swapFb();
}

/*!
  \brief project to a new Layout

  This method will update the offset and scale according to the new Layout given.

  \sa Layout
*/

MAGICS_NO_EXPORT void OpenGLDriver::project(const magics::Layout& layout) const
{
	//Create a new OpenGLLayoutNode
	OpenGLNode* parent=getNewLayoutParentInTree();
	//string name=parent->name() + "." + layout.name();
	string name=layout.name();
	OpenGLLayoutNode* gll =  new OpenGLLayoutNode(name, layout);	

	project(gll,parent,layout);		
	
}

/*!
  \brief project to a new Layout

  This method will update the offset and scale according to the new Layout given.

  \sa Layout
*/

MAGICS_NO_EXPORT void OpenGLDriver::project(const magics::PreviewLayout& layout) const
{
	//Create a new OpenGLPreviewLayoutNode
	OpenGLNode* parent=getNewLayoutParentInTree();
	//string name=parent->name() + "." + layout.name();
	string name=layout.name();
	OpenGLPreviewLayoutNode* gll =  new OpenGLPreviewLayoutNode(name, layout);	

	project(gll,parent,layout);
}

/*!
  \brief project to a new Layout

  This method will update the offset and scale according to the new Layout given.

  \sa Layout
*/

MAGICS_NO_EXPORT void OpenGLDriver::project(const magics::MagnifierLayout& layout) const
{
	//Create a new OpenGLPreviewLayoutNode
	OpenGLNode* parent=getNewLayoutParentInTree();
	//string name=parent->name() + "." + layout.name();
	string name=layout.name();
	OpenGLMagnifierLayoutNode* gll =  new OpenGLMagnifierLayoutNode(name, layout);	

	project(gll,parent,layout);
}

/*!
  \brief reproject out of the last Layout

  This method will update the offset and scale to the state they were before the
  last Layout was received.
  
  \sa UnLayout
*/
MAGICS_NO_EXPORT void OpenGLDriver::unproject() const
{
	Log::dev() << "--->UNPROJECT " << endl;
	
	//Finish the central dpl
	if(actDpl_ != 0 && actDpl_->compileEnd()==false)
	{
		Log::debug() << "Unproject error!!! Display list compilation failed! " << endl;	
	}
	
	//Verify current layout node in stack
	/*if(glLayoutStack_.top()->layout()->name() != unlayout.name())	
	{
		Log::error() << "--->UNPROJECT (" << unlayout.name() << 
				") Layout stack error!" << endl;
	}*/			
	
	Log::dev() << "   current node(before pop): " << currentGlNode_->name() << endl;
	Log::dev() << "   current layout: " << glLayoutStack_.top()->name() << endl;

	OpenGLLayoutNode *gll=glLayoutStack_.top();

	//Set current node to the layouts parent
	currentGlNode_=gll->parent();

	//Pop layout from the stack
	glLayoutStack_.pop();

	Log::dev() << "   current node(after pop): " << currentGlNode_->name() << endl;
	
	//Verify dimension stack
 	if(dimensionStack_.empty())
	{
		Log::error() << "--->UNPROJECT (" <<
			      ") Dimension stack error!" << endl;
		assert(dimensionStack_.empty() == false);
	}

	coordRatioX_= scalesX_.top();
	coordRatioY_= scalesY_.top();
	scalesX_.pop();
	scalesY_.pop();

	//Verify dimension stack
	dimensionY_ = dimensionStack_.top();
	dimensionStack_.pop();
	dimensionX_ = dimensionStack_.top();
	dimensionStack_.pop();
	
	//Add new node and start dpl for the central part
	addGlNode(currentGlNode_);
	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();

	checkError("OpenGLDriver::unproject()");

}

/*!
  \brief setup a new layer

  This method will setup a new layer. Layers enable overlays of entities
  of information.

  \sa PhysicalLayer
*/
MAGICS_NO_EXPORT void OpenGLDriver::newLayer(const Layer& layer) const
{
	OpenGLNode* parent=currentGlNode_;
	//string name=parent->name() + "." + layer.name();
	string name =  layer.name();
	OpenGLLayerNode* gll =  new OpenGLLayerNode(name, layer);	
	//glTree_->addNode(parent->name(),gll->name(),gll);	
	parent->addNode(gll);

	glLayerStack_.push(gll);
	currentGlNode_=gll;

	if(actDpl_) actDpl_->compileEnd();
	addGlNode(currentGlNode_);
	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();

	Log::debug() << "OpenGLDriver::new Layer " << layer.name() << " needs implementing" <<endl;
}

/*!
  \brief close the current layer

  This method will close an existing layer. This includes resets of existing boxes. 

  \sa UnPhysicalLayer PhysicalLayer
*/
MAGICS_NO_EXPORT void OpenGLDriver::closeLayer(const Layer& layer) const
{
	Log::dev() << "---> CLOSE LAYER " << endl;
	
	//Finish the central dpl
	if(actDpl_ != 0 && actDpl_->compileEnd()==false)
	{
		Log::debug() << "closeLayer error!!! Display list compilation failed! " << endl;	
	}
	
	Log::dev() << "   current node(before pop): " << currentGlNode_->name() << endl;
	Log::dev() << "   current layer: " << glLayerStack_.top()->name() << endl;

	OpenGLLayerNode *gll=glLayerStack_.top();

	assert(gll->layer().name() == layer.name());

	//Set current node to the layer parent
	currentGlNode_=gll->parent();

	//Pop layer from the stack
	glLayerStack_.pop();

	//Add new node and start dpl for the central part
	addGlNode(currentGlNode_);
	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();
}


/*!
  \brief sets a new colour

  This colour stays the default drawing colour until the painting in the 
  current box is finished.

  \sa Colour
*/
MAGICS_NO_EXPORT void OpenGLDriver::setNewColour(const Colour &colour) const
{
	if(currentColour_ == colour) return;
	currentColour_ = colour;
	glColor3f(currentColour_.red(),currentColour_.green(),currentColour_.blue());
}

/*!
  \brief sets a new line width

  This line width stays the default width until the painting in the 
  current box is finished.

  \sa setLineParameters()
*/
MAGICS_NO_EXPORT void OpenGLDriver::setNewLineWidth(const float width) const
{	
	currentLineWidth_ = width*lineWidthFactor_;
	glLineWidth(currentLineWidth_);
	
}

/*!
  \brief sets new properties of how lines are drawn

  These properties stay the default until the painting in the 
  current box is finished.

  \sa LineStyle

  param linestyle Object describing the line style
  param w width of the line

*/
MAGICS_NO_EXPORT int OpenGLDriver::setLineParameters(const LineStyle linestyle, const float w) const
{
	const int width = (w < 1) ? 1 : static_cast<int>(w);

	setNewLineWidth(width);

	if(linestyle==M_SOLID)            {glDisable(GL_LINE_STIPPLE);}
	else if (linestyle==M_DASH)       {glLineStipple(1,linePatterns_[1]); glEnable(GL_LINE_STIPPLE);}
	else if (linestyle==M_DOT)        {glLineStipple(1,linePatterns_[2]); glEnable(GL_LINE_STIPPLE);}
	else if (linestyle==M_CHAIN_DASH) {glLineStipple(1,linePatterns_[3]); glEnable(GL_LINE_STIPPLE);}
	else if (linestyle==M_CHAIN_DOT)  {glLineStipple(1,linePatterns_[4]); glEnable(GL_LINE_STIPPLE);}

	return 0;
}

/*!
  \brief renders polylines

  This method renders a polyline given as two float arrays. The two 
  arrays given as X and Y values have to be at least the length of
  <i>n</i>. All values beyond <i>n</i> will be ignored. The style is
  determined by what is described in the current LineStyle.

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y array of y values
*/
MAGICS_NO_EXPORT void OpenGLDriver::renderPolyline(const int n, float *x, float *y) const
{	
	float *xx = new float[n];
	float *yy = new float[n];
	
	for(int i = 0; i<n; i++)
	{
		xx[i] = projectX(x[i]);
		yy[i] = projectY(y[i]);
	}

	if(polylineAntialiasing_== false)
	{
		glPushAttrib(GL_ENABLE_BIT );
		glDisable(GL_BLEND);	
		glDisable(GL_LINE_SMOOTH);			
	}

	setNewPolygonMode(GL_LINE);
	Primitive_simple(n, xx, yy, GL_LINE_STRIP);
	glDisable(GL_LINE_STIPPLE);

	if(polylineAntialiasing_== false)
	{
		glPopAttrib();		
	}

	delete [] xx;
	delete [] yy;

	checkError("OpenGLDriver::renderPolyline()");
}

/*!
  \brief renders horizontal polylines

  This method renders a horizontal polyline given as two float arrays.
  This method is mainly thought for the scanline algorithm used by some 
  output drivers to shade areas. <i>n</i> has to be a multiple of 2!

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y y value
*/
MAGICS_NO_EXPORT void OpenGLDriver::renderPolyline2x(const int n, int* x, const int y) const
{	
  	setNewPolygonMode(GL_LINE);
	if ( n < 2 || (n%2) != 0 ) return;

	int *xx = x;
	const int n2 = n/2;
	for (int i=0; i<n2; i++)
	{
		int x0 = *(xx++);
		int x1 = *(xx++);

		glBegin(GL_LINES);
		  glVertex2f(x0,y);
		  glVertex2f(x1,y);		
		glEnd();
	}
	glDisable(GL_LINE_STIPPLE);

	checkError("OpenGLDriver::renderPolyline2x()");
}

/*!
  \brief renders a single line

  This method renders a polyline with two points.The style is
  determined by what is described in the current LineStyle.

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y array of y values
*/
MAGICS_NO_EXPORT void OpenGLDriver::renderPolyline2(const int n, float* x, float* y) const
{
 	setNewPolygonMode(GL_LINE);
	if(n != 2) return;

	if(x[0] == x[1] || y[0] == y[1])
	{
		glPushAttrib(GL_COLOR_BUFFER_BIT | GL_LINE_BIT | GL_HINT_BIT );
		glDisable(GL_BLEND);
		glDisable(GL_LINE_SMOOTH);
	}		

	Primitive_simple(n, x, y, GL_LINES);

	if(x[0] == x[1] || y[0] == y[1])
	{
		glPopAttrib();
	}

	glDisable(GL_LINE_STIPPLE);
	checkError("OpenGLDriver::renderPolyline2()");
}

/*!
  \brief renders a filled polygon

  This method renders a filled polygon. The style is
  determined by what is described in the current LineStyle.

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y array of y values
*/
MAGICS_NO_EXPORT void OpenGLDriver::renderSimplePolygon(const int n, float* x, float* y) const
{
	setNewPolygonMode(GL_FILL);	
		
	const bool closed = true;
	glHole *hpt = new glHole(n);
	GLdouble* 	u=hpt->GetData();
        GLdouble*	v;
	
	gluTessBeginPolygon(tobj_,NULL);
	gluTessBeginContour(tobj_);
	for(int i = 0; i<n; i++)
	{
		v = u;
		*(u++) = projectX(x[i]);
		*(u++) = projectY(y[i]);
		if(closed || (i<n-2) ) *(u++) = 0.;
		else *(u++) = 0.5;
		gluTessVertex(tobj_,v,v);
	}	
	gluTessEndContour(tobj_);	
	gluTessEndPolygon(tobj_);
	
	//delete [] hpt;
	delete hpt;

	checkError("OpenGLDriver::renderSimplePolygon()");	
}

/*!
  \brief renders text strings

  This method renders given text strings.

  \sa Text
  \param text object containing the strings and their description
*/

MAGICS_NO_EXPORT void OpenGLDriver::renderText(const Text& text) const
{ 			
	checkError("OpenGLDriver::renderText() 1");	
	
	//Finish the current display list
	if(actDpl_) actDpl_->compileEnd();

	checkError("OpenGLDriver::renderText() 2");	

	if(glLayoutStack_.top()->classType() != "OpenGLMagnifierLayoutNode")
	{
		//Create a display list for the text and compile it
		actDpl_=currentGlNode_->addDpl(M_DPL_TEXT);	
		actDpl_->compileStart();
		renderTextItem(&text);
		actDpl_->compileEnd();

		//Then create a display list for the text magnification. Do not compile it now!!					
		glPixelStorei(GL_UNPACK_ALIGNMENT,1); 		
		actDpl_=currentGlNode_->addDpl(M_DPL_TEXT_MAGNIFY);	
		actDpl_->text(text.clone());

		//Get the parent layout
		OpenGLLayoutNode *gll=glLayoutStack_.top();
		if(gll == 0)
		{
			Log::error() << "OpenGLDriver::renderText ---> Layout stack is empty !" << endl;
			assert(gll != 0);
		}
		actDpl_->parentLayout(gll);
	}
	else
	{
		//Create a display list for the text and compile it
		actDpl_=currentGlNode_->addDpl(M_DPL_TEXT_NOMAGNIFY);	
		actDpl_->text(text.clone());

		//Get the parent layout
		OpenGLLayoutNode *gll=glLayoutStack_.top();
		if(gll == 0)
		{
			Log::error() << "OpenGLDriver::renderText ---> Layout stack is empty !" << endl;
			assert(gll != 0);
		}
		actDpl_->parentLayout(gll);
	}

	//Start a new display list	
	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();

	checkError("OpenGLDriver::renderText() 3");				
}	

/*!
  \brief drawing a circle

  This method renders given text strings.

  The meaning of the last parameter <i>s</i> is as follows:
     - 0-8 determines how many quarters of the circle are filled. Starting from the top clock-wise.
     - 9 fills the whole circle but leaves a vertical bar empty in the middle of the circle.

  \param x X Position
  \param y Y Position
  \param r Radius of circle
  \param fill Style which determines how the circle is shaded

  \note Draw a circle using geometric drawing primitives Quadrics primitives were also tested but 
  gave worse results in terms of quality	
*/
MAGICS_NO_EXPORT void OpenGLDriver::circle(const float x, const float y, const float r, const int fill) const
{
	int i,j,k,nn;
	double an_step;
	double an=0.;
  
	//If the diameter is less than 1 pixel only a short line is
	//rendered						
	if(r < 0.5)
	{						
		//Draw the line segment
		glBegin(GL_LINES);
		glVertex2f(-0.5, 0.5);
		glVertex2f(0.5, -0.5);	
		glEnd();
	}
		
	//Otherwise a closed polygon is rendered. Its resolution 
	//depend on the radius (in pixel)	
	else
	{		
		if(r <= 2.1)
		{
			nn=4;
		}		
		else if(r <= 6.1)
		{
			nn=8;
		}		
		else if(r < 9.1)
		{
			nn=12;
		}	
		else if(r < 21)
		{
			nn=20;
		}	
		else
		{
			nn=180;
		}			

		glPushAttrib(GL_POLYGON_BIT);
				
		glFrontFace(GL_CW);
        	glEnable(GL_CULL_FACE);  
		glCullFace(GL_BACK);
			
		an_step=2.*PI/nn;
	
		if(fill >=0 && fill < 8) //circle or filled wedge)
		{
			//Circle
			glPolygonMode(GL_FRONT,GL_LINE);				
			an=PI/2.;
			glBegin(GL_LINE_LOOP);
			for (k=0; k< nn; k++)
	  		{            		
	    			glVertex2f(cos(an) * r, sin(an) * r);
				an-=an_step;
	  		}
			glEnd(); 				
									
			//Filled wedge
			if(fill > 0)
			{
				float eps=0.001; 
				glPolygonMode(GL_FRONT,GL_FILL); 									
				glBegin(GL_POLYGON);
				glVertex2f(0,0);
								
				an=PI/2.;
				while(an >= PI/2.- fill*PI/4.-eps)
				{            			
	    				glVertex2f(cos(an) * r, sin(an) * r);
					an-=an_step;
	  			}
				glEnd();  			
			}	
		}
			
		//Draw a filled circle
		if(fill == 8)
		{			
			glPolygonMode(GL_FRONT,GL_FILL);	
			an=PI/2.;
			glBegin(GL_POLYGON);																				
			for (k=0; k<= nn; k++)
	  		{     
	    			glVertex2f(cos(an) * r, sin(an) * r);
				an-=an_step;
	  		}
			glEnd();  	
		}
		
		else if(fill == 9)
		{
			/*setNewPolygonMode(GL_LINE);
			glBegin(GL_LINES);
			glVertex2f(0,r-1.);
			glVertex2f(0,r+1.);
			glEnd();*/
		}
				
		glPopAttrib();	
	}
}		

/*!
  \brief render pixmaps

  This method renders pixmaps. These are used for cell shading and raster input (GIFs and PNGs).

  \sa renderCellArray()

  param x0 x of lower corner
  param y0 y of lower corner
  param x1 x of higher corner
  param y1 y of higher corner
  param w width of pixmap
  param h height of pixmap
  param pixmap contents
  param hasAlpha indicates if alpha is used or not

*/
MAGICS_NO_EXPORT bool OpenGLDriver::renderPixmap(float x0,float y0,float x1,float y1,
                                            int w,int h,unsigned char* pixmap,
					    int , bool hasAlpha) const
{
	//Put it into a texture
	Log::debug() << "renderPixmap: " << x0 << " " << y0 << " " << x1 << " " << y1 << endl; 

	if(actDpl_) actDpl_->compileEnd();

	OpenGLTextureItem *tex=new OpenGLTextureItem();
	texList_.push_back(tex);

	tex->setDplMode(true);

	//GLubyte *pix=new GLubyte[w*h*3];
	//for(int i=0; i< w*h*3; i+=3)
	//{
	//	pix[i]=0;
	//	pix[i+1]=0;
	//	pix[i+2]=0;
	//}

	if(hasAlpha)
	{
		tex->generateFromImage(pixmap,w,h,GL_RGBA);
	}
	else
	{
		tex->transparent(false);
		tex->generateFromImage(pixmap,w,h,GL_RGB);
	}

	//delete pix;

	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);

	assert(glLayoutStack_.empty() == false);
	//OpenGLLayoutNode *gll=glLayoutStack_.top();


	actDpl_->setTexture(tex);
	actDpl_->setClipping(true);
	actDpl_->parentLayout(glLayoutStack_.top());				

	actDpl_->compileStart();

	//Clipping
	//assert(glLayoutStack_.empty() == false);
	//OpenGLLayoutNode *gll=glLayoutStack_.top();
	
	//glEnable(GL_SCISSOR_TEST);
	//glScissor(gll->winX(),gll->winY(),gll->winWidth(),gll->winHeight());
		   
	//map it

	//glPushMatrix();
	//glTranslatef(x0,y1,0.);
	//glRotatef(-180.,0.,0.,1.);
	//glTranslatef(x0,y1,0.);
	tex->mapTexture(x0,y1,x1,y0);
	//tex->mapTexture(-100,-50,100,50);
	//glPopMatrix();

	//glDisable(GL_SCISSOR_TEST);

	actDpl_->compileEnd();

	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();

	return true;

#if 0
	unsigned char *p = pixmap;	
	const float dx =  (x1 - x0)/w;
	//const float dy = -(y1 - y0)/h;   // Minus needed for Y axis correction
	const float dy = (y1 - y0)/h;

	const float X0 = x0;
	const float Y0 = y0;
	
	int r,g,b;

	float wx, wy;

	setNewPolygonMode(GL_FILL);
	for (int i=h-1;i>=0;i--)
	{
		for(int j=0;j<w; j++)
		{
		 	r = (int) *(p++);
			g = (int) *(p++);
			b = (int) *(p++);
			setNewColour(Colour(r/255.,g/255.,b/255.));

			//const int in = width*i+j;	 
		  	//const short c = image[in];
 
 		  //if(!(lt[c]=="undefined"))
		  //if(!(lt[c]=="none"))
		 // {			
			//setNewColour(Colour(lt[c].red(),lt[c].green(),lt[c].blue()));
			
			wx = x0+(j*dx);
			wy = y0+(i*dy);
			
			glRectf(wx,wy,wx+dx,wy+dy);						  
		 // }// point has colour
		}
	}


	//for(int i=h-1;i>=0;i--)
	//{
	//	for(int j=0;j<w; x0+=dx,j++)
	//	{
	//		r = (int) *(p++);
	//		g = (int) *(p++);
	//		b = (int) *(p++);
	//		setNewColour(Colour(r/255.,g/255.,b/255.));
	//		
	//		float x0 = X0+(j*dx);
	//		float y0 = Y0+(i*dy);
	//		float x1 = x0+dx;
	//		float y1 = y0-dy;
	//										
	//		glRectf(x0,y0,x1,y1);			
	//		
	//	}
	//	x0 = X0;
	//	y0 += dy;
	//}

	return true;
#endif

return true;

}	

//Log::debug() << "OpenGLDriver::renderPixmap needs implementing." <<endl;
//	return true;
//}

/*!
  \brief render cell arrays

  This method renders cell arrays, also called images in Magics language. These are 
  mainly used for satellite data.

  sa renderPixmap()

  param image Object containing an image
*/
MAGICS_NO_EXPORT bool OpenGLDriver::renderCellArray(const Image& image) const
{	
	setNewPolygonMode(GL_FILL);

	ColourTable &lt = image.getColourTable(); 
	const int width  = image.getNumberOfColumns();
	const int height = image.getNumberOfRows();
	const float x0 = projectX(image.getOrigin().x());
	const float y0 = projectY(image.getOrigin().y());
	const float x1 = projectX(image.getOrigin().x()+image.getWidth());
	const float y1 = projectY(image.getOrigin().y()+image.getHeight());
	const float dx = (x1-x0)/width;
	const float dy = -(y1-y0)/height;

	float wx, wy;

	for (int i=height-1;i>=0;i--)
	{
		for(int j=0;j<width; j++)
		{
		  const int in = width*i+j;	 
		  const short c = image[in];
 
 		  //if(!(lt[c]=="undefined"))
		  if(!(lt[c]=="none"))
		  {			
			setNewColour(Colour(lt[c].red(),lt[c].green(),lt[c].blue()));
			
			wx = x0+(j*dx);
			wy = y0+(i*dy);
			
			glRectf(wx,wy,wx+dx,wy+dy);						  
		  }// point has colour
		}
	}
	return true;
}


/*MAGICS_NO_EXPORT void OpenGLDriver::renderSymbols(const Symbol<PaperPoint>& symbol) const
{
	//debugOutput("Start symbols");
	
	if(symbol.getSymbol()=="logo_ecmwf")
		//renderLogo(projectX(symbol[0].x()),projectY(symbol[0].y()));
		return;
	else 
		BaseDriver::renderSymbols(symbol);
}*/


//============================================================
//
// OpenGLDriver specific methods from this point
//
//============================================================


/*!
  \brief prints debug output

  When Magics++ is compiled in debug mode these extra strings are printed.

  \note This can increase file and log file sizes if you run Magics++ in debug mode!

  \param s string to be printed
*/
MAGICS_NO_EXPORT void OpenGLDriver::debugOutput(const string &s) const
{
	Log::debug() << s << endl;
}

/*!
  \brief class information are given to the output-stream
*/
void OpenGLDriver::print(ostream& out)  const
{
	out << "OpenGLDriver[";
	out << "]";
}

/*! \brief Function to check if version of OpenGL is supported
    Could be improved if static. Because it will always have the same result!
*/
static bool supportsOpenGLVersion(int atLeastMajor, int atLeastMinor)
{
	const char* version = (const char*) glGetString(GL_VERSION);
	
	Log::info() << "OpenGL version: "  << glGetString(GL_VERSION) << endl;
	Log::info() << "OpenGL vendor: "   << glGetString(GL_VENDOR) << endl;
	Log::info() << "OpenGL renderer: " << glGetString(GL_RENDERER) << endl;
	
	int major, minor;
	if(sscanf(version, "%d.%d", &major, &minor) == 2)
	{
		if(major > atLeastMajor) return true;
		if(major == atLeastMajor && minor >= atLeastMinor) return true;
	}
	else Log::warning() << "OpenGLDriver: OpenGL version string is malformed!" << endl;
	return false;
}



void OpenGLDriver::setNewPolygonMode(const GLint p) const
{
        if(currentPolygonMode_ == p) return;

	currentPolygonMode_ = p;
	glPolygonMode(GL_FRONT_AND_BACK,currentPolygonMode_);	
}

/*!
   \brief Main function to redraw the OpenGL widget.
   
   This method is not only called by the X-Callbacks for resize, but also
   by memberfunctun such as newPage()!
   
*/
void OpenGLDriver::redisplay() const
{
	ready_ = false;
	//GLwDrawingAreaMakeCurrent(openGLArea_, openGLContext_);
	swapFb();
	ready_ = true;

	checkError("OpenGLDriver::redisplay()");
}

void OpenGLDriver::Primitive_cluster(int i, int n, float* cx, float* cy, GLenum style) const
{	
	setNewPolygonMode(GL_LINE);
	float* xx = cx;
        float* yy = cy;
	int j;

	glEnableClientState(GL_VERTEX_ARRAY);

	for ( j = 0; j<n; j++)
	{
		int k;

 		glBegin(style); 
		for ( k = 0; k<i; k++)
		{
			glVertex2f(xx[j*i+k],yy[j*i+k]);
		}
		glEnd();
	}
	checkError("OpenGLDriver::Primitive_cluster()");
}


void OpenGLDriver::Primitive_simple(int n, float* cx,float* cy, GLenum style) const
{
//	setNewPolygonMode(GL_LINE);
	float* xx = cx;
	float* yy = cy;

#ifndef OPENGL_11	
	glBegin(style);
	int j; 
	for ( j = 0; j<n; j++)
	{
		glVertex2f(xx[j],yy[j]);		
	}
	glEnd();
#else
	GLfloat* vert = new GLfloat[n*2];
	int run=0;
	for (int j = 0; j<n; j++)
	{
		vert[run] = xx[j]; run++;
		vert[run] = yy[j]; run++;
	}
 
	glPushClientAttrib(GL_CLIENT_VERTEX_ARRAY_BIT);
	glEnable(GL_LINE_SMOOTH);
	glEnableClientState(GL_VERTEX_ARRAY);

	glVertexPointer(2, GL_FLOAT,2*sizeof(GLfloat), vert);
	glDrawArrays(style, 0, n-1);

	glPopClientAttrib();

	delete [] vert;
#endif
	checkError("OpenGLDriver::Primitive_simple()");
}

void OpenGLDriver::renderFilledPolygon(const int n, float* x, float* y) const
{
	setNewPolygonMode(GL_FILL);	
	//gluTessBeginPolygon(tobj_, NULL);
		
	const bool closed = true;
	glHole *hpt = new glHole(n);
	GLdouble* 	u=hpt->GetData();
        GLdouble*	v;
	
	gluTessBeginPolygon(tobj_,NULL);
	gluTessBeginContour(tobj_);
	for(int i = 0; i<n; i++)
	{
		v = u;
		*(u++) = x[i];
		*(u++) = y[i];
		if(closed || (i<n-2) ) *(u++) = 0.;
		else *(u++) = 0.5;
		gluTessVertex(tobj_,v,v);
	}	
	gluTessEndContour(tobj_);	
	gluTessEndPolygon(tobj_);
	
	delete hpt;
}

MAGICS_NO_EXPORT void OpenGLDriver::renderTextForMagnifier(OpenGLLayoutNode *gll,
                         const Text *text, float mFactor, float tFactor)
{
	//If text is already compiled to dpl returns	
	//if(dpl->compiled() == true) return;	
	
	setCoordRatio(gll);

	//If the text is rendered for magnification the text position is mfactor times
	//further away from the oriogo in GL window coordinates. Set parameters for the	
	//translation!	
	float Xoff=0., Yoff=0, Xmin=0., Ymin=0.;	
	//Xoff=gll->layout()->x() * 0.01 * gll->dimensionX();
	//Yoff=gll->layout()->y() * 0.01 * gll->dimensionY();
	Xoff=gll->winX();
	Yoff=gll->winY();
	Xmin=gll->layout().minX();
	Ymin=gll->layout().minY();
	
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 ); 
	renderTextItem(text,mFactor,tFactor,Xoff,Yoff,Xmin,Ymin);
		
	resetCoordRatio();	
}

	
MAGICS_NO_EXPORT void OpenGLDriver::renderTextItem(const Text* text,
					float mFactor,float tFactor,float Xoff, float Yoff, float Xmin, float Ymin) const					       
{	
	Text *t=text->clone();

	if(tFactor > 1)
	{
		float transX=(tFactor-1.)*(Xoff+projectX(-Xmin));
		float transY=(tFactor-1.)*(Yoff+projectY(-Ymin));

		for(vector<PaperPoint>::iterator it=t->begin(); it != t->end(); it++)
		{ 
			float x=(*it).x();
			(*it).x((tFactor-1.)*(Xoff+projectX(-Xmin))+tFactor*projectX(x));

			float y=(*it).y();
			(*it).y((tFactor-1.)*(Yoff+projectY(-Ymin))+tFactor*projectY(y));
		}	
	}
	else
	{
		for(vector<PaperPoint>::iterator it=t->begin(); it != t->end(); it++)
		{ 
			float x=(*it).x();
			(*it).x(projectX(x));

			float y=(*it).y();
			(*it).y(projectY(y));
		}	
	}

	painter_->renderText(t,coordRatioY_,mFactor);	
}


//============================================
//
//   INTERACTIVE MODE
// 
//============================================



void OpenGLDriver::notify(MagicsZoomEvent& event)
{
	notifyObservers(&OpenGLDriverObserver::zoomControlStatusForOpenGLGui, event.set());
}

void OpenGLDriver::notify(MagicsMagnifierEvent& event)
{
	notifyObservers(&OpenGLDriverObserver::magnifierControlStatusForOpenGLGui,event.set());
}

void OpenGLDriver::notify(MagicsAntialiasingEvent& event)
{
	if(antialiasing_ != event.set())
	{
		antialiasing_= event.set();
		renderTree(stepToRender_);
	}
}

void OpenGLDriver::notify(MagicsRestoreFbEvent& event) 
{
	restoreFb(); 
}

void OpenGLDriver::notify(MagicsAnimationCurrentStepEvent& event) 
{
	renderTree(event.step()); 
}

void OpenGLDriver::notify(MagicsLayerUpdateEvent& event) 
{
	layerUpdate();
}

void OpenGLDriver::pickSelection(const SelectionMode &sm)
{
 	//pickObjectAttr_.copy(sm);
	setInteractiveMode(IAM_PICK); 	
}	

void OpenGLDriver::pointSelection(const SelectionMode &sm)
{
 	//pickObjectAttr_.copy(sm);	
	//setInteractiveMode(IAM_POINT); 
}  

void OpenGLDriver::lineSelection(const  SelectionMode &sm)
{	
	//pickObjectAttr_.copy(sm);
	//setInteractiveMode(IAM_LINE); 
}  

void OpenGLDriver::areaSelection(const  SelectionMode &sm)
{ 	
	pickObjectAttr_.copy(sm);		
	//setInteractiveMode(IAM_AREA); 
	setInteractiveMode(IAM_ZOOM);
}  

void OpenGLDriver::polylineSelection(const  SelectionMode &sm)
{
 	//pickObjectAttr_.copy(sm);
	//setInteractiveMode(IAM_POLYLINE);
}  

void OpenGLDriver::polygonSelection(const  SelectionMode &sm)
{
	//pickObjectAttr_.copy(sm);
	//setInteractiveMode(IAM_POLYGON);		
}  

/*!
  \brief Leaves interactive mode
  
  This method calls the generic method to leave the interactive 
  mode properly
  
*/
void OpenGLDriver::noMode()
{ 	
	setInteractiveMode(IAM_NONE);
}  

void OpenGLDriver::setInteractiveMode(InteractiveMode new_mode)
{
	notifyObservers(&OpenGLDriverObserver::setInteractiveMode,new_mode);
}	

void OpenGLDriver::restoreFb() 
{
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->buildFb();
	}
}

void OpenGLDriver::swapFb() const
{	
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->buildFb();
	}
	MagicsSwapBufferEvent event;
	notifyObservers(event);
	
}

void OpenGLDriver::swapFbWithCurrentContent() const
{	
	MagicsSwapBufferEvent event;
	notifyObservers(event);
}

void OpenGLDriver::resetFb() const
{
	//mapBgImageToFb(0,0,0,dimensionX_,dimensionY_,0,0,dimensionX_,dimensionY_);
}	


void OpenGLDriver::mapBgImageToFb(const int id) const
{
	//mapBgImageToFb(id,0,0,dimensionX_,dimensionY_,0,0,dimensionX_,dimensionY_);
}	


void OpenGLDriver::mapBgImageToFb(const int id, const int x0, const int y0, const int x1, const int y1) const
{
	//mapBgImageToFb(id,x0,y0,x1,y1,x0,y0,x1,y1);
}	


void OpenGLDriver::mapBgImageToFb( const int id, const int xs0, const int ys0, const int xs1, const int ys1,
				const int xt0, const int yt0, const int xt1, const int yt1) const
{
	GLuint texId = tex_bg_id_[id];
	
	if(glIsTexture(texId) == GL_FALSE) 
        {	 
	  return;	  
        }    
      
	//Map bg texture to the framebuffer
        glBindTexture(GL_TEXTURE_2D,texId);
		
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );		
	glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);
		
	glEnable(GL_TEXTURE_2D);
	
	glPushAttrib(GL_POLYGON_BIT);
	glFrontFace(GL_CCW);
	glEnable(GL_CULL_FACE);	
	glCullFace(GL_BACK);	
	glPolygonMode(GL_FRONT,GL_FILL);	
		 
    	glBegin( GL_QUADS );

	float tx0=static_cast<float>(xs0)/static_cast<float>(dimensionX_);
	float ty0=static_cast<float>(ys0)/static_cast<float>(dimensionY_);
	float tx1=static_cast<float>(xs1)/static_cast<float>(dimensionX_);
	float ty1=static_cast<float>(ys1)/static_cast<float>(dimensionY_);
    					    	
	glTexCoord2f(tx0, ty0 );
	glVertex2f(xt0,yt0);

	glTexCoord2f(tx1, ty0);    	
	glVertex2f(xt1,yt0);
		
	glTexCoord2f(tx1, ty1);    	
	glVertex2f(xt1,yt1);		

	glTexCoord2f(tx0,ty1);    	
	glVertex2f(xt0,yt1);

    	glEnd();
	   
	glPopAttrib();   
	   
	glDisable(GL_TEXTURE_2D);  
}


void OpenGLDriver::saveFbToBgImage(const int id) const
{
      GLuint texId = tex_bg_id_[id];	     
      
      if(glIsTexture(texId) == GL_TRUE) 
        {	 
	  glDeleteTextures(1,&texId);	  
        }
	
      glBindTexture(GL_TEXTURE_2D,texId);
      	           
      glCopyTexImage2D(GL_TEXTURE_2D,0, GL_RGBA, 
      	      0,0,dimensionX_, dimensionY_,0);
                
      //checkError("OpenGLDriver::saveFbToTexture()");      
}

//! Method to check if an OpenGL error occured.
/*!
  Method to check if an OpenGL error occured in the previous code.
  
  To do so the OpenGL function glGetError() is used.
  
  \param s String token to say where and what occured. 
*/
void OpenGLDriver::checkError(string s) const
{
	GLenum error;
	while( (error = glGetError()) != GL_NO_ERROR)
	{
		Log::error() <<"OpenGLDriver: in "<< s <<"  --> glError() reports: "<<gluErrorString(error) << "\n"; 
		assert(error != GL_NO_ERROR  ); 
	}
}


////////////////////////////////////////////////////////////////////////////////
//
//   O P E N G L   C A L L B A C K S
//
////////////////////////////////////////////////////////////////////////////////


void OpenGLDriver::inputEvent(MtInputEvent *event)
{
	notifyObservers(&OpenGLDriverObserver::inputEventDispatchFromDriver,event);
}

void OpenGLDriver::TessErrorCB(GLenum errorCode)
{
	const GLubyte *estring;
	estring = gluErrorString(errorCode);
	Log::error() << "OpenGLDriver --> Tessellation Error: "<< estring << endl;
}

//void OpenGLDriver::myCombine(GLdouble coords[3],GLdouble *data[4],GLfloat weight[4],GLdouble **dataOut)
void OpenGLDriver::TessCombineCB(GLdouble coords[3],GLdouble**,GLfloat*,GLdouble **dataOut)
{
	GLdouble *v;
	v = new GLdouble [3];
	v[0] = coords[0];
	v[1] = coords[1];
	v[2] = coords[2];
	*dataOut = v;
}

//==============================================================
//
// Observer handling
//
//==============================================================

void OpenGLDriver::registerObserver(OpenGLDriverObserver* observer)
{
	observers_.push_back(observer);
}
void OpenGLDriver::unregisterObserver(OpenGLDriverObserver* observer)
{
	observers_.erase(std::remove_if(observers_.begin(), observers_.end(), 
    	bind2nd(equal_to<OpenGLDriverObserver*>(), observer)), observers_.end());
	
}

void OpenGLDriver::notifyObservers(MagicsEvent& event) const
{
	Log::dev()<< "OpenGLDriver::notifyObservers-->" << event << endl;
	for (vector<OpenGLDriverObserver*>::const_iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
			(*observer)->notify(event);
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::Function function) const
{
    for (vector<OpenGLDriverObserver*>::const_iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)();
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::SelectionFunction function, SelectionObject* object)
{
    for (vector<OpenGLDriverObserver*>::iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)(object);
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::InteractiveModeFunction function, InteractiveMode m)
{
    for (vector<OpenGLDriverObserver*>::iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)(m);
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::BoolFunction function, bool b)
{
    for (vector<OpenGLDriverObserver*>::iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)(b);
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::IntFunction function, int ival)
{
    for (vector<OpenGLDriverObserver*>::iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)(ival);
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::StringFunction function, const string& sval)
{
    for (vector<OpenGLDriverObserver*>::iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)(sval);
}

void OpenGLDriver::notifyObservers(OpenGLDriverObserver::InputEventFunction function, MtInputEvent *event)
{
    for (vector<OpenGLDriverObserver*>::iterator observer = observers_.begin(); observer != observers_.end(); ++observer)
		((*observer)->*function)(event);
}

static SimpleObjectMaker<OpenGLDriver, BaseDriver> OpenGL_driver("OpenGL");


//==========================================================
//
// Visit the tree nodes
//
//==========================================================

void OpenGLDriver::visit(OpenGLNode& node)
{	
	(this->*visitFunc_)(node);
}

bool OpenGLDriver::visitEnter(OpenGLNode& node)
{	
	return (this->*visitEnterFunc_)(node);
}

bool OpenGLDriver::visitLeave(OpenGLNode& node)
{	
	return (this->*visitLeaveFunc_)(node);
}

//==========================================================
//
// Visitors: rendering the standard tree
//
//==========================================================

bool OpenGLDriver::visitEnter_renderTree(OpenGLNode& node)
{			
	node.preTransformDpl().call();

	if(node.classType() == "OpenGLLayerNode")
	{
		OpenGLLayerNode *layer= static_cast<OpenGLLayerNode*>(&node);

		if(layer->visible() == false) 
		{
			return false;
		}
		
		float alpha=layer->alpha();
		if(alpha != 1.)
		{
			glEnable(GL_BLEND); 
			glBlendFunc(GL_CONSTANT_ALPHA,GL_ONE_MINUS_CONSTANT_ALPHA);
			glBlendColor(1.,1.,1.,alpha);	
		
		}
	}
	else if(node.classType() == "OpenGLStepNode")
	{
		OpenGLStepNode *step = (OpenGLStepNode*)&node;
		if(stepToRender_ != step->id())
		{
			checkError("Enter> ");
			return false;					
		}
		
		return true;

	}	
	
	else if(node.classType() == "OpenGLPreviewLayoutNode"  || 
		node.classType() == "OpenGLMagnifierLayoutNode")
	{
		checkError("Enter> ");
		return false;
	}

	

	//Log::dev()  <<  "Executing: " << 	`	glb->name() << "[preTransform] " <<   endl;		
	
	return true;	
}



bool OpenGLDriver::visitLeave_renderTree(OpenGLNode& node)
{		
	if(node.classType() == "OpenGLLayerNode")
	{
		float alpha=static_cast<OpenGLLayerNode*>(&node)->alpha() ;
		if(alpha != 1.)
		{
			glDisable(GL_BLEND); 		
		}
	}
	//Log::dev()  <<  "Executing: " << glb->name() << "[postTransform]" << endl;
	//Log::dev()<<  "Executing: " << glb->name() << "[postTransform]" << endl;	
	
	node.postTransformDpl().call();

	return true;		
}


void OpenGLDriver::visit_renderTree(OpenGLNode& node)
{		

//Log::dev() <<  "Executing: " << glb->name() << " (objects: " << glb->dpl().size() << ")" << endl;
	Log::dev()<<  "Executing: " << node.name() << " (objects: " << node.dpl().size() << ")" << endl;

	std::set<mglDplType> s;
	s.insert(M_DPL_NORMAL);
	s.insert(M_DPL_TEXT);

	bool clipping;

	for(OpenGLDplManager::DplIterator it=node.dpl().firstItem(); it != node.dpl().lastItem(); it++)
	{
		if((*it)->type() == M_DPL_NORMAL || (*it)->type() == M_DPL_TEXT)
		{
			clipping=(*it)->clipping();
			if(clipping)
			{
				OpenGLLayoutNode *gll=(*it)->parentLayout();
				glEnable(GL_SCISSOR_TEST);
				glScissor(gll->winX(),gll->winY(),gll->winWidth(),gll->winHeight());
			}
			
			if((*it)->texture() != 0)
			{
				(*it)->texture()->bind();
			}
	
			(*it)->call();

			if((*it)->texture() != 0)
			{
				(*it)->texture()->unBind();
			}


			if(clipping)
			{
				glDisable(GL_SCISSOR_TEST);
			}
		}
	}		
	//node.dpl().call(s);	
}	


void OpenGLDriver::renderTree(int stepNo)
{
	//Get all avialable frames
	/*list<OpenGLNode*> fr;
	glTree_->findFrame(fr);

	Log::dev() << "---> renderTree: Number of frames: " << fr.size() << endl;
	Log::dev() << "---> renderTree: current node: " << 
				currentGlNode_->name() << endl;

	//Return if threre are no frames!
	if(fr.empty()) return;*/

	//setNewColour(Colour(1.,1.,1.));	

	checkError("OpenGLDriver::renderTree() 1");

	//Clear display

	glClear(GL_COLOR_BUFFER_BIT);

	//???
	currentPolygonMode_ = GL_LINE;
	glPolygonMode(GL_FRONT_AND_BACK,currentPolygonMode_);

	//Save the current step id
	int ori_step=stepToRender_;
	
	//Set the current step id to the new value
	stepToRender_=stepNo;

	stepObjectToRender_=0;

	//Get the Layout node of the frames
	//OpenGLFrameNode* fr_tmp = (OpenGLFrameNode*) (*fr.begin());
	//glLayoutStack_.push((OpenGLLayoutNode*)fr_tmp->parent());	

	list<OpenGLStepNode*> glsv;

	bool treeChanged=false;

	if(animationStepMap_.size()  != 0 ) 
	{		
		animationStepMap_.begin()->second->selectStep(stepToRender_,glsv);
	
		//If the step node was found
		for(list<OpenGLStepNode*>::iterator it=glsv.begin(); it != glsv.end(); it++)
		{
			OpenGLStepNode* gls =*it;						
			stepObjectToRender_= gls->step();
						
			if(gls->cached() == false)
			{			
				OpenGLLayoutNode* gll=gls->parentLayout();

				glLayoutStack_.push(gll);
				staLayouts_.push(&gll->layout());	
			
				setDimension(gll);

				currentGlNode_=gls;
				
				addGlNode(gls);

				actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
				actDpl_->compileStart();

				gls->step()->execute(*this,gll->layout());

				if(actDpl_==0)
				{
					addGlNode(gls);
					actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
					actDpl_->compileStart();
				}
			
				actDpl_->compileEnd();		

				gls->cached(true);
				
				resetDimension();
			
				//Remove layout			
				glLayoutStack_.pop();
				staLayouts_.pop();	

				treeChanged=true;			
			}
		}

		checkError("OpenGLDriver::renderTree() 2");
		//Pop the frames' Layout node 
		

	}
	
	checkError("OpenGLDriver::renderTree() 3");
	//Current node is the root
	//currentGlNode_=glTree_->root();
	currentGlNode_=glTree_;

	//OpenGL setting for rendering
	
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
	
	if(antialiasing_)
	{
		glPushAttrib(GL_COLOR_BUFFER_BIT | GL_LINE_BIT | GL_HINT_BIT );
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glEnable(GL_LINE_SMOOTH);
		glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);		
	}

	//Render the tree
	visitFunc_=&OpenGLDriver::visit_renderTree;
	visitEnterFunc_=&OpenGLDriver::visitEnter_renderTree;
	visitLeaveFunc_=&OpenGLDriver::visitLeave_renderTree;
	
	glTree_->print();

	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->fboBg()->bind();
	}

	glClear(GL_COLOR_BUFFER_BIT);	
	glTree_->accept(*this);
	
	//Save bg texture to fb texture
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->fboBg()->unBind();
		OpenGLCore::instance()->bgToFb();		
	}

	//Save fb content to bg texture
	else if(OpenGLCore::instance()->renderMode() == OpenGLCore::BackBufferMode)
	{
		OpenGLCore::instance()->fbToBg();
	}

	if(antialiasing_)
	{
		glPopAttrib();
	}

	/*glColor3f(1.,0.,0.);
	
	loadSymbols();
	for(unsigned int ii=0;ii<sym_.size();ii++)
	{
		glPushMatrix();
		glLoadIdentity();
		string id=sym_[ii].id;
		symbols_[id] = new OpenGLDplItem;
		generateSymbolDpl(symbols_[id],sym_[ii]);
		glTranslatef(50,20+ii*30,0.);		
		glScalef(20,20,1.);
		//glRectf(-0.5,-0.5,1.,1.);
		symbols_[id]->call();
		glPopMatrix();
	}*/
	
	if(treeChanged)
	{
		getLayerTree();
		MagicsLayerTreeEvent event(layerTree_);
		notifyObservers(event);	
	}

	checkError("OpenGLDriver::renderTree()");

	//saveFbToBgImage(0);
	//saveFbToBgImage(1);
	checkError("OpenGLDriver::renderTree() 6");
	//Notify observers about the change in current frame
	//if(frameToRender_ != ori_frame)
	//{
	notifyObservers(&OpenGLDriverObserver::currentFrameChanged);
	checkError("OpenGLDriver::renderTree() 7");
	//Swap the buffers!
	swapFb();

	//}
}	


//==========================================================
//
// Visitors: rendering the preview tree
//
//==========================================================

bool OpenGLDriver::visitEnter_renderPreview(OpenGLNode& node)
{						
	node.preTransformDpl().call();		
	//Log::dev()  <<  "Executing: " << glb->name() << "[preTransform] " <<   endl;		
	return true;
}


bool OpenGLDriver::visitLeave_renderPreview(OpenGLNode& node)
{			
	node.postTransformDpl().call();	
	//Log::dev()  <<  "Executing: " << glb->name() << "[postTransform]" << endl;			
}


void OpenGLDriver::visit_renderPreview(OpenGLNode& node)
{	
	//draw node				
	std::set<mglDplType> s;
	s.insert(M_DPL_NORMAL);
	s.insert(M_DPL_TEXT);
	
	node.dpl().call(s);
}	


void OpenGLDriver::renderPreview(OpenGLPreviewLayoutNode *glm,int tx, int ty,int tw, int th)
{	
	//Check if it is a valid node
	if(glm == 0) return;
	
	//get the preview plot pos and size
	int x=glm->winX(); 
	int y=glm->winY();	
	int w=glm->winWidth();	
	int h=glm->winHeight();	

	int tw_vp=tw*(dimensionX_/static_cast<float>(w));
	int th_vp=th*(dimensionY_/static_cast<float>(h));

	//glPushMatrix();
	//glLoadIdentity();
	//glPushAttrib(GL_CURRENT_BIT);
	//glColor3f(1.,0.,0.);
	//glRectf(tx,ty,tw,th);
	//glPopAttrib();
	//glPopMatrix();
	
       	//Set the viewport according to the preview plot pos and size
	glPushAttrib(GL_VIEWPORT_BIT);		
	glViewport(tx,ty,tw_vp,th_vp);
	glClear(GL_COLOR_BUFFER_BIT);

	//Render the whole tree with preview mode
	visitFunc_=&OpenGLDriver::visit_renderPreview;
	visitEnterFunc_=&OpenGLDriver::visitEnter_renderPreview;
	visitLeaveFunc_=&OpenGLDriver::visitLeave_renderPreview;

	glm->accept(*this);

	//Re-set viewport
	glPopAttrib();
}

void OpenGLDriver::addGlNode(OpenGLNode *parent) const
{
	string parentName=parent->name();
	
	stringstream sbuff;
	sbuff << parent->numOfChildren();

	//string name=parentName + "." + "gl_"+sbuff.str();
	string name="gl_"+sbuff.str();
	OpenGLNode *gln = new OpenGLNode(name);
	//glTree_->addNode(parentName,name,gln);
	parent->addNode(gln);

	currentGlNode_=gln;
}


void OpenGLDriver::setDimension(OpenGLLayoutNode *gll) const
{
	setDimension(gll->dimensionX(),gll->dimensionY());
}

void OpenGLDriver::setDimension(float dx, float dy) const
{
	dimensionStack_.push(dimensionX_);
	dimensionStack_.push(dimensionY_);
	dimensionX_=dx;
	dimensionY_=dy;
}

void OpenGLDriver::resetDimension() const
{
	dimensionY_  = dimensionStack_.top();
	dimensionStack_.pop();	
	dimensionX_  = dimensionStack_.top();
	dimensionStack_.pop();
}

void OpenGLDriver::setCoordRatio(OpenGLLayoutNode *gll) const
{
	setCoordRatio(gll->coordRatioX(),gll->coordRatioY());
}

void OpenGLDriver::setCoordRatio(float cx, float cy) const
{
	scalesX_.push(coordRatioX_);
	scalesY_.push(coordRatioY_);
	coordRatioX_=cx;
	coordRatioY_=cy;
}

void OpenGLDriver::resetCoordRatio() const
{
	coordRatioX_  = scalesX_.top();
	coordRatioY_  = scalesY_.top();
	scalesX_.pop();	
	scalesY_.pop();
}

MAGICS_NO_EXPORT void OpenGLDriver::project(OpenGLLayoutNode *gll,OpenGLNode *parent,
					    const magics::Layout& layout) const
{	
	Log::debug() << "---> PROJECT " << layout.name() <<  endl;

	//Close the current display list
	if(actDpl_) actDpl_->compileEnd();
	actDpl_=0;

	//Get layout geometry
	const float width    = layout.width()  * 0.01 * dimensionX_;
	const float height   = layout.height() * 0.01 * dimensionY_;
	const float x        = layout.x() * 0.01 * dimensionX_;
	const float y        = layout.y() * 0.01 * dimensionY_;
	
	//Get layout coordinate range
	const float minX = layout.minX();
	const float maxX = layout.maxX();
	const float minY = layout.minY();
	const float maxY = layout.maxY();
	const float rangeX = maxX - minX;
	const float rangeY = maxY - minY;

	//Set dimensions
	dimensionStack_.push(dimensionX_);
	dimensionStack_.push(dimensionY_);	
	dimensionX_ = width;
	dimensionY_ = height;

	gll->dimensionX(dimensionX_);
	gll->dimensionY(dimensionY_);

	//Find out gl window coordinates of the box. We need the 
	//parent layout geometry!

	OpenGLLayoutNode *parentLayout=0;

	if(!glLayoutStack_.empty())
	{
		parentLayout=glLayoutStack_.top();
		gll->window(parentLayout->winX()+x, parentLayout->winY()+y,
	                    width,height);
	}
	else
	{
		gll->window(x,y,width,height);
	}
	
	Log::debug() << "window" << gll->winX() << " " << gll->winY() << " " << gll->winWidth() << " " << gll->winHeight() << endl;

	//Add glLayoutNode to the tree and registers it!
	//glTree_->addNode(parent->name(),gll->name(),gll);		
	parent->addNode(gll);

	//OpenGLNode* currentGlNode_=gll;
	currentGlNode_=gll;
	glLayoutStack_.push(gll);	

	//Add animatonControl
	/*if(layout.animationRules() && !layout.animationRules()->empty())
	{
		OpenGLAnimationControl *anc= new OpenGLAnimationControl;
		animationStepMap_[gll]=anc;

		AnimationRules *rule = layout.animationRules();
		
		int i=0;
		for(vector<AnimationStep*>::iterator it=rule->begin(); it != rule->end(); it++)
		{
			AnimationStep* step=*it;
			assert(!step->label().empty());
			Log::debug() <<"step: " << step->label().front() << endl;

			std::ostringstream out;
   			out  << i;

			string name=gll->name() + "." + "step_" + out.str();					
			OpenGLStepNode *gls=new OpenGLStepNode(name,step);		

			gls->id(i);

			//glTree_->addNode(gll->name(),gls->name(),gls);	

			anc->addStepNode(gls);

			//glAnimationLayoutList_.push_back(gll);

			i++;
		}
	}*/

	//-------------------------------------
	// Generate dpl for pre-transform	
	//-------------------------------------	

	actDpl_=gll->addPreTransformDpl();

	actDpl_->compileStart();
			
	//Save mv matrix! There must be a glPopMatrix call in post-transform!
	glPushMatrix();	    

	//Here we have to compensate the translation by project(-XMin) etc. in the 
	//parent layout node!
	if(parentLayout)
	{
		glTranslatef(parentLayout->projectedMinX(),
			     parentLayout->projectedMinY(),0.);
		
		Log::dev() << "Translate redo: " << parentLayout->projectedMinX()  << " " << parentLayout->projectedMinY() << endl;	
	}

	//Translate the origo and put the new origo 
	//(in gl window coordinates!) to the stack 	
	glTranslatef(x,y,0.0);

	Log::dev() << "Translate: " << x << " " << y << endl;	 

	//Set the scaling factor
	//projectX and, projectY depend on coordRatio[X,Y]_ !!!      	
	const float xx = width/rangeX;
	const float yy = height/rangeY;

	scalesX_.push(coordRatioX_);
	scalesY_.push(coordRatioY_);
	coordRatioX_ = xx;
	coordRatioY_ = yy;

	gll->coordRatioX(coordRatioX_);
	gll->coordRatioY(coordRatioY_);

	glTranslatef(projectX(-minX),projectY(-minY),0.);	

	//Save the projected minX, minY
	gll->projectedMinX(projectX(minX));
	gll->projectedMinY(projectY(minY));
	gll->projectedMaxX(projectX(maxX));
	gll->projectedMaxY(projectY(maxY));

	Log::dev() << "Translate: " << projectX(-minX) << " " << projectY(-minY) << endl;	 

	/*glPushAttrib(GL_CURRENT_BIT | GL_POLYGON_BIT);
	glColor3f(0.,0.,1.);
	glPolygonMode(GL_FRONT, GL_LINE);
	glRectf(projectX(minX),projectY(minY),projectX(maxX),projectY(maxY));
	glPopAttrib();*/	

	//Finish dpl for pre-transform
	actDpl_->compileEnd();
	
	//-----------------------------------------------
	// Generate dpl for post-transform (un-project)
	//-----------------------------------------------

	actDpl_=gll->addPostTransformDpl();
	actDpl_->compileStart();
		
	//glPopAttrib();
	glPopMatrix();

	//Finish post-transform dpl 
	actDpl_->compileEnd();


	//Add new node and start dpl for the central part
	addGlNode(gll);
	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();

	emptyNode_=true;

	//Add animatonControl
	if(layout.animationRules() && !layout.animationRules()->empty())
	{
		OpenGLAnimationControl *anc= new OpenGLAnimationControl;
		animationStepMap_[gll]=anc;

		AnimationRules *rule = layout.animationRules();
		
		int i=0;
		for(vector<AnimationStep*>::iterator it=rule->begin(); it != rule->end(); it++)
		{
			AnimationStep* step=*it;

			//Log::debug() <<"step: " << step->label() << endl;

			std::ostringstream out;
   			out  << i;
			string name=currentGlNode_->name() + "." + "step_" + out.str();					

			OpenGLStepNode *gls=new OpenGLStepNode(name,step);		
			gls->id(i);

			anc->addStepDef(gls);

			i++;
		}
	}

	//Error check
	string s="OpenGLDriver::project()" + gll->name();
	checkError(s);

}		



MAGICS_NO_EXPORT OpenGLNode* OpenGLDriver::getNewLayoutParentInTree() const
{	
	//------------------------------------------
	// Generate OpenGLLayoutNode for the layout
	//------------------------------------------
	
	OpenGLLayoutNode* gll;
	OpenGLNode *parent=currentGlNode_;

	//If the current node is "empty" then step one level up in the
	// tree
	if(parent != glTree_ && parent->classType() == "OpenGLNode" && 
           parent->numOfChildren() == 0 && parent->dpl().getSize() == 1)
	{
		parent=parent->parent(); 		
		currentGlNode_=parent; //just in case
	}	
	return parent;
}



OpenGLAnimationControl* OpenGLDriver::animationControl()
{	
	if(animationStepMap_.size() ==0 ) return 0;

	return animationStepMap_.begin()->second;
}


MAGICS_NO_EXPORT void OpenGLDriver::redisplay(const PreviewLayout& layout) const 
{
	//Log::dev() << "BaseDriver::redisplay(const PreviewLayout& layout)" << endl;

	project(layout);
	layout.visit(*this);  // visit this ROOT layout! 
	unproject();
}

MAGICS_NO_EXPORT void OpenGLDriver::redisplay(const MagnifierLayout& layout) const 
{
	//Log::dev() << "OpenGLDriver::redisplay(const MagnifierLayout& layout)" << layout<<  endl;

	project(layout);
	layout.visit(*this);  // visit this ROOT layout! 
	unproject();

}

float OpenGLDriver::calcMagnifierMaxPower()
{
	if(animationStepMap_.empty()) return -1; 
	OpenGLAnimationControl *ac=animationStepMap_.begin()->second;

	float dx, dy;	
	ac->stepGridResolution(stepToRender_,dx,dy);

	list<OpenGLNode*> lst;
	glTree_->findMagnifierLayout(lst);

	if(lst.size() <=0 ) return -1;

	OpenGLMagnifierLayoutNode* gll=static_cast<OpenGLMagnifierLayoutNode*>(*(lst.begin())); 

	if(dx >0 && dy > 0) 
	{
		float wdx=70;
		float wdy=40;
		float ldx, ldy;	


		gll->winSizeToLayoutSize(wdx,ldx,true);
		gll->winSizeToLayoutSize(wdy,ldy,false);

		float magx=ldx/dx;
		float magy=ldy/dy;

		return (magx > magy)?magx:magy;
	}
	else
	{
		return -1;
	}
}


void OpenGLDriver::clearMagnifierLayout()
{
	list<OpenGLNode*> lst;
	glTree_->findMagnifierLayout(lst);

	if(lst.size() <=0 ) return;
	OpenGLMagnifierLayoutNode* gll=static_cast<OpenGLMagnifierLayoutNode*>(*(lst.begin()));

	gll->deleteChildren();
	gll->dpl().erase();
}


void OpenGLDriver::magnifierLayoutUpdate(float wx1, float wy1, float wx2, float wy2,float wdx, float wdy,float zoom)
{
	checkError("OpenGLDriver::magnifierLayoutUpdate() 1");

	list<OpenGLNode*> lst;
	glTree_->findMagnifierLayout(lst);
	
	magnifierZoom_=zoom;
		
	if(lst.size() <=0 ) return;

	OpenGLMagnifierLayoutNode* gll=(OpenGLMagnifierLayoutNode*) *(lst.begin());  

	vector<PaperPoint> pp;
	if(gll->winAreaToLayoutArea(wx1,wy1,wx2,wy2,pp) == true)
	{
		float ldx,ldy;

		gll->winSizeToLayoutSize(wdx,ldx,true);
		gll->winSizeToLayoutSize(wdy,ldy,false);

		Log::dev() << "GridValArea>  dx: " << fabsf(ldx) << " dy: " << fabsf(ldy) << endl;  
		for(int i=0; i <  pp.size() ; i++)
		{
			Log::dev() << " " << pp[i].x() << " " << pp[i].y() << endl;
		}
	
		//If the step node was found	
		currentGlNode_=gll;

		//Clear the contents
		gll->deleteChildren();
		gll->dpl().erase();
		
		glLayoutStack_.push(gll);	

		setCoordRatio(gll);

		//Reset colour
		setNewColour(Colour(-1.,-1.,-1.));

		actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
		actDpl_->compileStart();

		checkError("OpenGLDriver::magnifierLayoutUpdate() 2");

		gll->layout().redisplay(*this,stepObjectToRender_,pp,fabsf(ldx),fabsf(ldy));
		checkError("OpenGLDriver::magnifierLayoutUpdate() 3");
		actDpl_->compileEnd();	
		checkError("OpenGLDriver::magnifierLayoutUpdate() 4");
		glLayoutStack_.pop();

		//Reset colour
		setNewColour(Colour(-1.,-1.,-1.));

		resetCoordRatio();

		//Current node is the root
		//currentGlNode_=glTree_->root();
		currentGlNode_=glTree_;
	}		
}

MAGICS_NO_EXPORT void OpenGLDriver::renderTextSymbols(const TextSymbol& symbol) const
{	
	const int nPos  = symbol.size();
	const int nText = symbol.text().size();

	assert(nText == nPos);

	renderSymbols(symbol);

	float dy=9./(magnifierZoom_*coordRatioY_);

	//Loop for all positions
	for(unsigned int i=0;i< nPos ;i++)
	{
		Text text;			
		text.push_back(PaperPoint(symbol[i].x(),
					  symbol[i].y()+dy));

		text.setJustification(MCENTRE);
		text.setVerticalAlign(MBOTTOM);
		text.setAngle(0.);
		text.setBlanking(true);		
	
		text.addText(symbol.text().at(i),
			     symbol.font()); //75 dpi
				
		renderText(text);
	}
}

//Create display list
void OpenGLDriver::generateSymbolDpl(OpenGLSymbolItem *gldSym,svgSymbol sym) const
{
	setCoordRatio(1.,1.);

	//Symbol size in pixels!!!
	float symbolSize=gldSym->size();

	gldSym->compileStart();

	const unsigned int si = sym.elements.size();

	//setNewLineWidth(1.);
	const float pX = 1. / coordRatioX_;
	const float pY = 1. / coordRatioY_;

	const float scaling=symbolSize;
	
	for(unsigned int i=0;i<si;i++)  // for all elements in the symbol description
	{

		if(sym.elements[i].name == "circle")
		{
			const float r  = atof(sym.elements[i].attributes["r"].c_str())  * scaling;
			const float cx = atof(sym.elements[i].attributes["cx"].c_str()) * scaling;
			const float cy = atof(sym.elements[i].attributes["cy"].c_str()) * scaling;
			const int s = atoi(sym.elements[i].attributes["fill"].c_str());
			circle(cx,cy,r,s);
		}
		else if(sym.elements[i].name == "snowflake")
		{
			const float r = atof(sym.elements[i].attributes["r"].c_str()) * scaling;
			const float cx = atof(sym.elements[i].attributes["cx"].c_str()) * scaling;
			const float cy = atof(sym.elements[i].attributes["cy"].c_str()) * scaling;			
			snowflake(cx,cy,r);
		}
		else if(sym.elements[i].name == "drizzle")
		{
			const float r = atof(sym.elements[i].attributes["r"].c_str()) * scaling;
			const float cx = atof(sym.elements[i].attributes["cx"].c_str()) * scaling;
			const float cy = atof(sym.elements[i].attributes["cy"].c_str()) * scaling;
			drizzle(cx,cy,r);
		}
		else if(sym.elements[i].name == "triangle")
		{
			const float r = atof(sym.elements[i].attributes["r"].c_str()) * scaling;
			const int s = atoi(sym.elements[i].attributes["fill"].c_str());
			const float cx = atof(sym.elements[i].attributes["cx"].c_str()) * scaling;
			const float cy = atof(sym.elements[i].attributes["cy"].c_str()) * scaling;
			const int li = atoi(sym.elements[i].attributes["line"].c_str());
			triangle(cx,cy,r,s,li);
		}
		else if(sym.elements[i].name == "lightning")
		{
			const float r = atof(sym.elements[i].attributes["r"].c_str()) * scaling;
			const float cx = atof(sym.elements[i].attributes["cx"].c_str()) * scaling;
			const float cy = atof(sym.elements[i].attributes["cy"].c_str()) * scaling;
			lightning(cx,cy,r);
		}

		else if(sym.elements[i].name == "polyline")
		{			
			vector<PaperPoint> vPP;
			parsePoints(vPP, sym.elements[i].attributes["points"]);
						
			for(int s=0;s<vPP.size();s++)
			{
				vPP[s].x( vPP[s].x() * scaling);
				vPP[s].y( vPP[s].y() * scaling);
			 }	

			if(magCompare("none",sym.elements[i].attributes["fill"]) )
			{
				BaseDriver::renderPolyline(vPP);
			}
			else 
			{
				BaseDriver::renderSimplePolygon(vPP);
			}
			
		}
	}// endfor 

	gldSym->compileEnd();

	resetCoordRatio();
}


MAGICS_NO_EXPORT void OpenGLDriver::renderSymbols(const Symbol& symbol) const
{	
	setLineParameters(M_SOLID, 1.); // reset line

	// if still empty
	loadSymbols();
	
	if(sym_.size() == 0 ) 
	{
		//Log::error() << "BaseDriver::renderSymbols("<<symbol.getSymbol()<<")-> NO symbols available!"<<endl; return
		Log::error() << "BaseDriver::renderSymbols()-> NO symbols available!" << endl;
		return;
	}


	Log::dev() << "Symbols: " << symbol.size() << endl;

	const int noSymbols = sym_.size();
	svgSymbol sym;

	int ii=0;
	for(;ii<noSymbols;ii++)
	{
		sym = sym_[ii];
		if(sym.id==symbol.getSymbol()) break;
	}

	if(ii==noSymbols) sym = sym_[0];

	//Find out symbol size in pixels!!!!!!!
	float symbolSize = convertCM(symbol.getHeight());

	//Magnifier data mode
	if(glLayoutStack_.top()->classType() == "OpenGLMagnifierLayoutNode")
	{
		symbolSize/=magnifierZoom_;
	}

	OpenGLSymbolItem *actSymbol;

	//If needed compile a display list for the symbol
	//The size of this symbol is "1 cm"
	if((actSymbol=symbols_->getSymbol(sym.id,symbolSize)) == 0 )
	{
		actSymbol=symbols_->addSymbol(sym.id,symbolSize);

		//Compile the new symbol
		actDpl_->compileEnd();
		
		generateSymbolDpl(actSymbol,sym);

		actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
		actDpl_->compileStart();
	}

	//Set colour
	setNewColour(symbol.getColour());	
	
	//Find out symbol size in pixels

	//split into areas

	//Get the current layout node
	assert(glLayoutStack_.empty() == false);
	OpenGLLayoutNode *gll=glLayoutStack_.top();
	
	//Create a new area group object
	OpenGLAreaGroup *grp=gll->areaGroup();
	if(!grp)
	{
		grp=gll->createAreaGroup();
	}

	//For each symbol point assigns an area
	float x,y,wx,wy;
	for(unsigned int i=0;i<symbol.size();i++)
	{ 
		//Get win coordinates!!
		gll->layoutCoordToWinCoord(symbol[i].x(),symbol[i].y(),wx,wy);

		//Add the index to the proper group
		grp->addItemIndex(i,wx,wy);		
	}

	actDpl_->compileEnd();

	for(unsigned int i=0; i < grp->size(); i++)
	{
		if(grp->at(i)->size() >0)
		{
			//Compile the new symbol
			actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
			actDpl_->setAreaIndex(i);
			actDpl_->compileStart();

			for(list<unsigned int>::iterator it=grp->at(i)->begin(); it != grp->at(i)->end(); it++) 
			{
				x=symbol[*it].x();
				y=symbol[*it].y();

				glPushMatrix();	
				glTranslatef(projectX(x),projectY(y),0.);
				actSymbol->call();

				glPopMatrix();
			}

			actDpl_->compileEnd();
		}
	}

	//We do not need the indeces any more (only the groups)
	grp->clearItemIndex();
 	
	
	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();
}

void OpenGLDriver::redisplay(const StepRenderer& step) const
{
	actDpl_->compileEnd();

	//Current layout
	OpenGLLayoutNode* gll=glLayoutStack_.top();
	
	//Animation control
	//Here we have to find the closest animation control in the tree
	//Until we have only one the solution below is working!!!
	//!!!!!!!!!!!!!!!! 
	OpenGLAnimationControl *ac=animationStepMap_.begin()->second;

	int i=0;
	for(OpenGLAnimationControl::StepIterator it=ac->firstStepDef(); it != ac->lastStepDef(); it++)
	{
		AnimationStep* step=(*it)->step();
		
		std::ostringstream out;
   		out  << i;
		//string name=currentGlNode_->name() + "." + "step_" + out.str();					
		string name = "step_" + out.str();	

		OpenGLStepNode *gls=new OpenGLStepNode(name,step);		
		gls->id(i);
		gls->setParentLayout(gll);

		//glTree_->addNode(currentGlNode_->parent()->name(),gls->name(),gls);
		currentGlNode_->parent()->addNode(gls);

		ac->addStep(gls);

		i++;
	}
	
	addGlNode(currentGlNode_->parent());

	actDpl_=currentGlNode_->addDpl(M_DPL_NORMAL);
	actDpl_->compileStart();	
}


void OpenGLDriver::getLayerTree()
{
	while(!glLayerStack_.empty())
	{
		glLayerStack_.pop();
	}

	if(layerTree_)
		delete layerTree_;

	layerTree_=new OpenGLNode;

	visitFunc_=&OpenGLDriver::visit_getLayerTree;
	visitEnterFunc_=&OpenGLDriver::visitEnter_getLayerTree;
	visitLeaveFunc_=&OpenGLDriver::visitLeave_getLayerTree;

	glTree_->accept(*this);
}

bool OpenGLDriver::visitEnter_getLayerTree(OpenGLNode& node)
{		
	if(node.classType() == "OpenGLLayerNode")
	{
		OpenGLNode *parent;
		if(glLayerStack_.empty())
		{
			parent=layerTree_;
		}
		else
		{	
			parent = glLayerStack_.top();
		}

		OpenGLLayerNode *gll_ori = static_cast<OpenGLLayerNode*>(&node);
		OpenGLLayerNode *gll=new OpenGLLayerNode(gll_ori->name(),gll_ori->layer());

		//layerTree_->addNode(parent->name(),gll->name(),gll);		
		parent->addNode(gll);
		glLayerStack_.push(gll);	
	}

	/*if(node.classType() == "OpenGLStepNode")
	{
		OpenGLStepNode *step = (OpenGLStepNode*)&node;
		if(stepToRender_ != step->id())
		{
			checkError("Enter> ");
			return false;					
		}
		
		return true;

	}*/	
	
	if(node.classType() == "OpenGLPreviewLayoutNode"  || 
		node.classType() == "OpenGLMagnifierLayoutNode")
	{
		checkError("Enter> ");
		return false;
	}

	//Log::dev()  <<  "Executing: " << 	`	glb->name() << "[preTransform] " <<   endl;		
	
	return true;	
}



bool OpenGLDriver::visitLeave_getLayerTree(OpenGLNode& node)
{		
	if(node.classType() == "OpenGLLayerNode")
	{		
		glLayerStack_.pop();	
	}
	
	//Log::dev()  <<  "Executing: " << glb->name() << "[postTransform]" << endl;
	//Log::dev()<<  "Executing: " << glb->name() << "[postTransform]" << endl;	

	return true;		
}


void OpenGLDriver::visit_getLayerTree(OpenGLNode& node)
{		
}	

void OpenGLDriver::layerUpdate()
{
	list<OpenGLNode*> gll;
	glTree_->findLayer(gll);

	OpenGLLayerNode *glLayer, *layer;
        OpenGLNode*  layerBase;
	for(list<OpenGLNode*>::iterator it=gll.begin(); it != gll.end(); it++)
	{
		glLayer=static_cast<OpenGLLayerNode*>(*it);

		layerBase=0;
		layerTree_->findLayer(&layerBase,glLayer->name());
				
		if(layerBase)
		{			
			layer=static_cast<OpenGLLayerNode*>(layerBase);
			glLayer->setVisible(layer->visible());
			glLayer->setAlpha(layer->alpha());
		}
		else
		{
			Log::warning() << "OpenGLDriver::layerChanged() ---> Inconsisteny in layer tree!" << endl;
		}
	}

	renderTree(stepToRender_);
}	
