"""
This is the "gnome" module.

The gnome module provides wrappers for LDTP to make the write of Gnome tests easier. 
"""
import ooldtp
import ldtp 
from .main import Application

class Seahorse(Application):
    """
    Seahorse manages the Seahorse application.
    """
    WINDOW       = "frmPasswordsandEncryptionKeys"
    LAUNCHER     = "seahorse"
    MNU_NEWKEY   = "mnuNew"
    NEWKEY_DLG   = "Create New ..."
    BTN_CONTINUE = "btnContinue"
    TYPE_PGP            = "PGP Key"
    NEWPGP_DLG          = "dlgCreateaPGPKey"
    DLG_NEWPGP_FULLNAME = "txtFullName"
    DLG_NEWPGP_EMAIL    = "txtEmailAddress"
    DLG_NEWPGP_COMMENT  = "txtComment"
    BTN_NEWPGP_CREATE   = "btnCreate"
    DLG_NEWKEY_PASS     = "dlgPassphrasefor*"
    BTN_PASS_OK         = "btnOK"
    DLG_GENERATING_KEY  = "dlgGeneratingkey"
    DLG_CREATING_SSH    = "dlgCreatingSecureShellKey"
    TYPE_SSH            = "Secure Shell Key"
    NEWSSH_DLG          = "New Secure Shell Key" 
    DLG_NEWSSH_DESC     = "txtKeyDescription"
    BTN_NEWSSH_CREATE_AND_SETUP = "Create and Set Up"
    DLG_SET_UP          = "Set Up Computer for SSH Connection"
    TXT_SET_UP_COMPUTER = "txtThehostnameoraddressoftheserver."
    TXT_SET_UP_LOGIN    = "txtLoginName"
    BTN_SET_UP          = "btnSetUp"
    BTN_NEWSSH_CREATE   = "Just Create Key"
    TAB_PERSONAL_KEYS   = "My Personal Keys"
    TAB_LIST            = "ptl0"
    TBL_PERSONAL_KEYS   = "tbl0"
    LBL_DELETE_KEY_Q   = "lblOneormoreofthedeletedkeysareprivatekeys.Areyousureyouwanttoproceed?"

    def __init__(self):
        Application.__init__(self)
        self.generated_keys = []

    def new_key(self, key_type):
        """
        It opens up the list of available new keys, and select the one to create.
        
        @type key_type: string
        @param key_type: The type of key to create. 
        """
        
        seahorse = ooldtp.context(self.name)
        
        try:
            mnu_new_key = seahorse.getchild(self.MNU_NEWKEY)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new key menu was not found."

        try:
            mnu_new_key.selectmenuitem() 
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when selecting new key menu item."

        try:
            ldtp.waittillguiexist(self.NEWKEY_DLG)
            dlg_new_key = ooldtp.context(self.NEWKEY_DLG)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new key dialog was not found."

        try:
            table  = dlg_new_key.getchild(role = 'table')
            types_table = table[0]

            for i in range(0, types_table.getrowcount(), 1):
                text = types_table.getcellvalue(i, 1)
                candidate = text.split('\n')[0]
                if candidate == key_type:
                    types_table.selectrowindex(i)
                    break
                ldtp.wait(1)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "Error getting the key types table."

        try:
            btn_continue = dlg_new_key.getchild(self.BTN_CONTINUE)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The continue button at the new key dialog was not found."

        try:
            btn_continue.click() 
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when clicking the continue button."
        
    def new_pgp_key(self, full_name, email, comment, passphrase):
        """
        It creates a new PGP key with the default settings.

        TODO: Allow advanced options
        TODO: Check the list afterwards for the newly created key

        @type full_name: string 
        @param full_name: Full name to type for the PGP key

        @type email: string 
        @param email: Email to type for the PGP key

        @type comment: string 
        @param comment: Comment to type for the PGP key

        @type passphrase: string 
        @param passphrase: Passphrase to type for the PGP key
        """
        
        self.new_key(self.TYPE_PGP)

        try:
            ldtp.waittillguiexist(self.NEWPGP_DLG)
            dlg_new_pgp = ooldtp.context(self.NEWPGP_DLG)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new key dialog was not found."

        try:
            txt_field = dlg_new_pgp.getchild(self.DLG_NEWPGP_FULLNAME)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The " + txt_field + " text field was not found."
        try:
            txt_field.settextvalue(full_name)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was an error when writing the text."

        try:
            txt_field = dlg_new_pgp.getchild(self.DLG_NEWPGP_EMAIL)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The " + txt_field + " text field was not found."
        try:
            txt_field.settextvalue(email)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when writing the text."
   
        try:
            txt_field = dlg_new_pgp.getchild(self.DLG_NEWPGP_COMMENT)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The " + txt_field + " text field was not found."
        try:
            txt_field.settextvalue(comment)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when writing the text."

        try:
            btn_create = dlg_new_pgp.getchild(self.BTN_NEWPGP_CREATE)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The create button at the new PGP key dialog was not found."

        try:
            btn_create.click() 
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when clicking the create button."
       
        try:
            ldtp.waittillguiexist(self.DLG_NEWKEY_PASS)
            dlg_new_pgp_pass = ooldtp.context(self.DLG_NEWKEY_PASS)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new pgp key passphrase dialog was not found."

        try:
            ldtp.enterstring(passphrase)
            ldtp.enterstring("<tab>")
            ldtp.enterstring(passphrase)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "Error entering passphrase."
 
        try:
            btn_pass_ok = dlg_new_pgp_pass.getchild(self.BTN_PASS_OK)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The OK button at the new PGP key passphrase dialog was not found."

        try:
            btn_pass_ok.click() 
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when clicking the OK button."
 
        try:
            ldtp.waittillguiexist(self.DLG_GENERATING_KEY)
            while ldtp.guiexist(self.DLG_GENERATING_KEY) == 1:
                ldtp.wait(1)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new pgp generating key dialog was not found."

        # Add key name to generated key list, so we know to delete it later.
        self.generated_keys.append('  '.join([full_name, email, "'"+comment+"'"]))

    def new_ssh_key(self, description, passphrase, set_up = False, computer = '', login = ''):
        """
        It creates a new SSH key with the default settings.

        TODO: Setting up the key is not working yet

        @type description: string 
        @param description: Description to type in the SSH key

        @type passphrase: string 
        @param passphrase: Passphrase to type for the SSH key

        @type set_up: boolean 
        @param passphrase: True, to set up the SSH key

        @type computer: string 
        @param computer: URL or IP of the computer to set up the key
        
        @type login: string
        @param login: Login to use in the remote computer
        """
        
        self.new_key(self.TYPE_SSH)

        try:
            ldtp.waittillguiexist(self.NEWSSH_DLG)
            dlg_new_ssh = ooldtp.context(self.NEWSSH_DLG)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new key dialog was not found."

        try:
            txt_field = dlg_new_ssh.getchild(self.DLG_NEWSSH_DESC)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The " + self.DLG_NEWSSH_DESC + " text field was not found."
        try:
            txt_field.settextvalue(description)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was an error when writing the text."

        if set_up == True:
            try:
                btn_create = dlg_new_ssh.getchild(self.BTN_NEWSSH_CREATE_AND_SETUP)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The create button at the new PGP key dialog was not found."

        else:
            try:
                btn_create = dlg_new_ssh.getchild(self.BTN_NEWSSH_CREATE)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The create button at the new PGP key dialog was not found."

        try:
            btn_create.click() 
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when clicking the create button."
      
 
        try:
            ldtp.waittillguiexist(self.DLG_NEWKEY_PASS)
            dlg_new_key_pass = ooldtp.context(self.DLG_NEWKEY_PASS)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The new key passphrase dialog was not found."

        try:
            ldtp.enterstring(passphrase)
            ldtp.enterstring("<tab>")
            ldtp.enterstring(passphrase)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "Error entering passphrase."
 
        try:
            btn_pass_ok = dlg_new_key_pass.getchild(self.BTN_PASS_OK)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The OK button at the new key passphrase dialog was not found."

        try:
            btn_pass_ok.click() 
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was a problem when clicking the OK button."
 
        if set_up == True and login is not None:

            try:
                ldtp.waittillguiexist(self.DLG_SET_UP)
                dlg_set_up_computer = ooldtp.context(self.DLG_SET_UP)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The set up computer dialog was not found."

            try:
                txt_field = dlg_set_up_computer.getchild(self.TXT_SET_UP_LOGIN)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The " + self.TXT_SET_UP_LOGIN + " text field was not found."
            try:
                txt_field.settextvalue(login)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "There was an error when writing the text."

        if set_up == True:
            try:
                txt_field = dlg_set_up_computer.getchild(self.TXT_SET_UP_COMPUTER)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The " + self.TXT_SET_UP_COMPUTER + " text field was not found."
            try:
                txt_field.settextvalue(computer)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "There was an error when writing the text."

            try:
                btn_set_up = dlg_set_up_computer.getchild(self.BTN_SET_UP)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The set up button was not found."

            try:
                btn_set_up.click() 
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "There was a problem when clicking the set up button."
            
        try:
            while ldtp.guiexist(self.DLG_CREATING_SSH) == 1:
                ldtp.wait(1)
            
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The creating key dialog was not found."
        # Add key name to generated key list, so we know to delete it later.
        self.generated_keys.append(description)

        # It is too fast to grab the main window afterwards
        ldtp.wait(3)

    def go_to_tab(self, tab_name):
        """
        Go to the specified tab.
        """
        seahorse = ooldtp.context(self.name)
        page_list = seahorse.getchild(self.TAB_LIST)
        page_list.selecttab(self.TAB_PERSONAL_KEYS)
        

    def assert_exists_key(self, name, tab_name = None):
        """
        It checks that the KEY with description 'description' is
        part of the keys of the current user

        @type name: string
        @param name: The name of the key to search
        
        @type tab_name: string
        @param tab_name: The tab name to search for the key.
        """
        if not tab_name:
            tab_name = self.TAB_PERSONAL_KEYS

        self.go_to_tab(tab_name)
        seahorse = ooldtp.context(self.name)
        return bool(seahorse.doesrowexist(self.TBL_PERSONAL_KEYS, name))

    def remove_key(self, key):
        seahorse = ooldtp.context(self.name)

        tbl_personal_keys = seahorse.getchild(self.TBL_PERSONAL_KEYS)
        mnu_delete = seahorse.getchild('mnuDelete')
        dlg_question = ooldtp.context('dlgQuestion')
        
        try:
            tbl_personal_keys.selectrow(key)
        except ldtp.LdtpExecutionError:
            # Doesn't exist, return False.
            return False

        mnu_delete.selectmenuitem()
        # General 'private key' verification.
        dlg_question.waittillguiexist(self.LBL_DELETE_KEY_Q)
        dlg_question.click('btnDelete')
        dlg_question.waittillguinotexist(self.LBL_DELETE_KEY_Q)

        # Specific key type verification.
        dlg_question.waittillguiexist()
        dlg_question.click('btnDelete')
        dlg_question.waittillguinotexist()
        
        return True

    def remove_keys(self, keys=None, tab_name=None):
        self.go_to_tab(tab_name or self.TAB_PERSONAL_KEYS)

        keys = keys or self.generated_keys

        for key in keys:
            self.remove_key(key)
        
class GEdit(Application):
    """
    GEdit manages the Gedit application.
    """
    WINDOW     = "frm*gedit"
    TXT_FIELD  = "txt1"
    LAUNCHER   = "gedit"
    SAVE_DLG   = "dlgSave*"
    SAVE_DLG_TXT_NAME = "txtName"
    SAVE_DLG_BTN_SAVE = "btnSave"
    QUESTION_DLG = "dlgQuestion"
    QUESTION_DLG_BTN_SAVE = "btnSave"
    QUESTION_DLG_BTN_SAVE_AS = "btnSaveAs"
    QUESTION_DLG_BTN_CLOSE = "btnClosewithoutSaving"
    MNU_QUIT = "mnuQuit"
    MNU_CLOSE = "mnuClose"
    MNU_NEW = "mnuNew"

    def __init__(self):
        Application.__init__(self)


    def write_text(self, text):
        """
        It writes text to the current buffer of the Gedit window.

        @type text: string
        @param text: The text string to be written to the current buffer.
        """
        Application.write_text(self, text, self.TXT_FIELD)

    def save(self, filename):
        """
        It tries to save the current opened buffer to the filename passed as parameter.

        TODO: It does not manage the overwrite dialog yet.

        @type filename: string
        @param filename: The name of the file to save the buffer to.
        """
        Application.save(self)
        ooldtp.context(self.name)

        try:
            ldtp.waittillguiexist(self.SAVE_DLG)
            save_dialog = ooldtp.context(self.SAVE_DLG)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The Gedit save dialog was not found."
        try:
            save_dlg_txt_filename = save_dialog.getchild(self.SAVE_DLG_TXT_NAME)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The filename txt field in Gedit save dialog was not found."
        try:
            ldtp.wait(2)
            save_dlg_txt_filename.settextvalue(filename)
        except ldtp.LdtpExecutionError:
           raise ldtp.LdtpExecutionError, "We couldn't write text."

        try:
            save_dlg_btn_save = save_dialog.getchild(self.SAVE_DLG_BTN_SAVE)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The button Save in Gedit save dialog was not found."
        
        try:
            save_dlg_btn_save.click()
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "There was an error when pushing the Save button."

        ldtp.waittillguinotexist(self.SAVE_DLG)
        ldtp.wait(1)

    def close(self, save=False, filename=''):
        """
        Given a gedit window, it tries to close the application.
        By default, it closes without saving. This behaviour can be changed to save (or save as) on close.
         
        @type save: boolean
        @param save: If True, the edited file will be saved on close.

        @type filename: string
        @param filename: The file name to save the buffer to 
        """

        # Exit using the Quit menu 
        try:
            gedit = ooldtp.context(self.name)
            try:
                quit_menu = gedit.getchild(self.MNU_QUIT)
            except ldtp.LdtpExecutionError:
                raise ldtp.LdtpExecutionError, "The quit menu was not found."
            quit_menu.selectmenuitem()
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "Mmm, something went wrong when closing the application."

        question_dialog = None
        count = 0
        while not gedit.waittillguinotexist(guiTimeOut=1) and \
                count < 10:
            try:
                question_dialog = ooldtp.context(self.QUESTION_DLG)
            except:
                count += 1
            else:
                break

        # If the text has changed, the save dialog will appear
        if question_dialog:
            # Test if the file needs to be saved
            if save:
                try:
                    question_dlg_btn_save = question_dialog.getchild(self.QUESTION_DLG_BTN_SAVE)
                    question_dlg_btn_save.click()
                except ldtp.LdtpExecutionError:
                    # If the Save button was not found, we will try to find the Save As
                    try:
                        question_dlg_btn_save = question_dialog.getchild(self.QUESTION_DLG_BTN_SAVE_AS)
                        question_dlg_btn_save.click()
                    except ldtp.LdtpExecutionError:
                        raise ldtp.LdtpExecutionError, "The save or save as buttons in Gedit question dialog were not found."

                    try:
                        ldtp.waittillguiexist(self.SAVE_DLG)
                        save_dialog = ooldtp.context(self.SAVE_DLG)
                    except ldtp.LdtpExecutionError:
                        raise ldtp.LdtpExecutionError, "The Gedit save dialog was not found."
                    try:
                        save_dlg_txt_filename = save_dialog.getchild(self.SAVE_DLG_TXT_NAME)
                    except ldtp.LdtpExecutionError:
                        raise ldtp.LdtpExecutionError, "The filename txt field in Gedit save dialog was not found."
                    try:
                        ldtp.wait(2)
                        save_dlg_txt_filename.settextvalue(filename)
                    except ldtp.LdtpExecutionError:
                        raise ldtp.LdtpExecutionError, "There was an error when writing the text."

                    try:
                        save_dlg_btn_save = save_dialog.getchild(self.SAVE_DLG_BTN_SAVE)
                    except ldtp.LdtpExecutionError:
                        raise ldtp.LdtpExecutionError, "The save button in Gedit save dialog was not found."
        
                    try:
                        save_dlg_btn_save.click()
                    except ldtp.LdtpExecutionError:
                        raise ldtp.LdtpExecutionError, "There was an error when pushing the Save button."

                    ldtp.waittillguinotexist(self.SAVE_DLG)
            
            else:
                try:
                    question_dlg_btn_close = question_dialog.getchild(self.QUESTION_DLG_BTN_CLOSE)
                    question_dlg_btn_close.click()
                except ldtp.LdtpExecutionError:
                    raise ldtp.LdtpExecutionError, "It was not possible to click the close button."

            gedit.waittillguinotexist(guiTimeOut=20)
 
class PolicyKit(Application):
    """
    PolicyKit class manages the GNOME pop up that ask for password for admin activities.
    """
    WINDOW     = "dlg0"
    TXT_PASS   = "txtPassword"
    BTN_OK     = "btnOK"
    BTN_CANCEL = "btnCancel"


    def __init__(self, password):
        """
        UpdateManager class main constructor
        
        @type password: string
        @param password: User's password for administrative tasks.

        """
        Application.__init__(self)
        self.password = password
    
    def wait(self):
        """
        Wait for the pop up window asking for the password to appear.

        @return 1, if the gksu window exists, 0 otherwise.
        """
        return ldtp.waittillguiexist(self.name)
        
    def set_password(self):
        """
        It enters the password in the text field and clicks enter. 
        """
        
        ooldtp.context(self.name)
        
        try:
            ldtp.enterstring (self.password)
            ldtp.enterstring ("<enter>")
            
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The PolicyKit txt field for the password was not found."
   
# TODO: Change this to use ooldtp
#        try:
#            btnOK = polKit.getchild(self.BTN_OK)
#        except ldtp.LdtpExecutionError, msg:
#            raise ldtp.LdtpExecutionError, "The GtkSudo OK button was not found."
#          
#        btnOK.click()
    
        #This also have problems because of the lack of accesibiliy information
        #ldtp.waittillguinotexist (self.name)
        
    def cancel(self):
        polKit = ooldtp.context(self.name)

        try:
            cancelButton = polKit.getchild(self.BTN_CANCEL)
        except ldtp.LdtpExecutionError:
            raise ldtp.LdtpExecutionError, "The PolicyKit cancel button was not found."
          
        cancelButton.click()
        ldtp.waittillguinotexist (self.name)
